﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>

#include "testAccount_Util.h"

#include <nn/util/util_ScopeExit.h>

namespace
{

typedef bool (*Comparator)(
    const nn::account::Uid& U,
    const nn::account::Nickname& Name,
    const char* UserData, const size_t UserDataSize,
    const char* Image, const size_t ImageSize);

bool CompareProfileByEditor(
    const nn::account::Uid& U,
    const nn::account::Nickname& Name,
    const char* UserData, const size_t UserDataSize,
    const char* Image, const size_t ImageSize) NN_NOEXCEPT
{
    nn::account::ProfileEditor pedit;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        nn::account::GetProfileEditor(&pedit, U));

    NN_SDK_ASSERT(UserDataSize == nn::account::UserDataBytesMax);

    nn::account::Nickname name;
    char data[nn::account::UserDataBytesMax];

    pedit.GetNickname(&name);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::strncmp(Name.name, name.name, sizeof(name.name)));
    pedit.GetUserData(data, sizeof(data));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::memcmp(UserData, data, UserDataSize));

    if (Image)
    {
        char imageOut[256];
        size_t outSize;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            pedit.LoadProfileImage(&outSize, imageOut, sizeof(imageOut)));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(ImageSize, outSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::memcmp(Image, imageOut, outSize));
    }
    return true;
}

bool CompareProfileByReader(
    const nn::account::Uid& U,
    const nn::account::Nickname& Name,
    const char* UserData, const size_t UserDataSize,
    const char* Image, const size_t ImageSize) NN_NOEXCEPT
{
    NN_UNUSED(UserData);
    NN_SDK_ASSERT(UserDataSize == nn::account::UserDataBytesMax);

    nn::account::Nickname name;

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&name, U));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::strncmp(Name.name, name.name, sizeof(name.name)));

    if (Image)
    {
        char imageOut[256];
        size_t outSize;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::LoadProfileImage(&outSize, imageOut, sizeof(imageOut), U));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(ImageSize, outSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::memcmp(Image, imageOut, outSize));
    }
    return true;
}

bool TestProfileForUser(const nn::account::Uid& u, Comparator comp) NN_NOEXCEPT
{
    const nn::account::Nickname EmptyName = {""};
    static const char EmptyData[nn::account::UserDataBytesMax] = {0};
    const nn::account::Nickname n0 = {"n0"};
    char d0[nn::account::UserDataBytesMax];
    std::memset(d0, 0x3F, sizeof(d0));

    nn::account::ProfileEditor pedit;

    // 初期状態 → 空
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        nn::account::GetProfileEditor(&pedit, u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(comp(u, EmptyName, EmptyData, sizeof(EmptyData), nullptr, 0));

    // Flush せずに再読み込み → 変更反映されない。
    pedit.SetNickname(n0);
    pedit.SetUserData(d0, sizeof(d0));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        nn::account::GetProfileEditor(&pedit, u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(comp(u, EmptyName, EmptyData, sizeof(EmptyData), nullptr, 0));

    // Flush せずに Refresh → 変更反映されない。
    pedit.SetNickname(n0);
    pedit.SetUserData(d0, sizeof(d0));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pedit.Refresh());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(comp(u, EmptyName, EmptyData, sizeof(EmptyData), nullptr, 0));

    // Flush して再読み込み → 変更反映される。
    pedit.SetNickname(n0);
    pedit.SetUserData(d0, sizeof(d0));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pedit.Flush());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(comp(u, n0, d0, sizeof(d0), nullptr, 0));

    // FlushWithImage して再読み込み → 変更反映される。
    char image[128];
    std::memset(image, 0xA1, sizeof(image));
    pedit.SetNickname(EmptyName);
    pedit.SetUserData(EmptyData, sizeof(EmptyData));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pedit.FlushWithImage(image, sizeof(image)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(comp(u, EmptyName, EmptyData, sizeof(EmptyData), image, sizeof(image)));

    return true;
} // NOLINT(readability/fn_size)

} // ~namespace <anonymous>

bool TestProfile() NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        // テスト用の明示的な終了
        nn::account::Finalize();
    };

    int count;
    nn::account::Uid u;

    // 作成前
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, count);
    {
        nn::account::ProfileEditor pedit;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::account::ResultUserNotExist,
            nn::account::GetProfileEditor(&pedit, u));
    }

    // 作成開始
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestProfileForUser(u, CompareProfileByEditor));

    // 作成中断
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CancelUserRegistration(u));
    {
        nn::account::ProfileEditor pedit;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::account::ResultUserNotExist,
            nn::account::GetProfileEditor(&pedit, u));
    }

    // 作成後
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(1, count);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestProfileForUser(u, CompareProfileByEditor));

    // 削除
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, count);

    // 作成後(2)
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(1, count);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestProfileForUser(u, CompareProfileByReader));

    // 削除
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, count);
    return true;
} // NOLINT(readability/fn_size)
