﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiBaasAccessToken.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiPrivate.h>
#include "detail/account_ShimLibraryUtility.h"

#include "testAccount_Initializer.h"
#include "testAccount_ServiceUtil.h"
#include "testAccount_Tests.h"
#include "testAccount_Util.h"

#include <cstdlib>
#include <new>

#include <nn/friends/friends_Api.h>
#include <nn/friends/friends_ApiAdmin.h>
#include <nn/os.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/oe.h>
#endif

#include <nnt.h>

namespace
{
// Initialize With 用
typedef std::aligned_storage<sizeof(nnt::account::DefaultTestResource), NN_ALIGNOF(nnt::account::DefaultTestResource)>::type Storage;
Storage g_AccountResourceStorage;
} // ~namespace <anonymous>

extern "C" void nnMain()
{
    nn::friends::Initialize();
    nn::friends::DaemonSuspension friendsSuspension;
    nn::friends::SuspendDaemon(&friendsSuspension);

    int result = 1;
#if defined(NNT_ACCOUNT_NETWORK_TEST_ENABLE)
    nnt::account::InitializeNetworkTest();
    NN_UTIL_SCOPE_EXIT {
        // アカウントを削除
        nnt::account::Cleanup();
        nnt::account::FinalizeNetworkTest(result);
    };
#else
    nnt::account::Initialize();
    NN_UTIL_SCOPE_EXIT {
        // アカウントを削除
        nnt::account::Cleanup();
        nnt::account::Finalize(result);
    };
#endif
    // アカウントを削除
    nnt::account::Cleanup();

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    nn::oe::Initialize();
#endif

    result = RUN_ALL_TESTS();
}

TEST(AccountService, RegistrationPermission)
{
    // Initialize ---------------------------------------------
    {
        nn::account::Initialize();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };
        bool regOk;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::IsUserRegistrationRequestPermitted(&regOk));
        EXPECT_TRUE(regOk);
        nn::account::Finalize();

        nn::account::InitializeForSystemService();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::IsUserRegistrationRequestPermitted(&regOk));
        EXPECT_TRUE(regOk);
        nn::account::Finalize();

        nn::account::InitializeForAdministrator();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::IsUserRegistrationRequestPermitted(&regOk));
        EXPECT_TRUE(regOk);
    }

    // InitializeWith -----------------------------------------

    // サービスの作成
    auto pSrc = new(&g_AccountResourceStorage) nnt::account::DefaultTestResource;
    NN_UTIL_SCOPE_EXIT
    {
        pSrc->~ServiceResource();
    };

    {
        // クライアントライブラリの初期化
        nn::account::InitializeWith(
            pSrc->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        // 権限チェック -> OK
        nn::account::Uid uid;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::BeginUserRegistration(&uid));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CompleteUserRegistration(uid));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::DeleteUser(uid));

        // アプリの参照を増やす
        auto p = pSrc->CreateServiceObjectForApplication();

        // 権限チェック -> NG
        bool regOk;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::IsUserRegistrationRequestPermitted(&regOk));
        EXPECT_FALSE(regOk);

        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::BeginUserRegistration(&uid));
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::account::ResultRegistrationNotPermitted, nn::account::CompleteUserRegistration(uid));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CompleteUserRegistrationForcibly(uid));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::DeleteUser(uid));

        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(p->IsUserRegistrationRequestPermitted(&regOk, 0));
        EXPECT_TRUE(regOk);
    }
}

TEST(AccountService, Basic)
{
    ASSERT_TRUE(TestOpenClose());
    ASSERT_TRUE(TestRegistrationAsCommon());
    ASSERT_TRUE(TestRegistrationAsWatcher());
    ASSERT_TRUE(TestRegistrationAsManager());
    ASSERT_TRUE(TestProfile());

    nn::account::InitializeBaasAccessTokenAccessor();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::FinalizeBaasAccessTokenAccessor();
    };
};

TEST(AccountService, InitializeWith)
{
    // サービスの初期化
    auto pSrc = new(&g_AccountResourceStorage) nnt::account::DefaultTestResource;
    NN_UTIL_SCOPE_EXIT
    {
        pSrc->~ServiceResource();
    };

    // クライアントライブラリの初期化
    nn::account::InitializeWith(
        pSrc->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    // ASSERT_TRUE(TestOpenClose());
    ASSERT_TRUE(TestRegistrationAsCommon());
    ASSERT_TRUE(TestRegistrationAsWatcher());
    ASSERT_TRUE(TestRegistrationAsManager());
    ASSERT_TRUE(TestProfile());

    nn::account::InitializeBaasAccessTokenAccessorWith(
        pSrc->GetBaasAccessTokenAccessorPointer());
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::FinalizeBaasAccessTokenAccessor();
    };
};

#if defined(NNT_ACCOUNT_NETWORK_TEST_ENABLE)

TEST(AccountService, TrySelectUserWithoutInteraction)
{
    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    nn::account::Uid uid;
    nn::account::Uid users[2];
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::settings::system::AccountSettings conf;
    conf.userSelectorSettings.flags.Set<nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>(true);
    nn::settings::system::SetAccountSettings(conf);

    // 初期状態
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, false));
    EXPECT_FALSE(uid);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, true));
    EXPECT_FALSE(uid);

    // ユーザー 1
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::BeginUserRegistration(&users[0]));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CompleteUserRegistration(users[0]));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, false));
    EXPECT_TRUE(uid);
    EXPECT_EQ(users[0], uid);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, true));
    EXPECT_FALSE(uid);

    nn::account::NetworkServiceAccountAdministrator admin;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[0]));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, false));
    EXPECT_TRUE(uid);
    EXPECT_EQ(users[0], uid);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, true));
    EXPECT_TRUE(uid);
    EXPECT_EQ(users[0], uid);

    // 設定オフ
    conf.userSelectorSettings.flags.Set<nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>(false);
    nn::settings::system::SetAccountSettings(conf);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, false));
    EXPECT_FALSE(uid);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, true));
    EXPECT_FALSE(uid);

    conf.userSelectorSettings.flags.Set<nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>(true);
    nn::settings::system::SetAccountSettings(conf);

    // ユーザー 2
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::BeginUserRegistration(&users[1]));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CompleteUserRegistration(users[1]));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, false));
    EXPECT_FALSE(uid);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, true));
    EXPECT_FALSE(uid);

    // 設定オフ
    conf.userSelectorSettings.flags.Set<nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>(false);
    nn::settings::system::SetAccountSettings(conf);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, false));
    EXPECT_FALSE(uid);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::detail::TrySelectUserWithoutInteraction(&uid, true));
    EXPECT_FALSE(uid);
}

TEST(AccountService, ReAuthenticateService)
{
    const int TestCount = 10;

    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    NN_LOG("[nnt::account] Service ReAuthentication\n");
    for (int i = 0; i < TestCount; ++ i)
    {
        auto begin = nn::os::GetSystemTick();

        nn::account::AsyncContext ctx;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::AuthenticateServiceAsync(&ctx));
        nn::os::SystemEvent e;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        NN_UTIL_SCOPE_EXIT
        {
            ctx.Cancel();
            e.Wait();
        };

        e.Wait();
        NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctx.GetResult());
    }

    nnt::account::Cleanup();
}

#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_TEST)
TEST(AccountService, MultiLogin)
{
    const int NumUsers = nn::account::UserCountMax;

    // 作成
    nn::account::Uid users[NumUsers];
    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };
        for (int i = 0; i < NumUsers; ++ i)
        {
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::BeginUserRegistration(&users[i]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CompleteUserRegistration(users[i]));

            nn::account::NetworkServiceAccountAdministrator admin;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[i]));

            nn::account::AsyncContext ctx;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(admin.RegisterAsync(&ctx));
            nn::os::SystemEvent e;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
            NN_UTIL_SCOPE_EXIT
            {
                ctx.Cancel();
                e.Wait();
            };

            e.Wait();
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctx.GetResult());
        }
    }

    {
        nn::account::Initialize();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        nn::account::UserHandle handles[NumUsers];
        for (int i = 0; i < NumUsers; ++ i)
        {
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&handles[i], users[i]));
        }
        NN_UTIL_SCOPE_EXIT
        {
            for (int i = 0; i < NumUsers; ++ i)
            {
                nn::account::CloseUser(handles[i]);
            }
        };

        nn::account::AsyncContext ctxs[NumUsers];
        nn::os::SystemEvent es[NumUsers];
        for (int i = 0; i < NumUsers; ++ i)
        {
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(&ctxs[i], handles[i]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctxs[i].GetSystemEvent(&es[i]));
        }
        for (int i = 0; i < NumUsers; ++ i)
        {
            es[i].Wait();
        }
        for (int i = 0; i < NumUsers; ++ i)
        {
            bool done;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctxs[i].HasDone(&done));
            ASSERT_TRUE(done);
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctxs[i].GetResult());
        }

        nnt::account::Buffer buffers[NumUsers];
        for (int i = 0; i < NumUsers; ++ i)
        {
            size_t sizeActual;
            buffers[i] = nnt::account::Buffer(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::LoadNetworkServiceAccountIdTokenCache(&sizeActual, buffers[i].Get<char>(), buffers[i].GetSize(), handles[i]));
        }

        for (int i = 0; i < NumUsers; ++ i)
        {
            NN_LOG("User#%d: \"%s\"\n", i, buffers[i].Get<char>());
        }
    }
    nnt::account::Cleanup();
}
#endif // NNT_ACCOUNT_ENABLE_APPLICATION_TEST

#endif // NNT_ACCOUNT_NETWORK_TEST_ENABLE
