﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/user/account_UserRegistry.h>

#include "testAccount_Module.h"

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_USER_REGISTRY_SINGLE

#if defined(NNT_ACCOUNT_ENABLE_USER_REGISTRY_SINGLE)

// 単一ユーザー追加テスト用のユーティリティ
namespace
{

typedef std::shared_ptr<nn::account::user::UserRegistry> ManagerPtrType;

bool CheckNoOne(ManagerPtrType pManager)
{
    a::Uid users[a::UserCountMax];
    size_t count = pManager->GetUserCount();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(count == 0);

    std::memset(users, 0xFF, sizeof(users));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        pManager->ListAllUsers(users, sizeof(users) / sizeof(users[0])));
    for (auto& u: users)
    {
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(!u);
    }

    return true;
}

bool CheckOnlyOne(ManagerPtrType pManager, const a::Uid& uid)
{
    a::Uid users[a::UserCountMax];

    size_t count = pManager->GetUserCount();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(count == 1);

    bool existence;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pManager->GetUserExistence(&existence, uid));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(existence);

    std::memset(users, 0xFF, sizeof(users));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pManager->ListAllUsers(users, sizeof(users) / sizeof(users[0])));

    int validCount = 0;
    for (auto& u: users)
    {
        if (u)
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(u == uid);
            ++ validCount;
        }
    }
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(validCount == 1);

    return true;
}

} // ~namespace <anonymous>

// 登録のキャンセル
TEST(AccountUser, UserRegistry_CancelRegistration)
{
    const int TestCount = 1000;

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto pManager = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->Initialize(s));

    for (int i = 0; i < TestCount; i++)
    {
        // 登録の開始
        a::Uid pending;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->BeginUserRegistration(&pending));
        EXPECT_TRUE(pending);
        EXPECT_TRUE(CheckNoOne(pManager));

        // 登録完了まではユーザーは存在しない
        bool existence;
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(pManager->GetUserExistence(&existence, pending));
        EXPECT_FALSE(existence);

        // 登録のキャンセル
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->CancelUserRegistration(pending));
        EXPECT_TRUE(CheckNoOne(pManager));
    }
}

// 1人目のユーザーの追加
TEST(AccountUser, UserRegistry_RegisterSingle)
{
    const int TestCount = 1000;

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto pManager = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->Initialize(s));

    for (int i = 0; i < TestCount; i++)
    {
        // 登録の開始
        a::Uid pending;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(
            pManager->BeginUserRegistration(&pending));
        EXPECT_TRUE(pending);
        EXPECT_TRUE(CheckNoOne(pManager));

        // 登録完了まではユーザーは存在しない
        bool existence;
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(pManager->GetUserExistence(&existence, pending));
        EXPECT_FALSE(existence);

        // 登録の確定
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->CompleteUserRegistration(pending));
        EXPECT_TRUE(CheckOnlyOne(pManager, pending));

        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(pManager->GetUserExistence(&existence, pending));
        EXPECT_TRUE(existence);

        // 削除
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->DeleteUser(pending));
        EXPECT_TRUE(CheckNoOne(pManager));
    }
}

// 2人目のユーザーの作成中に1人目削除、その後、作成を確定
TEST(AccountUser, UserRegistry_RegisterSingle2)
{
    const int TestCount = 1000;

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto pManager = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->Initialize(s));

    for (int i = 0; i < TestCount; i++)
    {
        // 1 人目
        a::Uid user0;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->BeginUserRegistration(&user0));
        EXPECT_TRUE(user0);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->CompleteUserRegistration(user0));
        EXPECT_TRUE(CheckOnlyOne(pManager, user0));

        // 2 人目 (作成開始)
        a::Uid user1;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->BeginUserRegistration(&user1));
        EXPECT_TRUE(user1);
        EXPECT_TRUE(CheckOnlyOne(pManager, user0)); // 念の為 user0 ひとりであることを確認

        // 1 人目削除
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->DeleteUser(user0));
        EXPECT_TRUE(CheckNoOne(pManager));

        // 2 人目 (作成完了)
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->CompleteUserRegistration(user1));
        EXPECT_TRUE(CheckOnlyOne(pManager, user1));

        a::Uid users[a::UserCountMax];
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(
            pManager->ListAllUsers(users, a::UserCountMax));
        EXPECT_TRUE(users[0]);
        for (auto j = 1; j < a::UserCountMax; ++j)
        {
            EXPECT_FALSE(users[j]);
        }

        // 2 人目削除
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->DeleteUser(user1));
        EXPECT_TRUE(CheckNoOne(pManager));
    }
}

#endif // NNT_ACCOUNT_ENABLE_USER_REGISTRY_SINGLE
