﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief 実装で使用している detail/ByteUtil と detail/UuidGenerator を評価する。
 */

#include "detail/account_UuidUtil.h"
#include <nn/account/detail/account_InternalConfig.h>

#include "testAccount_RamFs.h"

#include <memory>

#include <nn/nn_Log.h>
#include <nnt/nntest.h>

namespace target = nn::account::detail;

#define NNT_ACCOUNT_ENABLE_UUID

#if defined(NNT_ACCOUNT_ENABLE_UUID)

TEST(AccountUser, Uuid_Validity)
{
    // Invalid
    {
        const target::Uuid invalidUuid = {{0}};
        EXPECT_FALSE(invalidUuid);
        {
            auto invalidUid = target::ConvertToUid(invalidUuid);
            EXPECT_FALSE(invalidUid);
            EXPECT_EQ(0, std::memcmp(&invalidUuid, &invalidUid, sizeof(target::Uuid)));

            auto invalidUuid2 = target::ConvertToUuid(invalidUid);
            EXPECT_FALSE(invalidUuid2);
            EXPECT_EQ(0, std::memcmp(&invalidUuid, &invalidUuid2, sizeof(target::Uuid)));
        }
    }

    // Runtime
    nnt::account::RamFs fs;
    {
        auto lock = fs.AcquireWriterLock();
        fs.Delete(target::IdGenerationContextPath);
    }
    const auto UuidVarietyCount = 1000;
    for (auto i = 0; i < UuidVarietyCount; ++ i)
    {
        target::Uuid uuid;
        {
            auto lock = fs.AcquireWriterLock();
            uuid = target::GenerateUuid(fs);
        }
        EXPECT_TRUE(uuid);
        {
            auto uid = target::ConvertToUid(uuid);
            EXPECT_TRUE(uid);
            EXPECT_EQ(0, std::memcmp(&uuid, &uid, sizeof(target::Uuid)));

            auto uuid2 = target::ConvertToUuid(uid);
            EXPECT_TRUE(uuid2);
            EXPECT_EQ(0, std::memcmp(&uuid, &uuid2, sizeof(target::Uuid)));
        }
    }
};

TEST(AccountUser, Uuid_String)
{
    // Invalid
    {
        const target::UuidString Expected = {{"00000000-0000-0000-0000-000000000000"}};

        const target::Uuid InvalidUuid = {{0}};
        ASSERT_FALSE(InvalidUuid);

        auto str = target::UuidToString(InvalidUuid);
        EXPECT_EQ('-', str.GetInnerValue()[8]); // '-'
        EXPECT_EQ('-', str.GetInnerValue()[13]); // '-'
        EXPECT_EQ('-', str.GetInnerValue()[18]); // '-'
        EXPECT_EQ('-', str.GetInnerValue()[23]); // '-'
        EXPECT_EQ(0, std::memcmp(&Expected, &str, sizeof(Expected)));

        auto uuid2 = target::UuidUnstring(str);
        EXPECT_EQ(0, std::memcmp(&InvalidUuid, &uuid2, sizeof(InvalidUuid)));
    }

    // Constant
    {
        target::Uuid Example = {{0x01234567, 0x89ABCDEF, 0xDEADBEEF, 0xBEADCAFE}};
        target::UuidString Expected = {{"01234567-cdef-89ab-efbe-beadcafedead"}};
        auto str = target::UuidToString(Example);
        EXPECT_EQ(0, std::memcmp(&Expected, &str, sizeof(Expected)));
    }
}

TEST(AccountUser, Uuid_Serialize)
{
    // Invalid
    {
        const target::Uuid InvalidUuid = {{0}};

        auto su = target::SerializeUuid(InvalidUuid);
        EXPECT_EQ(0, std::memcmp(&InvalidUuid, &su, sizeof(InvalidUuid)));
    }

    // Constant
    {
        const target::Uuid Example = {{
            0x01234567,
            0x89ABCDEF,
            0xDEADBEEF,
            0xBEADCAFE
        }};
#if NN_BUILD_CONFIG_ENDIAN_SUPPORTS_LITTLE
        const uint8_t ExampleInBytes[16] = {
            0x67, 0x45, 0x23, 0x01,
            0xEF, 0xCD, 0xAB, 0x89,
            0xEF, 0xBE, 0xAD, 0xDE,
            0xFE, 0xCA, 0xAD, 0xBE
        };
        ASSERT_EQ(0, std::memcmp(&Example, &ExampleInBytes, sizeof(Example)));

        const uint8_t Expected[16] = {
            0x01, 0x23, 0x45, 0x67,
            0xCD, 0xEF, 0x89, 0xAB,
            0xEF, 0xBE, 0xBE, 0xAD,
            0xCA, 0xFE, 0xDE, 0xAD
        };
#else
        const uint8_t Expected[16] = {
            0x01, 0x23, 0x45, 0x67,
            0x89, 0xAB, 0xCD, 0xEF,
            0xDE, 0xAD, 0xBE, 0xEF,
            0xBE, 0xAD, 0xCA, 0xFE
        };
        ASSERT_EQ(0, std::memcmp(&Example, &Expected, sizeof(Example)));
#endif

        auto su = target::SerializeUuid(Example);
        EXPECT_EQ(0, std::memcmp(&Expected, &su, sizeof(Expected)));

        auto uuid2 = target::DeserializeUuid(su);
        EXPECT_EQ(0, std::memcmp(&Example, &uuid2, sizeof(Example)));
    }

    // Runtime
    const auto UuidVarietyCount = 1000;
    nnt::account::RamFs fs;
    auto lock = fs.AcquireWriterLock();
    fs.Delete(target::IdGenerationContextPath);
    for (auto i = 0; i < UuidVarietyCount; ++ i)
    {
        auto uuid = target::GenerateUuid(fs);
        ASSERT_TRUE(uuid);

        auto uuid2 = target::DeserializeUuid(target::SerializeUuid(uuid));
        EXPECT_EQ(0, std::memcmp(&uuid, &uuid2, sizeof(uuid)));
    }
}

TEST(AccountUser, Uuid_Uniqueness)
{
    nnt::account::RamFs fs;
    auto lock = fs.AcquireWriterLock();
    fs.Delete(target::IdGenerationContextPath);

    const int UuidDuplicateCheckCount = 0x0FFF;
    std::unique_ptr<target::Uuid[]> uuids(new target::Uuid[UuidDuplicateCheckCount]);
    for (int i = 0; i < UuidDuplicateCheckCount; i++)
    {
        uuids[i] = target::GenerateUuid(fs);
    }

    for (int i = 0; i < UuidDuplicateCheckCount; ++ i)
    {
        EXPECT_TRUE(uuids[i]);
        for (int j = 0; j < i; ++ j)
        {
            EXPECT_NE(0, std::memcmp(&uuids[i], &uuids[j], sizeof(uuids[i])));
        }
    }
};

#endif // NNT_ACCOUNT_ENABLE_UUID
