﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "http/account_RedirectUriParser.h"

#include <nn/account/account_TypesForSystemServices.h>

#include "testAccount_RedirectUriParser.h"
#include "testAccount_Util.h"

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_URL_PARSER

#if defined(NNT_ACCOUNT_ENABLE_URL_PARSER)

namespace
{

const char *Uri[] = {
    "", // 空
    "a", // 1 文字
    "http://example.com", // example.com
    "http://example.com/", // example.com/
    "nintendo://account.nx.sys",
    "nintendo://0005000001234567.app",
    "!@$%^&*()-_=+[]{}|\\;:'\",<.>/", // uri は [#?] 以外 OK
    "_______________________________________________________________________________________________________________________________________________________________________________________________________________________________________________________________", // 長い
};

const char *Separator[] = {
    "?", // クエリパラメータ
    "#", // URL フラグメント
};

const struct Error
{
    const char* input;
    int failAt;
} Errors[] = {
    {"=", 0}, // keyの長さ = 0
    {"=v", 0}, // keyの長さ = 0
    {"k=v&", 1}, // keyの長さ = 0
    {"k=v&=", 1}, // keyの長さ = 0
    {"k=&=v", 1}, // keyの長さ = 0
    {"key=value&=value", 1}, // keyの長さ = 0
    {"k", 0}, // value との区切り文字がない
    {"k=&k", 1}, // value との区切り文字がない
    {"k=v&k", 1}, // value との区切り文字がない
    {"key=value&key", 1}, // value との区切り文字がない
};

template <int Count>
struct TestCase
    : a::http::RedirectUriParser<256>
{
    typedef a::http::RedirectUriParser<256> Base;

    char* input;
    size_t _size;

    char _uri[256];
    const char* _separator;
    const redirect::KV** _pPairs;
    int _count;
    int _current;
    bool _result;

    TestCase(const char* uri, const char* separator) NN_NOEXCEPT
        : Base(_uri)
        , input(nullptr)
        , _size(32 * 1024 * 1024)
        , _separator(separator)
    {
        strncpy(_uri, uri, sizeof(_uri));
        input = reinterpret_cast<char*>(malloc(_size));
        NN_ABORT_UNLESS_NOT_NULL(input);
        _pPairs = new const redirect::KV*[Count];
    }
    ~TestCase() NN_NOEXCEPT
    {
        if (input)
        {
            delete[] _pPairs;
            std::free(input);
            input = nullptr;
        }
    }
    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_ABORT_UNLESS(strnlen(key, keyLength) == keyLength);
        NN_ABORT_UNLESS(strnlen(value, valueLength) == valueLength);

        if (!_result)
        {
            return;
        }
        auto index = _current ++;
        auto& in = *_pPairs[index];
        _result = true
            && (strlen(in.k) == keyLength && std::strncmp(in.k, key, keyLength) == 0)
            && (strlen(in.v) == valueLength && std::strncmp(in.v, value, valueLength) == 0);

        if (!_result)
        {
            NN_LOG("Failed at \"%s=%s\" of \"%s\"\n", in.k, in.v, input);
        }
    }

    void Initialize(int* indice, int count) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(Count == count);

        auto l = nn::util::SNPrintf(input, _size, "%s%s", _uri, _separator);
        NN_ABORT_UNLESS(static_cast<size_t>(l) < _size);

        _count = count;
        for (int i = 0; i < count; ++ i)
        {
            auto& in = redirect::kvInputs[indice[i]];
            _pPairs[i] = &in;

            l += nn::util::SNPrintf(
                input + l, _size - l,
                i == 0 ? "%s=%s" : "&%s=%s",
                in.k, in.v);
            NN_ABORT_UNLESS(static_cast<size_t>(l) < _size);
        }
    }

    bool RunTest() NN_NOEXCEPT
    {
        _current = 0;
        _result = true;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(Base::Parse(input, strlen(input) + 1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(_result);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(_current == _count);
        return true;
    }
};
struct ErrorTestCase
    : a::http::RedirectUriParser<256>
{
    typedef a::http::RedirectUriParser<256> Base;

    char _uriBase[256];
    const char* _input;
    int _current;
    int _failAt;

    explicit ErrorTestCase(const char* uriBase) NN_NOEXCEPT
        : Base(_uriBase)
        , _input(nullptr)
    {
        strncpy(_uriBase, uriBase, sizeof(_uriBase));
    }
    bool RunTest(const char* input, int failAt) NN_NOEXCEPT
    {
        _input = input;
        _current = 0;
        _failAt = failAt;
        return !Base::Parse(input, strlen(input) + 1);
    }
    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_UNUSED(keyLength);
        NN_UNUSED(valueLength);
        auto index = _current ++;
        NN_ABORT_UNLESS(index < _failAt, "Unreachable: \"%s=%s\" of \"%s\"\n", key, value, _input);
    }
};

template <int Level>
bool TestForSpecificUriKind(const char* uri, const char* separator) NN_NOEXCEPT
{
    TestCase<Level> test(uri, separator);

    bool result = true;
    int indice[Level] = {};

    const auto InputLength = sizeof(redirect::kvInputs) / sizeof(redirect::kvInputs[0]);
    NN_STATIC_ASSERT(sizeof(redirect::kvInputs) / sizeof(redirect::kvInputs[0]) > 0);

    while (NN_STATIC_CONDITION(true))
    {
        ++ indice[0];
        for (int i = 0; i < Level; ++ i)
        {
            if (indice[i] >= InputLength)
            {
                // 現在のレベルのバリエーションを網羅した
                if (i + 1 < Level)
                {
                    // 次のレベルのバリエーションを増やす
                    -- indice[i];
                    ++ indice[i + 1];
                }
                else
                {
                    // すべてのバリエーションをテスト済み
                    return result;
                }
            }
            else
            {
                // 現在のレベルの次のバリエーションを見る
                break;
            }
        }

        test.Initialize(indice, Level);
        bool r = test.RunTest();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(r);
        result = (result && r);
    }
    NN_ABORT_UNLESS("Unreachable\n");
}

template <int Level>
bool RunTest() NN_NOEXCEPT
{
    bool result = true;
    for (auto uri : Uri)
    {
        for (auto separator : Separator)
        {
            bool r = TestForSpecificUriKind<Level>(uri, separator);
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(r);
            result = (result && r);
        }
    }
    return result;
}

bool RunTestError() NN_NOEXCEPT
{
    t::Buffer buffer(1024);
    bool result = true;
    for (auto uri : Uri)
    {
        for (auto separator : Separator)
        {
            for (auto e : Errors)
            {
                auto l = nn::util::SNPrintf(buffer.Get<char>(), buffer.GetSize(), "%s%s%s", uri, separator, e.input);
                NN_ABORT_UNLESS(l < static_cast<int>(buffer.GetSize()));
                ErrorTestCase test(uri);
                NNT_ACCOUNT_RETURN_FALSE_UNLESS(test.RunTest(buffer.Get<char>(), e.failAt));
            }
        }
    }
    return result;
}

} // ~namespace <anonymous>


TEST(AccountRedirectUriParser, Basic)
{
    EXPECT_TRUE(RunTest<1>());
    EXPECT_TRUE(RunTest<2>());
    EXPECT_TRUE(RunTest<3>());
    EXPECT_TRUE(RunTest<16>());
}

TEST(AccountRedirectUriParser, Error)
{
    /* テストされる入力:
        "?="
        "?=v"
        "?k=v&"
        "?k=v&="
        "?k=&=v"
        "?key=value&=value"
        "?k"
        "?k=&k"
        "?k=v&k"
        "?key=value&key"
        "#="
        "#=v"
        "#k=v&"
        "#k=v&="
        "#k=&=v"
        "#key=value&=value"
        "#k"
        "#k=&k"
        "#k=v&k"
        "#key=value&key"
        "a?="
        "a?=v"
        "a?k=v&"
        "a?k=v&="
        "a?k=&=v"
        "a?key=value&=value"
        "a?k"
        "a?k=&k"
        "a?k=v&k"
        "a?key=value&key"
        "a#="
        "a#=v"
        "a#k=v&"
        "a#k=v&="
        "a#k=&=v"
        "a#key=value&=value"
        "a#k"
        "a#k=&k"
        "a#k=v&k"
        "a#key=value&key"
        "http://example.com?="
        "http://example.com?=v"
        "http://example.com?k=v&"
        "http://example.com?k=v&="
        "http://example.com?k=&=v"
        "http://example.com?key=value&=value"
        "http://example.com?k"
        "http://example.com?k=&k"
        "http://example.com?k=v&k"
        "http://example.com?key=value&key"
        "http://example.com#="
        "http://example.com#=v"
        "http://example.com#k=v&"
        "http://example.com#k=v&="
        "http://example.com#k=&=v"
        "http://example.com#key=value&=value"
        "http://example.com#k"
        "http://example.com#k=&k"
        "http://example.com#k=v&k"
        "http://example.com#key=value&key"
        "http://example.com/?="
        "http://example.com/?=v"
        "http://example.com/?k=v&"
        "http://example.com/?k=v&="
        "http://example.com/?k=&=v"
        "http://example.com/?key=value&=value"
        "http://example.com/?k"
        "http://example.com/?k=&k"
        "http://example.com/?k=v&k"
        "http://example.com/?key=value&key"
        "http://example.com/#="
        "http://example.com/#=v"
        "http://example.com/#k=v&"
        "http://example.com/#k=v&="
        "http://example.com/#k=&=v"
        "http://example.com/#key=value&=value"
        "http://example.com/#k"
        "http://example.com/#k=&k"
        "http://example.com/#k=v&k"
        "http://example.com/#key=value&key"
        "nintendo://account.nx.sys?="
        "nintendo://account.nx.sys?=v"
        "nintendo://account.nx.sys?k=v&"
        "nintendo://account.nx.sys?k=v&="
        "nintendo://account.nx.sys?k=&=v"
        "nintendo://account.nx.sys?key=value&=value"
        "nintendo://account.nx.sys?k"
        "nintendo://account.nx.sys?k=&k"
        "nintendo://account.nx.sys?k=v&k"
        "nintendo://account.nx.sys?key=value&key"
        "nintendo://account.nx.sys#="
        "nintendo://account.nx.sys#=v"
        "nintendo://account.nx.sys#k=v&"
        "nintendo://account.nx.sys#k=v&="
        "nintendo://account.nx.sys#k=&=v"
        "nintendo://account.nx.sys#key=value&=value"
        "nintendo://account.nx.sys#k"
        "nintendo://account.nx.sys#k=&k"
        "nintendo://account.nx.sys#k=v&k"
        "nintendo://account.nx.sys#key=value&key"
        "nintendo://0005000001234567.app?="
        "nintendo://0005000001234567.app?=v"
        "nintendo://0005000001234567.app?k=v&"
        "nintendo://0005000001234567.app?k=v&="
        "nintendo://0005000001234567.app?k=&=v"
        "nintendo://0005000001234567.app?key=value&=value"
        "nintendo://0005000001234567.app?k"
        "nintendo://0005000001234567.app?k=&k"
        "nintendo://0005000001234567.app?k=v&k"
        "nintendo://0005000001234567.app?key=value&key"
        "nintendo://0005000001234567.app#="
        "nintendo://0005000001234567.app#=v"
        "nintendo://0005000001234567.app#k=v&"
        "nintendo://0005000001234567.app#k=v&="
        "nintendo://0005000001234567.app#k=&=v"
        "nintendo://0005000001234567.app#key=value&=value"
        "nintendo://0005000001234567.app#k"
        "nintendo://0005000001234567.app#k=&k"
        "nintendo://0005000001234567.app#k=v&k"
        "nintendo://0005000001234567.app#key=value&key"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?="
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?=v"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?k=v&"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?k=v&="
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?k=&=v"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?key=value&=value"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?k"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?k=&k"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?k=v&k"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/?key=value&key"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#="
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#=v"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#k=v&"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#k=v&="
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#k=&=v"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#key=value&=value"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#k"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#k=&k"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#k=v&k"
        "!@$%^&*()-_=+[]{}|\;:'",<.>/#key=value&key"
    */
    EXPECT_TRUE(RunTestError());
}

#endif // NNT_ACCOUNT_ENABLE_URL_PARSER
