﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "profile/account_ProfileAdaptor.h"

#include "testAccount_Module.h"

#include "detail/account_CacheUtil.h"
#include "detail/account_PathUtil.h"
#include "detail/account_UuidUtil.h"
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/profile/account_ProfileStorage.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>

namespace a = nn::account;
namespace p = nn::account::profile;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_ADAPTOR_DEFAULT
#define NNT_ACCOUNT_ENABLE_ADAPTOR_MII0_IMAGE0
#define NNT_ACCOUNT_ENABLE_ADAPTOR_MII0_IMAGE1
#define NNT_ACCOUNT_ENABLE_ADAPTOR_AUTHOR

namespace
{
const size_t TestDataCount = 32;
char g_Nicknames[TestDataCount][a::NicknameBytesMax + 1] = {{'\0'}};
std::pair<void*, size_t>* TestImages;

const char BadNickname[a::UserDataBytesMax + 1] = {0x1f, 0x03, 0x71, 0x4F};

struct TestDataInitializer
{
    TestDataInitializer() NN_NOEXCEPT
    {
        TestImages = new std::pair<void*, size_t>[TestDataCount];

        for (auto i = 0u; i < TestDataCount; ++ i)
        {
            // ニックネーム
            for (auto j = 0u; j < a::NicknameBytesMax; ++ j)
            {
                auto u8 = static_cast<uint8_t>(((i & 0xF) << 4) | (j & 0xF));
                g_Nicknames[i][j] = (j > i? '\0': (0x3B ^ u8)); // 最初は文字数少な目
            }
            g_Nicknames[i][a::NicknameBytesMax] = '\0';

            // 画像
            size_t bytes = 4096 + 13 * i;
            TestImages[i].first = malloc(bytes);
            TestImages[i].second = bytes;
            auto data = reinterpret_cast<char*>(TestImages[i].first);
            for (size_t s = 0; s < bytes; ++ s)
            {
                data[s] = static_cast<char>((s + i) % 0xFF);
            }
        }
    }
    ~TestDataInitializer() NN_NOEXCEPT
    {
        for (auto i = 0u; i < TestDataCount; ++ i)
        {
            free(TestImages[i].first);
        }

        delete[] TestImages;
    }
} g_TestDataInitializer;


// プロフィールが空のユーザーに対する ProfileAdaptor の挙動のテスト
void TestAdaptorDefaultValue(
    std::shared_ptr<a::profile::ProfileStorage> storage)
{
    // Default value
    a::Uid u = {{0x0123456789ABCDEFull, 0xDEADBEEFBEADCAFEull}};
    storage->Add(u);

    a::profile::ProfileAdaptor adaptor(storage.get(), u);

    a::baas::UserProfile profile;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor.AcquireProfile(&profile.base));
    EXPECT_EQ(0u, strnlen(profile.base.nickname, sizeof(profile.base.nickname)));
    EXPECT_EQ(8u + 224u, strnlen(profile.base.extraData, sizeof(profile.base.extraData)));

    const char Expected[] =
        "01606150"
        "iavN70VnASP+yt6tvu++rQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
        "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
        "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
        "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";
    EXPECT_EQ(0, std::memcmp(Expected, profile.base.extraData, sizeof(Expected)));

    size_t outSize;
    char rawData[168];
    auto rv = nn::util::Base64::FromBase64String(&outSize, rawData, sizeof(rawData), profile.base.extraData + 8, nn::util::Base64::Mode_NormalNoLinefeed);
    EXPECT_EQ(nn::util::Base64::Status_Success, rv);

    const uint8_t ExpectedRaw[] = {
        0x89, 0xab, 0xcd, 0xef, 0x45, 0x67, 0x01, 0x23, // Author (lo)
        0xfe, 0xca, 0xde, 0xad, 0xbe, 0xef, 0xbe, 0xad, // Author (hi)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // TimeStamp
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // _padding
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // _padding
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (0)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (8)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (16)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (24)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (32)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (40)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (48)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (56)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (64)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (72)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (80)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (88)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (96)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (104)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (112)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // UserData (120)
    };
    EXPECT_EQ(0, std::memcmp(ExpectedRaw, rawData, sizeof(ExpectedRaw)));

    // 画像はない
    a::detail::Uuid cacheId;
    t::Buffer buffer(a::ProfileImageBytesMax * 3);
    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultUserNotExist, adaptor.CreateProfileImageCache(&cacheId, buffer.GetAddress(), buffer.GetSize()));

    storage->Delete(u);
}

// ProfileBase が一致するか
bool CompareProfileBase(
    const a::profile::ProfileBase& lhs, const a::profile::UserData& lhs1,
    const a::profile::ProfileBase& rhs, const a::profile::UserData& rhs1) NN_NOEXCEPT
{
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(lhs.author, rhs.author);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(lhs.timeStamp, rhs.timeStamp);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::strncmp(lhs.nickname, rhs.nickname, sizeof(lhs.nickname)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::memcmp(lhs1.data, rhs1.data, sizeof(rhs1.data)));
    return true;
}
// 指定されたユーザーのプロフィール画像が、指定されたバッファの内容と一致するか
bool CompareProfileImage(
    const a::detail::AbstractLocalStorage& ls,
    const void* image, size_t imageSize,
    const a::Uid& uid) NN_NOEXCEPT
{
    char path[128];
    a::detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, ls.GetRootPath());

    size_t size;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ls.GetFileSystem().GetSize(&size, path));
    EXPECT_EQ(size, imageSize);

    size_t actualRead;
    t::Buffer buffer(size);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ls.GetFileSystem().Read(&actualRead, buffer.GetAddress(), buffer.GetSize(), path));
    EXPECT_EQ(size, actualRead);
    EXPECT_EQ(0, std::memcmp(image, buffer.GetAddress(), imageSize));

    return true;
}
// 指定されたユーザーのProfileBase とプロフィール画像が、指定されたバッファの内容と一致するか
bool CompareProfile(
    std::shared_ptr<a::profile::ProfileStorage> storage,
    const a::detail::AbstractLocalStorage& ls,
    const a::profile::ProfileBase& base, const a::profile::UserData& userData,
    const void* image, size_t imageSize,
    const a::Uid& uid) NN_NOEXCEPT
{
    a::profile::ProfileBase output;
    a::profile::UserData output1;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->GetProfile(&output, &output1, uid));
    EXPECT_TRUE(CompareProfileBase(base, userData, output, output1));
    if (image != nullptr)
    {
        EXPECT_TRUE(CompareProfileImage(ls, image, imageSize, uid));
    }

    return true;
}

// 指定されたパスのファイルの内容が、指定されたバッファの Base64 エンコード値と等しいか
bool CompareCachedImage(
    const a::detail::AbstractLocalStorage& ls,
    const void* image, size_t imageSize,
    const a::detail::Uuid& cacheId) NN_NOEXCEPT
{
    size_t size;
    auto r = a::detail::CacheUtil::LoadCacheFile<256 * 1024, a::ResultInsufficientBuffer>(&size, nullptr, 0, cacheId, ls);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(r);
    t::Buffer buffer(size + 1);
    r = a::detail::CacheUtil::LoadCacheFile<256 * 1024, a::ResultInsufficientBuffer>(&size, buffer.GetAddress(), buffer.GetSize(), cacheId, ls);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(r);
    buffer.Get<char>()[size] = '\0';

    size_t decodedSize;
    t::Buffer decoded(size);
    auto rv = nn::util::Base64::FromBase64String(
        &decodedSize, decoded.GetAddress(), decoded.GetSize(), buffer.Get<char>(), nn::util::Base64::Mode_NormalNoLinefeed);
    EXPECT_EQ(nn::util::Base64::Status_Success, rv);

    EXPECT_EQ(imageSize, decodedSize);
    EXPECT_EQ(0, std::memcmp(image, decoded.GetAddress(), imageSize));

    return true;
}

// タイムスタンプが古いプロフィールが意に反して変更されないか。新しいプロフィールは意図通りに反映されるか。
bool TestTimestampCheck(
    a::profile::ProfileAdaptor& adaptor,
    std::shared_ptr<a::profile::ProfileStorage> storage,
    const a::detail::AbstractLocalStorage& ls,
    const a::Uid& uid,
    const a::profile::ProfileBase& base, const a::profile::UserData& baseUserData,
    const void* image, size_t imageSize,
    a::baas::UserProfileBase baasData) NN_NOEXCEPT
{
    // ローカルのタイムスタンプを未来に設定する。他は同じにする
    a::profile::ProfileBase dummy;
    dummy = base;
    dummy.timeStamp = base.timeStamp + 1;
    a::profile::UserData dummyUserData = baseUserData;
    storage->Update(uid, dummy, dummyUserData, nullptr, 0);
    // 再度外部データを適用する (外部データが古いので失敗する)
    a::profile::ProfileBase remote;
    a::profile::UserData remoteUserData;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Decode(&remote, &remoteUserData, baasData));
    a::profile::ProfileAdaptor::EvaluationResult eval;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Evaluate(&eval, remote));
    EXPECT_EQ(a::profile::ProfileAdaptor::EvaluationResult_Export, eval); // author が一致するので常に Export
    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultProfileObsoleted, adaptor.Import(remote, remoteUserData, image, imageSize));

    // ローカルのタイムスタンプを同時刻に設定し、他はダミーデータで埋める
    dummy = base;
    dummy.timeStamp = base.timeStamp;
    std::strncpy(dummy.nickname, BadNickname, sizeof(dummy.nickname));
    std::memset(dummyUserData.data, 0xFA, sizeof(dummyUserData.data));
    storage->Update(uid, dummy, dummyUserData, nullptr, 0);
    // 再度外部データを適用する
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Decode(&remote, &remoteUserData, baasData));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Evaluate(&eval, remote));
    EXPECT_EQ(a::profile::ProfileAdaptor::EvaluationResult_Export, eval); // author が一致するので常に Export
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor.Import(remote, remoteUserData, image, imageSize));
    // 結果の確認
    EXPECT_TRUE(CompareProfile(storage, ls, base, baseUserData, image, imageSize, uid));

    // ローカルのタイムスタンプを過去に設定し、他はダミーデータで埋める
    dummy = base;
    dummy.timeStamp = base.timeStamp - 1;
    std::strncpy(dummy.nickname, BadNickname, sizeof(dummy.nickname));
    std::memset(dummyUserData.data, 0x3E, sizeof(dummyUserData.data));
    storage->Update(uid, dummy, dummyUserData, nullptr, 0);
    // 再度外部データを適用する
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Decode(&remote, &remoteUserData, baasData));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Evaluate(&eval, remote));
    EXPECT_EQ(a::profile::ProfileAdaptor::EvaluationResult_Export, eval); // author が一致するので常に Export
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor.Import(remote, remoteUserData, image, imageSize));
    // 結果の確認
    EXPECT_TRUE(CompareProfile(storage, ls, base, baseUserData, image, imageSize, uid));

    return true;
}

// Mii もプロフィール画像も持たないユーザーのプロフィールのシリアライズ, デシリアライズのテスト
void TestAdapterModification(
    std::shared_ptr<a::profile::ProfileStorage> storage,
    const a::detail::AbstractLocalStorage& ls,
    int* testDataIndice, size_t testDataCount)
{
    auto u = a::detail::ConvertToUid(ls.GenerateUuidWithContext());

    storage->Add(u);

    for (auto i = 0u; i < testDataCount; ++ i)
    {
        auto index = testDataIndice[i];

        // テストデータの初期設定
        a::profile::ProfileBase update;
        update.author = u;
        update.timeStamp = index + 1;
        std::strncpy(update.nickname, g_Nicknames[index], sizeof(update.nickname));
        a::profile::UserData userData;
        std::memset(userData.data, i, sizeof(userData.data));
        storage->Update(u, update, userData, nullptr, 0);

        // Adaptor 初期化
        a::profile::ProfileAdaptor adaptor(storage.get(), u);

        // Adaptor によってシリアライズされた値の取得
        a::baas::UserProfileBase profile;
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor.AcquireProfile(&profile));
        EXPECT_EQ(8u + 224u, strnlen(profile.extraData, sizeof(profile.extraData))); // 内容はとりあえず見ない

        // 画像はないため、キャッシュは取得できない
        a::detail::Uuid cacheId;
        {
            t::Buffer buffer(a::ProfileImageBytesMax * 3);
            NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
                a::ResultUserNotExist,
                adaptor.CreateProfileImageCache(&cacheId, buffer.GetAddress(), buffer.GetSize()));
        }

        //// ニックネームの検査
        EXPECT_EQ(0, std::strncmp(profile.nickname, g_Nicknames[index], sizeof(profile.nickname)));
        EXPECT_TRUE(strnlen(profile.nickname, sizeof(profile.nickname)) <= a::NicknameBytesMax);

        //// LocalData の検査
        EXPECT_TRUE(TestTimestampCheck(
            adaptor, storage, ls, u,
            update, userData, nullptr, 0u,
            profile));
    }
    storage->Delete(u);
}

// Mii は持たないが、プロフィール画像を持つユーザーのプロフィールのシリアライズ, デシリアライズのテスト
void TestAdapterModificationWithImage(
    std::shared_ptr<a::profile::ProfileStorage> storage, const a::detail::AbstractLocalStorage& ls,
    int* testDataIndice, size_t testDataCount)
{
    auto u = a::detail::ConvertToUid(ls.GenerateUuidWithContext());

    storage->Add(u);

    for (auto i = 0u; i < testDataCount; ++ i)
    {
        auto index = testDataIndice[i];

        // テストデータの初期設定
        a::profile::ProfileBase update;
        update.author = u;
        update.timeStamp = index + 1;
        std::strncpy(update.nickname, g_Nicknames[index], sizeof(update.nickname));
        a::profile::UserData userData;
        std::memset(userData.data, i, sizeof(userData.data));
        storage->Update(u, update, userData, TestImages[index].first, TestImages[index].second);

        {
            size_t actualSize;
            t::Buffer b(128 * 1024);
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(storage->LoadImage(&actualSize, b.GetAddress(), b.GetSize(), u));
        }

        // Adaptor 初期化
        a::profile::ProfileAdaptor adaptor(storage.get(), u);

        // Adaptor による、転送用の値の取得
        a::baas::UserProfileBase profile;
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor.AcquireProfile(&profile));
        EXPECT_EQ(8u + 224u, strnlen(profile.extraData, sizeof(profile.extraData)));

        // 画像のキャッシュの取得
        a::detail::Uuid cacheId;
        {
            t::Buffer buffer(a::ProfileImageBytesMax * 3);
            NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor.CreateProfileImageCache(
                &cacheId, buffer.GetAddress(), buffer.GetSize()));
        }
        NN_UTIL_SCOPE_EXIT
        {
            a::detail::CacheUtil::DeleteCacheFile(cacheId, ls);
        };
        EXPECT_TRUE(CompareCachedImage(ls, TestImages[index].first, TestImages[index].second, cacheId));

        //// ニックネームの検査
        ////  - UTF-8 が正しいか
        EXPECT_EQ(0, std::strncmp(profile.nickname, g_Nicknames[index], sizeof(profile.nickname)));
        EXPECT_TRUE(strnlen(profile.nickname, sizeof(profile.nickname)) <= a::NicknameBytesMax);

        //// LocalData の検査
        auto& anotherImage = TestImages[TestDataCount - index - 1];
        EXPECT_TRUE(TestTimestampCheck(
            adaptor, storage, ls, u,
            update, userData, anotherImage.first, anotherImage.second,
            profile));
    }
    storage->Delete(u);
}

}

#if defined(NNT_ACCOUNT_ENABLE_ADAPTOR_DEFAULT)
TEST(AccountProfile, ProfileAdaptor_DefaultValue)
{
    a::detail::LocalStorage<a::detail::DefaultFileSystem, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    // Default value
    TestAdaptorDefaultValue(storage);
}
TEST(AccountProfile, ProfileAdaptor_DefaultValueOnRamFs)
{
    const int TestCount = 100;

    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    // Default value
    for (auto i = 0; i < TestCount; ++ i)
    {
        TestAdaptorDefaultValue(storage);
    }
}
#endif

#if defined(NNT_ACCOUNT_ENABLE_ADAPTOR_MII0_IMAGE0)
TEST(AccountProfile, ProfileAdaptor_Base)
{
    a::detail::LocalStorage<a::detail::DefaultFileSystem, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    // Nickname modification
    const size_t IndiceLength = 1; // HostFs では 1 つ (RamFs ではぜんぶ見る)
    int indice[IndiceLength];
    for (auto i = 0u; i < IndiceLength; ++ i)
    {
        indice[i] = i;
    }
    TestAdapterModification(storage, s, indice, IndiceLength);
}
TEST(AccountProfile, ProfileAdaptor_BaseOnRamFs)
{
    const int TestCount = 100;

    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    // Nickname modification
    const size_t IndiceLength = TestDataCount;
    int indice[IndiceLength];
    for (auto i = 0u; i < IndiceLength; ++ i)
    {
        indice[i] = i;
    }
    // Default value
    for (auto i = 0; i < TestCount; ++ i)
    {
        TestAdapterModification(storage, s, indice, IndiceLength);
    }
}
#endif

#if defined(NNT_ACCOUNT_ENABLE_ADAPTOR_MII0_IMAGE1)
TEST(AccountProfile, ProfileAdaptor_BaseWithImage)
{
    a::detail::LocalStorage<a::detail::DefaultFileSystem, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    // Nickname modification
    const size_t IndiceLength = 1; // HostFs では 1 つ (RamFs ではぜんぶ見る)
    int indice[IndiceLength];
    for (auto i = 0u; i < IndiceLength; ++ i)
    {
        indice[i] = i;
    }
    TestAdapterModificationWithImage(storage, s, indice, IndiceLength);
}
TEST(AccountProfile, ProfileAdaptor_BaseWithImageOnRamFs)
{
    const int TestCount = 100;

    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    // Nickname modification
    const size_t IndiceLength = TestDataCount;
    int indice[IndiceLength];
    for (auto i = 0u; i < IndiceLength; ++ i)
    {
        indice[i] = i;
    }
    // Default value
    for (auto i = 0; i < TestCount; ++ i)
    {
        TestAdapterModificationWithImage(storage, s, indice, IndiceLength);
    }
}
#endif

#if defined(NNT_ACCOUNT_ENABLE_ADAPTOR_AUTHOR)
TEST(AccountProfile, ProfileAdaptor_Author)
{
    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    // 古いプロフィールの作成
    a::Uid uid0 = a::detail::ConvertToUid(s.GenerateUuidWithContext());
    storage->Add(uid0);
    a::profile::ProfileBase base0 = a::profile::DefaultProfileBase;
    base0.author = uid0;
    strncpy(base0.nickname, "user0", sizeof("user0"));
    a::profile::UserData user0;
    std::memset(user0.data, 0xF0, sizeof(user0.data));
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Update(uid0, base0, user0, nullptr, 0));
    // こちらは強制適用「可」にしておく
    a::profile::ProfileAdaptor adaptor0(storage.get(), uid0, true);

    a::baas::UserProfileBase profile;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor0.AcquireProfile(&profile));

    // 新しいプロフィールの作成
    a::Uid uid1 = a::detail::ConvertToUid(s.GenerateUuidWithContext());
    storage->Add(uid1);
    a::profile::ProfileBase base1 = a::profile::DefaultProfileBase;
    base1.author = uid1;
    base1.timeStamp = base0.timeStamp + 1;
    strncpy(base1.nickname, "user1", sizeof("user1"));
    a::profile::UserData user1;
    std::memset(user0.data, 0xF1, sizeof(user0.data));
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Update(uid1, base1, user1, nullptr, 0));
    // こちらは強制適用「不可」にしておく
    a::profile::ProfileAdaptor adaptor1(storage.get(), uid1, false);
    // タイムスタンプが古いと適用できない
    a::profile::ProfileBase remote;
    a::profile::UserData remoteUser;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor1.Decode(&remote, &remoteUser, profile));
    a::profile::ProfileAdaptor::EvaluationResult eval;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor1.Evaluate(&eval, remote));
    EXPECT_EQ(a::profile::ProfileAdaptor::EvaluationResult_Export, eval);
    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(
        a::ResultProfileObsoleted,
        adaptor1.Import(remote, remoteUser, nullptr, 0));

    a::baas::UserProfileBase another;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor1.AcquireProfile(&another));

    // 0 に 1 を適用する。 (0 は 1 になる)
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor0.Decode(&remote, &remoteUser, another));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor0.Evaluate(&eval, remote));
    EXPECT_EQ(a::profile::ProfileAdaptor::EvaluationResult_Import, eval);
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor0.Import(remote, remoteUser, nullptr, 0));
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->GetProfile(&base0, &user0, uid0));
    EXPECT_EQ(uid1, base0.author);
    EXPECT_EQ(base1.timeStamp, base0.timeStamp);
    EXPECT_EQ(0, strncmp(base0.nickname, "user1", sizeof("user1")));
    EXPECT_EQ(0, std::memcmp(user0.data, user1.data, sizeof(user0.data)));

    a::baas::UserProfileBase updated;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor0.AcquireProfile(&updated));
    EXPECT_EQ(0, strncmp(updated.nickname, "user1", sizeof("user1")));
    EXPECT_EQ(0, std::memcmp(user0.data, user1.data, sizeof(user0.data)));

    // 0 に 0_1 を適用する (0 は 1 のままの筈)
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor0.Decode(&remote, &remoteUser, updated));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor0.Evaluate(&eval, remote));
    EXPECT_EQ(a::profile::ProfileAdaptor::EvaluationResult_Import, eval);
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor0.Import(remote, remoteUser, nullptr, 0));
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->GetProfile(&base0, &user0, uid0));
    EXPECT_EQ(uid1, base0.author);
    EXPECT_EQ(base1.timeStamp, base0.timeStamp);
    EXPECT_EQ(0, strncmp(base0.nickname, "user1", sizeof("user1")));
    EXPECT_EQ(0, std::memcmp(user0.data, user1.data, sizeof(user0.data)));

    // 0 に 0 を適用する (0 は 0 に戻る)
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor0.Decode(&remote, &remoteUser, profile));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor0.Evaluate(&eval, remote));
    EXPECT_EQ(a::profile::ProfileAdaptor::EvaluationResult_Import, eval);
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(adaptor0.Import(remote, remoteUser, nullptr, 0));
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->GetProfile(&base0, &user0, uid0));
    EXPECT_EQ(uid0, base0.author);
    EXPECT_EQ(base0.timeStamp, base0.timeStamp);
    EXPECT_EQ(0, strncmp(base0.nickname, "user0", sizeof("user0")));
    a::profile::UserData expect0;
    std::memset(expect0.data, 0xF0, sizeof(expect0.data));
    EXPECT_EQ(0, std::memcmp(user0.data, expect0.data, sizeof(user0.data)));
}
#endif

#if defined(NNT_ACCOUNT_ENABLE_ADAPTOR_EVALUATE)

#endif
