﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_StateRetention.h>
#include <nn/account/account_Selector.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#include <nn/oe.h>
#endif

#include "testAccount_Util.h"

#include <nnt.h>
#include <nnt/nntest.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();
    testing::InitGoogleTest(&argc, argv);

    auto& listeners = testing::UnitTest::GetInstance()->listeners();
    auto* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    nn::oe::Initialize();
#endif

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    nn::account::Initialize();

    nnt::Exit(RUN_ALL_TESTS());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
}

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#if !defined(NNT_ACCOUNT_COMPATIBILITY_TEST)
TEST(AccoutApp, Preselection)
{
    const auto TestCount = 2;

    nn::account::Uid user = nn::account::InvalidUid;
    for (auto i = 0; i < TestCount; ++ i)
    {
        nn::account::UserHandle handle;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::OpenPreselectedUser(&handle));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetUserId(&user, handle));
        ASSERT_TRUE(user);
        nn::account::CloseUser(handle);
    }

    NN_LOG(
        ">> User %08x_%08x_%08x_%08x is preselected\n",
        static_cast<uint32_t>(user._data[0] >> 32),
        static_cast<uint32_t>(user._data[0] & 0xFFFFFFFFull),
        static_cast<uint32_t>(user._data[1] >> 32),
        static_cast<uint32_t>(user._data[1] & 0xFFFFFFFFull));
}
#endif // NNT_ACCOUNT_COMPATIBILITY_TEST
#endif // NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON

TEST(AccountApp, List)
{
    nn::account::Uid users[nn::account::UserCountMax];
    int userCount;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(
        &userCount, users, std::extent<decltype(users)>::value));

    nn::account::Uid qualifiedUsers[nn::account::UserCountMax];
    int qualifiedUserCount;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ListQualifiedUsers(
        &qualifiedUserCount, qualifiedUsers, std::extent<decltype(qualifiedUsers)>::value));
    ASSERT_TRUE(std::equal(users, users + userCount, qualifiedUsers));

    nn::account::Uid openUsers[nn::account::UserCountMax];
    int openUserCount;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ListOpenUsers(
        &openUserCount, openUsers, std::extent<decltype(openUsers)>::value));
    ASSERT_EQ(0, openUserCount);

#if defined(NNT_ACCOUNT_COMPATIBILITY_TEST)
    auto settings = nn::account::DefaultUserSelectionSettings;
    settings.isSkipEnabled = true;

    nn::account::Uid user;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ShowUserSelector(&user, settings));
#else
    auto user = users[0];
#endif

    nn::account::UserHandle handle;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&handle, user));
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(handle);
    };

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ListOpenUsers(
        &openUserCount, openUsers, std::extent<decltype(openUsers)>::value));
    ASSERT_EQ(1, openUserCount);
    ASSERT_TRUE(user == openUsers[0]);
}

TEST(AccountApp, NsaIdToken)
{
    int count;
    nn::account::Uid users[nn::account::UserCountMax];
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
    NN_LOG("Existing user count: %d\n", count);

#if defined(NNT_ACCOUNT_COMPATIBILITY_TEST)
    auto settings = nn::account::DefaultUserSelectionSettings;
    settings.isSkipEnabled = true;

    nn::account::Uid user;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ShowUserSelector(&user, settings));
#else
    auto user = users[0];
#endif

    NN_LOG(
        "User selected: %08x_%08x_%08x_%08x\n",
        static_cast<uint32_t>(user._data[0] >> 32),
        static_cast<uint32_t>(user._data[0] & 0xFFFFFFFFull),
        static_cast<uint32_t>(user._data[1] >> 32),
        static_cast<uint32_t>(user._data[1] & 0xFFFFFFFFull));

    nn::account::Nickname nickname;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNickname(&nickname, user));
    NN_LOG("Nickname: %s\n", nickname.name);

    nn::account::UserHandle handle;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&handle, user));
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(handle);
    };

    nn::account::Uid uid;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetUserId(&uid, handle));
    EXPECT_TRUE(uid == user);

    // トークンキャッシュの生成
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
    nn::nifm::SubmitNetworkRequestAndWait();
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    nn::account::AsyncContext ctx;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(
        &ctx, handle));
    nn::os::SystemEvent e;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    e.Wait();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ctx.GetResult());

    // トークンの取得
    size_t size;
    nnt::account::Buffer buffer(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::LoadNetworkServiceAccountIdTokenCache(
        &size, buffer.Get<char>(), buffer.GetSize(), handle));
    ASSERT_TRUE(size <= nn::account::NetworkServiceAccountIdTokenLengthMax);

    NN_LOG("ID token: \"%s\"\n", buffer.Get<char>());
}

TEST(AccoutApp, Retention)
{
    int count;
    nn::account::Uid users[nn::account::UserCountMax];
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
    NN_LOG("Existing user count: %d\n", count);

    for (int i = 0; i < count; ++ i)
    {
        const auto& user = users[i];
        NN_LOG(
            "User selected: %08x_%08x_%08x_%08x\n",
            static_cast<uint32_t>(user._data[0] >> 32),
            static_cast<uint32_t>(user._data[0] & 0xFFFFFFFFull),
            static_cast<uint32_t>(user._data[1] >> 32),
            static_cast<uint32_t>(user._data[1] & 0xFFFFFFFFull));

        nn::account::UserHandle handle;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&handle, user));
    }

    auto declaration = nn::account::DebugEnableInterprogramOpenUserRetention();

    // Push
    nn::account::PushOpenUsers();

    // Push したユーザーを取得
    nn::account::Uid pushedUsers[nn::account::UserCountMax];
    auto pushedCount = nn::account::ListPushedOpenUsers(pushedUsers, std::extent<decltype(pushedUsers)>::value);
    EXPECT_EQ(pushedCount, count);
    for (int i = 0; i < pushedCount; ++i)
    {
        auto p = std::find(users, users + count - 1, pushedUsers[i]);
        EXPECT_TRUE(p != nullptr);
        EXPECT_EQ(pushedUsers[i], *p);
    }

    // Pop
    nn::account::UserHandle pushedHandles[nn::account::UserCountMax];
    nn::account::PopOpenUsers(&pushedCount, pushedHandles, std::extent<decltype(pushedHandles)>::value);
    EXPECT_EQ(pushedCount, count);
    for (int i = 0; i < pushedCount; ++i)
    {
        nn::account::Uid uid;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetUserId(&uid, pushedHandles[i]));

        auto p = std::find(users, users + count - 1, uid);
        EXPECT_TRUE(p != nullptr);
        EXPECT_EQ(uid, *p);

        nn::account::CloseUser(pushedHandles[i]);
    }
}
