﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAccount_Initializer.h"
#include "testAccount_Util.h"

#include <curl/curl.h>
#include <cstdlib>
#include <cstring>

#include <nnt.h>
#include <nnt/nnt_Argument.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN)
#include <nn/nn_Windows.h>
#include <nn/nsd/nsd_ApiForTest.h>
#endif

#include <nn/nn_Abort.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/init.h>
#include <nn/nifm.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/socket.h>
#include <nn/time/time_Api.h>

namespace
{
NN_ALIGNAS(4096) uint8_t g_MallocBuffer[128 * 1024 * 1024];
} // ~namespace <anonymous>

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace nnt {
namespace account {

namespace {
const int SocketCount = 2;
const int SocketConcurrencyCount = SocketCount;

// Socket 用のリソース
nn::socket::ConfigDefaultWithConstrainedMemory
    <
        SocketCount, // tcpSocketCountMax
        0            // udpSocketCountMax
    >g_SocketConfigWithMemory(SocketConcurrencyCount);
} // ~namespace nnt::account::<anonymous>

void Initialize(bool networkRequired) NN_NOEXCEPT
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    //
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    // ファイルシステム
    nn::fs::SetAllocator(
        std::malloc,
        [](void* p, size_t s)->void
    {
        NN_UNUSED(s);
        std::free(p);
    });

    if (networkRequired)
    {
        // ソケット
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
        NN_ABORT_UNLESS(curl_global_init(CURL_GLOBAL_ALL) == CURLE_OK);

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN)
        // 環境を TD1 に固定
        nn::nsd::SetTd1EnvironmentForTest();
#endif
        // ネットワーク接続
        nn::nifm::SubmitNetworkRequestAndWait();
        NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

        // 環境が td1 であることの確認
        nn::nsd::EnvironmentIdentifier envId;
        nn::nsd::GetEnvironmentIdentifier(&envId);
        NN_ABORT_UNLESS(std::strncmp("td1", envId.value, sizeof(envId.value)) == 0);
    }

    TryLogNetworkTime();
    NN_LOG("[nnt::account] Test initialized successfully\n");
}

void Finalize(int result, bool networkRequired) NN_NOEXCEPT
{
    if (networkRequired)
    {
        // 通信機能の終了
        curl_global_cleanup();
        nn::socket::Finalize();
    }

    //
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());

    // テスト結果の反映
    nnt::Exit(result);
}

} // ~namespace nnt::account
}
