﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/nac/nac_Pad.h>

namespace nns {
namespace nac {

//---------------------------------------------------------------------------
//! @brief        コンストラクタです。
//---------------------------------------------------------------------------
Pad::Pad()
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    : m_Press(0)
    , m_PressPrev(0)
    , m_Trigger(0)
    , m_Release(0)
    , m_Channel(0)
    , leftX(0.f)
    , leftY(0.f)
    , rightX(0.f)
    , rightY(0.f)
    , m_IsConnected(false)
#else
    : m_PadState()
    , m_Press()
    , m_PressPrev()
    , m_Trigger()
    , m_Release()
#endif
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    , m_HWnd( nullptr )
    , m_IsPadEnabled( true )
    , m_IsUseKeyboardForcus( false )
#endif
{
}

//---------------------------------------------------------------------------
//! @brief        初期化
//---------------------------------------------------------------------------
bool Pad::Initialize( void* hWnd )
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    NW_UNUSED_VARIABLE( hWnd );
    PADInit();

    // m_PADStatuses の初期値を設定
    for( int i = 0; i < PAD_MAX_CONTROLLERS; ++i )
    {
        m_PADStatuses[i].err = PAD_ERR_NO_CONTROLLER;
    }

    m_Channel = 0;
#else
    nn::hid::InitializeDebugPad();

    InitializeNPad();

    m_Press.storage = 0;
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    nn::hid::InitializeKeyboard();
    m_HWnd = hWnd;
#endif

    return true;
}

void Pad::InitializeNPad()
{
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_WIN ) && !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    nn::hid::InitializeNpad();

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet( nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask );

    // 使用する Npad を設定

    m_NpadIds[0] = nn::hid::NpadId::No1;
    m_NpadIds[1] = nn::hid::NpadId::No2;
    m_NpadIds[2] = nn::hid::NpadId::No3;
    m_NpadIds[3] = nn::hid::NpadId::Handheld;

    nn::hid::SetSupportedNpadIdType( m_NpadIds, NPADID_NUM );
#endif
}

//---------------------------------------------------------------------------
//! @brief        開放処理
//---------------------------------------------------------------------------
bool Pad::Finalize()
{
    return true;
}

//---------------------------------------------------------------------------
//! @brief        Npadの有効な操作形態(NpadStyleSet)を取得
//---------------------------------------------------------------------------
void Pad::GetNpadState()
{
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_WIN ) && !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    //現在有効な操作形態(NpadStyleSet)を取得

    bool IsAnalogStickLData = false;
    bool IsAnalogStickRData = false;
    for(int i = 0; i < NPADID_NUM; i++)
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet( m_NpadIds[i] );
        m_NpadState[i].Reset();
        if ( !IsAnalogStickLData )
        {
            m_AnalogStickL.x = 0;
            m_AnalogStickL.y = 0;
        }
        if ( !IsAnalogStickRData )
        {
            m_AnalogStickR.x = 0;
            m_AnalogStickR.y = 0;
        }
        // ジョイコン操作が有効な場合
        if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            //最新のNpadのステートを取得
            nn::hid::GetNpadState(&(m_NpadJoyDualState[i]), m_NpadIds[i]);
            m_NpadState[i]        = m_NpadJoyDualState[i].buttons;

            // analogStickの確認
            if ( m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLRight>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLUp>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLLeft>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLDown>() )
            {
                m_AnalogStickL = m_NpadJoyDualState[i].analogStickL;
                IsAnalogStickLData = true;
            }

            if ( m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRRight>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRUp>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRLeft>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRDown>() )
            {
                m_AnalogStickR = m_NpadJoyDualState[i].analogStickR;
                IsAnalogStickRData = true;
            }

            m_IsNpadDataAvailable = true;
        }

        // 携帯機コントローラー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleHandheld>() == true)
        {
            //最新のNpadのステートを取得
            nn::hid::GetNpadState(&(m_NpadHandheldState), m_NpadIds[i]);
            m_NpadState[i]        = m_NpadHandheldState.buttons;

            // analogStickの確認
            if ( m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLRight>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLUp>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLLeft>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickLDown>() )
            {
                m_AnalogStickL = m_NpadHandheldState.analogStickL;
                IsAnalogStickLData = true;
            }

            if ( m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRRight>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRUp>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRLeft>() ||
                 m_NpadState[i].Test< nn::hid::NpadJoyButton::StickRDown>() )
            {
                m_AnalogStickR = m_NpadHandheldState.analogStickR;
                IsAnalogStickRData = true;
            }
            m_IsNpadDataAvailable = true;
        }
    }
#endif
}

//---------------------------------------------------------------------------
//! @brief       NPadの指定されたキーの有無を取得
//---------------------------------------------------------------------------
bool Pad::TestNapdStyleSet(int index )
{
    NN_UNUSED( index );
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    for(int i = 0; i < NPADID_NUM; i++)
    {
        if ( m_IsNpadDataAvailable && m_NpadState[i].Test( index ) )
        {
            return true;
        }
    }
#endif
    return false;
}

//---------------------------------------------------------------------------
//! @brief        毎フレームごとのアップデート
//---------------------------------------------------------------------------
bool Pad::Update( bool isSetFocus )
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    m_PressPrev = m_Press;

    PADRead( m_PADStatuses );
    PADClampCircle( m_PADStatuses );

    const PADStatus* padStatus = &(m_PADStatuses[m_Channel]);

    if ( padStatus->err == PAD_ERR_NONE )
    {
        u32 padHold = padStatus->button;

        m_Press = 0;

        // Holdの値を設定
        ChangeMask(MASK_A, padHold & PAD_BUTTON_A);
        ChangeMask(MASK_B, padHold & PAD_BUTTON_B);
        ChangeMask(MASK_X, padHold & PAD_BUTTON_X);
        ChangeMask(MASK_Y, padHold & PAD_BUTTON_Y);
        ChangeMask(MASK_R, padHold & PAD_TRIGGER_R);
        ChangeMask(MASK_L, padHold & PAD_TRIGGER_L);
        ChangeMask(MASK_Z, padHold & PAD_TRIGGER_Z);
        ChangeMask(MASK_START, padHold & PAD_BUTTON_START);
        ChangeMask(MASK_RIGHT, padHold & PAD_BUTTON_RIGHT);
        ChangeMask(MASK_LEFT, padHold & PAD_BUTTON_LEFT);
        ChangeMask(MASK_UP, padHold & PAD_BUTTON_UP);
        ChangeMask(MASK_DOWN, padHold & PAD_BUTTON_DOWN);

        // アナログスティックを設定
        NormalizeStickValueCafe( &leftX, &leftY, padStatus->stickX, padStatus->stickY );
        NormalizeStickValueCafe( &rightX, &rightY, padStatus->substickX, padStatus->substickY) ;

        m_IsConnected = true;
    }
    else
    {
        m_IsConnected = false;
    }

    m_Trigger = (m_Press ^ m_PressPrev) & m_Press;
    m_Release = (m_Press ^ m_PressPrev) & m_PressPrev;
#else
    m_PressPrev = m_Press;
    m_IsNpadDataAvailable = false;

    // キーボードの状態を取得する
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_Press.storage = 0;

    bool isGetState = false;
    if ( m_IsUseKeyboardForcus )
    {
        isGetState = isSetFocus;
    }
    else
    {
        if ( reinterpret_cast<HWND>( m_HWnd ) == GetForegroundWindow() ) isGetState = true;
    }

    if ( isGetState )
    {
        nn::hid::KeyboardState keyboardState;
        nn::hid::GetKeyboardState(&keyboardState);

        if ( m_IsPadEnabled )
        {
            nn::hid::GetDebugPadState(&m_PadState);

            m_Press.SetBit(IDX_A,      m_PadState.buttons.Test< ::nn::hid::DebugPadButton::A >()      || keyboardState.keys.Test< ::nn::hid::KeyboardKey::A >());
            m_Press.SetBit(IDX_B,      m_PadState.buttons.Test< ::nn::hid::DebugPadButton::B >()      || keyboardState.keys.Test< ::nn::hid::KeyboardKey::B >());
            m_Press.SetBit(IDX_X,      m_PadState.buttons.Test< ::nn::hid::DebugPadButton::X >()      || keyboardState.keys.Test< ::nn::hid::KeyboardKey::X >());
            m_Press.SetBit(IDX_Y,      m_PadState.buttons.Test< ::nn::hid::DebugPadButton::Y >()      || keyboardState.keys.Test< ::nn::hid::KeyboardKey::Y >());
            m_Press.SetBit(IDX_Z,      m_PadState.buttons.Test< ::nn::hid::DebugPadButton::ZR >()     || keyboardState.keys.Test< ::nn::hid::KeyboardKey::E >());
            m_Press.SetBit(IDX_L,      m_PadState.buttons.Test< ::nn::hid::DebugPadButton::L >()      || keyboardState.keys.Test< ::nn::hid::KeyboardKey::L >());
            m_Press.SetBit(IDX_R,      m_PadState.buttons.Test< ::nn::hid::DebugPadButton::R >()      || keyboardState.keys.Test< ::nn::hid::KeyboardKey::R >());
            m_Press.SetBit(IDX_START,  m_PadState.buttons.Test< ::nn::hid::DebugPadButton::Start >()  || keyboardState.keys.Test< ::nn::hid::KeyboardKey::S >());
            m_Press.SetBit(IDX_SELECT, m_PadState.buttons.Test< ::nn::hid::DebugPadButton::Select >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::D >());

            m_Press.SetBit(IDX_LEFT,   m_PadState.buttons.Test< ::nn::hid::DebugPadButton::Left >()   || keyboardState.keys.Test< ::nn::hid::KeyboardKey::LeftArrow >());
            m_Press.SetBit(IDX_UP,     m_PadState.buttons.Test< ::nn::hid::DebugPadButton::Up >()     || keyboardState.keys.Test< ::nn::hid::KeyboardKey::UpArrow >());
            m_Press.SetBit(IDX_RIGHT,  m_PadState.buttons.Test< ::nn::hid::DebugPadButton::Right >()  || keyboardState.keys.Test< ::nn::hid::KeyboardKey::RightArrow >());
            m_Press.SetBit(IDX_DOWN,   m_PadState.buttons.Test< ::nn::hid::DebugPadButton::Down >()   || keyboardState.keys.Test< ::nn::hid::KeyboardKey::DownArrow >());

            m_Press.SetBit(IDX_LS_SHIFT,   keyboardState.keys.Test< ::nn::hid::KeyboardKey::LeftShift >());
            m_Press.SetBit(IDX_RS_SHIFT,   keyboardState.keys.Test< ::nn::hid::KeyboardKey::RightShift >());

            m_Press.SetBit(IDX_0, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D0 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad0 >() );
            m_Press.SetBit(IDX_1, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D1 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad1 >() );
            m_Press.SetBit(IDX_2, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D2 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad2 >() );
            m_Press.SetBit(IDX_3, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D3 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad3 >() );
            m_Press.SetBit(IDX_4, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D4 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad4 >() );
            m_Press.SetBit(IDX_5, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D5 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad5 >() );
            m_Press.SetBit(IDX_6, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D6 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad6 >() );
            m_Press.SetBit(IDX_C, keyboardState.keys.Test< ::nn::hid::KeyboardKey::C >());
            m_Press.SetBit(IDX_E, keyboardState.keys.Test< ::nn::hid::KeyboardKey::E >());
            m_Press.SetBit(IDX_F, keyboardState.keys.Test< ::nn::hid::KeyboardKey::F >());
            m_Press.SetBit(IDX_G, keyboardState.keys.Test< ::nn::hid::KeyboardKey::G >());
        }
        else
        {
            m_Press.SetBit(IDX_A,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::A >());
            m_Press.SetBit(IDX_B,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::B >());
            m_Press.SetBit(IDX_X,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::X >());
            m_Press.SetBit(IDX_Y,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::Y >());
            m_Press.SetBit(IDX_Z,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::E >());
            m_Press.SetBit(IDX_L,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::L >());
            m_Press.SetBit(IDX_R,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::R >());
            m_Press.SetBit(IDX_START,  keyboardState.keys.Test< ::nn::hid::KeyboardKey::S >());
            m_Press.SetBit(IDX_SELECT, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D >());

            m_Press.SetBit(IDX_LEFT,    keyboardState.keys.Test< ::nn::hid::KeyboardKey::LeftArrow >());
            m_Press.SetBit(IDX_UP,      keyboardState.keys.Test< ::nn::hid::KeyboardKey::UpArrow >());
            m_Press.SetBit(IDX_RIGHT,   keyboardState.keys.Test< ::nn::hid::KeyboardKey::RightArrow >());
            m_Press.SetBit(IDX_DOWN,    keyboardState.keys.Test< ::nn::hid::KeyboardKey::DownArrow >());

            m_Press.SetBit(IDX_LS_SHIFT,   keyboardState.keys.Test< ::nn::hid::KeyboardKey::LeftShift >());
            m_Press.SetBit(IDX_RS_SHIFT,   keyboardState.keys.Test< ::nn::hid::KeyboardKey::RightShift >());
            m_Press.SetBit(IDX_0, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D0 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad0 >() );
            m_Press.SetBit(IDX_1, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D1 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad1 >() );
            m_Press.SetBit(IDX_2, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D2 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad2 >() );
            m_Press.SetBit(IDX_3, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D3 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad3 >() );
            m_Press.SetBit(IDX_4, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D4 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad4 >() );
            m_Press.SetBit(IDX_5, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D5 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad5 >() );
            m_Press.SetBit(IDX_6, keyboardState.keys.Test< ::nn::hid::KeyboardKey::D6 >() || keyboardState.keys.Test< ::nn::hid::KeyboardKey::NumPad6 >() );
            m_Press.SetBit(IDX_C, keyboardState.keys.Test< ::nn::hid::KeyboardKey::C >());
            m_Press.SetBit(IDX_E, keyboardState.keys.Test< ::nn::hid::KeyboardKey::E >());
            m_Press.SetBit(IDX_F, keyboardState.keys.Test< ::nn::hid::KeyboardKey::F >());
            m_Press.SetBit(IDX_G, keyboardState.keys.Test< ::nn::hid::KeyboardKey::G >());

        }
    }
#else
    nn::hid::GetDebugPadState(&m_PadState);
    GetNpadState();

    m_Press.storage = 0;
    m_Press.SetBit(IDX_A,      m_PadState.buttons.Test< nn::hid::DebugPadButton::A >()      || TestNapdStyleSet( nn::hid::NpadJoyButton::A::Index ) );
    m_Press.SetBit(IDX_B,      m_PadState.buttons.Test< nn::hid::DebugPadButton::B >()      || TestNapdStyleSet( nn::hid::NpadJoyButton::B::Index ) );
    m_Press.SetBit(IDX_X,      m_PadState.buttons.Test< nn::hid::DebugPadButton::X >()      || TestNapdStyleSet( nn::hid::NpadJoyButton::X::Index ) );
    m_Press.SetBit(IDX_Y,      m_PadState.buttons.Test< nn::hid::DebugPadButton::Y >()      || TestNapdStyleSet( nn::hid::NpadJoyButton::Y::Index ) );
    m_Press.SetBit(IDX_Z,      m_PadState.buttons.Test< nn::hid::DebugPadButton::ZR >()     || TestNapdStyleSet( nn::hid::NpadJoyButton::ZR::Index ) );
    m_Press.SetBit(IDX_L,      m_PadState.buttons.Test< nn::hid::DebugPadButton::L >()      || TestNapdStyleSet( nn::hid::NpadJoyButton::L::Index ) );
    m_Press.SetBit(IDX_R,      m_PadState.buttons.Test< nn::hid::DebugPadButton::R >()      || TestNapdStyleSet( nn::hid::NpadJoyButton::R::Index ) );
    m_Press.SetBit(IDX_START,  m_PadState.buttons.Test< nn::hid::DebugPadButton::Start >()  || TestNapdStyleSet( nn::hid::NpadJoyButton::Plus::Index ) );
    m_Press.SetBit(IDX_SELECT, m_PadState.buttons.Test< nn::hid::DebugPadButton::Select >() || TestNapdStyleSet( nn::hid::NpadJoyButton::Minus::Index ) );

    m_Press.SetBit(IDX_LEFT,   m_PadState.buttons.Test< nn::hid::DebugPadButton::Left >()   || TestNapdStyleSet( nn::hid::NpadJoyButton::Left::Index ) );
    m_Press.SetBit(IDX_UP,     m_PadState.buttons.Test< nn::hid::DebugPadButton::Up >()     || TestNapdStyleSet( nn::hid::NpadJoyButton::Up::Index ) );
    m_Press.SetBit(IDX_RIGHT,  m_PadState.buttons.Test< nn::hid::DebugPadButton::Right >()  || TestNapdStyleSet( nn::hid::NpadJoyButton::Right::Index ) );
    m_Press.SetBit(IDX_DOWN,   m_PadState.buttons.Test< nn::hid::DebugPadButton::Down >()   || TestNapdStyleSet( nn::hid::NpadJoyButton::Down::Index ) );
#endif

    m_Trigger.storage = (m_Press.storage ^ m_PressPrev.storage) & m_Press.storage;
    m_Release.storage = (m_Press.storage ^ m_PressPrev.storage) & m_PressPrev.storage;
#endif

    return true;
} // NOLINT(impl/function_size)

//---------------------------------------------------------------------------
//! @brief        マスク値で与えたボタンの何れかが押されているかを判定します。
//!
//! @param[in]    mask       判定するボタンのマスク値です。
//!
//! @return       押されている場合、 true を返します。
//---------------------------------------------------------------------------
bool Pad::IsHold(uint32_t mask) const
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    return (mask & m_Press) != 0;
#else
    return (mask & m_Press.storage) != 0;
#endif
}

//---------------------------------------------------------------------------
//! @brief        マスク値で与えたボタンの何れかがこのフレームに押されているかを判定します。
//!
//! @param[in]    mask       判定するボタンのマスク値です。
//!
//! @return       押されている場合、 true を返します。
//---------------------------------------------------------------------------
bool Pad::IsTrigger(uint32_t mask) const
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    return (mask & m_Trigger) != 0;
#else
    return (mask & m_Trigger.storage) != 0;
#endif
}

//---------------------------------------------------------------------------
//! @brief        マスク値で与えたボタンの何れかがこのフレームで離されているかを判定します。
//!
//! @param[in]    mask       判定するボタンのマスク値です。
//!
//! @return       離されている場合、 true を返します。
//---------------------------------------------------------------------------
bool Pad::IsRelease(uint32_t mask) const
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    return (mask & m_Release) != 0;
#else
    return (mask & m_Release.storage) != 0;
#endif
}

//---------------------------------------------------------------------------
//! @brief        左アナログスティックのX値を取得します。
//!
//! @return       左アナログスティックのX値を返します。
//---------------------------------------------------------------------------
float Pad::GetLeftStickX()
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    return leftX;
#else
    if ( m_IsNpadDataAvailable )
    {
        return static_cast< float >( m_AnalogStickL.x ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
    else
    {
        return static_cast< float >( m_PadState.analogStickL.x ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
#endif
}

//---------------------------------------------------------------------------
//! @brief        左アナログスティックのY値を取得します。
//!
//! @return       左アナログスティックのY値を返します。
//---------------------------------------------------------------------------
float Pad::GetLeftStickY()
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    return leftY;
#else

    if ( m_IsNpadDataAvailable )
    {
        return static_cast< float >( m_AnalogStickL.y ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
    else
    {
        return static_cast< float >( m_PadState.analogStickL.y ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
#endif
}

//---------------------------------------------------------------------------
//! @brief        右アナログスティックのX値を取得します。
//!
//! @return       右アナログスティックのX値を返します。
//---------------------------------------------------------------------------
float Pad::GetRightStickX()
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    return rightX;
#else
    if ( m_IsNpadDataAvailable )
    {
        return static_cast< float >( m_AnalogStickR.x ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
    else
    {
        return static_cast< float >( m_PadState.analogStickR.x ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
#endif
}

//---------------------------------------------------------------------------
//! @brief        右アナログスティックのY値を取得します。
//!
//! @return       右アナログスティックのY値を返します。
//---------------------------------------------------------------------------
float Pad::GetRightStickY()
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    return rightY;
#else
    if ( m_IsNpadDataAvailable )
    {
        return static_cast< float >( m_AnalogStickR.y ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
    else
    {
        return static_cast< float >( m_PadState.analogStickR.y ) / static_cast< float >( ::nn::hid::AnalogStickMax );
    }
#endif
}

}
}
