﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/mm/mm_MoviePlayerUtils.h>
/**
 * @brief
 * StreamReader utility class for reading data from client reader interface.
 *
 * @details
 * Purpose:
 *     The main purpose of StreamReader class is to provide a client specific file i/o implementation. The client needs to
 * provide implementation for ClientStreamReader class public interface. The implementation can be a memory, network or
 * any other file system.
 *
 */

bool StreamReader::Open(const char *filename, void** fileHandle)
{
    if( ( filename == nullptr ) || ( fileHandle == nullptr ) )
    {
        return false;
    }

    mFileHandle = new nn::fs::FileHandle;
    if( mFileHandle == NULL )
    {
        NN_SDK_LOG("Out of memory error for Stream handle");
        return false;
    }
    mFileHandle->handle = NULL;
    nn::Result result = nn::fs::OpenFile(mFileHandle, filename, nn::fs::OpenMode_Read);
    if( result.IsFailure() )
    {
        NN_SDK_LOG("Failed to open %s", filename);
        return false;
    }
    else
    {
        *fileHandle = mFileHandle;
        nn::fs::GetFileSize(&fileLength, *mFileHandle);
        NN_SDK_LOG("* Open file success and file length is :%d \n", fileLength);
        return true;
    }
}

size_t StreamReader::ReadAt(void* fileHandle, int64_t offset, void *data, size_t size)
{
    size_t dataRead = 0;
    if( ( fileHandle == nullptr ) || ( offset < 0 ) || ( data == nullptr ) || ( size <= 0 ))
    {
        return dataRead;
    }
    else
    {
        nn::fs::FileHandle fHandle;
        fHandle = *( nn::fs::FileHandle* )fileHandle;
        if( fHandle.handle == mFileHandle->handle )
        {
            nn::fs::ReadFile(&dataRead, fHandle, offset, data, size);
        }
        else
        {
            NN_SDK_LOG("* Invalid handle received\n");
        }
        return dataRead;
    }
}

bool StreamReader::Close(void* fileHandle)
{
    if( fileHandle != nullptr )
    {
        nn::fs::FileHandle fHandle;
        fHandle = *( nn::fs::FileHandle* )fileHandle;
        if( fHandle.handle == mFileHandle->handle )
        {
            nn::fs::CloseFile(fHandle);
            NN_SDK_LOG("File closed :%d \n", fHandle.handle);
            fHandle.handle = NULL;
            delete mFileHandle;
            return true;
        }
        else
        {
            NN_SDK_LOG("* Invalid handle received\n");
            return false;
        }
    }
    else
    {
        return false;
    }
}

void StreamReader::GetSize(void* fileHandle, int64_t *size)
{
    if( ( fileHandle != nullptr ) || ( size != nullptr ) )
    {
        nn::fs::FileHandle fHandle;
        fHandle = *( nn::fs::FileHandle* )fileHandle;
        if( fHandle.handle == mFileHandle->handle )
        {
            // nn::fs::GetFileSize() has performance implications.
            *size = fileLength;
        }
        else
        {
            NN_SDK_LOG("Invalid handle received\n");
        }
    }
}

/**
 * @brief
 * MediaClock utility class for AV sync.
 * @details
 *
 * Purpose:
 *     The main purpose of MediaClock class is to provide APIs to perform audio video synchronization for rendering
 * purpose. Same MediaClock instance is passed to Audio renderer and video renderer. Audio renderer will update anchor
 * time after each audio frame is presented to renderer. Video renderer will drop video frames if it is too late.
 *
 */
MediaClock::MediaClock()
    : m_Lock { false },
      m_AnchorTimeMediaUs(-1),
      m_AnchorTimeRealUs(-1),
      m_MaxTimeMediaUs(INT64_MAX),
      m_MediaStartTimeUs(-1)
{
}

void MediaClock::SetMediaStartTime(int64_t mediaStartTimeUs)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    m_MediaStartTimeUs = mediaStartTimeUs;
}

movie::Status MediaClock::UpdateAnchorTime(int64_t anchorTimeMediaUs, int64_t anchorTimeRealUs, int64_t maxTimeMediaUs)
{
    if( anchorTimeMediaUs < 0 || anchorTimeRealUs < 0 )
        return movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    int64_t nowUs = Clock::GetNowUs();
    int64_t nowMediaUs = anchorTimeMediaUs + ( nowUs - anchorTimeRealUs );
    if( nowMediaUs < 0 )
        return movie::Status_UnknownError;
    m_AnchorTimeRealUs = nowUs;
    m_AnchorTimeMediaUs = nowMediaUs;
    m_MaxTimeMediaUs = maxTimeMediaUs;
    return movie::Status_Success;
}

void MediaClock::ClearAnchorTime()
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    m_AnchorTimeMediaUs = -1;
    m_AnchorTimeRealUs = -1;
}

movie::Status MediaClock::GetRealTimeForMediaTime(int64_t targetMediaUs, int64_t *outRealUs)
{
    if( outRealUs == NULL )
        return movie::Status_ErrorBadValue;
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    int64_t nowUs = Clock::GetNowUs();
    int64_t nowMediaUs = 0;
    movie::Status status = GetMediaTime(nowUs, &nowMediaUs, true);
    if( status != movie::Status_Success )
        return status;
    *outRealUs = ( targetMediaUs - nowMediaUs ) + nowUs;
    return movie::Status_Success;
}

movie::Status MediaClock::GetMediaTime(int64_t realUs, int64_t *outMediaUs, bool allowPastMaxTime) const
{
    if( m_AnchorTimeRealUs == -1 )
        return movie::Status_UnknownError;
    int64_t mediaUs = m_AnchorTimeMediaUs + ( realUs - m_AnchorTimeRealUs );
    if( mediaUs > m_MaxTimeMediaUs && !allowPastMaxTime )
        mediaUs = m_MaxTimeMediaUs;
    if( mediaUs < m_MediaStartTimeUs )
        mediaUs = m_MediaStartTimeUs;
    if( mediaUs < 0 )
        mediaUs = 0;
    *outMediaUs = mediaUs;
    return movie::Status_Success;
}

/**
 * @brief
 * MemoryTracker utility class to track memory allocations.
 * @details
 *
 * Purpose:
 * The main purpose of MemoryTracker class is to track memory allocations by various allocators.
 * Note: This is a debug utility and will incur performance issues when used.
 * The tracker reports memory allocation statistics and any memory leaks.
 */
MemoryTracker::MemoryTracker()
    : m_Lock { false },
      m_CreationTime(nn::os::GetSystemTick()),
      m_Total(0),
      m_MaxSize(0),
      m_Id(0),
      m_InfoUsed{}
{
}

void MemoryTracker::Track(void *memory, size_t size, const char* threadName)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    for( int i = 0; i < s_MaxAllocations; ++i )
    {
        if( !m_InfoUsed[ i ] )
        {
            m_Infos[ i ] = AllocationInfo(size, m_Id++, memory, threadName);
            m_InfoUsed[ i ] = true;
            break;
        }
    }
    m_Total += size;
    m_MaxSize = std::max(m_MaxSize, m_Total);
}

void MemoryTracker::Untrack(void *memory)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( memory == nullptr )
        return;
    AllocationInfo *info = NULL;
    for( int i = 0; i < s_MaxAllocations; ++i )
    {
        if( m_Infos[ i ].address == memory )
        {
            info = m_Infos + i;
            m_InfoUsed[ i ] = false;
            break;
        }
    }
    if( info != nullptr )
        m_Total -= info->allocSize;
}

void MemoryTracker::OutputUsage()
{
    for( int i = 0; i < s_MaxAllocations; ++i )
    {
        if( m_InfoUsed[ i ] != NULL )
        {
            AllocationInfo *info = m_Infos + i;
            nn::TimeSpan t = ( info->time - m_CreationTime ).ToTimeSpan();
            NN_LOG("\n Memory Leaked !!!!!!! ");
            NN_LOG("\n Memory address = 0x%p, Size = %lu", info->address, info->allocSize);
            NN_LOG(" Allocation id = %lu, Allcoation time = %ld.%ld", info->allocId, t.GetSeconds(), t.GetMilliSeconds() % 1000);
            NN_LOG(" Thread name = \"%s\" \n", info->allocThreadName);
        }
    }
    NN_LOG(" Current size: %lu Max size: %lu \n", m_Total, m_MaxSize);
}

MemoryTracker::AllocationInfo::AllocationInfo(size_t size, size_t id, void *address, const char* threadName)
    : allocSize(size),
      allocId(id),
      address(address),
      time(nn::os::GetSystemTick())
{
    if( threadName != nullptr )
    {
        int length = strlen(threadName);
        if( length > nn::os::ThreadNameLengthMax )
            length = nn::os::ThreadNameLengthMax;
        memcpy(allocThreadName, threadName, length);
    }
}
