﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief
 * AudioOutRenderer class abstracting nn::audio::AudioOut
 *
 * @details
 *
 * Purpose:
 *     AudioOutRenderer class uses nn::audio::AudioOut for playing back of PCM audio data. AudioOut supports only
 * limited sampling rates. If the input sample rate fed to AudioOut is not unsupported, nn::audio::ResamplerType is
 * used for resampling input to supported sample rate. By default AudioOutRenderer plays PCM audio at 48kHz, all other
 * sample rates are resampled to 48kHz.
 *
 * Setup:
 *     A client using AudioOutRenderer needs to create an instance of the class and initialize
 * it. Default values are set during instantiation and initialization. Client can open the renderer with sample rate
 * and number of audio channels. Open() API will indicate whether a resampling is needed. If resampling is needed
 * client has to call resampling APIs before submitting data to AudioOut for rendering. When AudioOutRenderer Open()
 * API is called, it will create an instance of AudioOut. If resampling is needed resampler will be setup. Buffers
 * needed for AudioOut will be allocated. Zero filled empty buffers are sent to AudioOut.
 *
 * Main Processing:
 *    When client calls Start() API, AudioOutRenderer will start AudioOut. AudioOutRenderer does not have a thread.
 * Client need to query for empty buffers using GetEmptyAudioOutBuffer() API. Fill empty buffers with PCM data. If
 * needed submit PCM data for resampling using ResampleAudio() API, otherwise submit PCM data for rendering using
 * AppendFilledAudioOutBuffer() API Client can use GetPlayedOutAudioDurationUs() API to get audio samples played in
 * microseconds.
 *
 * Teardown:
 *     When a client is done submitting PCM data it should call Stop() API which internally stops AudioOut. Close() API
 * will release resources allocated by AudioOutRenderer.
 *
 */
#include <nns/mm/mm_AudioOutRenderer.h>
AudioOutRenderer::AudioOutRenderer()
    : m_Channels(-1),
    m_InputSampleRate(-1),
    m_InputChannels(-1),
    m_FrameSize(0),
    m_AudioOutStarted(false),
    m_AudioOutOpened(false),
    m_PresentationTimeUs(0),
    m_SampleSize(0),
    m_BufferSize(8192 * 4),
    m_AudioOutBuffer { },
    m_InBuffers { },
    m_ResamplerBuffer(nullptr),
    m_ResamplerOutBuffers { },
    m_ResampleAudio(false),
    m_OutputSampleRate(48000),
    m_OutputChannels(2),
    m_ResamplerOutBufferSize(0)
{
}

AudioOutRenderer::~AudioOutRenderer()
{
}

movie::Status AudioOutRenderer::Initialize()
{
    m_Channels = 0;
    m_FrameSize = sizeof(short);
    m_PresentationTimeUs = 0ll;
    m_SampleSize = 16;
    return movie::Status_Success;
}

movie::Status AudioOutRenderer::Open(uint32_t sampleRate, int channelCount, bool *resampleAudio, int audioFrameSize)
{
    nn::Result nnResult = nn::ResultSuccess();
    if( resampleAudio == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    m_InputSampleRate = sampleRate;
    m_InputChannels = channelCount;
    m_OutputSampleRate = sampleRate;
    m_OutputChannels = channelCount;
    int defaultSampleRate = 48000;
    m_FrameSize = m_Channels * sizeof(short);
    if( channelCount > m_Channels )
    {
        m_Channels = channelCount;
    }
    nn::audio::AudioOutParameter parameter{};
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = defaultSampleRate;
    parameter.channelCount = m_Channels;

    nnResult = nn::audio::OpenDefaultAudioOut(&m_AudioOut, parameter);
    if( nnResult.IsSuccess() )
    {
        m_Channels = nn::audio::GetAudioOutChannelCount(&m_AudioOut);
    }
    m_SampleFormat = nn::audio::GetAudioOutSampleFormat(&m_AudioOut);
    m_SampleSize = nn::audio::GetSampleByteSize(m_SampleFormat);
    int audioOutSampleRate = nn::audio::GetAudioOutSampleRate(&m_AudioOut);
    if( m_InputSampleRate != audioOutSampleRate )
    {
        size_t resamplerBufferSize = nn::audio::GetRequiredBufferSizeForResampler(m_OutputChannels);

        m_ResamplerBuffer = aligned_alloc(nn::audio::AudioOutBuffer::AddressAlignment, resamplerBufferSize);
        if( m_ResamplerBuffer == nullptr )
        {
            return movie::Status_OutOfMemory;
        }
        memset(m_ResamplerBuffer, 0, resamplerBufferSize);
        nn::audio::InitializeResampler(&m_Resampler, m_ResamplerBuffer, resamplerBufferSize, m_InputSampleRate, defaultSampleRate, m_OutputChannels);
        m_ResampleAudio = true;
    }

    // Allocate audio out buffers and submit to AudioOut
    size_t resamplerOutBufferSize = 0;
    if( m_ResampleAudio == true )
    {
        const int inFrameSampleCount = audioFrameSize / m_SampleSize;
        const auto outSampleCount = nn::audio::GetResamplerOutputSampleCount(&m_Resampler, inFrameSampleCount);
        resamplerOutBufferSize = nn::util::align_up(outSampleCount * sizeof(int16_t) * m_Channels, nn::audio::AudioOutBuffer::SizeGranularity);
        m_ResamplerOutBufferSize = resamplerOutBufferSize;
    }
    int dataSize = 516;
    int numOutSamples = 0;
    for( int i = 0; i < s_NumBuffers; ++i )
    {
        m_InBuffers[ i ] = aligned_alloc(nn::audio::AudioOutBuffer::AddressAlignment, m_BufferSize);
        if( m_InBuffers[ i ] == nullptr )
        {
            return movie::Status_OutOfMemory;
        }
        memset(m_InBuffers[ i ], 0, m_BufferSize);
        if( m_ResampleAudio == true )
        {
            m_ResamplerOutBuffers[ i ] = aligned_alloc(nn::audio::AudioOutBuffer::AddressAlignment, resamplerOutBufferSize);
            if( m_ResamplerOutBuffers[ i ] == nullptr )
            {
                return movie::Status_OutOfMemory;
            }
            memset(m_ResamplerOutBuffers[ i ], 0, resamplerOutBufferSize);
            nn::audio::ProcessResamplerBuffer(&m_Resampler, &numOutSamples, static_cast<int16_t*>( m_ResamplerOutBuffers[ i ] ), resamplerOutBufferSize, static_cast<const int16_t*>( m_InBuffers[ i ] ), dataSize);
            int outputDataSize = numOutSamples * channelCount * static_cast<int>( nn::audio::GetSampleByteSize(m_SampleFormat) );
            nn::audio::SetAudioOutBufferInfo(&m_AudioOutBuffer[ i ], m_ResamplerOutBuffers[ i ], resamplerOutBufferSize, outputDataSize);
        }
        else
        {
            nn::audio::SetAudioOutBufferInfo(&m_AudioOutBuffer[ i ], m_InBuffers[ i ], m_BufferSize, dataSize);
        }
        nn::audio::AppendAudioOutBuffer(&m_AudioOut, &m_AudioOutBuffer[ i ]);
    }
    *resampleAudio = m_ResampleAudio;
    m_AudioOutOpened = true;
    return movie::Status_Success;
}

movie::Status AudioOutRenderer::ResampleAudio(void *resamplerBuffer, void *inputDataBuffer, size_t inputDataSize, size_t *outputDataSize)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( ( resamplerBuffer == nullptr ) || ( inputDataBuffer == nullptr ) || ( outputDataSize == nullptr ) )
    {
        return movie::Status_ErrorBadValue;
    }
    int numOutSamples = 0;
    if( m_ResampleAudio == true )
    {
        nn::audio::ProcessResamplerBuffer(&m_Resampler, &numOutSamples, static_cast<int16_t*>( resamplerBuffer ), m_ResamplerOutBufferSize, static_cast<const int16_t*>( inputDataBuffer ), inputDataSize / ( m_OutputChannels * 2 ));
        *outputDataSize =  numOutSamples * m_OutputChannels * static_cast<int>( nn::audio::GetSampleByteSize(m_SampleFormat) );
        movieStatus = movie::Status_Success;
    }
    return movieStatus;
}

movie::Status AudioOutRenderer::GetResampledOutBufferSize(size_t *size)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( size == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    if( m_ResampleAudio == true )
    {
        *size = m_ResamplerOutBufferSize;
        movieStatus = movie::Status_Success;
    }
    return movieStatus;
}

movie::Status AudioOutRenderer::Start()
{
    nn::Result nnResult = nn::ResultSuccess();
    movie::Status movieStatus = movie::Status_UnknownError;

    nnResult = nn::audio::StartAudioOut(&m_AudioOut);
    if( nnResult.IsSuccess() )
    {
        movieStatus = movie::Status_Success;
        m_AudioOutStarted = true;
    }
    return movieStatus;
}

movie::Status AudioOutRenderer::GetEmptyAudioOutBuffer(nn::audio::AudioOutBuffer** audioOutBuffer, void **dataBuffer, size_t *outSize)
{
    movie::Status movieStatus = movie::Status_OutputBufferNotAvailable;
    if( ( audioOutBuffer != nullptr ) && ( dataBuffer != nullptr ) && ( outSize != nullptr ) )
    {
        // Check whether AudioOut has any empty buffer available
        *audioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&m_AudioOut);
        if( *audioOutBuffer == nullptr )
        {
            *dataBuffer = nullptr;
            *outSize = 0;
            return movieStatus;
        }
        *dataBuffer = nn::audio::GetAudioOutBufferDataPointer(*audioOutBuffer);
        *outSize = nn::audio::GetAudioOutBufferDataSize(*audioOutBuffer);
        if( ( *dataBuffer == nullptr ) || ( *outSize <= 0 ) )
        {
            return movieStatus;
        }
        return movie::Status_Success;
    }
    return movieStatus;
}

movie::Status AudioOutRenderer::AppendFilledAudioOutBuffer(nn::audio::AudioOutBuffer* audioOutBuffer, void *buffer, size_t size, size_t FilledSize)
{
    movie::Status movieStatus = movie::Status_Success;

    // Set new audio buffer to AudioOut for playback
    if( (audioOutBuffer != nullptr) && ( buffer != nullptr ) && ( size > 0 ) )
    {
        nn::audio::SetAudioOutBufferInfo(audioOutBuffer, buffer, size, FilledSize);
        nn::audio::AppendAudioOutBuffer(&m_AudioOut, audioOutBuffer);
    }
    return movieStatus;
}

movie::Status AudioOutRenderer::Stop()
{
    // Stop AudioOut.
    if( ( m_AudioOutOpened == true ) && ( m_AudioOutStarted == true ) )
    {
        nn::audio::StopAudioOut(&m_AudioOut);
        m_AudioOutStarted = false;
    }
    return movie::Status_Success;
}

movie::Status AudioOutRenderer::Close()
{
    // Close AudioOut.
    if( m_AudioOutOpened == true )
    {
        nn::audio::CloseAudioOut(&m_AudioOut);
        m_AudioOutOpened = false;
    }
    if( m_ResamplerBuffer != nullptr )
    {
        free(m_ResamplerBuffer);
        m_ResamplerBuffer = nullptr;
    }
    for( int i = 0; i < s_NumBuffers; ++i )
    {
        if( m_InBuffers[ i ] != nullptr )
        {
            free(m_InBuffers[ i ]);
            m_InBuffers[ i ] = nullptr;
        }
        if( m_ResamplerOutBuffers[ i ] != nullptr )
        {
            free(m_ResamplerOutBuffers[ i ]);
            m_ResamplerOutBuffers[ i ] = nullptr;
        }
    }
    return movie::Status_Success;
}

int64_t AudioOutRenderer::GetPlayedOutAudioDurationUs(int64_t nowUs)
{
    return nowUs;
}
