﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Button.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerAddon.h>
#include <nns/hid/hid_ControllerManager.h>

namespace nns { namespace hid {

const float Controller::DefaultStickDpadDownThreshold = 0.5f;

const float Controller::DefaultStickDpadUpThreshold = 0.25f;

const nn::util::Float2 Controller::InvalidPointer =
{
    {
        {
            std::numeric_limits<float>::lowest(),
            std::numeric_limits<float>::lowest()
        }
    }
};

Controller::Controller(ControllerManager* pManager) NN_NOEXCEPT
    : m_pManager(pManager)
    , m_Pointer(InvalidPointer)
    , m_LeftStickDpadDownThreshold(DefaultStickDpadDownThreshold)
    , m_RightStickDpadDownThreshold(DefaultStickDpadDownThreshold)
    , m_LeftStickDpadUpThreshold(DefaultStickDpadUpThreshold)
    , m_RightStickDpadUpThreshold(DefaultStickDpadUpThreshold)
    , m_IsPointerBoundaryDefined(false)
    , m_PointerBoundaryMin(InvalidPointer)
    , m_PointerBoundaryMax(InvalidPointer)
{
    NN_ASSERT_NOT_NULL(m_pManager);
    m_Buttons.Reset();
    m_ButtonsDown.Reset();
    m_ButtonsUp.Reset();
    m_ButtonsRepeat.Reset();
    m_LeftStick.x = 0.0f;
    m_LeftStick.y = 0.0f;
    m_RightStick.x = 0.0f;
    m_RightStick.y = 0.0f;
    m_PointerPhase.Reset();
    for (size_t i = 0; i < static_cast<size_t>(ButtonCountMax); ++i)
    {
        m_ButtonsDurationCounts[i] = 0;
        m_RepeatDelayCounts[i] = DefaultRepeatDelayCount;
        m_RepeatIntervalCounts[i] = DefaultRepeatIntervalCount;
    }
}

Controller::~Controller() NN_NOEXCEPT {}

ControllerAddon* Controller::GetControllerAddon(
    ControllerAddonId id, int index) const NN_NOEXCEPT
{
    // 添字が範囲外ならば NULL を返します。
    if (index < 0 && m_ControllerAddons.size() <= static_cast<size_t>(index))
    {
        return 0;
    }

    for (std::vector<ControllerAddon*>::const_iterator it =
             m_ControllerAddons.begin(); it != m_ControllerAddons.end(); ++it)
    {
        ControllerAddon* pControllerAddon = *it;

        // 指定のコントローラ識別子を持つか判定します。
        if (pControllerAddon->GetControllerAddonId() == id)
        {
            // 目的の添字に到達したならばコントローラを返します。
            if (index == 0)
            {
                return pControllerAddon;
            }

            // コントローラが見つかったので添字を進めます。
            --index;
        }
    }

    // 条件を満たすコントローラが見つからなかった場合は NULL を返します。
    return 0;
}

void Controller::SetButtonsRepeatOption(
    ButtonSet buttons, uint8_t delayCount, uint8_t intervalCount) NN_NOEXCEPT
{
    for (int i = 0; i < ButtonCountMax; ++i)
    {
        if (buttons.Test(i))
        {
            // 指定されたデジタルボタンにリピートを開始するまでのフレーム数を設定します。
            m_RepeatDelayCounts[i] = delayCount;

            // 指定されたデジタルボタンにリピートの間隔となるフレーム数を設定します。
            m_RepeatIntervalCounts[i] = intervalCount;
        }
    }
}

void Controller::SetLeftStickDpadThreshold(float down, float up) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(down, 0.0f, 1.0f);
    NN_ASSERT_MINMAX(up, 0.0f, 1.0f);
    NN_ASSERT_LESS_EQUAL(up, down);
    m_LeftStickDpadDownThreshold = down;
    m_LeftStickDpadUpThreshold = up;
}

void Controller::SetRightStickDpadThreshold(float down, float up) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(down, 0.0f, 1.0f);
    NN_ASSERT_MINMAX(up, 0.0f, 1.0f);
    NN_ASSERT_LESS_EQUAL(up, down);
    m_RightStickDpadDownThreshold = down;
    m_RightStickDpadUpThreshold = up;
}

void Controller::SetPointerBoundary(
    const nn::util::Float2& min, const nn::util::Float2& max) NN_NOEXCEPT
{
    NN_ASSERT_GREATER(min.x, std::numeric_limits<float>::lowest());
    NN_ASSERT_GREATER(min.y, std::numeric_limits<float>::lowest());
    NN_ASSERT_GREATER_EQUAL(max.x, min.x);
    NN_ASSERT_GREATER_EQUAL(max.y, min.y);

    // 有効範囲が再設定されたことを記録します。
    m_PointerPhase.Set<PointerPhase::Reconfigured>(true);

    // 有効範囲の状態を定義済みに変更します。
    m_IsPointerBoundaryDefined = true;

    // 有効範囲を設定します。
    m_PointerBoundaryMin = min;
    m_PointerBoundaryMax = max;
}

void Controller::UpdateBase(
    ButtonSet buttons,
    const nn::util::Float2& leftStick, const nn::util::Float2& rightStick,
    bool isTouched, const nn::util::Float2& pointer) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(leftStick.x, -1.0f, 1.0f);
    NN_ASSERT_MINMAX(leftStick.y, -1.0f, 1.0f);
    NN_ASSERT_MINMAX(rightStick.x, -1.0f, 1.0f);
    NN_ASSERT_MINMAX(rightStick.y, -1.0f, 1.0f);

    // ポインティングデバイスの座標が有効範囲内にあるか否かを判定します。
    bool isInside = this->IsInsidePointerBoundary(pointer);

    // 左スティックの十字キー割り当てを実施します。
    buttons |= this->GetDpadMask(Button::LsUp::Index) &
               this->GetStickDpad(
                   leftStick,
                   m_LeftStickDpadDownThreshold,
                   m_LeftStickDpadUpThreshold,
                   m_Buttons,
                   Button::LsUp::Index);

    // 右スティックの十字キー割り当てを実施します。
    buttons |= this->GetDpadMask(Button::RsUp::Index) &
               this->GetStickDpad(
                   rightStick,
                   m_RightStickDpadDownThreshold,
                   m_RightStickDpadUpThreshold,
                   m_Buttons,
                   Button::RsUp::Index);

    // ポインティングデバイスのデジタルボタン割り当てが必要ならば実施します。
    if (isInside)
    {
        buttons.Set<Button::Touch>(isTouched);
    }

    // 新たに押下されたデジタルボタンを最新の値に更新します。
    m_ButtonsDown = ~m_Buttons & buttons;

    // 新たに開放されたデジタルボタンを最新の値に更新します。
    m_ButtonsUp = m_Buttons & ~buttons;

    // 押下中のデジタルボタンを最新の値に更新します。
    m_Buttons = buttons;

    // デジタルボタンのリピート状態をリセットします。
    m_ButtonsRepeat.Reset();

    for (int i = 0; i < ButtonCountMax; ++i)
    {
        if (buttons.Test(i))
        {
            // リピートが必要ならば割り当てます。
            if (this->NeedsRepeat(i))
            {
                m_ButtonsRepeat.Set(i, true);
            }

            // デジタルボタンが押下状態であれば経過フレーム数を更新します。
            ++m_ButtonsDurationCounts[i];
        }
        else
        {
            // デジタルボタンが押下状態でなければ経過フレーム数をリセットします。
            m_ButtonsDurationCounts[i] = 0;
        }
    }

    // スティックの座標を更新します。
    m_LeftStick = leftStick;
    m_RightStick = rightStick;

    // ポインティングデバイスの状態を更新します。
    this->UpdatePointer(pointer, isInside);

    // 管理対象のアドオンを更新します。
    for (std::vector<ControllerAddon*>::iterator it = m_ControllerAddons.begin();
         it != m_ControllerAddons.end(); ++it)
    {
        (*it)->Update();
    }
}

ButtonSet Controller::GetStickDpad(
    const nn::util::Float2& stick, float down, float up,
    ButtonSet prevButtons,
    int dpadPosition) const NN_NOEXCEPT
{
    const float length2 = stick.x * stick.x + stick.y * stick.y;

    // 開放状態と見做す閾値未満であれば十字キーは押下されません。
    if (length2 < up * up)
    {
        return ButtonSet();
    }

    // 押下状態と見做す閾値未満かつ前回も押下状態でなかったならば十字キーは押下されません。
    if ((length2 < down * down) &&
        (!prevButtons.Test(dpadPosition + Dpad_Up) &&
         !prevButtons.Test(dpadPosition + Dpad_Down) &&
         !prevButtons.Test(dpadPosition + Dpad_Left) &&
         !prevButtons.Test(dpadPosition + Dpad_Right)))
    {
        return ButtonSet();
    }

    // スティックの座標を時計回りに 22.5 度回転させます。
    const float x = stick.x * 0.9238795f - stick.y * 0.3826834f;
    const float y = stick.x * 0.3826834f + stick.y * 0.9238795f;

    // 座標系を 8 つの領域に分割して十字キーの押下状態へ割り当てます。
    ButtonSet buttons = {};
    if (x >= 0.0f)
    {
        if (y >= 0.0f)
        {
            buttons.Set(dpadPosition + Dpad_Right, true);
            if (x < y)
            {
                buttons.Set(dpadPosition + Dpad_Up, true);
            }
        }
        else
        {
            buttons.Set(dpadPosition + Dpad_Down, true);
            if (x >= -y)
            {
                buttons.Set(dpadPosition + Dpad_Right, true);
            }
        }
    }
    else
    {
        if (y >= 0.0f)
        {
            buttons.Set(dpadPosition + Dpad_Up, true);
            if (-x >= y)
            {
                buttons.Set(dpadPosition + Dpad_Left, true);
            }
        }
        else
        {
            buttons.Set(dpadPosition + Dpad_Left, true);
            if (x > y)
            {
                buttons.Set(dpadPosition + Dpad_Down, true);
            }
        }
    }

    return buttons;
}

ButtonSet Controller::GetDpadMask(int dpadPosition) const NN_NOEXCEPT
{
    // 十字キーが割り当てられていなければ 0 を返します。
    if (dpadPosition < 0)
    {
        return ButtonSet();
    }

    // 指定位置の十字キーのビットを全てオンにします。
    ButtonSet buttons = {};
    buttons.Set(dpadPosition + Dpad_Up, true);
    buttons.Set(dpadPosition + Dpad_Down, true);
    buttons.Set(dpadPosition + Dpad_Left, true);
    buttons.Set(dpadPosition + Dpad_Right, true);

    return buttons;
}

bool Controller::NeedsRepeat(int position) const NN_NOEXCEPT
{
    NN_ASSERT_LESS(position, ButtonCountMax);

    const uint32_t duration = m_ButtonsDurationCounts[position];
    const uint8_t delay = m_RepeatDelayCounts[position];
    const uint8_t interval = m_RepeatIntervalCounts[position];

    // 指定されたデジタルボタンにリピートが設定されているか確認します。
    if (interval > 0)
    {
        // リピート条件を満たしているか判定します。
        if (duration >= delay && (duration - delay) % interval == 0)
        {
            return true;
        }
    }

    return false;
}

bool Controller::IsInsidePointerBoundary(const nn::util::Float2& pointer
                                         ) const NN_NOEXCEPT
{
    // ポインティングデバイスの有効範囲が設定されているか確認します。
    if (m_IsPointerBoundaryDefined)
    {
        // 指定された座標がポインティングデバイスの有効範囲に入っているか判定します。
        if (m_PointerBoundaryMin.x <= pointer.x &&
            m_PointerBoundaryMin.y <= pointer.y &&
            pointer.x <= m_PointerBoundaryMax.x &&
            pointer.y <= m_PointerBoundaryMax.y)
        {
            return true;
        }
    }

    return false;
}

void Controller::UpdatePointer(const nn::util::Float2& pointer,
                               bool isInside) NN_NOEXCEPT
{
    if (isInside)
    {
        //!< ポインティングデバイスの座標を最新の値に更新します。
        m_Pointer = pointer;
    }
    else if (m_PointerPhase.Test<PointerPhase::Reconfigured>())
    {
        //!< 有効範囲が再設定されている場合は過去の座標を無効化します。
        m_Pointer = GetInvalidPointer();
    }

    //!< ポインティングデバイスのフェーズを最新の値に更新します。
    m_PointerPhase.Set<PointerPhase::Began>(
        isInside && !m_PointerPhase.Test<PointerPhase::On>());
    m_PointerPhase.Set<PointerPhase::Ended>(
        !isInside && m_PointerPhase.Test<PointerPhase::On>());
    m_PointerPhase.Set<PointerPhase::On>(isInside);
    m_PointerPhase.Set<PointerPhase::Reconfigured>(false);
}

}} // namespace nns::hid
