﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfxLogPrimitiveRendererGfxRes.h"
#include "gfxLogPrimitiveRendererShaderRes.h"

namespace nns
{
namespace gfxLog
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! 実装です。
//------------------------------------------------------------------------------
uint32_t Shader::GetDefaultVertexFormat(
    ShaderVariation shaderVariation
) NN_NOEXCEPT
{
    uint32_t vertexFormat = 0;
    switch( shaderVariation )
    {
    case ShaderVariation_VertexColor:
        vertexFormat = VertexFormat_Pos | VertexFormat_Color;
        break;
    case ShaderVariation_SolidColor:
        vertexFormat = VertexFormat_Pos;
        break;
    case ShaderVariation_Texture:
        vertexFormat = VertexFormat_Pos | VertexFormat_Uv | VertexFormat_Color;
        break;
    case ShaderVariation_TextureArray:
        vertexFormat = VertexFormat_Pos | VertexFormat_Uv | VertexFormat_Color;
        break;
    case ShaderVariation_Cubemap:
        vertexFormat = VertexFormat_Pos | VertexFormat_Uv | VertexFormat_Color;
        break;
    case ShaderVariation_CountMax:
    default: NN_UNEXPECTED_DEFAULT;
    }
    return vertexFormat;
}

bool Shader::Create(
    Shader* pShader,
    nn::gfx::ResShaderFile* pResShaderFile,
    int variationIndex,
    uint32_t vertexFormat,
    nn::gfx::Device* pGfxDevice,
    nn::util::BytePtr& pMemory ) NN_NOEXCEPT
{
#if NN_GFX_IS_TARGET_GL
    nn::gfx::ShaderCodeType shaderCodeType = nn::gfx::ShaderCodeType_Source;
#elif NN_GFX_IS_TARGET_NVN
    nn::gfx::ShaderCodeType shaderCodeType = nn::gfx::ShaderCodeType_Binary;
#else
#error Unsupported Platform
#endif

    nn::gfx::ResShaderContainer* pContainer = pResShaderFile->GetShaderContainer();
    nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation(variationIndex);
    if( NULL != pVariation )
    {
        nn::gfx::Shader* pGfxShader =
            pVariation->GetResShaderProgram( shaderCodeType )->GetShader();
        NN_ASSERT_NOT_NULL(pGfxShader);

        pShader->SetVertexShader( pGfxShader );
        pShader->SetPixelShader( pGfxShader );

        // リソースから頂点ステートを生成します。
        if (!CreateVertexState( pShader,
                                vertexFormat,
                                pGfxShader,
                                pGfxDevice,
                                pMemory))
        {
            return false;
        }

        pShader->UpdateSystemSlots();
        return true;
    }
    return false;
}

int Shader::InitializeVertexStateInfo(
    nn::gfx::VertexState::InfoType*      info,
    nn::gfx::VertexAttributeStateInfo*   attribs,
    nn::gfx::VertexBufferStateInfo*      buffers,
    uint32_t                             vertexFormat,
    nn::gfx::Shader*                     pVertexShader ) NN_NOEXCEPT
{
    int attribCount = 0;

    info->SetDefault();

    int slot = -1;

    if ( pVertexShader != NULL )
    {
        slot = pVertexShader->GetInterfaceSlot(
            nn::gfx::ShaderStage_Vertex,
            nn::gfx::ShaderInterfaceType_Input,
            "i_Position"
        );
    }
    else
    {
        slot = 0;
    }

    if( vertexFormat & VertexFormat_Pos && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_Position" );
        attrib.SetBufferIndex( 0 );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attrib.SetOffset( 0 );

        nn::gfx::VertexBufferStateInfo& buffer = buffers[ attribCount ];
        buffer.SetDefault();
        buffer.SetStride( sizeof(nn::util::Float3) );

        attribCount++;
    }

    if ( pVertexShader != NULL )
    {
        slot = pVertexShader->GetInterfaceSlot(
            nn::gfx::ShaderStage_Vertex,
            nn::gfx::ShaderInterfaceType_Input,
            "i_Color"
        );
    }
    else
    {
        slot = 0;
    }

    if( vertexFormat & VertexFormat_Color && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_Color" );
        attrib.SetBufferIndex( 1 );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_32_32_Float );
        attrib.SetOffset( 0 );

        nn::gfx::VertexBufferStateInfo& buffer = buffers[ attribCount ];
        buffer.SetDefault();
        buffer.SetStride( sizeof(nn::util::Float4) );

        attribCount++;
    }

    if ( pVertexShader != NULL )
    {
        slot = pVertexShader->GetInterfaceSlot(
            nn::gfx::ShaderStage_Vertex,
            nn::gfx::ShaderInterfaceType_Input,
            "i_TexCoord"
        );
    }
    else
    {
        slot = 0;
    }

    if( vertexFormat & VertexFormat_Uv && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_TexCoord" );
        attrib.SetBufferIndex( 2 );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attrib.SetOffset( 0 );

        nn::gfx::VertexBufferStateInfo& buffer = buffers[ attribCount ];
        buffer.SetDefault();
        buffer.SetStride( sizeof(nn::util::Float2) );

        attribCount++;
    }

    if( 0 == attribCount )
    {
        return 0;
    }

    info->SetVertexAttributeStateInfoArray( attribs, attribCount );
    info->SetVertexBufferStateInfoArray( buffers, attribCount );

    return attribCount;
}

bool Shader::CreateVertexState(
    Shader* pShader,
    uint32_t vertexFormat,
    nn::gfx::Shader* pVertexShader,
    nn::gfx::Device* pGfxDevice,
    nn::util::BytePtr& pMemory ) NN_NOEXCEPT
{
    nn::gfx::VertexState::InfoType info;
    nn::gfx::VertexAttributeStateInfo attribs[ VertexAttribute_CountMax ];
    // インタリーブではなく、属性ごとに頂点バッファを持ちます。
    nn::gfx::VertexBufferStateInfo buffers[ VertexAttribute_CountMax ];

    int attribCount = InitializeVertexStateInfo(
        &info,
        attribs,
        buffers,
        vertexFormat,
        pVertexShader
    );
    if ( attribCount == 0 )
    {
        return false;
    }

    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );

    pShader->m_VertexState.SetMemory( pMemory.Get(), size );
    pMemory.Advance( size );
    pShader->m_VertexState.Initialize( pGfxDevice, info, pVertexShader );
    return true;
}

void Shader::Cleanup( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT
{
    m_VertexState.Finalize( pGfxDevice );
    SetDefault();
}

Shader::Shader() NN_NOEXCEPT
    :   m_VertexFormat(VertexFormat_Default),
        m_ShaderVariationIndex(0),
        m_ShaderSlotCount(0),
        m_SlotView(0)
{
    for (int i = 0; i < Constants_ModelSlotCountMax; i++)
    {
        m_SlotModel[i] = 0;
    }
    for (int i = 0; i < Constants_UserSlotCountMax; i++)
    {
        m_SlotUser[i] = 0;
    }
    for (int i = 0; i < Constants_SamplerSlotCountMax; i++)
    {
        m_SlotSamplerPs[i] = 0;
    }
    for (int i = 0; i < nn::gfx::ShaderStage_End; i++)
    {
        m_pGfxShader[i] = NULL;
    }
    SetDefault();
}

void Shader::SetDefault() NN_NOEXCEPT
{
    for( int idx = 0; idx < nn::gfx::ShaderStage_End; ++idx )
    {
        m_pGfxShader[ idx ] = NULL;
    }

    m_SlotView = ShaderSlotNone;
    for( int idx = 0; idx < Constants_ModelSlotCountMax; ++idx )
    {
        m_SlotModel[ idx ] = ShaderSlotNone;
    }
    for( int idx = 0; idx < Constants_UserSlotCountMax; ++idx )
    {
        m_SlotUser[ idx ] = ShaderSlotNone;
    }
}

void Shader::SetVertexShader( nn::gfx::Shader* pGfxShader ) NN_NOEXCEPT
{
    m_pGfxShader[ nn::gfx::ShaderStage_Vertex ] = pGfxShader;
}

void Shader::SetPixelShader( nn::gfx::Shader* pGfxShader ) NN_NOEXCEPT
{
    m_pGfxShader[ nn::gfx::ShaderStage_Pixel ] = pGfxShader;
}

nn::gfx::Shader* Shader::GetVertexShader() NN_NOEXCEPT
{
    return m_pGfxShader[ nn::gfx::ShaderStage_Vertex ];
}

nn::gfx::Shader* Shader::GetPixelShader() NN_NOEXCEPT
{
    return m_pGfxShader[ nn::gfx::ShaderStage_Pixel ];
}

const nn::gfx::VertexState* Shader::GetVertexState() const NN_NOEXCEPT
{
    return &m_VertexState;
}

void Shader::UpdateSystemSlots() NN_NOEXCEPT
{
    // スロットを取得します。
    nn::gfx::Shader* pVertexShader = m_pGfxShader[ nn::gfx::ShaderStage_Vertex ];
    nn::gfx::Shader* pPixelShader = m_pGfxShader[ nn::gfx::ShaderStage_Pixel ];

    if( pVertexShader != NULL )
    {
        m_SlotView = pVertexShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Vertex,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "View" );

        m_SlotModel[0] = pVertexShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Vertex,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "Model" );

        m_SlotUser[0] = pVertexShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Vertex,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "User" );
    }

    if( pPixelShader != NULL )
    {
        m_SlotModel[1] = pPixelShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "Model" );

        m_SlotUser[1] = pPixelShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "User" );

        m_SlotSamplerPs[0] = pPixelShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_Sampler, "texture0" );

        m_SlotSamplerPs[1] = pPixelShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_Sampler, "textureArray" );

        m_SlotSamplerPs[2] = pPixelShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_Sampler, "textureCubemap" );
    }
}

int Shader::GetSlotView() const NN_NOEXCEPT
{
    return m_SlotView;
}

int Shader::GetSlotModel(int index) const NN_NOEXCEPT
{
    return m_SlotModel[index];
}

int Shader::GetSlotSamplerPs(int index) const NN_NOEXCEPT
{
    return m_SlotSamplerPs[index];
}

} // namespace PrimitiveRenderer
} // namespace gfxLog
} // namespace nns
