﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Log.h>
#include "gfxLogPrimitiveRendererGfxRes.h"
#include "gfxLogPrimitiveRendererMeshRes.h"

namespace nns
{
namespace gfxLog
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! 内部変数です。
//------------------------------------------------------------------------------
namespace
{

const size_t VertexStrideSizeTable[ VertexAttribute_CountMax ] =
{
    sizeof(nn::util::Float3), // pos
    sizeof(nn::util::Float4), // color
    sizeof(nn::util::Float2), // uv
};

// 頂点数
const int VertexQuadCount = 4;

const int VertexCounts[ ShapeType_CountMax ] =
{
    VertexQuadCount,            // solid quad
};

// インデックス数
const int IndexSolidQuadCount = 6;
const int IndexWiredQuadCount = 5;

const int IndexCounts[ ShapeType_CountMax ] =
{
    IndexSolidQuadCount,           // solid quad
};
} // namespace


//------------------------------------------------------------------------------
//! 内部関数の実装です。
//------------------------------------------------------------------------------

//! @brief コンストラクタです。
PrimitiveMesh::PrimitiveMesh() NN_NOEXCEPT
:   m_pGpuBuffer(NULL),
    m_pIndexBufferPointer(NULL),
    m_VertexCount(0),
    m_IndexCount(0)
{
    for( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        m_pVertexBufferPointer[idx] = NULL;
    }
}

bool PrimitiveMesh::Initialize( GpuBuffer* pGpuBuffer,
                                int numVertices,
                                int numIndices,
                                VertexFormat vertexFormat ) NN_NOEXCEPT

{
    m_pGpuBuffer = pGpuBuffer;
    m_VertexCount = numVertices;
    m_IndexCount = numIndices;

    const size_t bufferAlignment = m_pGpuBuffer->GetBufferAlignment();

    // 現状のallocate済サイズを取得する。
    // 全体で何バイトのメモリが必要なのか、アラインメントを考慮して計算する。
    size_t bufferSize = m_pGpuBuffer->GetAllocatedSize();

    // 必要なサイズを、アラインメントを考慮して計算する
    for ( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        if ( vertexFormat & (1 << idx) )
        {
            const VertexAttribute attrib = static_cast<VertexAttribute>(idx);
            bufferSize =
                nn::util::align_up(bufferSize + GetVertexMemorySize(attrib), bufferAlignment);
        }
    }

    // インデックスバッファのサイズ分追加。アラインメントを考慮
    bufferSize = nn::util::align_up(bufferSize + sizeof(uint32_t) * m_IndexCount, bufferAlignment);

    size_t allocateSize = bufferSize - m_pGpuBuffer->GetAllocatedSize();

    // 必要なメモリを確保
    void* pBuffer = m_pGpuBuffer->Allocate(allocateSize);

    // メモリ確保に失敗した場合は、falseを返す。
    if ( pBuffer == NULL )
    {
        NN_LOG("[GfxPrimitiveRenderer] Warning!! pBuffer is NULL\n");
        return false;
    }

    // 各頂点バッファのポインタを計算する。
    for( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        if ( vertexFormat & (1 << idx) )
        {
            const VertexAttribute attrib = static_cast<VertexAttribute>(idx);
            m_pVertexBufferPointer[idx] =
                nn::util::BytePtr( pBuffer ).AlignUp(bufferAlignment).Get();
            pBuffer = nn::util::BytePtr( m_pVertexBufferPointer[idx] )
                .Advance( GetVertexMemorySize(attrib) )
                .Get();
        }
        else
        {
            m_pVertexBufferPointer[idx] = NULL;
        }
    }

    // インデックスバッファのポインタを計算する。
    m_pIndexBufferPointer = nn::util::BytePtr( pBuffer ).AlignUp(bufferAlignment).Get();

    return true;
}

int PrimitiveMesh::GetIndexCount() const NN_NOEXCEPT
{
    return m_IndexCount;
}

size_t PrimitiveMesh::GetVertexStride( VertexAttribute attrib ) const NN_NOEXCEPT
{
    NN_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    return VertexStrideSizeTable[ attrib ];
}

size_t PrimitiveMesh::GetVertexMemorySize( VertexAttribute attrib ) const NN_NOEXCEPT
{
    NN_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    size_t elementSize = 0;
    switch( attrib )
    {
        case VertexAttribute_Pos:
            elementSize = sizeof(nn::util::Float3);
            break;
        case VertexAttribute_Uv:
            elementSize = sizeof(nn::util::Float2);
            break;
        case VertexAttribute_Color:
            elementSize = sizeof(nn::util::Float4);
            break;
        case VertexAttribute_CountMax:
        default: NN_UNEXPECTED_DEFAULT;
            break;
    }
    return elementSize * m_VertexCount;
}

void* PrimitiveMesh::GetVertexBufferCpuAddress( VertexAttribute attribute ) const NN_NOEXCEPT
{
    NN_ASSERT(0 <= attribute && attribute < VertexAttribute_CountMax);
    return m_pVertexBufferPointer[ attribute ];

}

uint32_t* PrimitiveMesh::GetIndexBufferCpuAddress() const NN_NOEXCEPT
{
    return static_cast<uint32_t*>(m_pIndexBufferPointer);
}

//! @brief 指定された頂点属性のメモリサイズを取得します。
//! @param[in] attrib   頂点属性です。
//! @param[out] pGpuAddress
void PrimitiveMesh::GetVertexBufferGpuAddress(
    VertexAttribute attrib,
    nn::gfx::GpuAddress* pGpuAddress
) const NN_NOEXCEPT
{
    NN_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    m_pGpuBuffer->GetGpuAddress( pGpuAddress, m_pVertexBufferPointer[ attrib ] );
}

bool PrimitiveMesh::IsValidVertexBuffer(
    VertexAttribute attrib
) const NN_NOEXCEPT
{
    NN_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    return m_pVertexBufferPointer[ attrib ] != NULL;
}

bool PrimitiveMesh::IsValidIndexBuffer() const NN_NOEXCEPT
{
    return m_pIndexBufferPointer != NULL;
}

void PrimitiveMesh::GetIndexBufferGpuAddress(
    nn::gfx::GpuAddress* pGpuAddress
) const NN_NOEXCEPT
{
    m_pGpuBuffer->GetGpuAddress( pGpuAddress, m_pIndexBufferPointer );
}

template<typename T>
bool PrimitiveMesh::CopyVertexBuffer(
    VertexAttribute attribute,
    const T* pPodData,
    size_t size
) NN_NOEXCEPT
{
    NN_ASSERT(0 <= attribute && attribute < VertexAttribute_CountMax);
    T* pDst = static_cast<T*>(GetVertexBufferCpuAddress( attribute ));
    if( pDst != NULL )
    {
        std::memcpy( pDst, pPodData, size );
        return true;
    }
    return false;
}

bool PrimitiveMesh::CopyIndexBuffer(
    const uint32_t* pPodData,
    size_t size
) NN_NOEXCEPT
{
    uint32_t* pDst = GetIndexBufferCpuAddress();
    if( pDst != NULL )
    {
        std::memcpy( pDst, pPodData, size );
        return true;
    }
    return false;
}

MeshSet::MeshSet() NN_NOEXCEPT
{
}

PrimitiveMesh* MeshSet::Get( ShapeType shapeType ) NN_NOEXCEPT
{
    NN_ASSERT(0 <= shapeType && shapeType < ShapeType_CountMax);
    return &(m_Mesh[ shapeType ]);
}

size_t MeshSet::CalculateMemoryPoolSize( const size_t alignmentSize ) NN_NOEXCEPT
{
    size_t totalSize = 0;

    // 頂点バッファのサイズ
    for ( int i = 0; i < ShapeType_CountMax; i++ )
    {
        size_t shapeSize =0;
        shapeSize += VertexCounts[i] *
            (sizeof(nn::util::Float3) + sizeof(nn::util::Float2) + sizeof(nn::util::Float4));
        shapeSize += 3 * alignmentSize;
        shapeSize += IndexCounts[i] * sizeof(uint32_t);
        shapeSize += alignmentSize;

        totalSize += shapeSize;
    }


    return totalSize;
}

bool CreateQuad(
    float width,
    float height,
    bool bWired,
    GpuBuffer* pGpuBuffer,
    PrimitiveMesh* pMeshBuffer ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGpuBuffer);
    NN_ASSERT_NOT_NULL(pMeshBuffer);

    const uint32_t VertexIndicesWired[ 5 ] =
    {
        0, 1, 2, 3, 0,
    };

    const uint32_t VertexIndices[ 6 ] =
    {
        0, 2, 1,
        1, 2, 3
    };

    const nn::util::Float3 VertexPos[ VertexQuadCount ] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER(   0.f, height, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( width, height, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER(   0.f,    0.f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( width,    0.f, 0.f )
    };

    const nn::util::Float2 VertexUv[ VertexQuadCount ] =
    {
        NN_UTIL_FLOAT_2_INITIALIZER( 0.f, 0.f ),
        NN_UTIL_FLOAT_2_INITIALIZER( 1.f, 0.f ),
        NN_UTIL_FLOAT_2_INITIALIZER( 0.f, 1.f ),
        NN_UTIL_FLOAT_2_INITIALIZER( 1.f, 1.f )
    };

    const nn::util::Float4 VertexColor[ VertexQuadCount ] =
    {
        NN_UTIL_FLOAT_4_INITIALIZER( 1.0f, 1.0f, 1.0f, 1.0f ),
        NN_UTIL_FLOAT_4_INITIALIZER( 1.0f, 1.0f, 1.0f, 1.0f ),
        NN_UTIL_FLOAT_4_INITIALIZER( 1.0f, 1.0f, 1.0f, 1.0f ),
        NN_UTIL_FLOAT_4_INITIALIZER( 1.0f, 1.0f, 1.0f, 1.0f ),
    };

    // インデックス数
    const int NumIndices = ( bWired ) ? IndexWiredQuadCount : IndexSolidQuadCount;

    // 頂点フォーマット
    VertexFormat vertexFormat = ( bWired ) ? VertexFormat_Pos : VertexFormat_Default;

    if (!pMeshBuffer->Initialize(pGpuBuffer, VertexQuadCount, NumIndices, vertexFormat))
    {
        return false;
    }

    // インデックスデータをコピー
    const size_t IndexMemorySize = ( bWired ) ? sizeof(VertexIndicesWired) : sizeof(VertexIndices);
    pMeshBuffer->CopyIndexBuffer( ( bWired ) ? VertexIndicesWired : VertexIndices,
        IndexMemorySize);

    // 頂点データをコピー
    pMeshBuffer->CopyVertexBuffer( VertexAttribute_Pos, VertexPos, sizeof(VertexPos) );
    if (!bWired)
    {
        pMeshBuffer->CopyVertexBuffer( VertexAttribute_Uv, VertexUv, sizeof(VertexUv) );
        pMeshBuffer->CopyVertexBuffer( VertexAttribute_Color, VertexColor, sizeof(VertexColor) );
    }

    return true;
}

// デフォルトメッシュの初期化
bool GraphicsResource::InitializeDefaultMeshSet(
    nn::gfx::Device*       pGfxDevice,
    nn::gfx::MemoryPool*   pMemoryPool,
    ptrdiff_t              memoryPoolOffset,
    size_t                 memoryPoolSize ) NN_NOEXCEPT
{
    GpuBuffer::InitializeArg arg;
    arg.SetBufferCount(1);
    arg.SetGpuAccessFlag( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer );

    const size_t Alignment = GpuBuffer::GetGpuBufferAlignement(pGfxDevice, arg);
    const size_t RequiredSize = MeshSet::CalculateMemoryPoolSize(Alignment);
    arg.SetBufferSize(RequiredSize);

    size_t offset = memoryPoolOffset;

    // アライメントを揃えます。
    if ( offset != 0 )
    {
        offset = nn::util::align_up( offset, Alignment);
    }

    // メモリサイズをチェックします。
    if( (offset - memoryPoolOffset) + RequiredSize >= memoryPoolSize )
    {
        return false;
    }

    if (!m_GpuBuffer.Initialize(pGfxDevice, arg, pMemoryPool, offset))
    {
        return false;
    }

    m_GpuBuffer.Map(0);

    PrimitiveMesh* pMeshBuffer = NULL;
    pMeshBuffer = m_DefaultMeshSet.Get(ShapeType_Quad);
    if (!CreateQuad(1.f, 1.f, false, &m_GpuBuffer, pMeshBuffer))
    {
        return false;
    }
    m_GpuBuffer.Unmap();

    return true;
}


// デフォルトメッシュの破棄
void GraphicsResource::FinalizeDefaultMeshSet(
    nn::gfx::Device* pGfxDevice
) NN_NOEXCEPT
{
    m_GpuBuffer.Finalize( pGfxDevice );
}

} // namespace PrimitiveRenderer
} // namespace gfxLog
} // namespace nns
