﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfxLogPrimitiveRendererGfxRes.h"

namespace nns
{
namespace gfxLog
{
namespace PrimitiveRenderer
{

#if NN_GFX_IS_TARGET_GL
    #if defined( NN_BUILD_TARGET_PLATFORM_ADDRESS_32 )
        #include "../../../Resources/Gfx/gfxLog/PrimitiveRenderer_BuildinShader_Generic32.h"
    #else
        #include "../../../Resources/Gfx/gfxLog/PrimitiveRenderer_BuildinShader_Generic.h"
    #endif
#elif NN_GFX_IS_TARGET_NVN
    #if defined( NN_BUILD_TARGET_PLATFORM_ADDRESS_32 )
        #include "../../../Resources/Gfx/gfxLog/PrimitiveRenderer_BuildinShader_NX32.h"
    #else
        #include "../../../Resources/Gfx/gfxLog/PrimitiveRenderer_BuildinShader_NX.h"
    #endif
#else
    #error Unsupported Platform
#endif

//  プリミティブレンダラが利用するグラフィックスリソースを初期化します。
bool GraphicsResource::Initialize(
    nn::gfx::Device*                        pGfxDevice,
    void*                                   pMemory,
    size_t                                  memorySize,
    nn::gfx::MemoryPool*                    pMemoryPool,
    ptrdiff_t                               memoryPoolOffset,
    size_t                                  memoryPoolSize ) NN_NOEXCEPT
{

    // メモリサイズが足りているかチェック
    if ( memorySize < GetRequiredMemorySize( pGfxDevice ) )
    {
        return false;
    }

    // メモリプールのサイズが足りているかチェック
    if ( memoryPoolSize < GetRequiredMemoryPoolSize( pGfxDevice ) )
    {
        return false;
    }

    nn::util::BytePtr memoryBytePtr( pMemory );

    // プリセットの描画ステートを初期化します。
    if ( !InitializeRenderState(
        pGfxDevice,
        memoryBytePtr,
        memorySize - memoryBytePtr.Distance(pMemory)))
    {
        return false;
    }

    // デフォルトシェーダを初期化します。
    if (!InitializeShader(
        pGfxDevice,
        PrimitiveRendererBuildinShader,
        memoryBytePtr,
        memorySize - memoryBytePtr.Distance(pMemory)))
    {
        return false;
    }

    // デフォルトメッシュを生成します。
    if (!InitializeDefaultMeshSet(
        pGfxDevice,
        pMemoryPool,
        memoryPoolOffset,
        memoryPoolSize ))
    {
        return false;
    }

    return true;
}


//  プリミティブレンダラが利用するグラフィックスリソースを破棄します。
void GraphicsResource::Finalize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT
{
    // デフォルト・メッシュを破棄します。
    FinalizeDefaultMeshSet( pGfxDevice );

    // デフォルトシェーダを破棄します。
    FinalizeShader( pGfxDevice );

    // 描画ステートを破棄します。
    FinalizeRenderState( pGfxDevice );
}

void GraphicsResource::InitializeRenderStateInfo(
    nn::gfx::BlendState::InfoType& info,
    nn::gfx::BlendTargetStateInfo& targetInfo,
    int                            blendType ) NN_NOEXCEPT
{
    info.SetDefault();
    targetInfo.SetDefault();
    switch(blendType)
    {
        case BlendType_Normal:
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
            targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
            targetInfo.SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
            targetInfo.SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        break;

        case BlendType_Add:
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_One );
            targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
            targetInfo.SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_One );
            targetInfo.SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        break;

        case BlendType_Sub:
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_One );
            targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_ReverseSubtract );
            targetInfo.SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_One );
            targetInfo.SetAlphaBlendFunction( nn::gfx::BlendFunction_ReverseSubtract );
        break;

        case BlendType_Mul:
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_Zero );
            targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_SourceColor );
            targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
            targetInfo.SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_Zero );
            targetInfo.SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_SourceColor );
            targetInfo.SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        break;

        case BlendType_Opacity:
            targetInfo.SetBlendEnabled(false);
        break;
        default: NN_UNEXPECTED_DEFAULT;
    }

    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
}

bool GraphicsResource::InitializeRenderState(
    nn::gfx::Device*     pGfxDevice,
    nn::util::BytePtr&   pMemory,
    size_t               memorySize ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGfxDevice);
    NN_ASSERT( memorySize >= GetRequiredMemorySizeForBlendState() );

    // 必要なメモリサイズに達していなかったら、何もせずにreturnする。
    if ( memorySize < GetRequiredMemorySizeForBlendState() )
    {
        return false;
    }

    // Blend
    for( int idxType = 0; idxType < BlendType_CountMax; ++idxType )
    {
        nn::gfx::BlendState::InfoType info;
        nn::gfx::BlendTargetStateInfo targetInfo;
        InitializeRenderStateInfo(info, targetInfo, idxType);
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        m_BlendState[ idxType ].SetMemory( pMemory.Get(), size );
        pMemory.Advance( size );
        m_BlendState[ idxType ].Initialize( pGfxDevice, info );
    }

    // DepthStencil
    for( int idxType = 0; idxType < DepthStencilType_CountMax; ++idxType )
    {
        nn::gfx::DepthStencilState::InfoType info;
        info.SetDefault();
        info.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_LessEqual );
        switch(idxType)
        {
            case DepthStencilType_DepthWriteTest:
                info.SetDepthWriteEnabled(true);
                info.SetDepthTestEnabled(true);
            break;

            case DepthStencilType_DepthWrite:
                info.SetDepthWriteEnabled(true);
                info.SetDepthTestEnabled(false);
            break;

            case DepthStencilType_DepthTest:
                info.SetDepthWriteEnabled(false);
                info.SetDepthTestEnabled(true);
            break;

            case DepthStencilType_DepthNoWriteTest:
                info.SetDepthWriteEnabled(false);
                info.SetDepthTestEnabled(false);
            break;
            default: NN_UNEXPECTED_DEFAULT;
        }
        m_DepthStencilState[ idxType ].Initialize( pGfxDevice, info );
    }

    // Rasterizer
    for( int idxTopology = 0; idxTopology < nn::gfx::PrimitiveTopologyType_End; ++idxTopology )
    {
        for( int idxCull = 0; idxCull < nn::gfx::CullMode_End; ++idxCull )
        {
            for( int idxFill = 0; idxFill < nn::gfx::FillMode_End; ++idxFill )
            {
                nn::gfx::RasterizerState::InfoType info;
                info.SetDefault();
                info.SetPrimitiveTopologyType(
                    static_cast<nn::gfx::PrimitiveTopologyType>(idxTopology)
                );
                info.SetCullMode( static_cast<nn::gfx::CullMode>(idxCull) );
                info.SetFillMode( static_cast<nn::gfx::FillMode>(idxFill) );
                m_RasterizerState[ idxTopology ][ idxCull ][ idxFill ].Initialize(
                    pGfxDevice,
                    info
                );
            }
        }
    }

    return true;
}

void GraphicsResource::FinalizeRenderState( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( pGfxDevice );

    for( int idxType = 0; idxType < BlendType_CountMax; ++idxType )
    {
        m_BlendState[ idxType ].Finalize( pGfxDevice );
    }

    for( int idxType = 0; idxType < DepthStencilType_CountMax; ++idxType )
    {
        m_DepthStencilState[ idxType ].Finalize( pGfxDevice );
    }

    for( int idxTopology = 0; idxTopology < nn::gfx::PrimitiveTopologyType_End; ++idxTopology )
    {
        for( int idxCull = 0; idxCull < nn::gfx::CullMode_End; ++idxCull )
        {
            for( int idxFill = 0; idxFill < nn::gfx::FillMode_End; ++idxFill )
            {
                m_RasterizerState[ idxTopology ][ idxCull ][ idxFill ].Finalize( pGfxDevice );
            }
        }
    }
}

bool GraphicsResource::InitializeShader(
    nn::gfx::Device*   pGfxDevice,
    void*              pShaderBinaryFile,
    nn::util::BytePtr& pMemory,
    size_t             memorySize ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( pGfxDevice );
    NN_ASSERT_NOT_NULL( pShaderBinaryFile );
    NN_ASSERT( memorySize >= GetRequiredMemorySizeForShader() );

    // 必要なメモリサイズでなかったら、何もせずにreturnする。
    if ( memorySize < GetRequiredMemorySizeForShader() )
    {
        return false;
    }

    // シェーダバイナリの初期化を行います。
    m_pResShaderFile = nn::gfx::ResShaderFile::ResCast( pShaderBinaryFile );
    NN_ASSERT_NOT_NULL( m_pResShaderFile );
    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );
    pContainer->Initialize( pGfxDevice );

    nn::gfx::ResShaderVariation* pFirstVariation = pContainer->GetResShaderVariation( 0 );
    NN_ASSERT_NOT_NULL( pFirstVariation );

    nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_Success;

#if NN_GFX_IS_TARGET_GL
    m_ShaderCodeType = nn::gfx::ShaderCodeType_Source;
    shaderResult = pFirstVariation->GetResShaderProgram( m_ShaderCodeType )->Initialize(
        pGfxDevice
    );
#else
    m_ShaderCodeType = nn::gfx::ShaderCodeType_Binary;
    shaderResult = pFirstVariation->GetResShaderProgram( m_ShaderCodeType )->Initialize(
        pGfxDevice
    );

    // バイナリで初期化失敗した場合は、ソースで初期化を試します。
    if ( shaderResult != nn::gfx::ShaderInitializeResult_Success )
    {
        m_ShaderCodeType = nn::gfx::ShaderCodeType_Source;
        shaderResult = pFirstVariation->GetResShaderProgram( m_ShaderCodeType )->Initialize(
            pGfxDevice
        );
    }
#endif

    NN_ASSERT_EQUAL( shaderResult, nn::gfx::ShaderInitializeResult_Success );

    // 全てのバリエーションを初期化します。
    for( int idxVariation = 1, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxVariation );
        NN_ASSERT_NOT_NULL( pVariation );
        shaderResult = pVariation->GetResShaderProgram( m_ShaderCodeType )->Initialize(
            pGfxDevice
        );
        NN_ASSERT_EQUAL( shaderResult, nn::gfx::ShaderInitializeResult_Success );
    }

    for( int idxVariation = 0; idxVariation < ShaderVariation_CountMax; ++idxVariation )
    {
        ShaderVariation shaderVariation = static_cast<ShaderVariation>( idxVariation );
        Shader* pShader = &m_DefaultShader[ idxVariation ];

        // シェーダバリエーションの頂点フォーマットを取得します。
        uint32_t vertexFormat = Shader::GetDefaultVertexFormat( shaderVariation );

        // デフォルトシェーダを生成します。
        if (!Shader::Create( pShader,
                             m_pResShaderFile,
                             shaderVariation,
                             vertexFormat,
                             pGfxDevice,
                             pMemory))
        {
            return false;
        }

        pShader->UpdateSystemSlots();
    }

    return true;
}

void GraphicsResource::FinalizeShader( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT
{
    // シェーダバイナリを破棄します。
    NN_ASSERT_NOT_NULL( m_pResShaderFile );
    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );

    for( int idxVariation = 0; idxVariation < ShaderVariation_CountMax; ++idxVariation )
    {
        m_DefaultShader[ idxVariation ].Cleanup( pGfxDevice );

        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxVariation );
        NN_ASSERT_NOT_NULL( pVariation );
        nn::gfx::ResShaderProgram* pShaderProgram = pVariation->GetResShaderProgram(
            m_ShaderCodeType
        );
        pShaderProgram->Finalize( pGfxDevice );
    }
    pContainer->Finalize( pGfxDevice );
    m_pResShaderFile = NULL;
}

size_t GraphicsResource::GetRequiredMemoryPoolSize(
    nn::gfx::Device* pGfxDevice
) NN_NOEXCEPT
{
    GpuBuffer::InitializeArg arg;
    arg.SetBufferCount(1);
    arg.SetGpuAccessFlag( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer );
    const size_t bufferAlign = GpuBuffer::GetGpuBufferAlignement(pGfxDevice, arg);

    nn::gfx::MemoryPool::InfoType mpInfo;
    mpInfo.SetDefault();
    mpInfo.SetMemoryPoolProperty(
        nn::gfx::MemoryPoolProperty_CpuUncached |
        nn::gfx::MemoryPoolProperty_GpuCached );
    size_t memoryPoolSize  = nn::util::align_up(
        MeshSet::CalculateMemoryPoolSize(bufferAlign),
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( pGfxDevice, mpInfo )
    );

    memoryPoolSize += bufferAlign;
    return memoryPoolSize;
}

size_t GraphicsResource::GetMemoryPoolAlignment(
    nn::gfx::Device* pGfxDevice
) NN_NOEXCEPT
{
    GpuBuffer::InitializeArg arg;
    arg.SetBufferCount(1);
    arg.SetGpuAccessFlag( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

    return GpuBuffer::GetGpuBufferAlignement(pGfxDevice, arg);
}

size_t GraphicsResource::GetRequiredMemorySizeForBlendState() NN_NOEXCEPT
{
    size_t requiredMemorySize = 0;

    // Blend
    for( int idxType = 0; idxType < BlendType_CountMax; ++idxType )
    {
        nn::gfx::BlendState::InfoType info;
        nn::gfx::BlendTargetStateInfo targetInfo;
        InitializeRenderStateInfo(info, targetInfo, idxType);
        requiredMemorySize += nn::gfx::BlendState::GetRequiredMemorySize( info );
        requiredMemorySize += nn::gfx::BlendState::RequiredMemoryInfo_Alignment;
    }

    return requiredMemorySize;
}

size_t GraphicsResource::GetRequiredMemorySizeForShader() NN_NOEXCEPT
{
    size_t requiredMemorySize = 0;

    for( int idxVariation = 0; idxVariation < ShaderVariation_CountMax; ++idxVariation )
    {
        ShaderVariation shaderVariation = static_cast<ShaderVariation>( idxVariation );
        // シェーダバリエーションの頂点フォーマットを取得します。
        uint32_t vertexFormat = Shader::GetDefaultVertexFormat( shaderVariation );

        nn::gfx::VertexState::InfoType info;
        nn::gfx::VertexAttributeStateInfo attribs[ VertexAttribute_CountMax ];
        // インタリーブではなく、属性ごとに頂点バッファを持ちます。
        nn::gfx::VertexBufferStateInfo buffers[ VertexAttribute_CountMax ];

        int attribCount = Shader::InitializeVertexStateInfo(
            &info,
            attribs,
            buffers,
            vertexFormat,
            NULL
        );
        if ( attribCount == 0 )
        {
            continue;
        }

        requiredMemorySize += nn::gfx::VertexState::GetRequiredMemorySize( info );
        requiredMemorySize += nn::gfx::VertexState::RequiredMemoryInfo_Alignment;
    }

    return requiredMemorySize;
}

size_t GraphicsResource::GetRequiredMemorySize(
    nn::gfx::Device*   pGfxDevice
) NN_NOEXCEPT
{
    NN_UNUSED( pGfxDevice );

    size_t requiredMemorySize = 0;

    requiredMemorySize += GetRequiredMemorySizeForBlendState();
    requiredMemorySize += GetRequiredMemorySizeForShader();

    return requiredMemorySize;
}

size_t GraphicsResource::GetRequiredMemoryAlignment(
    nn::gfx::Device* pGfxDevice
) NN_NOEXCEPT
{
    NN_UNUSED( pGfxDevice );

    size_t requiredAlignment = 1;
    requiredAlignment = std::max(static_cast<size_t>(
        nn::gfx::BlendState::RequiredMemoryInfo_Alignment),
        requiredAlignment
    );
    requiredAlignment = std::max(
        static_cast<size_t>(nn::gfx::VertexState::RequiredMemoryInfo_Alignment),
        requiredAlignment
    );
    return requiredAlignment;
}

Shader* GraphicsResource::GetDefaultShader(
    const ShaderVariation shaderVariation
) NN_NOEXCEPT
{
    return &m_DefaultShader[ shaderVariation ];
}

MeshSet* GraphicsResource::GetDefaultMeshSet() NN_NOEXCEPT
{
    return &m_DefaultMeshSet;
}

} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
