﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cctype>
#include <cstdarg>
#include <nn/gfx.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nns/nns_Log.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif
#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Color.h>

#include "gfxLogGraphicsSystem.h"
#include "gfxLogLogOut.h"

namespace {
#define TEXT_BLACK      "\x1b[30m"
#define TEXT_RED        "\x1b[31m"
#define TEXT_GREEN      "\x1b[32m"
#define TEXT_YELLO      "\x1b[33m"
#define TEXT_BLUE       "\x1b[34m"
#define TEXT_MAGENTA    "\x1b[35m"
#define TEXT_CYAN       "\x1b[36m"
#define TEXT_WHITE      "\x1b[37m"
#define TEXT_DEFAULT    "\x1b[39m"

#define BACK_BLACK      "\x1b[40m"
#define BACK_RED        "\x1b[41m"
#define BACK_GREEN      "\x1b[42m"
#define BACK_YELLO      "\x1b[43m"
#define BACK_BLUE       "\x1b[44m"
#define BACK_MAGENTA    "\x1b[45m"
#define BACK_CYAN       "\x1b[46m"
#define BACK_WHITE      "\x1b[47m"
#define BACK_DEFAULT    "\x1b[49m"

struct ResultOfParseExcapeSequence
{
    ColorIndex  colorIndex;
    size_t nextPosition;
};

const nn::util::Color4u8 DefaultTextColor = nn::util::Color4u8::White();
const nn::util::Color4u8 DefaultBackColor = nn::util::Color4u8::Black();
const nn::util::Color4u8 ScreenBackColor = nn::util::Color4u8::Black();

/**
 * @brief LogOut で使用する変数を管理するクラス
 *
 * @details LogOut で使用する変数を管理するクラス。シングルトンです。
 */
class LogOutResourceManager
{
    NN_DISALLOW_COPY(LogOutResourceManager);
    NN_DISALLOW_MOVE(LogOutResourceManager);
public:
    static LogOutResourceManager& GetInstance() NN_NOEXCEPT;

    void SetCustomDraw(bool isCustomDraw) NN_NOEXCEPT
    {
        m_IsCustomDraw = isCustomDraw;
    }

    bool IsCustomDraw() const NN_NOEXCEPT
    {
        return m_IsCustomDraw;
    }

    void SetCustomDrawOnLoad(bool isCustomDrawOnLoad) NN_NOEXCEPT
    {
        m_IsCustomDrawOnLoad = isCustomDrawOnLoad;
    }

    bool IsCustomDrawOnLoad() const NN_NOEXCEPT
    {
        return m_IsCustomDrawOnLoad;
    }

    void SetDevice(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        m_pDevice = pDevice;
    }

    nn::gfx::Device* GetDevice() const NN_NOEXCEPT
    {
        return m_pDevice;
    }

    void SetCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
    {
        m_pCommandBuffer = pCommandBuffer;
    }

    nn::gfx::CommandBuffer* GetCommandBuffer() const NN_NOEXCEPT
    {
        return m_pCommandBuffer;
    }

    void SetDebugFontTextWriter(nn::gfx::util::DebugFontTextWriter* pDebugFontWriter) NN_NOEXCEPT
    {
        m_pDebugFontWriter = pDebugFontWriter;
    }

    nn::gfx::util::DebugFontTextWriter* GetDebugFontTextWriter() const NN_NOEXCEPT
    {
        return m_pDebugFontWriter;
    }

private:
    LogOutResourceManager() NN_NOEXCEPT
        : m_IsCustomDraw(false),
          m_IsCustomDrawOnLoad(false),
          m_pDevice(NULL),
          m_pCommandBuffer(NULL),
          m_pDebugFontWriter(NULL)
    {
    }

private:
    bool m_IsCustomDraw;
    bool m_IsCustomDrawOnLoad;
    nn::gfx::Device* m_pDevice;
    nn::gfx::CommandBuffer* m_pCommandBuffer;
    nn::gfx::util::DebugFontTextWriter* m_pDebugFontWriter;
};

LogOutResourceManager& LogOutResourceManager::GetInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(LogOutResourceManager, s_LogOutResourceManager);
    return s_LogOutResourceManager;
}

nn::mem::StandardAllocator& GetStandardAllocator() NN_NOEXCEPT;

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
/**
 * @brief グラフィックスの確保関数です。
 *
 * @param[in]    size         確保サイズ
 * @param[in]    alignment    アライメント
 * @param[in]    pUserData    ユーザーデータ
 * @return 確保したメモリを返します
 * @details グラフィックスの確保関数です。
 */
void* Allocate( size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    return GetStandardAllocator().Allocate(size, alignment);
}

/**
 * @brief グラフィックスの解放関数です。
 *
 * @param[in]    addr         解放するメモリ
 * @param[in]    pUserData    ユーザーデータ
 * @return 確保したメモリを返します
 * @details グラフィックスの解放関数です。
 */
void Free( void* addr, void* pUserData) NN_NOEXCEPT
{
    GetStandardAllocator().Free( addr );
}

/**
 * @brief グラフィックスの再確保関数です。
 *
 * @param[in]    addr         既存のメモリ
 * @param[in]    newSize      新しい確保サイズ
 * @param[in]    pUserData    ユーザーデータ
 * @return 確保したメモリを返します
 * @details グラフィックスの再確保関数です。
 */
void* Reallocate( void* addr, size_t newSize, void* pUserData) NN_NOEXCEPT
{
    return GetStandardAllocator().Reallocate( addr, newSize );
}
#endif

/**
 * @brief アロケーターのインスタンスを返す関数です。
 *
 * @return アロケーターのインスタンス
 * @details アロケーターのインスタンスを返す関数です。
 */
nn::mem::StandardAllocator& GetStandardAllocator() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::mem::StandardAllocator, s_StandardAllocator);
    NN_FUNCTION_LOCAL_STATIC(nn::os::Mutex, s_Mutex, (false));

    s_Mutex.Lock();
    NN_FUNCTION_LOCAL_STATIC(bool, s_IsInitialized, = false);

    if (!s_IsInitialized)
    {
        const size_t MemorySize = 128 * 1024 * 1024;
        NN_ALIGNAS(4096) static nn::Bit8 s_Memory[MemorySize];

        s_StandardAllocator.Initialize(
            s_Memory, MemorySize
        );

        s_IsInitialized = true;
    }
    s_Mutex.Unlock();
    return s_StandardAllocator;
}

/**
 * @brief グラフィックスの初期化関数です。
 *
 * @details グラフィックスの初期化関数です。
 */
void InitializeGraphics() NN_NOEXCEPT
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
        nv::InitializeGraphics(GetStandardAllocator().Allocate(GraphicsSystemMemorySize, 1),
            GraphicsSystemMemorySize);
    }
    // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリアロケータを設定します。
    nv::SetGraphicsDevtoolsAllocator(Allocate, Free, Reallocate, NULL);
#endif
}

/**
 * @brief エスケープシーケンスを解析します。
 *
 * @param[in] String        解析対象の文字列
 * @param[in] startPosition String の文字列の解析開始位置
 * @return    見つかったエスケープシーケンスの enum 値を返します。
 * @details   エスケープシーケンスをパターンマッチングして指定したエスケープシーケンス
 *            と一致した場合 enum 値を返します。
 *            指定したエスケープシーケンスにマッチしない場合 ColorIndex_Invalid を返します。
 */
ResultOfParseExcapeSequence ParseEscapeSequence(
    const std::string& String,
    size_t startPosition) NN_NOEXCEPT
{
    struct Table
    {
        const char* Pattern;
        size_t Length;
        ColorIndex ColorIndex;
    };
    const Table table[] = {
        { TEXT_BLACK   , sizeof(TEXT_BLACK) - 1,    ColorIndex_TextBlack },   // 文字色: 黒
        { TEXT_RED     , sizeof(TEXT_RED) - 1,      ColorIndex_TextRed },     // 文字色: 赤
        { TEXT_GREEN   , sizeof(TEXT_GREEN) - 1,    ColorIndex_TextGreen },   // 文字色: 緑
        { TEXT_YELLO   , sizeof(TEXT_YELLO) - 1,    ColorIndex_TextYellow },  // 文字色: 黄
        { TEXT_BLUE    , sizeof(TEXT_BLUE) - 1,     ColorIndex_TextBlue },    // 文字色: 青
        { TEXT_MAGENTA , sizeof(TEXT_MAGENTA) - 1,  ColorIndex_TextMagenta }, // 文字色: マゼンタ
        { TEXT_CYAN    , sizeof(TEXT_CYAN) - 1,     ColorIndex_TextCyan },    // 文字色: シアン
        { TEXT_WHITE   , sizeof(TEXT_WHITE) - 1,    ColorIndex_TextWhite },   // 文字色: 白
        { TEXT_DEFAULT , sizeof(TEXT_DEFAULT) - 1,  ColorIndex_TextDefault }, // 文字色: デフォルト
        { BACK_BLACK   , sizeof(BACK_BLACK) - 1,    ColorIndex_BackBlack },   // 背景色: 黒
        { BACK_RED     , sizeof(BACK_RED) - 1,      ColorIndex_BackRed },     // 背景色: 赤
        { BACK_GREEN   , sizeof(BACK_GREEN) - 1,    ColorIndex_BackGreen },   // 背景色: 緑
        { BACK_YELLO   , sizeof(BACK_YELLO) - 1,    ColorIndex_BackYellow },  // 背景色: 黄
        { BACK_BLUE    , sizeof(BACK_BLUE) - 1,     ColorIndex_BackBlue },    // 背景色: 青
        { BACK_MAGENTA , sizeof(BACK_MAGENTA) - 1,  ColorIndex_BackMagenta }, // 背景色: マゼンタ
        { BACK_CYAN    , sizeof(BACK_CYAN) - 1,     ColorIndex_BackCyan },    // 背景色: シアン
        { BACK_WHITE   , sizeof(BACK_WHITE) - 1,    ColorIndex_BackWhite },   // 背景色: 白
        { BACK_DEFAULT , sizeof(BACK_DEFAULT) - 1,  ColorIndex_BackDefault }, // 背景色: デフォルト
    };

    ResultOfParseExcapeSequence result = {};
    result.colorIndex = ColorIndex_Invalid;
    result.nextPosition = startPosition;

    for (int i = 0; i < NN_ARRAY_SIZE(table); i++)
    {
        if (String.compare(startPosition, table[i].Length, table[i].Pattern) == 0)
        {
            result.colorIndex = table[i].ColorIndex;
            result.nextPosition = startPosition + table[i].Length;
            return result;
        }
    }
    return result;
}
}

namespace nns {
namespace gfxLog {

/**
 * @brief LogOut のシングルトンのインスタンスを返します。
 *
 * @return    シングルトンのインスタンス
 * @details   LogOut のシングルトンのインスタンスを返します。
 */
LogOut& LogOut::GetInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(LogOut, s_LogOut);
    return s_LogOut;
}

/**
 * @brief LogOut のコンストラクタです。
 *
 * @details   LogOut のコンストラクタです。
 */
LogOut::LogOut() NN_NOEXCEPT
    : m_IsCustomDraw(LogOutResourceManager::GetInstance().IsCustomDraw())
    , m_IsRunning(false)
    , m_pDevice(LogOutResourceManager::GetInstance().GetDevice())
    , m_pCommandBuffer(LogOutResourceManager::GetInstance().GetCommandBuffer())
    , m_pPrimitiveRenderer(NULL)
    , m_pDebugFontWriter(LogOutResourceManager::GetInstance().GetDebugFontTextWriter())
    , m_pGraphicsSystem(NULL)
    , m_TempDisplayText()
    , m_DisplayText()
    , m_TextColorToSet(DefaultTextColor)
    , m_BackColorToSet(DefaultBackColor)
    , m_TextColor(DefaultTextColor)
    , m_BackColor(DefaultBackColor)
    , m_Mutex(false)
    , m_pThreadStack(NULL)
    , m_Thread()
    , m_EventType()
    , m_IdealCore(nn::os::IdealCoreUseDefaultValue)
    , m_Current(0)
    , m_TempOneLine()
    , m_LeftToSet(0.0f)
    , m_TopToSet(0.0f)
    , m_WidthToSet(0.0f)
    , m_HeightToSet(0.0f)
    , m_Left(0.0f)
    , m_Top(0.0f)
    , m_Width(0.0f)
    , m_Height(0.0f)
    , m_PosX(0.0f)
{
    // Draw のためにロード時の設定を保持します
    LogOutResourceManager& logOutResoure = LogOutResourceManager::GetInstance();
    logOutResoure.SetCustomDrawOnLoad(logOutResoure.IsCustomDraw());

    Initialize();
}

/**
 * @brief LogOut のデストラクタです。
 *
 * @details   LogOut のデストラクタです。
 */
LogOut::~LogOut() NN_NOEXCEPT
{
    FinalizePrimitiveRenderer();
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    if (!m_IsCustomDraw)
    {
        nn::os::SignalEvent(&m_EventType);
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);

        GetStandardAllocator().Free(m_pThreadStack);

        m_pGraphicsSystem->FinalizeDebugFont();
        m_pGraphicsSystem->Finalize();
        delete m_pGraphicsSystem;
        m_pGraphicsSystem = NULL;
    }
#endif
}

/**
 * @brief LogOut を初期化します。
 *
 * @details   LogOut を初期化します。コンストラクタから呼ばれます。
 */
void LogOut::Initialize() NN_NOEXCEPT
{
    if (m_IsCustomDraw)
    {
        NN_ABORT_UNLESS_NOT_NULL(
            LogOutResourceManager::GetInstance().GetCommandBuffer());
        NN_ABORT_UNLESS_NOT_NULL(
            LogOutResourceManager::GetInstance().GetDebugFontTextWriter());
    }
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    else
    {
        NN_ABORT_UNLESS_EQUAL(
            LogOutResourceManager::GetInstance().GetCommandBuffer(),
            static_cast<nn::gfx::CommandBuffer*>(NULL));
        NN_ABORT_UNLESS_EQUAL(
            LogOutResourceManager::GetInstance().GetDebugFontTextWriter(),
            static_cast<nn::gfx::util::DebugFontTextWriter*>(NULL));
    }
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    if (!m_IsCustomDraw)
    {
        nn::os::InitializeEvent(&m_EventType, false, nn::os::EventClearMode_ManualClear);
        InitializeGraphics();

        m_pGraphicsSystem = new GraphicsSystem();
        m_pGraphicsSystem->SetApplicationHeap(&GetStandardAllocator());
        m_pGraphicsSystem->Initialize();
        m_pGraphicsSystem->InitializeDebugFont();
        m_pDevice = &m_pGraphicsSystem->GetDevice();
    }
#endif

    m_Left = 0.0f;
    m_Top = 0.0f;
    if (m_IsCustomDraw)
    {
        m_Width = 1280.0f;
        m_Height = 720.0f;
    }
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    else
    {
        m_Width = static_cast<float>(m_pGraphicsSystem->GetDisplayWidth());
        m_Height = static_cast<float>(m_pGraphicsSystem->GetDisplayHeight());
    }
#endif
    InitializePrimitiveRenderer();

    m_LeftToSet = m_Left;
    m_TopToSet = m_Top;
    m_WidthToSet = m_Width;
    m_HeightToSet = m_Height;

    m_PosX = 0.0f;

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    if (!m_IsCustomDraw)
    {
        m_pThreadStack = reinterpret_cast<char*>(
            GetStandardAllocator().Allocate(
                ThreadStackSize,
                nn::os::ThreadStackAlignment
            )
        );

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &m_Thread, ThreadFunc, this, m_pThreadStack,
            ThreadStackSize, nn::os::DefaultThreadPriority));
    }
#endif
}

/**
* @brief LogOut スレッドを開始します。
*
* @details   スレッドが実行済みの場合は何もしません。
*/
void LogOut::Start() NN_NOEXCEPT
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    if (!m_IsCustomDraw && !m_IsRunning)
    {
        nn::Bit64 affinityMask = static_cast<nn::Bit64>(0x1) << m_IdealCore;
        if (m_IdealCore == nn::os::IdealCoreDontCare
            || m_IdealCore == nn::os::IdealCoreUseDefaultValue
            || m_IdealCore == nn::os::IdealCoreNoUpdate)
        {
            // コア指定がコア番号ではない場合は現在の affinityMask を使用する
            int idealCore;
            nn::os::GetThreadCoreMask(&idealCore, &affinityMask, &m_Thread);
        }

        // スレッドが動作するコアを設定
        nn::os::SetThreadCoreMask(
            &m_Thread,
            m_IdealCore,
            affinityMask);

        // スレッドをStart
        nn::os::StartThread(&m_Thread);
        m_IsRunning = true;
    }
#endif
}

/**
 * @brief 文字列を文字単位に分割する。CR, LF の解析も行う
 *
 * @param[in] Data    対象文字列
 * @return    分割した文字列の vector を返します。
 * @details   文字列を文字単位に分割する。CR, LF の解析も行う
 */
std::vector<LogOut::StringData> LogOut::Split(const LogOut::StringData& Data) const NN_NOEXCEPT
{
    std::vector<LogOut::StringData> dataArray;
    StringData stringData;
    const char CarriageReturn = '\r';
    const char LineFeed = '\n';

    const std::string String = Data.GetString();
    for (size_t i = 0; i < Data.GetString().size(); i++)
    {
        const char Ch = String[i];
        if (Ch == CarriageReturn || Ch == LineFeed)
        {
            if (Ch == CarriageReturn)
            {
                stringData.SetLineEnding(LineEnding_CarriageReturn);
            }
            else
            {
                stringData.SetLineEnding(LineEnding_LineFeed);
            }
            stringData.SetString("");
            stringData.SetTextColor(Data.GetTextColor());
            stringData.SetBackColor(Data.GetBackColor());
            dataArray.push_back(stringData);
        }
        else
        {
            stringData.SetString(Data.GetString().substr(i, 1));
            stringData.SetLineEnding(LineEnding_NO);
            stringData.SetTextColor(Data.GetTextColor());
            stringData.SetBackColor(Data.GetBackColor());
            dataArray.push_back(stringData);
        }
    }
    return dataArray;
}

/**
 * @brief エスケープシーケンスの色指定をもとにテキスト色、背景色を設定します。
 *
 * @param[in] ColorIndex    色指定
 * @details   エスケープシーケンスの色指定をもとにテキスト色、背景色を設定します。
 */
void LogOut::SetTextAndBackColor(const ColorIndex ColorIndex) NN_NOEXCEPT
{
    switch (ColorIndex)
    {
    case ColorIndex_TextBlack: // 文字色: 黒
        m_TextColor = nn::util::Color4u8::Black();
        break;
    case ColorIndex_TextRed: // 文字色: 赤
        m_TextColor = nn::util::Color4u8::Red();
        break;
    case ColorIndex_TextGreen: // 文字色: 緑
        m_TextColor = nn::util::Color4u8::Green();
        break;
    case ColorIndex_TextYellow: // 文字色: 黄
        m_TextColor = nn::util::Color4u8::Yellow();
        break;
    case ColorIndex_TextBlue: // 文字色: 青
        m_TextColor = nn::util::Color4u8::Blue();
        break;
    case ColorIndex_TextMagenta: // 文字色: マゼンタ
        m_TextColor = nn::util::Color4u8(255, 0, 255, 255);
        break;
    case ColorIndex_TextCyan: // 文字色: シアン
        m_TextColor = nn::util::Color4u8(0, 255, 255, 255);
        break;
    case ColorIndex_TextWhite: // 文字色: 白
        m_TextColor = nn::util::Color4u8::White();
        break;
    case ColorIndex_TextDefault: // 文字色: 標準に戻す
        m_TextColor = DefaultTextColor;
        break;
    case ColorIndex_BackBlack: // 背景色: 黒
        m_BackColor = nn::util::Color4u8::Black();
        break;
    case ColorIndex_BackRed: // 背景色: 赤
        m_BackColor = nn::util::Color4u8::Red();
        break;
    case ColorIndex_BackGreen: // 背景色: 緑
        m_BackColor = nn::util::Color4u8::Green();
        break;
    case ColorIndex_BackYellow: // 背景色: 黄
        m_BackColor = nn::util::Color4u8::Yellow();
        break;
    case ColorIndex_BackBlue: // 背景色: 青
        m_BackColor = nn::util::Color4u8::Blue();
        break;
    case ColorIndex_BackMagenta: // 背景色: マゼンタ
        m_BackColor = nn::util::Color4u8(255, 0, 255, 255);
        break;
    case ColorIndex_BackCyan: // 背景色: シアン
        m_BackColor = nn::util::Color4u8(0, 255, 255, 255);
        break;
    case ColorIndex_BackWhite: // 背景色: 白
        m_BackColor = nn::util::Color4u8::White();
        break;
    case ColorIndex_BackDefault: // 背景色: 標準に戻す
        m_BackColor = DefaultBackColor;
        break;
    default:
        break;
    }
}


/**
 * @brief エスケープシーケンスを解析して分割します。
 *
 * @param[in] String    解析対象の文字列 (改行コードを含む場合あり)
 * @return    分割した文字列の vectorを返します
 * @details   色指定を行うエスケープシーケンスをもとに文字列を分割して返します。
 *            エスケープシーケンスは解析後の文字列には含まれません。
 */
std::vector<LogOut::StringData> LogOut::SplitEscapeSequences(const std::string& String) NN_NOEXCEPT
{
    const char Escape = 0x1b;
    std::vector<LogOut::StringData> returnData;
    StringData stringData;

    size_t previousPosition = 0;
    for (size_t i = 0; i < String.size();)
    {
        if (String[i] == Escape)
        {
            ResultOfParseExcapeSequence result = ParseEscapeSequence(String, i);
            if (result.colorIndex != ColorIndex_Invalid)
            {
                // 有効なエスケープシーケンスを見つけたのでエスケープシーケンスの前
                // までの文字列をキューに追加します
                stringData.SetString(String.substr(previousPosition, i - previousPosition));
                stringData.SetLineEnding(LineEnding_NO); // 設定した値は利用されない
                stringData.SetTextColor(m_TextColor);
                stringData.SetBackColor(m_BackColor);
                returnData.push_back(stringData);

                SetTextAndBackColor(result.colorIndex);
                previousPosition = result.nextPosition;
                NN_ABORT_UNLESS_GREATER(result.nextPosition, i);
                i = result.nextPosition;

                m_TextColorToSet = m_TextColor;
                m_BackColorToSet = m_BackColor;
            }
            else
            {
                i++;
            }
        }
        else
        {
            i++;
        }
    }

    const std::string lastString = String.substr(previousPosition);
    if (!lastString.empty())
    {
        stringData.SetString(lastString);
        stringData.SetLineEnding(LineEnding_NO); // 設定した値は利用されない
        stringData.SetTextColor(m_TextColor);
        stringData.SetBackColor(m_BackColor);
        returnData.push_back(stringData);
    }
    return returnData;
}

/**
 * @brief 画面に入りきらない文字列を捨てます。
 *
 * @param[in] stringData    余計な文字列を捨てる対象の文字列 vector
 * @pre       引数で渡された stringData に対して必要なロックを取得していること
 * @details   画面に収まる行数をもとに表示できない文字列を捨てます。
 */
void LogOut::LimitLines(std::vector<StringData>& stringData) NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = GetDebugFont();
    const float DisplayHeight = m_Height;
    const float FontHeight = debugFontTextWriter.GetFontHeight();
    const size_t ScreenLines = static_cast<size_t>(DisplayHeight / FontHeight);

    if (!stringData.empty())
    {
        size_t lines = 0;
        for (int i = static_cast<int>(stringData.size()) - 1; i >= 0; i--)
        {
            if (stringData[i].GetLineEnding() == LineEnding_LineFeed ||
                stringData[i].GetLineEnding() == LineEnding_AutoLineFeed)
            {
                lines++;
            }
            else if (i == static_cast<int>(stringData.size()) - 1)
            {
                lines++;
            }
            if (lines > ScreenLines)
            {
                stringData.erase(stringData.begin(), stringData.begin() + i + 1);
                break;
            }
        }
    }
}

/**
 * @brief 自動整形を行います。
 *
 * @param[in] String    分割したい文字列
 * @param[in] Width     表示できる幅
 * @return    画面に収まる文字列と収まらなかった文字列のペアを返します。
 * @details   自動整形を行います。
 */
LogOut::StringPair LogOut::WrapString(const std::string& String, const float Width) NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = GetDebugFont();
    std::vector<std::string> wrapping;


    float totalWidth = 0.0f;
    float previousTotalWidth = 0.0f;
    std::string currentString;
    std::string nextString;
    StringPair pair;

    for (size_t i = 0; i < String.size(); i++)
    {
        currentString = nextString;
        nextString += String[i];

        std::string currentCharString;
        currentCharString = String[i];

        // 次の文字を文字列に追加して超えないか判定します
        totalWidth += debugFontTextWriter.CalculateStringWidth(currentCharString.c_str());
        if (totalWidth > Width)
        {
            pair.SetString(currentString);
            pair.SetRemain(String.substr(i));
            pair.SetTotalFontWidth(previousTotalWidth);
            return pair;
        }
        previousTotalWidth = totalWidth;
    }

    pair.SetString(String);
    pair.SetRemain("");
    pair.SetTotalFontWidth(previousTotalWidth);
    return pair;
}

/**
* @brief CR を処理する
*
* @param[in] destination    出力結果
* @param[in] data           処理対象の vector
* @pre       引数に対して必要なロックを取得していること
* @details   CR を処理して、処理途中の行を m_TempOneLine に入れる
*/
void LogOut::ProcessCR(
    std::vector<StringData>& destination,
    const std::vector<StringData>& data
) NN_NOEXCEPT
{
    if (!data.empty())
    {
        for (size_t i = 0; i < data.size(); i++)
        {
            if (data[i].GetLineEnding() == LineEnding_CarriageReturn)
            {
                m_Current = 0;
            }
            else if (data[i].GetLineEnding() == LineEnding_LineFeed ||
                     data[i].GetLineEnding() == LineEnding_AutoLineFeed)
            {
                m_TempOneLine.push_back(data[i]);
                for (size_t j = 0; j < m_TempOneLine.size(); j++)
                {
                    destination.push_back(m_TempOneLine[j]);
                }

                m_TempOneLine.clear();
                m_Current = 0;
            }
            else
            {
                if (m_Current >= m_TempOneLine.size())
                {
                    m_TempOneLine.push_back(data[i]);
                }
                else
                {
                    m_TempOneLine[m_Current] = data[i];
                }
                m_Current++;
            }
        }
    }
}

/**
* @brief 同じ文字色の文字列を連結する。
*
* @param[in] stringData    表示用文字列 vector
* @details   同じ文字色の文字列を連結する。改行をまたいで連結はしない。
*/
void LogOut::Concat(std::vector<StringData>& stringData) NN_NOEXCEPT
{
    std::vector<StringData> tempArray;
    if (!stringData.empty())
    {
        StringData temp = stringData[0];
        for (size_t i = 1; i < stringData.size(); i++)
        {
            if (temp.GetLineEnding() != LineEnding_NO)
            {
                // 前の文字列が改行を含む場合、連結できないのでキューに入れる
                // そして現在処理中の要素を更新する
                tempArray.push_back(temp);
                temp = stringData[i];
            }
            else if (temp.GetTextColor() == stringData[i].GetTextColor() &&
                     temp.GetBackColor() == stringData[i].GetBackColor())
            {
                // 前の文字列と処理中の文字列の色が一致していた場合、文字列を連結し
                // 処理中の文字列の改行コードを temp に設定する
                const std::string string2 = stringData[i].GetString();
                const std::string string = temp.GetString() + string2;
                temp.SetString(string);
                temp.SetLineEnding(stringData[i].GetLineEnding());

                if (temp.GetLineEnding() != LineEnding_NO)
                {
                    // 処理中の文字列(temp)が改行を含む場合、連結できないのでキューに入れる
                    // そして現在処理中の文字列を空にして、改行コードをなしに設定して次の
                    // 処理に備える
                    tempArray.push_back(temp);
                    temp.SetString("");
                    temp.SetLineEnding(LineEnding_NO);
                }
            }
            else
            {
                // 文字色または背景色が違うので連結できない
                tempArray.push_back(temp);
                temp = stringData[i];
            }
        }
        tempArray.push_back(temp);
        stringData = tempArray;
    }
}

/**
 * @brief 引数で指定した文字列の vector を自動整形を行って m_DisplayText に詰めます。
 *
 * @param[in] Data    表示したい文字列の vector
 * @pre       引数に対して必要なロックを取得していること
 * @details   画面に収まる行数をもとに表示できない文字列を捨てます。
 */
void LogOut::WrapStrings(std::vector<LogOut::StringData>& Data) NN_NOEXCEPT
{
    const float Right = m_Left + m_Width;
    std::vector<LogOut::StringData> temp;

    std::string string;
    StringPair pair;
    for( size_t i = 0; i < Data.size(); i++)
    {
        string = Data[i].GetString();
        do
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(Right, m_PosX);
            pair = WrapString(string, Right - m_PosX);
            m_PosX += pair.GetTotalFontWidth();

            StringData stringData;
            stringData.SetString(pair.GetString());

            if (!pair.GetRemain().empty())
            {
                // 自動整形で改行が挿入される場合
                stringData.SetLineEnding(LineEnding_AutoLineFeed);
            }
            else
            {
                // オリジナルの改行コードを設定する
                stringData.SetLineEnding(Data[i].GetLineEnding());
            }
            stringData.SetTextColor(Data[i].GetTextColor());
            stringData.SetBackColor(Data[i].GetBackColor());
            temp.push_back(stringData);

            // 自動改行または自動整形で改行が入ったら X座標をリセットします
            if (stringData.GetLineEnding() != LineEnding_NO)
            {
                m_PosX = m_Left;
            }

            // 次のループのために設定します
            string = pair.GetRemain();
        } while (!pair.GetRemain().empty());
    }

    Data = temp;
}

/**
 * @brief テキスト色を設定します。
 *
 * @param[in] Color    設定する色
 * @details   次に描画するときの使用するテキスト色を設定します。
 */
void LogOut::InternalSetTextColor(const nn::util::Color4u8& Color) NN_NOEXCEPT
{
    m_Mutex.Lock();
    m_TextColorToSet = Color;
    m_Mutex.Unlock();
}

/**
 * @brief テキスト色を設定します。 (static 関数)
 *
 * @param[in] Color    設定する色
 * @details   次に描画するときの使用するテキスト色を設定します。
 */
void LogOut::SetTextColor(const nn::util::Color4u8& Color) NN_NOEXCEPT
{
    LogOut& logOut = LogOut::GetInstance();
    logOut.InternalSetTextColor(Color);
    logOut.Start();
}

/**
 * @brief 背景色を設定します。
 *
 * @param[in] Color    設定する色
 * @details   次に描画するときの使用する背景色を設定します。
 */
void LogOut::InternalSetBackgroudColor(const nn::util::Color4u8& Color) NN_NOEXCEPT
{
    m_Mutex.Lock();
    m_BackColorToSet = Color;
    m_Mutex.Unlock();
}

/**
 * @brief 背景色を設定します。 (static 関数)
 *
 * @param[in] Color    設定する色
 * @details   次に描画するときの使用する背景色を設定します。
 */
void LogOut::SetBackgroudColor(const nn::util::Color4u8& Color) NN_NOEXCEPT
{
    LogOut& logOut = LogOut::GetInstance();
    logOut.InternalSetBackgroudColor(Color);
    logOut.Start();
}

/**
 * @brief 描画領域を設定します。
 *
 * @param[in]    Left    左のX座標
 * @param[in]    Top     上のY座標
 * @param[in]    Width   幅
 * @param[in]    Height  高さ
 * @details   次に描画するときの使用する矩形(描画領域)を設定します。
 */
void LogOut::InternalSetDrawRect(
    const float Left,
    const float Top,
    const float Width,
    const float Height
) NN_NOEXCEPT
{
    m_Mutex.Lock();
    m_LeftToSet = Left;
    m_TopToSet = Top;
    m_WidthToSet = Width;
    m_HeightToSet = Height;
    m_Mutex.Unlock();
}

/**
 * @brief 描画領域を設定します。 (static 関数)
 *
 * @param[in]    Left    左のX座標
 * @param[in]    Top     上のY座標
 * @param[in]    Width   幅
 * @param[in]    Height  高さ
 * @details   次に描画するときの使用する矩形(描画領域)を設定します。
 */
void LogOut::SetDrawRect(
    const float Left,
    const float Top,
    const float Width,
    const float Height
) NN_NOEXCEPT
{
    LogOut& logOut = LogOut::GetInstance();
    logOut.InternalSetDrawRect(Left, Top, Width, Height);
    logOut.Start();
}

/**
* @brief ログ描画用スレッドのコアを設定します。
*
* @param[in] idealCore     優先して動作するコア番号
* @details ログ描画用スレッドのコア割り当てを設定します。
*/
void LogOut::InternalSetThreadCoreNumber(int idealCore) NN_NOEXCEPT
{
    m_Mutex.Lock();
    m_IdealCore = idealCore;
    m_Mutex.Unlock();
}

/**
* @brief ログ描画用スレッドのコアを設定します。
*
* @param[in] idealCore     優先して動作するコア番号
* @details ログ描画用スレッドのコア割り当てを設定します。
*/
void LogOut::SetThreadCoreNumber(int idealCore) NN_NOEXCEPT
{
    LogOut& logOut = LogOut::GetInstance();
    logOut.InternalSetThreadCoreNumber(idealCore);
    logOut.Start();
}

/**
 * @brief 表示範囲に収まるようにクリッピングします。
 *
 * @param[in]    String  対象文字列
 * @param[in]    Width   幅
 * @return クリッピングした文字列
 * @details 指定した幅に収まる文字列を返します。
 */
std::string LogOut::ClipString(const std::string& String, const float Width) NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = GetDebugFont();
    std::string tempString = String;
    float fontWidth = debugFontTextWriter.CalculateStringWidth(tempString.c_str());
    while (fontWidth > Width)
    {
        if (!tempString.empty())
        {
            tempString.erase(tempString.size() - 1, 1);
            fontWidth = debugFontTextWriter.CalculateStringWidth(tempString.c_str());
        }
    }
    return tempString;
}

/**
 * @brief 文字列をキューに追加します。
 *
 * @param[in]    String    対象文字列
 * @details 文字列をログ出力用に追加します。
 */
void LogOut::AddToQueue(const std::string& String) NN_NOEXCEPT
{
    m_Mutex.Lock();
    m_TextColor = m_TextColorToSet;
    m_BackColor = m_BackColorToSet;

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    // NX の場合、常にキューに追加します。
    const bool IsDraw = true;
#else
    // Windows の場合カスタムドローが設定されていない場合、キューに追加しません。
    const bool IsDraw = m_IsCustomDraw;
#endif
    if (IsDraw)
    {
        m_Left = m_LeftToSet;
        m_Top = m_TopToSet;
        m_Width = m_WidthToSet;
        m_Height = m_HeightToSet;
        m_PosX = m_LeftToSet;

        std::vector<StringData> data = SplitEscapeSequences(String);
        for (size_t i = 0; i < data.size(); i++)
        {
            // 文字列を分割します
            std::vector<LogOut::StringData> charArray = Split(data[i]);

            // CR を処理する
            ProcessCR(m_TempDisplayText, charArray);
        }

        // 余計な文字列を整理します
        LimitLines(m_TempDisplayText);

        // 文字列を連結します
        Concat(m_TempDisplayText);

        m_DisplayText = m_TempDisplayText;
        for (size_t j = 0; j < m_TempOneLine.size(); j++)
        {
            m_DisplayText.push_back(m_TempOneLine[j]);
        }

        // 自動改行を処理します
        WrapStrings(m_DisplayText);

        // 余計な文字列を整理します
        LimitLines(m_DisplayText);

        // 文字列を連結します
        Concat(m_DisplayText);
    }
    m_Mutex.Unlock();
}

/**
 * @brief 描画します。
 *
 * @details キューに入れた文字列を画面に描画します。SetConfiguration が呼ばれた場合、
 *          コマンドの発行だけで実際の描画はアプリケーションで行います。
 */
void LogOut::InternalDraw() NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = GetDebugFont();
    const float FontHeight = debugFontTextWriter.GetFontHeight();
    std::vector<StringData> displayText;

    m_Mutex.Lock();
    float posX = m_Left;
    float posY = m_Top;
    const float Right = m_Left + m_Width;
    displayText = m_DisplayText;
    m_Mutex.Unlock();

    nn::gfx::CommandBuffer& commandBuffer = GetCommandBuffer();
    PrimitiveRenderer::Renderer& primitiveRender = GetPrimitiveRenderer();

    if (!m_IsCustomDraw)
    {
        m_pGraphicsSystem->BeginDraw();
    }

    primitiveRender.Update(0);
    debugFontTextWriter.SetScale(1.0f, 1.0f);

    primitiveRender.SetColor(ScreenBackColor);
    primitiveRender.Draw2DRect(&commandBuffer,
        m_Left,
        m_Top,
        m_Width,
        m_Height
    );

    for (size_t i = 0; i < displayText.size(); i++)
    {
        if (!displayText[i].GetString().empty())
        {
            const std::string ClippedString = ClipString(displayText[i].GetString(), Right - posX);
            const float TextWidth = debugFontTextWriter.CalculateStringWidth(ClippedString.c_str());
            primitiveRender.SetColor(displayText[i].GetBackColor());
            primitiveRender.Draw2DRect(&commandBuffer,
                posX,
                posY,
                TextWidth,
                FontHeight
            );

            NN_ABORT_UNLESS_LESS_EQUAL(posX, m_Left + m_Width);
            NN_ABORT_UNLESS_LESS_EQUAL(posY, m_Top + m_Height);
            debugFontTextWriter.SetTextColor(displayText[i].GetTextColor());
            debugFontTextWriter.SetCursor(posX, posY);
            debugFontTextWriter.Print(ClippedString.c_str());

            const float StringWidth = debugFontTextWriter.CalculateStringWidth(
                ClippedString.c_str()
            );
            posX += StringWidth;
        }

        if (displayText[i].GetLineEnding() == LineEnding_LineFeed ||
            displayText[i].GetLineEnding() == LineEnding_AutoLineFeed)
        {
            posX = m_Left;
            posY += FontHeight;
        }
        NN_ABORT_UNLESS_LESS_EQUAL(posX, m_Left + m_Width);
        NN_ABORT_UNLESS_LESS_EQUAL(posY, m_Top + m_Height);
    }

    debugFontTextWriter.Draw(&commandBuffer);
    if (!m_IsCustomDraw)
    {
        m_pGraphicsSystem->EndDraw();
    }
}

/**
 * @brief 描画コマンドを発行します。 (static 関数)
 *
 * @details 描画コマンドを発行します。SetConfiguration が呼ばれた場合のみ処理を行います。
 */
void LogOut::WriteCommand() NN_NOEXCEPT
{
    LogOut& logOut = LogOut::GetInstance();
    if (LogOutResourceManager::GetInstance().IsCustomDrawOnLoad())
    {
        logOut.InternalDraw();
    }
    logOut.Start();
}

/**
 * @brief 指定した文字列を描画し、ログ出力します。(static 関数)
 *
 * @param[in]    String    対象文字列
 * @details 指定した文字列を描画し、ログ出力します。
 */
void LogOut::Log(const std::string& String) NN_NOEXCEPT
{
    // Target Manager にログを出力します
    NN_LOG("%s", String.c_str());

    LogOut& logOut = LogOut::GetInstance();
    logOut.AddToQueue(String);
    logOut.Start();
}

/**
 * @brief 描画スレッドの実装関数です。
 *
 * @details 描画スレッドの実装関数です。SetConfiguration が呼ばれなかった場合のみ呼ばれます。
 */
void LogOut::ThreadFuncImpl() NN_NOEXCEPT
{
    while (!nn::os::TryWaitEvent(&m_EventType))
    {
        InternalDraw();
    }
}

/**
 * @brief スレッドのエントリーポイントです。
 *
 * @param[in]    pArg      スレッド引数
 * @details 描画スレッドのエントリーポイントです。
 * SetConfiguration が呼ばれなかった場合のみ呼ばれます。
 */
void LogOut::ThreadFunc(void* pArg) NN_NOEXCEPT
{
    LogOut * pLogOut = reinterpret_cast<LogOut*>(pArg);
    pLogOut->ThreadFuncImpl();
}

/**
 * @brief デバイス、コマンドバッファ、デバッグフォントライターを設定します。
 *
 * @param[in]    pDevice               デバイス
 * @param[in]    pCommandBuffer        コマンドバッファ
 * @param[in]    pDebugFontWriter      デバッグフォントライター
 * @details アプリケーションで描画したい場合にデバイス、コマンドバッファ、
 *          デバッグフォントライターを設定します。
 */
void LogOut::SetConfiguration(
    nn::gfx::Device* pDevice,
    nn::gfx::CommandBuffer* pCommandBuffer,
    nn::gfx::util::DebugFontTextWriter* pDebugFontWriter) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pDebugFontWriter);

    LogOutResourceManager& logOutResoure = LogOutResourceManager::GetInstance();
    logOutResoure.SetCustomDraw(true);
    logOutResoure.SetDevice(pDevice);
    logOutResoure.SetCommandBuffer(pCommandBuffer);
    logOutResoure.SetDebugFontTextWriter(pDebugFontWriter);
}

/**
 * @brief コマンドバッファを取得します。
 *
 * @return コマンドバッファを返します。
 * @details 内部実装関数です。
 */
nn::gfx::CommandBuffer& LogOut::GetCommandBuffer() NN_NOEXCEPT
{
    if (m_IsCustomDraw)
    {
        return *m_pCommandBuffer;
    }
    else
    {
        return m_pGraphicsSystem->GetCommandBuffer();
    }
}

/**
 * @brief デバッグフォントライターを取得します。
 *
 * @return デバッグフォントライターを返します。
 * @details 内部実装関数です。
 */
nn::gfx::util::DebugFontTextWriter& LogOut::GetDebugFont() NN_NOEXCEPT
{
    if (m_IsCustomDraw)
    {
        return *m_pDebugFontWriter;
    }
    else
    {
        return m_pGraphicsSystem->GetDebugFont();
    }
}


/**
 * @brief プリミティブレンダラーを取得します。
 *
 * @return プリミティブレンダラーを返します。
 * @details 内部実装関数です。
 */
PrimitiveRenderer::Renderer& LogOut::GetPrimitiveRenderer() const NN_NOEXCEPT
{
    return *m_pPrimitiveRenderer;
}

/**
* @brief プリミティブレンダラーの初期化を行います。
*
* @details プリミティブレンダラーの初期化を行います。
*/
void LogOut::InitializePrimitiveRenderer() NN_NOEXCEPT
{
    PrimitiveRenderer::RendererInfo info;
    info.SetDefault();

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )します
    info.SetMultiBufferQuantity(2);
    info.SetAllocator(AllocateFunction, &GetStandardAllocator());

    // PrimitiveRendererのインスタンス
    if (m_pDevice)
    {
        m_pPrimitiveRenderer = PrimitiveRenderer::CreateRenderer(m_pDevice, info);

        m_pPrimitiveRenderer->SetScreenWidth(static_cast<int>(m_Width));
        m_pPrimitiveRenderer->SetScreenHeight(static_cast<int>(m_Height));
    }
}

/**
* @brief プリミティブレンダラーの終了処理を行います。
*
* @details プリミティブレンダラーの終了処理を行います。
*/
void LogOut::FinalizePrimitiveRenderer() NN_NOEXCEPT
{
    if (m_pDevice)
    {
        PrimitiveRenderer::DestroyRenderer(
            m_pPrimitiveRenderer,
            m_pDevice,
            FreeFunction,
            &GetStandardAllocator()
        );
    }
}


/**
* @brief 確保関数です。
*
* @param[in]    size         確保サイズ
* @param[in]    alignment    アライメント
* @param[in]    pUserData    ユーザーデータ
* @return 確保したメモリを返します。
* @details 確保関数です。
*/
void* LogOut::AllocateFunction(size_t size,
    size_t alignment,
    void* pUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserData);
    return reinterpret_cast<
        nn::mem::StandardAllocator*>(pUserData)->Allocate(size, alignment);
}

/**
* @brief 解放関数です。
*
* @param[in]    ptr          解放するメモリ
* @param[in]    pUserData    ユーザーデータ
* @details 解放関数です。
*/
void LogOut::FreeFunction(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserData);
    reinterpret_cast<nn::mem::StandardAllocator*>(pUserData)->Free(ptr);
}

} // namespace gfxLog
} // namespace nns
