﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nns/gfx0/gfx0_DriverShim.h>
#include <nns/gfx0/gfx0_Driver.h>

namespace nns { namespace gfx0 {

namespace {

//-----------------------------------------------------------------------------
// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
nn::sf::SharedPointer<IGfx0Driver>  g_Gfx0Driver;

}   // namespace


//-----------------------------------------------------------------------------
//  グラフィックドライバのサービスの使用を開始します。DFC 用です。
//
void InitializeGfx0DriverByDfcForTest() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_Gfx0Driver);

    g_Gfx0Driver = CreateGfx0DriverByDfc();

    auto result = g_Gfx0Driver->WaitServerReady();
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
}


//-----------------------------------------------------------------------------
//  グラフィックドライバのサービスの使用を開始します。HIPC 用です。
//
void InitializeGfx0Driver() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_Gfx0Driver);

    g_Gfx0Driver = CreateGfx0DriverByHipc();

    auto result = g_Gfx0Driver->WaitServerReady();
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
}


//-----------------------------------------------------------------------------
//  グラフィックドライバのサービスの使用を終了します。
//
void FinalizeGfx0Driver() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Gfx0Driver);
    g_Gfx0Driver = nullptr;
}


//-----------------------------------------------------------------------------
//  スケルトン用のダミー関数
//
int32_t DummyFuncGfx0(int32_t x, int32_t y) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Gfx0Driver);

    int32_t retValue;
    auto result = g_Gfx0Driver->AddValue(&retValue, x, y);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);

    return retValue;
}



//-----------------------------------------------------------------------------
//  作成済みの nn::os::TransferMemory をサーバへ提供します。
//
void AttachTransferMemory(nn::os::TransferMemoryType* pTransferMemory, size_t size) NN_NOEXCEPT
{
    auto handle = nn::os::DetachTransferMemory( pTransferMemory );
    NN_SDK_ASSERT( handle != nn::os::InvalidNativeHandle );

    auto result = g_Gfx0Driver->AttachTransferMemoryImpl( nn::sf::NativeHandle(handle, true), size);

    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
}


//-----------------------------------------------------------------------------
//  特定のメモリ領域をデバイスアドレス空間用のメモリとしてサーバへ提供します。
//
void AttachDeviceMemory(uintptr_t address, size_t size) NN_NOEXCEPT
{
    auto processHandle = nn::dd::GetCurrentProcessHandle();

    auto result = g_Gfx0Driver->AttachDeviceMemory( nn::sf::NativeHandle(processHandle, false), address, size);

    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
}


//-----------------------------------------------------------------------------
//  サーバが作成した nn::os::SharedMemory を取得します。
//
void QuerySharedMemory(nn::os::SharedMemoryType* pSharedMemory) NN_NOEXCEPT
{
    // サーバプロセスは ilp64 のプロセスかもしれないので、
    // size の型は size_t ではなく uint64_t としておく。
    nn::sf::NativeHandle    handle;
    uint64_t                size;
    auto result = g_Gfx0Driver->QuerySharedMemoryImpl(&handle, &size);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_SDK_ASSERT(size < 0x100000000ull);
    NN_UNUSED(result);

    nn::os::AttachSharedMemory(pSharedMemory, size & 0xffffffff, handle.GetOsHandle(), handle.IsManaged());
    handle.Detach();
}


//-----------------------------------------------------------------------------
//  サーバが作成した nn::os::SystemEvent を取得します。
//
void QuerySystemEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    nn::sf::NativeHandle    handle;
    auto result = g_Gfx0Driver->QuerySystemEventImpl(&handle);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);

    nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();
}


//-----------------------------------------------------------------------------
//  サーバ内のデモプログラムの内部状態を１世代分進める
//
void AdvanceDemo() NN_NOEXCEPT
{
    g_Gfx0Driver->AdvanceDemo();
}


//-----------------------------------------------------------------------------

}} // namespace nns::gfx0

