﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>

namespace nns { namespace gfx0 {

//-----------------------------------------------------------------------------
//  IGfx0DriverSample クラス
//
//  TransferMemory、SharedMemory、SystemEvent を保持しており、
//  それぞれ IPC 先のプロセスと共有する。
//-----------------------------------------------------------------------------

class Gfx0DriverSample
{
private:
    nn::os::TransferMemoryType  m_TransferMemory;
    nn::os::SharedMemoryType    m_SharedMemory;
    nn::os::SystemEventType     m_SystemEvent;

    void*   m_TransferMemoryAddress;
    bool    m_TransferMemoryAvailable;

public:
    static const size_t    SharedMemorySize = 8192;

public:
    Gfx0DriverSample() NN_NOEXCEPT;
    ~Gfx0DriverSample() NN_NOEXCEPT;

    // Transfer Memory
    void AttachTransferMemory(nn::os::NativeHandle handle, bool isManaged, size_t size) NN_NOEXCEPT;

    void* GetTransferMemoryAddress() const
    {
        if (!m_TransferMemoryAvailable)
        {
            return NULL;
        }
        return m_TransferMemoryAddress;
    }

    // Shared Memory
    nn::os::NativeHandle GetSharedMemoryHandle() NN_NOEXCEPT
    {
        return nn::os::GetSharedMemoryHandle( &m_SharedMemory );
    }

    void* GetSharedMemoryAddress() NN_NOEXCEPT
    {
        return nn::os::GetSharedMemoryAddress( &m_SharedMemory );
    }

    uint64_t GetSharedMemorySize() NN_NOEXCEPT
    {
        return nn::os::GetSharedMemorySize( &m_SharedMemory );
    }


    // System Event
    nn::os::NativeHandle GetSystemEventHandle() NN_NOEXCEPT
    {
        return nn::os::GetReadableHandleOfSystemEvent( &m_SystemEvent );
    }

    void SignalSystemEvent() NN_NOEXCEPT
    {
        return nn::os::SignalSystemEvent( &m_SystemEvent );
    }
};

//-----------------------------------------------------------------------------

}}  // namespace nns::gfx0

