﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/dd.h>
#include <nn/result/result_HandlingUtility.h>

#include "gfx0_DriverImpl.h"

namespace nns { namespace gfx0 {

//-----------------------------------------------------------------------------
//  For IPC Test
//
nn::Result Gfx0DriverImpl::WaitServerReady() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}


nn::Result Gfx0DriverImpl::AddValue(nn::sf::Out<int32_t> pOut, int32_t x, int32_t y) NN_NOEXCEPT
{
    pOut.Set( x + y );

    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
//  TransferMemory をサーバへ提供する実装
//
nn::Result Gfx0DriverImpl::AttachTransferMemoryImpl(nn::sf::NativeHandle sfHandle, uint64_t size) NN_NOEXCEPT
{
    m_DriverSample.AttachTransferMemory( sfHandle.GetOsHandle(), sfHandle.IsManaged(), size & 0xffffffff );
    sfHandle.Detach();

    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
//  DeviceAddressSpace をサーバ側で利用する
//
nn::Result Gfx0DriverImpl::AttachDeviceMemory(nn::sf::NativeHandle sfHandle, uint64_t address, uint64_t size) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || \
    defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || \
    defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
    nn::dd::DeviceAddressSpaceType  das;

    // DeviceAddressSpace を構築
    auto result = nn::dd::CreateDeviceAddressSpace(&das, 0x100000000ull);
    NN_SDK_ASSERT( result.IsSuccess() );

    result = nn::dd::MapDeviceAddressSpaceNotAligned(
                            &das,
                            sfHandle.GetOsHandle(),
                            address,
                            size,
                            static_cast<nn::dd::DeviceVirtualAddress>(address),
                            nn::dd::MemoryPermission_ReadWrite);
    NN_SDK_ASSERT( result.IsSuccess() );

    result = nn::dd::AttachDeviceAddressSpace(&das, nn::dd::DeviceName_Sata);
    NN_SDK_ASSERT( result.IsSuccess() );

    // Do Nothing in this skeleton code.


    // DeviceAddressSpace を破棄
    nn::dd::DetachDeviceAddressSpace(&das, nn::dd::DeviceName_Sata);

    nn::dd::UnmapDeviceAddressSpace(
                            &das,
                            sfHandle.GetOsHandle(),
                            address,
                            size,
                            static_cast<nn::dd::DeviceVirtualAddress>(address));

    nn::dd::DestroyDeviceAddressSpace(&das);
#else
    NN_UNUSED(sfHandle);
    NN_UNUSED(address);
    NN_UNUSED(size);
#endif

    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
//  SharedMemory をサーバから取得する実装
//
nn::Result Gfx0DriverImpl::QuerySharedMemoryImpl(nn::sf::Out<nn::sf::NativeHandle> pOutHandle, nn::sf::Out<uint64_t> pOutSize) NN_NOEXCEPT
{
    auto   handle = m_DriverSample.GetSharedMemoryHandle();
    uint64_t size = m_DriverSample.GetSharedMemorySize();

    pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );
    pOutSize.Set( size );

    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
//  SystemEvent をサーバから取得する実装
//
nn::Result Gfx0DriverImpl::QuerySystemEventImpl(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    auto handle = m_DriverSample.GetSystemEventHandle();
    pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
//  デモプログラムを１世代分進める。具体的には以下の通り。
//  1)  クライアントプロセスから受取った TransferMemory の
//      先頭 32bit の int 値をインクリメントする。
//  2)  1) の int 値を SharedMemory 領域に ASCII 文字列で配置する。
//  3)  SystemEvent をシグナル化する。
//
void Gfx0DriverImpl::AdvanceDemo() NN_NOEXCEPT
{
    // 1) の処理
    int* valueAddress = reinterpret_cast<int*>(m_DriverSample.GetTransferMemoryAddress());
    int  value = *valueAddress;

    ++value;
    *valueAddress = value;

    // 2) の処理
    char*   dst = reinterpret_cast<char*>(m_DriverSample.GetSharedMemoryAddress());
    *dst++ = '0';
    *dst++ = 'x';
    for (int i=0; i<8; ++i)
    {
        char c  = static_cast<char>( (value >> ((7 - i) * 4)) & 0xf );
        *dst++ = (c < 10) ? ('0' + c) : ('A' + c - 10);
    }
    *dst = '\0';

    // 3) の処理
    m_DriverSample.SignalSystemEvent();
}


//-----------------------------------------------------------------------------

}}  // namespace nns::gfx0

