﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/gfx/gfx_PrimitiveRendererGfxRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nn/nn_SdkLog.h>
#include <nn/gfx/util/gfx_PrimitiveShape.h>

namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! 内部変数です。
//------------------------------------------------------------------------------
namespace
{

const size_t VertexStrideSizeTable[ VertexAttribute_CountMax ] =
{
    sizeof( nn::util::Float3 ),   // pos
    sizeof( nn::util::Float4 ),   // color
    sizeof( nn::util::Float2 ),   // uv
    sizeof( nn::util::Float3 ),   // normal
};


static const nn::gfx::util::PrimitiveShapeFormat VertexFormat_P    = static_cast< nn::gfx::util::PrimitiveShapeFormat >( nn::gfx::util::PrimitiveShapeFormat_Pos );
static const nn::gfx::util::PrimitiveShapeFormat VertexFormat_PN   = static_cast< nn::gfx::util::PrimitiveShapeFormat >( nn::gfx::util::PrimitiveShapeFormat_Pos | nn::gfx::util::PrimitiveShapeFormat_Normal );
static const nn::gfx::util::PrimitiveShapeFormat VertexFormat_PUv = static_cast< nn::gfx::util::PrimitiveShapeFormat >( nn::gfx::util::PrimitiveShapeFormat_Pos | nn::gfx::util::PrimitiveShapeFormat_Uv );
static const nn::gfx::util::PrimitiveShapeFormat VertexFormat_PNUv = static_cast< nn::gfx::util::PrimitiveShapeFormat >( nn::gfx::util::PrimitiveShapeFormat_Pos | nn::gfx::util::PrimitiveShapeFormat_Normal | nn::gfx::util::PrimitiveShapeFormat_Uv );

static const int LargeSphereSliceNum = 16;
static const int LargeSphereStackNum = 8;
static const int SmallSphereSliceNum = 8;
static const int SmallSphereStackNum = 4;

static const int LargeDiskDivideNum = 32;
static const int SmallDiskDivideNum = 16;

static const int UpperHalfSphereDivideNum = 10;

static const int PipeDivideNum = 10;
static const int CylinderDivideNum = 10;
static const int ConeDivideNum = 10;


// 頂点数
static const int VertexTriangleCount = 3;
static const int VertexPointCount = 1;
static const int VertexLineCount = 2;

// インデックス数
static const int IndexSolidTriangleCount = 3;
static const int IndexWiredTriangleCount = 4;
static const int IndexPointCount = 1;
static const int IndexLineCount = 2;

static const size_t ShapeClassSizeList[ShapeType_CountMax] =
{
    sizeof(nn::gfx::util::QuadShape),       // solid quad
    sizeof(nn::gfx::util::QuadShape),       // solid textured quad
    sizeof(nn::gfx::util::QuadShape),       // wired quad
    sizeof(nn::gfx::util::CubeShape),       // solid cube
    sizeof(nn::gfx::util::CubeShape),       // textured cube
    sizeof(nn::gfx::util::CubeShape),       // wired cube
    sizeof(nn::gfx::util::SphereShape),     // large solid sphere
    sizeof(nn::gfx::util::SphereShape),     // large texturedsphere
    sizeof(nn::gfx::util::SphereShape),     // large wired sphere
    sizeof(nn::gfx::util::SphereShape),     // small solid sphere
    sizeof(nn::gfx::util::SphereShape),     // small textured sphere
    sizeof(nn::gfx::util::SphereShape),     // small wired sphere
    0,                                      // line
    0,                                      // point
    0,                                      // solid triangle
    0,                                      // wired triangle
    sizeof(nn::gfx::util::CircleShape),     // large solid circle
    sizeof(nn::gfx::util::CircleShape),     // large textured circle
    sizeof(nn::gfx::util::CircleShape),     // large wired circle
    sizeof(nn::gfx::util::CircleShape),     // small solid circle
    sizeof(nn::gfx::util::CircleShape),     // small textured circle
    sizeof(nn::gfx::util::CircleShape),     // small wired circle
    sizeof(nn::gfx::util::QuadShape),       // screen quad
    sizeof(nn::gfx::util::QuadShape),       // flip screen quad
    sizeof(nn::gfx::util::HemiSphereShape), // solid upper half sphere
    sizeof(nn::gfx::util::HemiSphereShape), // textured upper half sphere
    sizeof(nn::gfx::util::HemiSphereShape), // wired upper half sphere
    sizeof(nn::gfx::util::PipeShape),       // solid pipe
    sizeof(nn::gfx::util::PipeShape),       // textured pipe
    sizeof(nn::gfx::util::PipeShape),       // wired pipe
    sizeof(nn::gfx::util::CylinderShape),   // solid cylinder
    sizeof(nn::gfx::util::CylinderShape),   // textured cylinder
    sizeof(nn::gfx::util::CylinderShape),   // wired cylinder
    sizeof(nn::gfx::util::ConeShape),       // solid cone
    sizeof(nn::gfx::util::ConeShape),       // textured cone
    sizeof(nn::gfx::util::ConeShape),       // wired cone
};
} // namespace


//------------------------------------------------------------------------------
//! 内部関数の実装です。
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//! 外部関数の実装です。
//------------------------------------------------------------------------------
void ConvertUint8x4ToFloat4( nn::util::Float4* pOutValue,
                             const nn::util::Uint8x4 source )
{
    pOutValue->v[0] = static_cast<float>(source.v[0]) / 255.f;
    pOutValue->v[1] = static_cast<float>(source.v[1]) / 255.f;
    pOutValue->v[2] = static_cast<float>(source.v[2]) / 255.f;
    pOutValue->v[3] = static_cast<float>(source.v[3]) / 255.f;
}

void ConvertMatrix4x3ToFloatT4x4( nn::util::FloatT4x4* pOut4x4,
                                  const nn::util::Matrix4x3fType& inMatrix4x3 )
{
    nn::util::Matrix4x4fType modelMtx4x4f;
    nn::util::MatrixConvert(&modelMtx4x4f, inMatrix4x3);
    nn::util::MatrixStore(pOut4x4, modelMtx4x4f);
}

//! @brief コンストラクタです。
PrimitiveMesh::PrimitiveMesh() : m_pGpuBuffer(nullptr),
                                 m_IndexBufferPointer(nullptr),
                                 m_VertexCount(0),
                                 m_IndexCount(0)
{
    for( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        m_VertexBufferPointer[idx] = nullptr;
    }
}

PrimitiveMesh::~PrimitiveMesh()
{
}

bool PrimitiveMesh::Initialize( GpuBuffer* pGpuBuffer,
                                int numVertices,
                                int numIndices,
                                VertexFormat vertexFormat )

{
    m_pGpuBuffer = pGpuBuffer;
    m_VertexCount = numVertices;
    m_IndexCount = numIndices;

    const size_t bufferAlignment = m_pGpuBuffer->GetBufferAlignment();

    // 現状のallocate済サイズを取得する。
    // 全体で何バイトのメモリが必要なのか、アラインメントを考慮して計算する。
    size_t bufferSize = m_pGpuBuffer->GetAllocatedSize();

    // 必要なサイズを、アラインメントを考慮して計算する
    for ( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        if ( vertexFormat & (1 << idx) )
        {
            const VertexAttribute attrib = static_cast<VertexAttribute>(idx);
            bufferSize = nn::util::align_up(bufferSize + GetVertexMemorySize(attrib), bufferAlignment);
        }
    }

    // インデックスバッファのサイズ分追加。アラインメントを考慮
    bufferSize = nn::util::align_up(bufferSize + sizeof(uint32_t) * m_IndexCount, bufferAlignment);

    size_t allocateSize = bufferSize - m_pGpuBuffer->GetAllocatedSize();

    // 必要なメモリを確保
    void* pBuffer = m_pGpuBuffer->Allocate(allocateSize);

    // メモリ確保に失敗した場合は、falseを返す。
    if ( pBuffer == nullptr )
    {
        NN_SDK_LOG("[GfxPrimitiveRenderer] Warning!! pBuffer is NULL\n");
        return false;
    }

    // 各頂点バッファのポインタを計算する。
    for( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        if ( vertexFormat & (1 << idx) )
        {
            const VertexAttribute attrib = static_cast<VertexAttribute>(idx);
            m_VertexBufferPointer[idx] = nn::util::BytePtr( pBuffer ).AlignUp(bufferAlignment).Get();
            pBuffer = nn::util::BytePtr( m_VertexBufferPointer[idx] ).Advance( GetVertexMemorySize(attrib) ).Get();
        }
        else
        {
            m_VertexBufferPointer[idx] = nullptr;
        }
    }

    // インデックスバッファのポインタを計算する。
    m_IndexBufferPointer = nn::util::BytePtr( pBuffer ).AlignUp(bufferAlignment).Get();

    return true;
}

int PrimitiveMesh::GetVertexCount() const
{
    return m_VertexCount;
}

int PrimitiveMesh::GetIndexCount() const
{
    return m_IndexCount;
}

size_t PrimitiveMesh::GetVertexStride( VertexAttribute attrib ) const
{
    NN_SDK_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    return VertexStrideSizeTable[ attrib ];
}

size_t PrimitiveMesh::GetVertexMemorySize( VertexAttribute attrib ) const
{
    NN_SDK_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    size_t elementSize = 0;
    switch( attrib )
    {
        case VertexAttribute_Pos:
            elementSize = sizeof(nn::util::Float3);
            break;
        case VertexAttribute_Uv:
            elementSize = sizeof(nn::util::Float2);
            break;
        case VertexAttribute_Color:
            elementSize = sizeof(nn::util::Float4);
            break;
        case VertexAttribute_Normal:
            elementSize = sizeof( nn::util::Float3 );
            break;
        case VertexAttribute_CountMax:
        default: NN_UNEXPECTED_DEFAULT;
            break;
    }
    return elementSize*m_VertexCount;
}

void* PrimitiveMesh::GetVertexBufferCpuAddress( VertexAttribute attribute )
{
    NN_SDK_ASSERT(0 <= attribute && attribute < VertexAttribute_CountMax);
    return m_VertexBufferPointer[ attribute ];

}

uint32_t* PrimitiveMesh::GetIndexBufferCpuAddress()
{
    return static_cast<uint32_t*>(m_IndexBufferPointer);
}

//! @brief 指定された頂点属性のメモリサイズを取得します。
//! @param[in] attrib   頂点属性です。
//! @param[out] pGpuAddress
void PrimitiveMesh::GetVertexBufferGpuAddress( VertexAttribute attrib, nn::gfx::GpuAddress* pGpuAddress) const
{
    NN_SDK_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    m_pGpuBuffer->GetGpuAddress( pGpuAddress, m_VertexBufferPointer[ attrib ] );
}

bool PrimitiveMesh::IsValidVertexBuffer( VertexAttribute attrib ) const
{
    NN_SDK_ASSERT(0 <= attrib && attrib < VertexAttribute_CountMax);
    return m_VertexBufferPointer[ attrib ] != nullptr;
}

bool PrimitiveMesh::IsValidIndexBuffer() const
{
    return m_IndexBufferPointer != nullptr;
}

void PrimitiveMesh::GetIndexBufferGpuAddress( nn::gfx::GpuAddress* pGpuAddress ) const
{
    m_pGpuBuffer->GetGpuAddress( pGpuAddress, m_IndexBufferPointer );
}

VertexFormat PrimitiveMesh::GetVertexFormat() const
{
    VertexFormat vertexFormat = static_cast<VertexFormat>(0);

    // 各頂点バッファのポインタチェック
    for( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        if ( m_VertexBufferPointer[idx] != nullptr )
        {
            vertexFormat = static_cast<VertexFormat>(vertexFormat | static_cast<VertexFormat>(1 << idx));
        }
    }

    return vertexFormat;
}

template<typename T>
bool PrimitiveMesh::CopyVertexBuffer( VertexAttribute attribute, const T* pPodData, size_t size )
{
    NN_SDK_ASSERT(0 <= attribute && attribute < VertexAttribute_CountMax);
    T* pDst = static_cast<T*>(GetVertexBufferCpuAddress( attribute ));
    if( pDst != nullptr )
    {
        std::memcpy( pDst, pPodData, size );
        return true;
    }
    return false;
}

bool PrimitiveMesh::CopyIndexBuffer( const uint32_t* pPodData, size_t size )
{
    uint32_t* pDst = GetIndexBufferCpuAddress();
    if( pDst != nullptr )
    {
        std::memcpy( pDst, pPodData, size );
        return true;
    }
    return false;
}

MeshSet::MeshSet()
{
}

MeshSet::~MeshSet()
{
}

PrimitiveMesh* MeshSet::Get(ShapeType shapeType)
{
    NN_SDK_ASSERT(0 <= shapeType && shapeType < ShapeType_CountMax);
    return &(m_Mesh[ shapeType ]);
}

nn::gfx::util:: PrimitiveShape* MeshSet::GetShape(ShapeType shapeType)
{
    NN_SDK_ASSERT(0 <= shapeType && shapeType < ShapeType_CountMax);
    return m_pShapes[ shapeType ];
}

template <typename T>
void GetPrimitiveShapeVertexIndexSize(
    int* vertexCount, int* IndexCount,
    nn::gfx::util::PrimitiveShapeFormat format,
    nn::gfx::PrimitiveTopology topology )
{
    *vertexCount = 0;
    *IndexCount  = 0;

    T pCreatedShape( format, topology );
    *vertexCount = pCreatedShape.GetVertexCount();
    *IndexCount  = pCreatedShape.GetIndexCount();
}

template <typename T>
void GetPrimitiveShapeVertexIndexSize(
    int* vertexCount, int* IndexCount,
    nn::gfx::util::PrimitiveShapeFormat format,
    nn::gfx::PrimitiveTopology topology,
    int param1 )
{
    *vertexCount = 0;
    *IndexCount  = 0;

    T pCreatedShape( format, topology, param1 );
    *vertexCount = pCreatedShape.GetVertexCount();
    *IndexCount  = pCreatedShape.GetIndexCount();
}

template <typename T>
void GetPrimitiveShapeVertexIndexSize(
    int* vertexCount, int* IndexCount,
    nn::gfx::util::PrimitiveShapeFormat format,
    nn::gfx::PrimitiveTopology topology,
    int param1, int param2 )
{
    *vertexCount = 0;
    *IndexCount  = 0;

    T pCreatedShape( format, topology, param1, param2 );
    *vertexCount = pCreatedShape.GetVertexCount();
    *IndexCount  = pCreatedShape.GetIndexCount();
}

size_t CalculateShapeSize( int vertexCount, int indexCount, const size_t alignmentSize )
{
    size_t shapeSize = 0;

    shapeSize += vertexCount * ( sizeof( nn::util::Float3 ) + sizeof( nn::util::Float3 ) + sizeof( nn::util::Float2 ) );
    shapeSize += 3 * alignmentSize;
    shapeSize += indexCount * sizeof( uint32_t );
    shapeSize += alignmentSize;

    return shapeSize;
}

size_t MeshSet::CalculateMemoryPoolSize( const size_t alignmentSize )
{
    size_t totalSize = 0;

    // 頂点バッファ、インデックスバッファのサイズ
    // 各要素が最大の要素を使用されたことを想定して使用サイズを計算する。
    int vertexCount = 0;
    int indexCount  = 0;

    // ShapeType_Quad
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::QuadShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_QuadTextured
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::QuadShape>( &vertexCount, &indexCount, VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_QuadWired
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::QuadShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineStrip );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_Cube
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CubeShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CubeTextured
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CubeShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CubeWired
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CubeShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineList );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_Sphere
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::SphereShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, LargeSphereSliceNum, LargeSphereStackNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_SphereTextured
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::SphereShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, LargeSphereSliceNum, LargeSphereStackNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_SphereWired
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::SphereShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, LargeSphereSliceNum, LargeSphereStackNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_SphereCoarse
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::SphereShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, SmallSphereSliceNum, SmallSphereStackNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_SphereTexturedCoarse
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::SphereShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, SmallSphereSliceNum, SmallSphereStackNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_SphereWiredCoarse
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::SphereShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, SmallSphereSliceNum, SmallSphereStackNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_Line:
    totalSize += CalculateShapeSize( VertexLineCount, IndexLineCount, alignmentSize );

    // ShapeType_Point:
    totalSize += CalculateShapeSize( VertexPointCount, IndexPointCount, alignmentSize );

    // ShapeType_Triangle:
    totalSize += CalculateShapeSize( VertexTriangleCount, IndexSolidTriangleCount, alignmentSize );

    // ShapeType_TriangleWired:
    totalSize += CalculateShapeSize( VertexTriangleCount, IndexWiredTriangleCount, alignmentSize );

    // ShapeType_Circle:
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CircleShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, LargeDiskDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CircleTextured:
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CircleShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, LargeDiskDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

     // ShapeType_CircleWired:
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CircleShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineStrip, LargeDiskDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CircleCoarse:
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CircleShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, SmallDiskDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CircleTexturedCoarse:
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CircleShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, SmallDiskDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CircleWiredCoarse:
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CircleShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineStrip, SmallDiskDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_ScreenQuad
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::QuadShape>( &vertexCount, &indexCount, VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_ScreenYFlipQuad
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::QuadShape>( &vertexCount, &indexCount, VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_UpperHalfSphere
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::HemiSphereShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, UpperHalfSphereDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_UpperHalfSphereTextured
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::HemiSphereShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, UpperHalfSphereDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_UpperHalfSphereWired
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::HemiSphereShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, UpperHalfSphereDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_Pipe
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::PipeShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, PipeDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_PipeTextured
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::PipeShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, PipeDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_PipeWired
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::PipeShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, PipeDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_Cylinder
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CylinderShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, CylinderDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CylinderTextured
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CylinderShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, CylinderDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_CylinderWired
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::CylinderShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, CylinderDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_Cone
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::ConeShape>( &vertexCount, &indexCount, VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, ConeDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_ConeTextured
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::ConeShape>( &vertexCount, &indexCount, VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, ConeDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_ConeWired
    GetPrimitiveShapeVertexIndexSize<nn::gfx::util::ConeShape>( &vertexCount, &indexCount, VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, ConeDivideNum );
    totalSize += CalculateShapeSize( vertexCount, indexCount, alignmentSize );

    // ShapeType_User
    totalSize += CalculateShapeSize( 0, 0, alignmentSize );

    return totalSize;
}  //NOLINT(impl/function_size)

size_t MeshSet::GetRequiredMemorySize()
{
    size_t  result = 0;

    for (int i = 0; i < ShapeType_CountMax; i++)
    {
        result += ShapeClassSizeList[i];
    }
    return result;
}

// 各メッシュ生成の実装です。
bool CreateTriangle(float width, float height, bool bWired, GpuBuffer* pGpuBuffer, PrimitiveMesh* pMeshBuffer)
{
    NN_SDK_ASSERT_NOT_NULL(pGpuBuffer);
    NN_SDK_ASSERT_NOT_NULL(pMeshBuffer);

    static const uint32_t VertexIndicesWired[4] =
    {
        0, 1, 2, 0,
    };

    static const uint32_t VertexIndices[3] =
    {
        0, 2, 1
    };

    const nn::util::Float3 VertexPos[VertexTriangleCount] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER(0.f, height, 0.f),
        NN_UTIL_FLOAT_3_INITIALIZER(width, height, 0.f),
        NN_UTIL_FLOAT_3_INITIALIZER(0.f,    0.f, 0.f)
    };

    const nn::util::Float3 VertexNormal[VertexTriangleCount] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER(0.f, 0.f, 1.f),
        NN_UTIL_FLOAT_3_INITIALIZER(0.f, 0.f, 1.f),
        NN_UTIL_FLOAT_3_INITIALIZER(0.f, 0.f, 1.f),
    };

    const nn::util::Float2 VertexUv[VertexTriangleCount] =
    {
        NN_UTIL_FLOAT_2_INITIALIZER(0.f, 0.f),
        NN_UTIL_FLOAT_2_INITIALIZER(1.f, 0.f),
        NN_UTIL_FLOAT_2_INITIALIZER(0.f, 1.f),
    };

    // インデックス数
    const int numIndices = (bWired) ? IndexWiredTriangleCount : IndexSolidTriangleCount;

    // 頂点フォーマット
    VertexFormat vertexFormat = (bWired) ? VertexFormat_Pos : VertexFormat_Default;

    if (pMeshBuffer->Initialize(pGpuBuffer, VertexTriangleCount, numIndices, vertexFormat) == false)
    {
        return false;
    }

    // インデックスデータをコピー
    const size_t indexMemorySize = (bWired) ? sizeof(VertexIndicesWired) : sizeof(VertexIndices);
    pMeshBuffer->CopyIndexBuffer((bWired) ? VertexIndicesWired : VertexIndices,
        indexMemorySize);

    // 頂点データをコピー
    pMeshBuffer->CopyVertexBuffer(VertexAttribute_Pos, VertexPos, sizeof(VertexPos));
    if (!bWired)
    {
        pMeshBuffer->CopyVertexBuffer(VertexAttribute_Uv, VertexUv, sizeof(VertexUv));
        pMeshBuffer->CopyVertexBuffer(VertexAttribute_Normal, VertexNormal, sizeof(VertexNormal));
    }

    return true;
}

bool CreatePoint( GpuBuffer* pGpuBuffer,
                  PrimitiveMesh* pMeshBuffer )
{
    NN_SDK_ASSERT_NOT_NULL(pGpuBuffer);
    NN_SDK_ASSERT_NOT_NULL(pMeshBuffer);

    static const nn::util::Float3 VertexPointPos[1] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER( 0.f, 0.f, 0.f )
    };

    if ( pMeshBuffer->Initialize(pGpuBuffer, VertexPointCount, IndexPointCount, VertexFormat_Pos) == false )
    {
        return false;
    }

    uint32_t* pIndexData = pMeshBuffer->GetIndexBufferCpuAddress();
    for( int idx = 0; idx < IndexPointCount; ++idx )
    {
        pIndexData[ idx ] = idx;
    }

    pMeshBuffer->CopyVertexBuffer( VertexAttribute_Pos, VertexPointPos,
                                   VertexPointCount * sizeof(nn::util::Float3) );

    return true;
}

bool CreateLine( GpuBuffer* pGpuBuffer,
                 PrimitiveMesh* pMeshBuffer )
{
    NN_SDK_ASSERT_NOT_NULL( pGpuBuffer );
    NN_SDK_ASSERT_NOT_NULL( pMeshBuffer );

    static const nn::util::Float3 VertexLinePos[VertexLineCount] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER( 0.f, 0.f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( 1.f, 0.f, 0.f ),
    };

    static const nn::util::Float2 VertexUv[VertexLineCount] =
    {
        NN_UTIL_FLOAT_2_INITIALIZER( 0.f, 0.5f ),
        NN_UTIL_FLOAT_2_INITIALIZER( 1.f, 0.5f ),
    };

    if ( pMeshBuffer->Initialize(pGpuBuffer, VertexLineCount, IndexLineCount, VertexFormat_Default) == false )
    {
        return false;
    }

     uint32_t* pIndexData = pMeshBuffer->GetIndexBufferCpuAddress();
    for( int idx = 0; idx < IndexLineCount; ++idx )
    {
        pIndexData[ idx ] = idx;
    }

    size_t vertexMemorySize = VertexLineCount * sizeof(nn::util::Float3);
    pMeshBuffer->CopyVertexBuffer( VertexAttribute_Pos, VertexLinePos, vertexMemorySize );

    nn::util::Float2* pUv = static_cast<nn::util::Float2*>(pMeshBuffer->GetVertexBufferCpuAddress( VertexAttribute_Uv ));
    for( int idx = 0; idx < VertexLineCount; ++idx )
    {
        pUv[ idx]     = VertexUv[ idx % 2 ];
    }

    return true;
}

bool MeshSet::Initialize(GpuBuffer* pGpuBuffer, nn::util::BytePtr& pMemory, size_t memorySize)
{
    NN_SDK_ASSERT(memorySize >= GetRequiredMemorySize());

    // メッシュ用のメモリ領域が足りるかチェックする。
    if (memorySize < MeshSet::GetRequiredMemorySize())
    {
        return false;
    }

    //  ShapeType_Quad
    CreateShape<nn::gfx::util::QuadShape>( m_pShapes[ ShapeType_Quad ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory );

    //  ShapeType_QuadTextured
    CreateShape<nn::gfx::util::QuadShape>( m_pShapes[ ShapeType_QuadTextured ], VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory );

    //  ShapeType_QuadWired
    CreateShape<nn::gfx::util::QuadShape>( m_pShapes[ ShapeType_QuadWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineStrip, pGpuBuffer, pMemory );

    //  ShapeType_Cube
    CreateShape<nn::gfx::util::CubeShape>( m_pShapes[ ShapeType_Cube ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory );

    //  ShapeType_CubeTextured
    CreateShape<nn::gfx::util::CubeShape>( m_pShapes[ ShapeType_CubeTextured ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory );

    //  ShapeType_CubeWired
    CreateShape<nn::gfx::util::CubeShape>( m_pShapes[ ShapeType_CubeWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, pGpuBuffer, pMemory );

    //  ShapeType_Sphere
    CreateShape<nn::gfx::util::SphereShape>( m_pShapes[ ShapeType_Sphere ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, LargeSphereSliceNum, LargeSphereStackNum );

    //  ShapeType_SphereTextured
    CreateShape<nn::gfx::util::SphereShape>( m_pShapes[ ShapeType_SphereTextured ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, LargeSphereSliceNum, LargeSphereStackNum );

    //  ShapeType_SphereWired
    CreateShape<nn::gfx::util::SphereShape>( m_pShapes[ ShapeType_SphereWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, pGpuBuffer, pMemory, LargeSphereSliceNum, LargeSphereStackNum );

    //  ShapeType_SphereCoarse
    CreateShape<nn::gfx::util::SphereShape>( m_pShapes[ShapeType_SphereCoarse], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, SmallSphereSliceNum, SmallSphereStackNum);

    //  ShapeType_SphereTexturedCoarse
    CreateShape<nn::gfx::util::SphereShape>( m_pShapes[ShapeType_SphereTexturedCoarse], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, SmallSphereSliceNum, SmallSphereStackNum);

    //  ShapeType_SphereWiredCoarse
    CreateShape<nn::gfx::util::SphereShape>( m_pShapes[ShapeType_SphereWiredCoarse], VertexFormat_PN, nn::gfx::PrimitiveTopology_LineList, pGpuBuffer, pMemory, SmallSphereSliceNum, SmallSphereStackNum);

    //  ShapeType_Line
    //  ShapeType_Point
    //  ShapeType_Triangle
    //  ShapeType_TriangleWired
    //  Point と Line と Triangle は PrimitiveShape に用意されていないので、PrimitiveShape 置き換え前の機能を利用する。

    //  ShapeType_Circle
    CreateShape<nn::gfx::util::CircleShape>( m_pShapes[ ShapeType_Circle ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, LargeDiskDivideNum );

    //  ShapeType_CircleTextured
    CreateShape<nn::gfx::util::CircleShape>( m_pShapes[ ShapeType_CircleTextured ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, LargeDiskDivideNum );

    //  ShapeType_CircleWired
    CreateShape<nn::gfx::util::CircleShape>( m_pShapes[ ShapeType_CircleWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineStrip, pGpuBuffer, pMemory, LargeDiskDivideNum );

    //  ShapeType_CircleCoarse
    CreateShape<nn::gfx::util::CircleShape>( m_pShapes[ ShapeType_CircleCoarse ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, SmallDiskDivideNum );

    //  ShapeType_CircleCoarseTextured
    CreateShape<nn::gfx::util::CircleShape>( m_pShapes[ ShapeType_CircleTexturedCoarse ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, SmallDiskDivideNum );

    //  ShapeType_CircleWiredCoarse
    CreateShape<nn::gfx::util::CircleShape>( m_pShapes[ ShapeType_CircleWiredCoarse ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineStrip, pGpuBuffer, pMemory, SmallDiskDivideNum );

    //  ShapeType_ScreenQuad
    CreateShape<nn::gfx::util::QuadShape>( m_pShapes[ ShapeType_ScreenQuad ], VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory );

    //  ShapeType_ScreenYFlipQuad
    CreateShape<nn::gfx::util::QuadShape>( m_pShapes[ ShapeType_ScreenYFlipQuad ], VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory );

    //  ShapeType_UpperHalfSphere
    CreateShape<nn::gfx::util::HemiSphereShape>( m_pShapes[ ShapeType_UpperHalfSphere ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, UpperHalfSphereDivideNum );

    //  ShapeType_UpperHalfSphereTextured
    CreateShape<nn::gfx::util::HemiSphereShape>( m_pShapes[ ShapeType_UpperHalfSphereTextured ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, UpperHalfSphereDivideNum );

    //  ShapeType_UpperHalfSphereWired
    CreateShape<nn::gfx::util::HemiSphereShape>( m_pShapes[ ShapeType_UpperHalfSphereWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, pGpuBuffer, pMemory, UpperHalfSphereDivideNum );

    //  ShapeType_Pipe
    CreateShape<nn::gfx::util::PipeShape>( m_pShapes[ ShapeType_Pipe ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, PipeDivideNum );

    //  ShapeType_PipeTextured
    CreateShape<nn::gfx::util::PipeShape>( m_pShapes[ ShapeType_PipeTextured ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, PipeDivideNum );

    //  ShapeType_PipeWired
    CreateShape<nn::gfx::util::PipeShape>( m_pShapes[ ShapeType_PipeWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, pGpuBuffer, pMemory, PipeDivideNum );

    //  ShapeType_Cylinder
    CreateShape<nn::gfx::util::CylinderShape>( m_pShapes[ ShapeType_Cylinder ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, CylinderDivideNum );

    //  ShapeType_CylinderTextured
    CreateShape<nn::gfx::util::CylinderShape>( m_pShapes[ ShapeType_CylinderTextured ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, CylinderDivideNum );

    //  ShapeType_CylinderWired
    CreateShape<nn::gfx::util::CylinderShape>( m_pShapes[ ShapeType_CylinderWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, pGpuBuffer, pMemory, CylinderDivideNum );

    //  ShapeType_Cone
    CreateShape<nn::gfx::util::ConeShape>( m_pShapes[ ShapeType_Cone ], VertexFormat_PN, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, ConeDivideNum );

    //  ShapeType_ConeTextured
    CreateShape<nn::gfx::util::ConeShape>( m_pShapes[ ShapeType_ConeTextured ], VertexFormat_PNUv, nn::gfx::PrimitiveTopology_TriangleList, pGpuBuffer, pMemory, ConeDivideNum );

    //  ShapeType_ConeWired
    CreateShape<nn::gfx::util::ConeShape>( m_pShapes[ ShapeType_ConeWired ], VertexFormat_P, nn::gfx::PrimitiveTopology_LineList, pGpuBuffer, pMemory, ConeDivideNum );

    return true;
}

void MeshSet::AllocateShapeGpuMemory(
    nn::gfx::util::PrimitiveShape* pShape,
    GpuBuffer* pGpuBuffer)
{
    size_t vertexBufferSize = pShape->GetVertexBufferSize();
    size_t indexBufferSize = pShape->GetIndexBufferSize();
    pShape->Calculate(
        pGpuBuffer->Allocate(vertexBufferSize), vertexBufferSize,
        pGpuBuffer->Allocate(indexBufferSize), indexBufferSize);
}

// デフォルトメッシュの初期化
bool GraphicsResource::InitializeDefaultMeshSet( nn::gfx::Device*       pGfxDevice,
                                                 nn::gfx::MemoryPool*   pMemoryPool,
                                                 ptrdiff_t              memoryPoolOffset,
                                                 size_t                 memoryPoolSize,
                                                 nn::util::BytePtr&     pMemory,
                                                 size_t                 memorySize)
{
    nns::gfx::GpuBuffer::InitializeArg arg;
    arg.SetBufferCount(1);
    arg.SetGpuAccessFlag( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer );

    const size_t alignment = nns::gfx::GpuBuffer::GetGpuBufferAlignement(pGfxDevice, arg);
    const size_t requiredSize = MeshSet::CalculateMemoryPoolSize(alignment);
    arg.SetBufferSize(requiredSize);

    size_t offset = memoryPoolOffset;

    // アライメントを揃えます。
    if ( offset != 0 )
    {
        offset = nn::util::align_up( offset, alignment );
    }

    // メモリサイズをチェックします。
    if( (offset - memoryPoolOffset) + requiredSize >= memoryPoolSize )
    {
        return false;
    }

    if ( m_GpuBuffer.Initialize(pGfxDevice, arg, pMemoryPool, offset) == false )
    {
        return false;
    }

    m_GpuBuffer.Map(0);

    m_DefaultMeshSet.Initialize(&m_GpuBuffer, pMemory, memorySize);

    //  ShapeType_Point
    PrimitiveMesh* pMeshBuffer = m_DefaultMeshSet.Get(ShapeType_Point);
    if (CreatePoint(&m_GpuBuffer, pMeshBuffer) == false)
    {
        return false;
    }

    //  ShapeType_Line
    pMeshBuffer = m_DefaultMeshSet.Get(ShapeType_Line);
    if (CreateLine(&m_GpuBuffer, pMeshBuffer) == false)
    {
        return false;
    }

    pMeshBuffer = m_DefaultMeshSet.Get(ShapeType_Triangle);
    if (CreateTriangle(1.f, 1.f, false, &m_GpuBuffer, pMeshBuffer) == false)
    {
        return false;
    }

    pMeshBuffer = m_DefaultMeshSet.Get(ShapeType_TriangleWired);
    if (CreateTriangle(1.f, 1.f, true, &m_GpuBuffer, pMeshBuffer) == false)
    {
        return false;
    }

    m_GpuBuffer.Unmap();

    return true;
} //NOLINT(impl/function_size)


// デフォルトメッシュの破棄
void GraphicsResource::FinalizeDefaultMeshSet( nn::gfx::Device* pGfxDevice )
{
    m_GpuBuffer.Finalize( pGfxDevice );
}

} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
