﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/gfx_MemoryPool.h>

#include <nns/dbgui/dbgui_InterfaceInfo.h>



namespace nns { namespace dbgui {

class Allocator
{
    NN_DISALLOW_COPY(Allocator);

private:
    struct GpuMemory
    {
        nn::gfx::MemoryPool     m_MemoryPool;
        void*                   pBuffer;
        size_t                  size;
        ptrdiff_t               offset;
    };

    GpuMemory                                   m_GpuMemory;

    nn::AlignedAllocateFunctionWithUserData     m_pMemoryCallbackAllocate;
    nn::FreeFunctionWithUserData                m_pMemoryCallbackFree;
    void*                                       m_pMemoryCallbackUserData;


public:
            Allocator();
            ~Allocator();


    void    Initialize(
                nn::AlignedAllocateFunctionWithUserData pMemoryCallbackAllocate,
                nn::FreeFunctionWithUserData pMemoryCallbackFree,
                void* pMemoryCallbackUserData);
    void    Finalize(nn::gfx::Device* pDevice);

    void    InitializeGpuMemoryPool(
                nn::gfx::Device* pDevice, size_t requiredPoolMemorySize, int memoryPoolPropertyFlags);
    void    FinalizeGpuMemoryPool(nn::gfx::Device* pDevice);

    void*   AllocateMemory(size_t size, size_t alignement);
    void    FreeMemory(void* pMemory);

    template<typename T>
    T*      Allocate();
    template<typename T>
    void    Free(T* pMemory);
    template<typename T>

    T*      AllocateArray(int count);
    template<typename T>
    void    FreeArray(T* pMemory, int count);

    ptrdiff_t               AllocateGpuMemoryPool(size_t size, size_t alignement);
    nn::gfx::MemoryPool*    GetGpuMemoryPool();
};


inline void* Allocator::AllocateMemory(size_t size, size_t alignement)
{
    if (size == 0)
        return nullptr;

    void* pMemory = m_pMemoryCallbackAllocate(size, alignement, m_pMemoryCallbackUserData);
    NN_SDK_ASSERT_NOT_NULL(pMemory);
    return pMemory;
}

inline void Allocator::FreeMemory(void* pMemory)
{
    m_pMemoryCallbackFree(pMemory, m_pMemoryCallbackUserData);
}

template<typename T>
inline T* Allocator::Allocate()
{
    void* pMemory = AllocateMemory(sizeof(T), alignof(T));
    return new (pMemory) T();
}

template<typename T>
inline void Allocator::Free(T* pMemory)
{
    pMemory->~T();
    FreeMemory(pMemory);
}

template<typename T>
inline T* Allocator::AllocateArray(int count)
{
    T* pMemory = reinterpret_cast<T*>(AllocateMemory(sizeof(T) * count, alignof(T)));
    for (int instanceIndex = 0; instanceIndex < count; ++instanceIndex)
    {
        new (pMemory + instanceIndex) T();
    }

    return pMemory;
}

template<typename T>
inline void Allocator::FreeArray(T* pMemory, int count)
{
    for (int instanceIndex = 0; instanceIndex < count; ++instanceIndex)
    {
        (pMemory + instanceIndex)->~T();
    }

    FreeMemory(pMemory);
}

inline nn::gfx::MemoryPool* Allocator::GetGpuMemoryPool()
{
    return &m_GpuMemory.m_MemoryPool;
}


} } // namespace nns { namespace dbgui {
