﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{DebugTextRenderer.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a singleton class that
 *  renders text to the screen. This class sets
 *  up the font textures, provides functionality
 *  to control size and placement of text, and sets
 *  up a command buffer to draw the text to a given
 *  render target.
 */

#include <nn/nn_Assert.h>
#include <vector>
#include <nvntutorial/DebugTextRenderer.h>
#include <nvntutorial/AssetFileDataHolder.h>
#include <nvntutorial/AssetFileLoadingHelper.h>
#include <nvntutorial/TextureIDManager.h>
#include <nvntutorial/DebugTextData.h>
#include <nvntutorial/ShaderHeaders/FontShaderDataHelper.h>
#include <nvntutorial/TutorialUtil.h>

static const uint32_t g_CommandMemorySize = 65536;
static const uint32_t g_ControlMemorySize = 65536;

/*
 * DebugTextRenderer Constructor
 * -----------------------------
 * Set initial values, initialize font data.
 */

DebugTextRenderer::DebugTextRenderer() :
    m_pDevice(NULL),
    m_pQueue(NULL),
    m_pManagedCommandBuffer(NULL),
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_UsingSyncManager(NULL),
    m_pVertexDataPool(NULL),
    m_pUniformBufferPool(NULL),
    m_pFontTexturePool(NULL),
    m_pFontTextureBufferPool(NULL),
    m_pTextureIDManager(NULL)
{
    memset(m_StringSize,                0, sizeof(m_StringSize));
    memset(m_FontPositionData,          0, sizeof(m_FontPositionData));
    memset(m_FontTextureCoordinateData, 0, sizeof(m_FontTextureCoordinateData));
    memset(m_FontMagScale,              0, sizeof(m_FontMagScale));
    memset(m_FontColor,                 0, sizeof(m_FontColor));
    memset(m_FontUniformDataVS,         0, sizeof(m_FontUniformDataVS));
    memset(m_FontUniformDataPSColor,    0, sizeof(m_FontUniformDataPSColor));
    memset(m_FontUniformDataPSSampler,  0, sizeof(m_FontUniformDataPSSampler));
    memset(m_FontTexture,               0, sizeof(m_FontTexture));
    memset(&m_FontIndexData,            0, sizeof(m_FontIndexData));

    GetFontCharData(&m_FontF, s_FontHeaderF, (uint32_t*)s_CharDataF, s_FontImageDataF);
    GetFontCharData(&m_FontP, s_FontHeaderP, (uint32_t*)s_CharDataP, s_FontImageDataP);
}

/*
 * DebugTextRenderer::Init
 * -----------------------
 * Sets up NVN data (command buffer, font textures, vertex buffers,
 * uniform buffers, render states), loads the font shader, and sets
 * some default values for text rendering.
 */
void DebugTextRenderer::Init(NVNdevice* pDevice, NVNqueue* pQueue, TextureIDManager* pTextureIDManager, FrameBufferedSyncManager* pSyncManager)
{
    m_pDevice = pDevice;
    m_pQueue = pQueue;
    m_pTextureIDManager = pTextureIDManager;

        /*
         * Queries the device for the proper control and command
         * memory alignment for a command buffer.
         */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

        /*
         * If a sync manager is provided, allow it to properly manage the command memory
         * swapping through its own syncs.  Otherwise, manage swapping ourselves
         */
    if (pSyncManager)
    {
        m_UsingSyncManager = true;
    }

    if (m_UsingSyncManager)
    {
        m_pManagedCommandBuffer = new ManagedCommandBuffer(pDevice, g_CommandMemorySize, g_ControlMemorySize);
        pSyncManager->RegisterMemoryManager(m_pManagedCommandBuffer);
    }
    else
    {
        if (!nvnCommandBufferInitialize(&m_CommandBuffer, m_pDevice))
        {
            NN_ASSERT(0 && "nvnCommandBufferInitialize failed");
        }

        /* Setup the command buffer memory. */
        m_pCommandMemoryPool = new MemoryPool();
        m_pCommandMemoryPool->Init(
            NULL,
            Align(g_CommandMemorySize, commandBufferCommandAlignment),
            NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
            pDevice);

        m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);
        m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
        nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        if (!nvnSyncInitialize(&m_CommandBufferSync, m_pDevice))
        {
            NN_ASSERT(0, "nvnSyncInitialize failed");
        }
    }

        /* Setup the render states. */

        /* Blend State */
    nvnBlendStateSetDefaults(&m_BlendState);
    nvnBlendStateSetBlendTarget(&m_BlendState, 0);
    nvnBlendStateSetBlendEquation(&m_BlendState, NVNblendEquation::NVN_BLEND_EQUATION_ADD, NVNblendEquation::NVN_BLEND_EQUATION_ADD);
    nvnBlendStateSetBlendFunc(&m_BlendState, NVNblendFunc::NVN_BLEND_FUNC_SRC_ALPHA, NVNblendFunc::NVN_BLEND_FUNC_ONE_MINUS_SRC_ALPHA,
                                             NVNblendFunc::NVN_BLEND_FUNC_SRC_ALPHA, NVNblendFunc::NVN_BLEND_FUNC_ONE_MINUS_SRC_ALPHA);

        /* Channel Mask State */
    nvnChannelMaskStateSetDefaults(&m_ChannelMaskState);

        /* Color State */
    nvnColorStateSetDefaults(&m_ColorState);
    nvnColorStateSetBlendEnable(&m_ColorState, 0, NVN_TRUE);
    nvnColorStateSetLogicOp(&m_ColorState, NVNlogicOp::NVN_LOGIC_OP_COPY);

        /* Depth Stencil State */
    nvnDepthStencilStateSetDefaults(&m_DepthStencilState);
    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_FALSE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_FALSE);
    nvnDepthStencilStateSetStencilTestEnable(&m_DepthStencilState, NVN_FALSE);

        /* Multisample state */
    nvnMultisampleStateSetDefaults(&m_MultisampleState);

        /* Polygon state */
    nvnPolygonStateSetDefaults(&m_PolygonState);
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_NONE);
    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CCW);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVNpolygonMode::NVN_POLYGON_MODE_FILL);


        /* Create the builders */
    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, m_pDevice);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    nvnBufferBuilderSetDevice(&m_BufferBuilder, m_pDevice);
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

    nvnTextureBuilderSetDevice(&m_TextureBuilder, m_pDevice);
    nvnTextureBuilderSetDefaults(&m_TextureBuilder);

        /* Load the font shader asset file. */
    m_pAssetLoader = new AssetFileLoadingHelper(m_pDevice, NULL);
    m_pDataHolder = m_pAssetLoader->LoadAssetFile("fontShader.out");

        /* Setup the position vertex attribute. */
    m_PositionLocation = FontShader::Attributes::GetAttributeLocation("a_position");
    NN_ASSERT(m_PositionLocation >= 0, "Position Vertex Attribute index is negative\n");

    NVNvertexAttribState* positionAttribute = &m_VertexAttributeStates[m_PositionLocation];
    nvnVertexAttribStateSetDefaults(positionAttribute);
    nvnVertexAttribStateSetFormat(positionAttribute, NVNformat::NVN_FORMAT_RGB32F, 0);
    nvnVertexAttribStateSetStreamIndex(positionAttribute, m_PositionLocation);

    NVNvertexStreamState* positionStream = &m_VertexStreamStates[m_PositionLocation];
    nvnVertexStreamStateSetStride(positionStream, sizeof(float) * 3);

        /* Setup the texture coordinate vertex attribute. */
    m_TextureCoordinateLocation = FontShader::Attributes::GetAttributeLocation("a_texCoord");
    NN_ASSERT(m_TextureCoordinateLocation >= 0, "Texture Coordinate Vertex Attribute index is negative\n");

    NVNvertexAttribState* textureCoordinateAttribute = &m_VertexAttributeStates[m_TextureCoordinateLocation];
    nvnVertexAttribStateSetDefaults(textureCoordinateAttribute);
    nvnVertexAttribStateSetFormat(textureCoordinateAttribute, NVNformat::NVN_FORMAT_RG32F, 0);
    nvnVertexAttribStateSetStreamIndex(textureCoordinateAttribute, m_TextureCoordinateLocation);

    NVNvertexStreamState* textureCoordinateStream = &m_VertexStreamStates[m_TextureCoordinateLocation];
    nvnVertexStreamStateSetStride(textureCoordinateStream, sizeof(float) * 2);

    uint32_t indexElements = g_FontMaxBufferSize * 6;

        /* Create the vertex data memory pool. */
    m_pVertexDataPool = new MemoryPool();
    m_pVertexDataPool->Init(
        NULL,
        (sizeof(float) * 3 + sizeof(float) * 2) * 4 * g_FontMaxBufferSize * g_MaxNumLines + sizeof(uint16_t) * indexElements,
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        m_pDevice);

        /* Create the uniform buffer memory pool. */
    m_pUniformBufferPool = new MemoryPool();
    m_pUniformBufferPool->Init(
        NULL,
       (Align(sizeof(FontShader::BlockColorFSUniformBlockData), 256) +
        Align(sizeof(FontShader::BlockSamplerFSUniformBlockData), 256) +
        Align(sizeof(FontShader::BlockVSUniformBlockData), 256)) * g_MaxNumLines,
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        m_pDevice);

        /* Create the vertex buffers for the font. */
    for(uint32_t i = 0; i < g_MaxNumLines; ++i)
    {
        ptrdiff_t vertexPoolOffset = m_pVertexDataPool->GetNewMemoryChunkOffset(sizeof(float) * 3 * 4 * g_FontMaxBufferSize, 4);
        CreateBuffer(&m_FontPositionData[i], m_pVertexDataPool->GetMemoryPool(), sizeof(float) * 3 * 4 * g_FontMaxBufferSize, vertexPoolOffset);

        vertexPoolOffset = m_pVertexDataPool->GetNewMemoryChunkOffset(sizeof(float) * 2 * 4 * g_FontMaxBufferSize, 4);
        CreateBuffer(&m_FontTextureCoordinateData[i], m_pVertexDataPool->GetMemoryPool(), sizeof(float) * 2 * 4 * g_FontMaxBufferSize, vertexPoolOffset);

        ptrdiff_t uniformPoolOffset = m_pUniformBufferPool->GetNewMemoryChunkOffset(sizeof(FontShader::BlockVSUniformBlockData), 256);
        CreateBuffer(&m_FontUniformDataVS[i], m_pUniformBufferPool->GetMemoryPool(), sizeof(FontShader::BlockVSUniformBlockData), uniformPoolOffset);

        uniformPoolOffset = m_pUniformBufferPool->GetNewMemoryChunkOffset(sizeof(FontShader::BlockColorFSUniformBlockData), 256);
        CreateBuffer(&m_FontUniformDataPSColor[i], m_pUniformBufferPool->GetMemoryPool(), sizeof(FontShader::BlockColorFSUniformBlockData), uniformPoolOffset);

        uniformPoolOffset = m_pUniformBufferPool->GetNewMemoryChunkOffset(sizeof(FontShader::BlockSamplerFSUniformBlockData), 256);
        CreateBuffer(&m_FontUniformDataPSSampler[i], m_pUniformBufferPool->GetMemoryPool(), sizeof(FontShader::BlockSamplerFSUniformBlockData), uniformPoolOffset);
    }

        /* Create the index buffer. */
    CreateBuffer(&m_FontIndexData,
                 m_pVertexDataPool->GetMemoryPool(),
                 sizeof(uint16_t) * indexElements,
                 m_pVertexDataPool->GetNewMemoryChunkOffset(sizeof(uint16_t) * indexElements, 2));

        /* Setup the indices. */
    uint16_t remap[] = { 0, 1, 3, 3, 0, 2 };

    for (uint32_t i = 0; i < indexElements; ++i)
    {
        reinterpret_cast<uint16_t*>(m_FontIndexData.m_pBufferMap)[i] = (uint16_t)(i / 6 * 4 + remap[i % 6]);
    }

        /* Allocate memory for the font textures. */
    size_t fontMemSize = Align(sizeof(s_FontImageDataF), 512) + Align(sizeof(s_FontImageDataP), 512);

    m_pFontTexturePool = new MemoryPool();
    m_pFontTexturePool->Init(NULL, fontMemSize, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, m_pDevice);

    m_pFontTextureBufferPool = new MemoryPool();
    m_pFontTextureBufferPool->Init(NULL, fontMemSize, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, m_pDevice);

        /* Create the font textures. */
    ptrdiff_t fontPOffset = m_pFontTextureBufferPool->GetNewMemoryChunkOffset(sizeof(s_FontImageDataP), 512);
    InitFontTexture(&m_FontP, fontPOffset);

    ptrdiff_t fontFOffset = m_pFontTextureBufferPool->GetNewMemoryChunkOffset(sizeof(s_FontImageDataF), 512);
    InitFontTexture(&m_FontF, fontFOffset);

    uint32_t sizeP = static_cast<uint32_t>(m_FontP.m_FontTextureHeight * m_FontP.m_FontTextureWidth * m_FontP.m_Channel);
    uint32_t sizeF = static_cast<uint32_t>(m_FontF.m_FontTextureHeight * m_FontF.m_FontTextureWidth * m_FontF.m_Channel);

    memcpy(m_FontP.m_FontTextureData.m_TextureBufferData.m_pBufferMap, m_FontP.m_pFontImageData, sizeP);
    memcpy(m_FontF.m_FontTextureData.m_TextureBufferData.m_pBufferMap, m_FontF.m_pFontImageData, sizeF);

    NVNcommandBuffer* pCmdBuf = NULL;
    if (!m_UsingSyncManager)
    {
        nvnQueueFenceSync(m_pQueue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
        nvnQueueFlush(m_pQueue);
        nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        pCmdBuf = &m_CommandBuffer;
    }
    else
    {
        pCmdBuf = m_pManagedCommandBuffer->GetCommandBuffer();
    }


        /* Copy the font texture data into an NVNtexture. */
    nvnCommandBufferBeginRecording(pCmdBuf);
    {
        NVNbufferAddress fontPBufferAddress = nvnBufferGetAddress(m_FontP.m_FontTextureData.m_TextureBufferData.m_pObject);
        NVNcopyRegion fontPCopyRegion;
        fontPCopyRegion.xoffset = 0;
        fontPCopyRegion.yoffset = 0;
        fontPCopyRegion.zoffset = 0;
        fontPCopyRegion.width   = static_cast<uint32_t>(m_FontP.m_FontTextureWidth);
        fontPCopyRegion.height  = static_cast<uint32_t>(m_FontP.m_FontTextureHeight);
        fontPCopyRegion.depth   = 1;

        NVNcopyFlags fontPCopyFlags = NVN_COPY_FLAGS_NONE;
        nvnCommandBufferCopyBufferToTexture(
            pCmdBuf,
            fontPBufferAddress,
            m_FontP.m_FontTextureData.m_pTexture,
            NULL,
            &fontPCopyRegion,
            fontPCopyFlags);

        NVNbufferAddress fontFBufferAddress = nvnBufferGetAddress(m_FontF.m_FontTextureData.m_TextureBufferData.m_pObject);
        NVNcopyRegion fontFCopyRegion;
        fontFCopyRegion.xoffset = 0;
        fontFCopyRegion.yoffset = 0;
        fontFCopyRegion.zoffset = 0;
        fontFCopyRegion.width   = static_cast<uint32_t>(m_FontF.m_FontTextureWidth);
        fontFCopyRegion.height  = static_cast<uint32_t>(m_FontF.m_FontTextureHeight);
        fontFCopyRegion.depth   = 1;

        NVNcopyFlags fontFCopyFlags = NVN_COPY_FLAGS_NONE;
        nvnCommandBufferCopyBufferToTexture(
            pCmdBuf,
            fontFBufferAddress,
            m_FontF.m_FontTextureData.m_pTexture,
            NULL,
            &fontFCopyRegion,
            fontFCopyFlags);
    }

    m_CommandHandle = nvnCommandBufferEndRecording(pCmdBuf);
    nvnQueueSubmitCommands(m_pQueue, 1, &m_CommandHandle);

        /* Set some default values for the renderer. */
    SetSpacing(true);
    SetGridSize(60.0f, 24.0f);
    SetColor(1.0f, 1.0f, 1.0f, 1.0f);
    SetZValue(-1.0f);

    m_Enabled = true;
}//NOLINT(impl/function_size)

/*
 * DebugTextRenderer::InitFontTexture
 * ----------------------------------
 * Sets up the NVN data for the font texture.
 */
void DebugTextRenderer::InitFontTexture(FontData* pData, ptrdiff_t poolOffset)
{
    uint32_t width   = (uint32_t)pData->m_FontTextureWidth;
    uint32_t height  = (uint32_t)pData->m_FontTextureHeight;
    uint32_t depth   = 1;
    uint32_t channel = (uint32_t)pData->m_Channel;
    uint32_t size    = width*height*depth*channel;

    uint32_t samplerID, textureID;

    if (!pData->m_FontTextureData.m_pSampler)
    {
        pData->m_FontTextureData.m_pSampler = new NVNsampler;
    }
    else
    {
        nvnSamplerFinalize(pData->m_FontTextureData.m_pSampler);
    }

        /* Create the sampler, register it, and grab its ID. */
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);
    nvnSamplerBuilderSetMinMagFilter(&m_SamplerBuilder, NVN_MIN_FILTER_LINEAR, NVN_MAG_FILTER_LINEAR);
    nvnSamplerBuilderSetWrapMode(&m_SamplerBuilder, NVN_WRAP_MODE_CLAMP, NVN_WRAP_MODE_CLAMP, NVN_WRAP_MODE_CLAMP);
    nvnSamplerInitialize(pData->m_FontTextureData.m_pSampler, &m_SamplerBuilder);
    samplerID = m_pTextureIDManager->RegisterSampler(pData->m_FontTextureData.m_pSampler);

    if (!pData->m_FontTextureData.m_pTexture)
    {
        pData->m_FontTextureData.m_pTexture = new NVNtexture;
    }
    else
    {
        nvnTextureFinalize(pData->m_FontTextureData.m_pTexture);
    }

        /* Create the texture and get its ID. */
    nvnTextureBuilderSetDefaults(&m_TextureBuilder);
    nvnTextureBuilderSetTarget(&m_TextureBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetLevels(&m_TextureBuilder, 1);
    nvnTextureBuilderSetFormat(&m_TextureBuilder, NVN_FORMAT_R8);
    nvnTextureBuilderSetSize2D(&m_TextureBuilder, width, height);
    nvnTextureBuilderSetDepth(&m_TextureBuilder, depth);
    nvnTextureBuilderSetSamples(&m_TextureBuilder, 0);
    uint64_t storage = nvnTextureBuilderGetStorageSize(&m_TextureBuilder);
    NN_ASSERT(storage);
    nvnTextureBuilderSetStorage(&m_TextureBuilder, m_pFontTexturePool->GetMemoryPool(), poolOffset);
    nvnTextureInitialize(pData->m_FontTextureData.m_pTexture, &m_TextureBuilder);
    textureID = m_pTextureIDManager->RegisterTexture(pData->m_FontTextureData.m_pTexture);


    if (!pData->m_FontTextureData.m_TextureBufferData.m_pObject)
    {
        pData->m_FontTextureData.m_TextureBufferData.m_pObject = new NVNbuffer;
    }
    else
    {
        nvnBufferFinalize(pData->m_FontTextureData.m_TextureBufferData.m_pObject);
    }

        /* Create the NVNbuffer to hold the texture data to be copied into an NVNtexture. */
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);
    nvnBufferBuilderSetStorage(&m_BufferBuilder, m_pFontTextureBufferPool->GetMemoryPool(), poolOffset, size);
    nvnBufferInitialize(pData->m_FontTextureData.m_TextureBufferData.m_pObject, &m_BufferBuilder);
    pData->m_FontTextureData.m_TextureBufferData.m_Address = nvnBufferGetAddress(pData->m_FontTextureData.m_TextureBufferData.m_pObject);
    pData->m_FontTextureData.m_TextureBufferData.m_pBufferMap = nvnBufferMap(pData->m_FontTextureData.m_TextureBufferData.m_pObject);

        /* Get the texture's combined texture handle. */
    pData->m_FontTextureData.m_TextureHandle = nvnDeviceGetTextureHandle(m_pDevice, textureID, samplerID);
}

/*
 * DebugTextRenderer::CleanBufferData
 * ----------------------------------
 * Cleans up a given FontBufferData struct.
 */
void DebugTextRenderer::CleanBufferData(FontBufferData& bufferData)
{
    if(bufferData.m_pObject)
    {
        nvnBufferFinalize(bufferData.m_pObject);
        delete bufferData.m_pObject;
        bufferData.m_pObject = NULL;
        bufferData.m_pBufferMap = NULL;
        bufferData.m_Address = 0;
    }
}

/*
 * DebugTextRenderer::CleanUp
 * --------------------------
 * Cleans up all the NVN data and mempory allocated
 * for the text renderer.
 */
void DebugTextRenderer::CleanUp()
{
    m_pDevice = NULL;
    m_pQueue = NULL;
    m_pTextureIDManager = NULL;

    for(uint32_t i = 0; i < g_MaxNumLines; ++i)
    {
        CleanBufferData(m_FontPositionData[i]);
        CleanBufferData(m_FontTextureCoordinateData[i]);
        CleanBufferData(m_FontUniformDataVS[i]);
        CleanBufferData(m_FontUniformDataPSColor[i]);
        CleanBufferData(m_FontUniformDataPSSampler[i]);

        if(m_FontTexture[i] && m_FontTexture[i]->m_pTexture)
        {
            nvnTextureFinalize(m_FontTexture[i]->m_pTexture);
            delete m_FontTexture[i]->m_pTexture;
            m_FontTexture[i]->m_pTexture = NULL;

            m_FontTexture[i]->m_TextureHandle = 0;

            CleanBufferData(m_FontTexture[i]->m_TextureBufferData);
        }

        if (m_FontTexture[i] && m_FontTexture[i]->m_pSampler)
        {
            nvnSamplerFinalize(m_FontTexture[i]->m_pSampler);
            delete m_FontTexture[i]->m_pSampler;
            m_FontTexture[i]->m_pSampler = NULL;
        }
    }

    if(m_FontP.m_FontTextureData.m_pSampler)
    {
        nvnSamplerFinalize(m_FontP.m_FontTextureData.m_pSampler);
        delete m_FontP.m_FontTextureData.m_pSampler;
        m_FontP.m_FontTextureData.m_pSampler = NULL;
    }

    if (m_FontP.m_FontTextureData.m_pTexture)
    {
        nvnTextureFinalize(m_FontP.m_FontTextureData.m_pTexture);
        delete m_FontP.m_FontTextureData.m_pTexture;
        m_FontP.m_FontTextureData.m_pTexture = NULL;
    }

    CleanBufferData(m_FontP.m_FontTextureData.m_TextureBufferData);

    if(m_FontF.m_FontTextureData.m_pSampler)
    {
        nvnSamplerFinalize(m_FontF.m_FontTextureData.m_pSampler);
        delete m_FontF.m_FontTextureData.m_pSampler;
        m_FontF.m_FontTextureData.m_pSampler = NULL;
    }

    if (m_FontF.m_FontTextureData.m_pTexture)
    {
        nvnTextureFinalize(m_FontF.m_FontTextureData.m_pTexture);
        delete m_FontF.m_FontTextureData.m_pTexture;
        m_FontF.m_FontTextureData.m_pTexture = NULL;
    }

    CleanBufferData(m_FontF.m_FontTextureData.m_TextureBufferData);

    if(m_FontIndexData.m_pObject)
    {
        nvnBufferFinalize(m_FontIndexData.m_pObject);
        delete m_FontIndexData.m_pObject;
        m_FontIndexData.m_pObject = NULL;
        m_FontIndexData.m_Address = 0;
        m_FontIndexData.m_pBufferMap = NULL;
    }

    if (m_UsingSyncManager)
    {
        delete m_pManagedCommandBuffer;
        m_pManagedCommandBuffer = NULL;
    }
    else
    {
        if (m_pControlPool)
        {
            AlignedDeallocate(m_pControlPool);
            m_pControlPool = NULL;
        }

        nvnCommandBufferFinalize(&m_CommandBuffer);

        nvnSyncFinalize(&m_CommandBufferSync);

        if (m_pCommandMemoryPool)
        {
            m_pCommandMemoryPool->Shutdown();
            delete m_pCommandMemoryPool;
            m_pCommandMemoryPool = NULL;
        }
    }

    if(m_pVertexDataPool)
    {
        m_pVertexDataPool->Shutdown();
        delete m_pVertexDataPool;
        m_pVertexDataPool = NULL;
    }

    if(m_pUniformBufferPool)
    {
        m_pUniformBufferPool->Shutdown();
        delete m_pUniformBufferPool;
        m_pUniformBufferPool = NULL;
    }

    if(m_pAssetLoader)
    {
        delete m_pAssetLoader;
        m_pAssetLoader = NULL;
    }

    if(m_pDataHolder)
    {
        delete m_pDataHolder;
        m_pDataHolder = NULL;
    }

    if(m_pFontTexturePool)
    {
        m_pFontTexturePool->Shutdown();
        delete m_pFontTexturePool;
        m_pFontTexturePool = NULL;
    }

    if(m_pFontTextureBufferPool)
    {
        m_pFontTextureBufferPool->Shutdown();
        delete m_pFontTextureBufferPool;
        m_pFontTextureBufferPool= NULL;
    }
}//NOLINT(impl/function_size)

/*
 * DebugTextRenderer Destructor
 * ----------------------------
 * Empty destructor.
 */
DebugTextRenderer::~DebugTextRenderer()
{
}

/*
 * DebugTextRenderer::GetFontCharData
 * ----------------------------------
 * Sets up the font data.
 */
void DebugTextRenderer::GetFontCharData(FontData* pData, const uint32_t* pFontHeader, const uint32_t* pCharData, const uint8_t* pFontImageData)
{
    float lineHeight;
    float maxCharWidth = 0.0f;
    float maxCharHeight = 0.0f;

        /* Check Font Texture Data */
    NN_ASSERT(pFontHeader != NULL && pCharData != NULL && pFontImageData != NULL && "No texture data.\n");

        /* Check char data buffer */
    if (pData->m_pCharDataBuffer)
    {
            /* Skip Data Initialization */
        return;
    }

        /* Set Font Texture Data Information */
    pData->m_FontTextureWidth  = (float)pFontHeader[0];
    pData->m_FontTextureHeight = (float)pFontHeader[1];
    pData->m_Channel           = (uint32_t)pFontHeader[2];
    lineHeight                 = (float)pFontHeader[4];
    pData->m_NumCharData       = pFontHeader[5];
    pData->m_GridOffsetX       = 0;
    pData->m_GridOffsetY       = 0;

    if (pData == &m_FontP)
    {
        pData->m_pCharDataBuffer = m_CharDataBufferP;
    }
    else
    {
        pData->m_pCharDataBuffer = m_CharDataBufferF;
    }

        /* Check the max number */
    NN_ASSERT(pData->m_NumCharData <= g_MaxNumChars && "Font has over the max number of characters.\n");

        /* Format of data is: id, x, y, width, height, xOffset, yOffset, xAdvance */
    for (uint32_t i = 0; i < pData->m_NumCharData; ++i)
    {
        uint32_t id       = pCharData[i * 8 + 0];
        uint32_t x        = pCharData[i * 8 + 1];
        uint32_t y        = pCharData[i * 8 + 2];
        uint32_t w        = pCharData[i * 8 + 3];
        uint32_t h        = pCharData[i * 8 + 4];
        int xOffset       = (int)pCharData[i * 8 + 5];
        int yOffset       = (int)pCharData[i * 8 + 6];
        uint32_t xAdvance = pCharData[i * 8 + 7];
        float charHeight  = (float)(h + yOffset);

        pData->m_pCharDataBuffer[i].m_Id       = id;
        pData->m_pCharDataBuffer[i].m_MinS     = (float)x / pData->m_FontTextureWidth;
        pData->m_pCharDataBuffer[i].m_MinT     = (float)(pData->m_FontTextureHeight - h - y) / pData->m_FontTextureHeight;
        pData->m_pCharDataBuffer[i].m_MaxS     = (float)(x + w) / pData->m_FontTextureWidth;
        pData->m_pCharDataBuffer[i].m_MaxT     = (float)(pData->m_FontTextureHeight - y) / pData->m_FontTextureHeight;
        pData->m_pCharDataBuffer[i].m_MinX     = (float)xOffset;
        pData->m_pCharDataBuffer[i].m_MinY     = (float)(lineHeight - yOffset - h);
        pData->m_pCharDataBuffer[i].m_MaxX     = (float)(xOffset + w);
        pData->m_pCharDataBuffer[i].m_MaxY     = (float)(lineHeight - yOffset);
        pData->m_pCharDataBuffer[i].m_XAdvance = (float)xAdvance;

            /* Set max height of char in GL cordinate space */
        if(charHeight > maxCharHeight)
        {
            maxCharHeight = charHeight;
        }

        if (pData->m_pCharDataBuffer[i].m_XAdvance > maxCharWidth)
        {
            maxCharWidth = (float)pData->m_pCharDataBuffer[i].m_XAdvance;
        }
    }

        /* Set grid offsetX */
    pData->m_GridOffsetX = maxCharWidth;

        /* Set grid offsetY */
    pData->m_GridOffsetY = maxCharHeight;

        /* Set pointer of Font Image Data */
    pData->m_pFontImageData = (uint8_t*)pFontImageData;
}

/*
 * DebugTextRenderer::CreateBuffer
 * -------------------------------
 * Create an NVNbuffer with the specified structure.
 */
void DebugTextRenderer::CreateBuffer(FontBufferData* pData, NVNmemoryPool* pPool, size_t size, ptrdiff_t offset)
{
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);
    nvnBufferBuilderSetStorage(&m_BufferBuilder, pPool, offset, size);

    if (pData->m_pObject)
    {
        nvnBufferFinalize(pData->m_pObject);
    }
    else
    {
        pData->m_pObject = new NVNbuffer;
    }

    if (!nvnBufferInitialize(pData->m_pObject, &m_BufferBuilder))
    {
        NN_ASSERT(0, "Failed to initialize buffer");
    }

    pData->m_Address = nvnBufferGetAddress(pData->m_pObject);
    pData->m_pBufferMap = nvnBufferMap(pData->m_pObject);
}

/*
 * DebugTextRenderer::SetSpacing
 * -----------------------------
 * Sets the spacing for the file
 */
void DebugTextRenderer::SetSpacing(bool proportional)
{
        /* Check initialization. */
    NN_ASSERT(m_FontP.m_pCharDataBuffer && "Need to call DebugTextRenderer::Init first");

        /* Update proportional boolean. */
    m_Proportional = proportional;

        /* Switch which scale we use. */
    if (proportional)
    {
        m_Font.m_pCharMagScale = m_Font.m_CharMagScaleP;
    }
    else
    {
        m_Font.m_pCharMagScale = m_Font.m_CharMagScaleF;
    }
}

/*
 * DebugTextRenderer::SetGridSize
 * ------------------------------
 * Sets the size of the grid that the text is rendered in.
 */
void DebugTextRenderer::SetGridSize(float xGrid, float yGrid)
{
        /* Check initialization. */
    NN_ASSERT(m_FontP.m_pCharDataBuffer && "Need to call DebugTextRenderer::Init first");

        /* Update scale */
    UpdateScale(&m_Font, xGrid, yGrid);
}

/*
 * DebugTextRenderer::UpdateScale
 * ------------------------------
 * Sets the scale of the text.
 */
void DebugTextRenderer::UpdateScale(FontRenderData* pData, float scaleX, float scaleY)
{
        /* Calculate char scale */
    pData->m_CharMagScaleF[0] = 2.0f / (scaleX * m_FontF.m_GridOffsetX);
    pData->m_CharMagScaleF[1] = 2.0f / (scaleY * m_FontF.m_GridOffsetY);

    pData->m_CharMagScaleP[0] = 2.0f / (scaleX * m_FontP.m_GridOffsetX);
    pData->m_CharMagScaleP[1] = 2.0f / (scaleY * m_FontP.m_GridOffsetY);
}

/*
 * DebugTextRenderer::SetColor
 * ---------------------------
 * Sets the color of the text.
 */
void DebugTextRenderer::SetColor(float r, float g, float b, float a)
{
        /*  Check initialization */
    NN_ASSERT(m_FontP.m_pCharDataBuffer && "Need to call DebugTextRenderer::Init first");

        /*  Update color */
    m_Font.m_Color[0] = r;
    m_Font.m_Color[1] = g;
    m_Font.m_Color[2] = b;
    m_Font.m_Color[3] = a;
}

/*
 * DebugTextRenderer::SetZValue
 * ----------------------------
 * Set the text's z value.
 */
void DebugTextRenderer::SetZValue(float zValue)
{
        /* Check initialization */
    NN_ASSERT(m_FontP.m_pCharDataBuffer && "Need to call DebugTextRenderer::Init first");

        /* Update depth value */
    m_Font.m_Depth = zValue;
}

/*
 * DebugTextRenderer::UpdateVertexBuffers
 * --------------------------------------
 * Updates the vertex buffers for a new string of characters.
 */
void DebugTextRenderer::UpdateVertexBuffers(const char* pStr, uint32_t strLength, float x, float y)
{
    uint32_t i = 0;
    float cursorOffset = 0;
    float gridOffset = 0;

    static float s_PositionBuffer[g_FontMaxBufferSize * 3 * 4];
    static float s_TextureCoordinateBuffer[g_FontMaxBufferSize * 2 * 4];

    FontRenderData* pFontData = &m_Font;
    FontData* pFont = GetCurrentFont();

    while (i < strLength)
    {
        for(; i < strLength; i++)
        {
                /* Get character id */
            uint32_t id = (uint32_t)pStr[i];
            uint32_t index = 0;

                /* Set index offset of buffers */
            uint32_t posSt = i * 12;
            uint32_t texSt = i * 8;

                /* Check "\n" */
            if(id == 10)
            {
                gridOffset -= (float)pFont->m_GridOffsetY;
                cursorOffset = 0;
                i++;
                break;
            }
            else if(id >= 32 && id <= 127)
            {
                    /* Get index of character id */
                index = BSearchIndex(pFont, id);
            }

                /* Set Vertex position */
            s_PositionBuffer[posSt + 0] = (float)(pFont->m_pCharDataBuffer[index].m_MinX + cursorOffset + x);
            s_PositionBuffer[posSt + 3] = (float)s_PositionBuffer[posSt + 0];
            s_PositionBuffer[posSt + 6] = (float)(pFont->m_pCharDataBuffer[index].m_MaxX + cursorOffset + x);
            s_PositionBuffer[posSt + 9] = (float)s_PositionBuffer[posSt + 6];

            cursorOffset += pFont->m_pCharDataBuffer[index].m_XAdvance;

            s_PositionBuffer[posSt + 1] = (float)(pFont->m_pCharDataBuffer[index].m_MinY + gridOffset + y);
            s_PositionBuffer[posSt + 4] = (float)(pFont->m_pCharDataBuffer[index].m_MaxY + gridOffset + y);
            s_PositionBuffer[posSt + 7] = (float)s_PositionBuffer[posSt + 1];
            s_PositionBuffer[posSt + 10] = (float)s_PositionBuffer[posSt + 4];

            s_PositionBuffer[posSt + 2] = (float)pFontData->m_Depth;
            s_PositionBuffer[posSt + 5] = (float)pFontData->m_Depth;
            s_PositionBuffer[posSt + 8] = (float)pFontData->m_Depth;
            s_PositionBuffer[posSt + 11] = (float)pFontData->m_Depth;

                /* Set Texture coordinate */
            s_TextureCoordinateBuffer[texSt + 0] = (float)pFont->m_pCharDataBuffer[index].m_MinS;
            s_TextureCoordinateBuffer[texSt + 2] = (float)pFont->m_pCharDataBuffer[index].m_MinS;
            s_TextureCoordinateBuffer[texSt + 4] = (float)pFont->m_pCharDataBuffer[index].m_MaxS;
            s_TextureCoordinateBuffer[texSt + 6] = (float)pFont->m_pCharDataBuffer[index].m_MaxS;

            s_TextureCoordinateBuffer[texSt + 1] = (float)pFont->m_pCharDataBuffer[index].m_MinT;
            s_TextureCoordinateBuffer[texSt + 3] = (float)pFont->m_pCharDataBuffer[index].m_MaxT;
            s_TextureCoordinateBuffer[texSt + 5] = (float)pFont->m_pCharDataBuffer[index].m_MinT;
            s_TextureCoordinateBuffer[texSt + 7] = (float)pFont->m_pCharDataBuffer[index].m_MaxT;
        }
    }

        /* Fill vertex buffer with vertex data. */
    memcpy(m_FontPositionData[m_FontNumLines].m_pBufferMap, s_PositionBuffer, sizeof(float) * 3 * 4 * strLength);
    memcpy(m_FontTextureCoordinateData[m_FontNumLines].m_pBufferMap, s_TextureCoordinateBuffer, sizeof(float) * 2 * 4 * strLength);

    m_StringSize[m_FontNumLines] = strLength;
}

/*
 * DebugTextRenderer::UpdateTextureBuffer
 * --------------------------------------
 * Update the texture buffer with the current texture.
 */
void DebugTextRenderer::UpdateTextureBuffer(FontTextureData* texture)
{
    m_FontTexture[m_FontNumLines] = texture;
}

/*
 * DebugTextRenderer::UpdateUniformBuffers
 * ---------------------------------------
 * Update the uniform buffers for the frame.
 */
void DebugTextRenderer::UpdateUniformBuffers(float* magScale, float* color, NVNtextureHandle* textureHandle)
{
    memcpy(m_FontUniformDataVS[m_FontNumLines].m_pBufferMap, magScale, sizeof(float) * 2);

    memcpy(m_FontUniformDataPSColor[m_FontNumLines].m_pBufferMap, color, sizeof(float) * 4);

    memcpy(m_FontUniformDataPSSampler[m_FontNumLines].m_pBufferMap, textureHandle, sizeof(NVNtextureHandle));
}

/*
 * DebugTextRenderer::BSearchIndex
 * -------------------------------
 * Find the index of a given character.
 */
uint32_t DebugTextRenderer::BSearchIndex(FontData* pFont, uint32_t id)
{
    uint32_t first;
    uint32_t last;
    uint32_t index = 0;

    if (id >= 32 && id < 32 + pFont->m_NumCharData && pFont->m_pCharDataBuffer[id - 32].m_Id == id)
    {
        return id - 32;
    }

    first = 0;
    last = pFont->m_NumCharData - 1;

    while(first <= last)
    {
        uint32_t mid = first + (last - first) / 2;

        if (pFont->m_pCharDataBuffer[mid].m_Id < id)
        {
            first = mid + 1;
        }
        else if (id < pFont->m_pCharDataBuffer[mid].m_Id)
        {
            last = mid - 1;
        }
        else
        {
            index = mid;
            break;
        }
    }
    return index;
}

/*
 * DebugTextRenderer::GetCurrentFont
 * ---------------------------------
 * Get which font is in use.
 */
DebugTextRenderer::FontData* DebugTextRenderer::GetCurrentFont()
{
    if(m_Proportional)
    {
        return &m_FontP;
    }
    else
    {
        return &m_FontF;
    }
}

/*
 * DebugTextRenderer::GetCharSize
 * ------------------------------
 * Get the current character size.
 */
void DebugTextRenderer::GetCharSize(float* pCharWidth, float* pCharHeight)
{
    FontData* pFont = GetCurrentFont();

        /* Get size of font */
    *pCharWidth  = pFont->m_GridOffsetX;
    *pCharHeight = pFont->m_GridOffsetY;
}

/*
 * DebugTextRenderer::SetDrawEnable
 * --------------------------------
 * Enable/disable drawing.
 */
void DebugTextRenderer::SetDrawEnable(bool enable)
{
    m_Enabled = enable;
}

/*
 * DebugTextRenderer::IsEnabled
 * ----------------------------
 * Check whether text rendering is enabled.
 */
bool DebugTextRenderer::IsEnabled()
{
    return m_Enabled;
}

/*
 * DebugTextRenderer::Puts
 * -----------------------
 * Render a string of text at a given line/column.
 */
void DebugTextRenderer::Puts(float column, float line, const char* pStr)
{
    uint32_t stringLength;
    float offsetX;
    float offsetY;
    FontData* pFont = GetCurrentFont();
    FontRenderData* pFontData = &m_Font;

        /* Don't draw if fonts are disabled */
    if (!m_Enabled)
    {
        return;
    }

        /* Check the initialize */
    NN_ASSERT(pStr && "Need to initialize str.\n");
    NN_ASSERT(pFont->m_pCharDataBuffer && "Need to call DebugTextRenderer::Init first.\n");

    stringLength = (uint32_t)(strlen((const char *)pStr));

        /* Calculate offsets */
    offsetX = (float)(column * pFont->m_GridOffsetX);
    offsetY = -(float)(line * pFont->m_GridOffsetY);

        /* Update Texture Buffer */
    UpdateTextureBuffer(&pFont->m_FontTextureData);

        /* Update Vertex Buffers */
    UpdateVertexBuffers(pStr, stringLength, offsetX, offsetY - pFont->m_GridOffsetY);

        /* Update Uniform Buffers */
    UpdateUniformBuffers(pFontData->m_pCharMagScale, pFontData->m_Color, &pFont->m_FontTextureData.m_TextureHandle);

    ++m_FontNumLines;
}

/*
 * DebugTextRenderer::Printf
 * -------------------------
 * Render a formatted string of text at a given line/column.
 */
void DebugTextRenderer::Printf(float column, float line, const char* pFmt, ...)
{
    char str[g_FontMaxBufferSize];
    va_list args;
    int stringSize;

        /* Don't draw if fonts are disabled */
    if (!m_Enabled)
    {
        return;
    }


        /* Get string */
    va_start(args, pFmt);
    stringSize = vsnprintf(str, g_FontMaxBufferSize, pFmt, args);

        /* Assert for over string size */
    if ( stringSize < 0 )
    {
        NN_ASSERT(!"String is too long\n");
    }

    va_end(args);

    Puts( column, line, str );
}

/*
 * DebugTextRenderer::Draw
 * -----------------------
 * Render the text to a given render target.
 */
void DebugTextRenderer::Draw(NVNtexture* renderTarget, int width, int height)
{
    NVNcommandBuffer* pCmdBuf = NULL;

    if (!m_UsingSyncManager)
    {
        nvnQueueFenceSync(m_pQueue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
        nvnQueueFlush(m_pQueue);
        nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Reset the command/control memory for the command buffer. */
        nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
        nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        pCmdBuf = &m_CommandBuffer;
    }
    else
    {
        pCmdBuf = m_pManagedCommandBuffer->GetCommandBuffer();
    }

    nvnCommandBufferBeginRecording(pCmdBuf);
    {
            /* Set the sampler and texture pool. */
        m_pTextureIDManager->SetSamplerPool(pCmdBuf);
        m_pTextureIDManager->SetTexturePool(pCmdBuf);

            /* Sets the render target. */
        nvnCommandBufferSetRenderTargets(pCmdBuf, 1, &renderTarget, NULL, NULL, NULL);

        nvnCommandBufferSetScissor(pCmdBuf, 0, 0, width, height);
        nvnCommandBufferSetViewport(pCmdBuf, 0, 0, width, height);

            /* Bind the render states. */
        nvnCommandBufferBindBlendState(pCmdBuf, &m_BlendState);
        nvnCommandBufferBindChannelMaskState(pCmdBuf, &m_ChannelMaskState);
        nvnCommandBufferBindColorState(pCmdBuf, &m_ColorState);
        nvnCommandBufferBindDepthStencilState(pCmdBuf, &m_DepthStencilState);
        nvnCommandBufferBindMultisampleState(pCmdBuf, &m_MultisampleState);
        nvnCommandBufferBindPolygonState(pCmdBuf, &m_PolygonState);
        nvnCommandBufferSetSampleMask(pCmdBuf, static_cast<uint32_t>(~0));

            /* Bind the vertex states. */
        nvnCommandBufferBindVertexAttribState(pCmdBuf, 2, m_VertexAttributeStates);
        nvnCommandBufferBindVertexStreamState(pCmdBuf, 2, m_VertexStreamStates);

            /* Bind the shader program. */
        m_pDataHolder->GetProgramData()[0]->BindShaderProgram(pCmdBuf);

            /* Bind the vertex/uniform buffers and draw. */
        for(uint32_t i = 0; i < m_FontNumLines; ++i)
        {
            nvnCommandBufferBindVertexBuffer(pCmdBuf, m_PositionLocation, m_FontPositionData[i].m_Address, sizeof(float) * 3 * 4 * m_StringSize[i]);
            nvnCommandBufferBindVertexBuffer(pCmdBuf, m_TextureCoordinateLocation, m_FontTextureCoordinateData[i].m_Address, sizeof(float) * 2 * 4 * m_StringSize[i]);

            nvnCommandBufferBindUniformBuffer(pCmdBuf, (NVNshaderStage::NVN_SHADER_STAGE_VERTEX), FontShader::BlockVSUniformBlockData::GetBinding(NVNshaderStage::NVN_SHADER_STAGE_VERTEX), m_FontUniformDataVS[i].m_Address, sizeof(float) * 2);
            nvnCommandBufferBindUniformBuffer(pCmdBuf, (NVNshaderStage::NVN_SHADER_STAGE_FRAGMENT), FontShader::BlockColorFSUniformBlockData::GetBinding(NVNshaderStage::NVN_SHADER_STAGE_FRAGMENT), m_FontUniformDataPSColor[i].m_Address, sizeof(float) * 4);
            nvnCommandBufferBindUniformBuffer(pCmdBuf, (NVNshaderStage::NVN_SHADER_STAGE_FRAGMENT), FontShader::BlockSamplerFSUniformBlockData::GetBinding(NVNshaderStage::NVN_SHADER_STAGE_FRAGMENT), m_FontUniformDataPSSampler[i].m_Address, sizeof(NVNtextureHandle));

            nvnCommandBufferDrawElements(pCmdBuf, NVN_DRAW_PRIMITIVE_TRIANGLES, NVN_INDEX_TYPE_UNSIGNED_SHORT, m_StringSize[i] * 6, m_FontIndexData.m_Address);
        }

    }

        /* Grab and submit the command handle. */
    m_CommandHandle = nvnCommandBufferEndRecording(pCmdBuf);
    nvnQueueSubmitCommands(m_pQueue, 1, &m_CommandHandle);

    m_FontNumLines = 0;
}
