﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AssetFileLoadingHelper.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a helper class used to load in asset files
 *  generated by Tutorial02. The asset file has three potential
 *  sections: compiled shaders, packaged textures, and model
 *  model data. The exact structure of the asset file can
 *  be seen in OutputFileHeaders.h.
 */

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

#include <nvntutorial/AssetFileLoadingHelper.h>
#include <nvntutorial/OutputFileHeaders.h>
#include <nvntutorial/TutorialUtil.h>
#include <nvntutorial/TextureIDManager.h>
#include <nvntutorial/MemoryPool.h>
#include <algorithm>

/*
 * AssetFileLoadingHelper Constructor
 * ----------------------------------
 * Sets up the file loader with a pointer to the NVNdevice and
 * the texture ID manager for the application.
 */
AssetFileLoadingHelper::AssetFileLoadingHelper(NVNdevice* pDevice, TextureIDManager* pTextureIDManager) :
    m_pDevice(pDevice),
    m_pTextureIDManager(pTextureIDManager),
    m_pFileHead(NULL),
    m_FileSize(0)
{
}

/*
* AssetFileLoadingHelper Destructor
* ---------------------------------
* Cleans up the file loader.
*/
AssetFileLoadingHelper::~AssetFileLoadingHelper()
{
}

/*
 * AssetFileLoadingHelper::LoadAssetFile
 * -------------------------------------
 * Loads the binary asset file, checks for each section of
 * the file, and interprets the data for the sections that
 * exist.
 */
AssetFileDataHolder* AssetFileLoadingHelper::LoadAssetFile(const char* pFilename)
{
    NN_ASSERT(pFilename != NULL, "File name to load NULL");

        /* Reads in the binary file and grabs a pointer to its head. */
    nn::fs::FileHandle fileHandle;
    std::string rom("rom:/");

    nn::Result res = nn::fs::OpenFile(&fileHandle, (rom + pFilename).c_str(), nn::fs::OpenMode_Read);
    NN_ASSERT(res.IsSuccess());

    res = nn::fs::GetFileSize(&m_FileSize, fileHandle);
    NN_ASSERT(res.IsSuccess());

    m_pFileHead = reinterpret_cast<char*>(AlignedAllocate(static_cast<size_t>(m_FileSize + 1), 8));
    memset(m_pFileHead, 0, static_cast<size_t>(m_FileSize + 1));
    size_t out;
    res = nn::fs::ReadFile(&out, fileHandle, 0, m_pFileHead, static_cast<size_t>(m_FileSize));
    NN_ASSERT(res.IsSuccess());

    OutputFileHeader* pFileHeader = reinterpret_cast<OutputFileHeader*>(m_pFileHead);
    NN_ASSERT(pFileHeader != NULL, "Failed to read file header")

        /* Creates a default buffer builder. */
    nvnBufferBuilderSetDevice(&m_BufferBuilder, m_pDevice);
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

        /* Creates a default texture builder. */
    nvnTextureBuilderSetDevice(&m_TextureBuilder, m_pDevice);
    nvnTextureBuilderSetDefaults(&m_TextureBuilder);

        /* Creates a default sampler builder. */
    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, m_pDevice);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    m_pDataHolder = new AssetFileDataHolder();

        /* Check for the shader section of the file and interpret the data. */
    if (pFileHeader->m_ShaderBlockOffset > 0)
    {
        char* pShaderBlockHead = m_pFileHead + pFileHeader->m_ShaderBlockOffset;
        LoadShaders(pShaderBlockHead);
    }

        /* Check for the texture section of the file and interpret the data. */
    if (pFileHeader->m_TextureBlockOffset > 0)
    {
        char* pTextureBlockHead = m_pFileHead + pFileHeader->m_TextureBlockOffset;
        LoadTextures(pTextureBlockHead);
    }

        /* Check for the model section of the file and interpret the data. */
    if (pFileHeader->m_ModelBlockOffset > 0)
    {
        char* pModelBlockHead = m_pFileHead + pFileHeader->m_ModelBlockOffset;
        LoadModels(pModelBlockHead);
    }

    nn::fs::CloseFile(fileHandle);

    AlignedDeallocate(m_pFileHead);
    m_pFileHead = NULL;

    return m_pDataHolder;
}

/*
 * AssetFileLoadingHelper::LoadSpecializedShaders
 * -------------------------------------
 * Generates the specialized versions of a set of shaders
 * passed in and loaded at runtime into a number of unique
 * shader programs, new NVNProgramData objects are loaded
 * into the AssetFileLoadingHelper class used to call
 * this function
 */
void AssetFileLoadingHelper::LoadSpecializedShaders(
    const char** ppShaderSources,
    const std::vector<NVNshaderStage>& shaderStages,
    GLSLCspecializationBatch* specializationBatch,
    const char* shaderProgramName,
    int nameLength,
    ReflectionBlockInfoFunction blockFunction,
    ReflectionUniformInfoFunction uniformFunction,
    AttributeReflectionInfoFunction attributeFunction)
{
    NN_ASSERT(ppShaderSources != NULL, "No shader source passed in\n");
    NN_ASSERT(shaderStages.size() > 0, "No shader stages passed in\n");
    NN_ASSERT(specializationBatch != NULL, "No shader specialization batch passed in\n");
    NN_ASSERT(blockFunction != NULL && uniformFunction != NULL, "No reflection functions passed to shader specialization\n");

        /*
         * The compile object contains the input shader source, options for compilation,
         * and all the results (error logs in the result of failure) of each step of
         * compilation.  Valid until the object is used for another regular Compile
         * operation, Pre-Specialized + Specialized operation, or until the object
         * is finalized.
         */
    GLSLCcompileObject compileObject;
    uint8_t initializeResult = glslcInitialize(&compileObject);
    NN_ASSERT(initializeResult == 1, "Failed to initialize glslc compile object\n");

        /* Contains input array of shader source */
    GLSLCinput& glslcInput = compileObject.input;
    glslcInput.sources = ppShaderSources;
    glslcInput.count = static_cast<uint8_t>(shaderStages.size());
    glslcInput.stages = &shaderStages[0];

        /* Various options available for compiling the shaders */
    GLSLCoptions& glslcOptions = compileObject.options;
    glslcOptions.optionFlags.glslSeparable = shaderStages.size() < 2;
    glslcOptions.optionFlags.outputAssembly = false;
    // NOTE: This must be set to true for now.
    glslcOptions.optionFlags.outputGpuBinaries = true;
    glslcOptions.optionFlags.outputPerfStats = true;
    glslcOptions.optionFlags.outputShaderReflection = true;
    glslcOptions.optionFlags.outputDebugInfo = GLSLC_DEBUG_LEVEL_G0;

        /*
         * Denotes whether to output gpu binaries of
         * a smaller size for exclusive use on the NX
         * device. This option has no effect on binaries
         * generated on Windows, which will continue to
         * contain the data necessary for both Windows
         * and NX use.
         */
#ifdef _WIN32
    glslcOptions.optionFlags.outputThinGpuBinaries = 0;
#else
    glslcOptions.optionFlags.outputThinGpuBinaries = 1;
#endif

        /*
         * The first step of shader specialization, the pre-specialized compile
         * function takes the raw shader source and generates intermediary
         * data that will be used in the specialized compile function below.
         *
         * NOTE: When using glslc online, it is required that glslcSetAllocator
         * is called with function pointers to an allocator and deallocator function
         * for glslc functions (bar glslcInitialize) to work.
         */
    if(!glslcCompilePreSpecialized(&compileObject))
    {
        NN_LOG(compileObject.lastCompiledResults->compilationStatus->infoLog);
        NN_ASSERT(0, "Shader specialization precompilation failed\n");
    }

        /*
         * Reflection information is ALWAYS generated by the pre-specialized function
         * above, regardless of whether the outputShaderReflection flag is set in the
         * glslc options above.  The reflection header contains offsets into the header
         * itself that contain the actual reflection information.
         */
    GLSLCprogramReflectionHeader* pReflectionHeader = compileObject.reflectionSection;
    const uint8_t* pReflectionData = reinterpret_cast<const uint8_t*>(pReflectionHeader) + pReflectionHeader->common.dataOffset;
    const char* pStringPool = reinterpret_cast<const char*>(pReflectionData) + pReflectionHeader->stringPoolOffset;

        /*
         * Grabs a pointer to the reflection information of the first uniform block
         * listed.
         */
    const GLSLCuniformBlockInfo* pUniformBlock = reinterpret_cast<const GLSLCuniformBlockInfo *>(pReflectionData + pReflectionHeader->uniformBlockOffset);
    for (uint32_t i = 0; i < pReflectionHeader->numUniformBlocks; ++i)
    {
            /*
             * Grab the binding location of the uniform block.  Each of the
             * entries in the bindings array corresponds to the enum values
             * in NVNshaderStage and lists the binding location of the block
             * in that stage;  a value of -1 means the block is not bound
             * in that stage.
             */
        int32_t bindingLocation = -1;
        for(uint32_t j = 0; j < GLSLC_NUM_SHADER_STAGES; ++j)
        {
            if(pUniformBlock->bindings[j] > -1)
            {
                bindingLocation = pUniformBlock->bindings[j];
                break;
            }
        }

            /*
             * Pass the reflection information (binding location, block size, and name)
             * through the function pointer passed in.  Additional reflection parameters
             * can be found in the definition of the GLSLCuniformBlockInfo struct.
             */
        if(bindingLocation > -1)
        {
            std::string blockName = pStringPool + pUniformBlock->nameInfo.nameOffset;
                /* Call reflection function */
            blockFunction(pUniformBlock->stagesReferencedIn, bindingLocation, pUniformBlock->size, blockName);
        }

            /* Advance to the next uniform block's reflection info */
        ++pUniformBlock;
    }

        /*
         * Grabs a pointer to the reflection information of the first uniform variable
         * listed.  The uniforms in this part of the reflection header come from all
         * uniform blocks in the shaders provided and no guarantee is made to their
         * ordering.
         */
    const GLSLCuniformInfo* pUniformData = reinterpret_cast<const GLSLCuniformInfo *>(pReflectionData + pReflectionHeader->uniformOffset);
    for (uint32_t j = 0; j < pReflectionHeader->numUniforms; ++j)
    {
        std::string uniformName = pStringPool + pUniformData->nameInfo.nameOffset;

            /*
             * Pass the reflection information (stages uniform is referenced in, offset in uniform block, and name)
             * through the function pointer passed in.  Additional reflection parameters can be found in the
             * definition of the GLSLCuniformInfo struct.
             */
        uniformFunction(pUniformData->stagesReferencedIn, pUniformData->blockOffset, 0, uniformName);

            /* Advance to the next uniform variable's reflection info */
        ++pUniformData;
    }

        /*
         * Grabs a pointer to the reflection information of the shader
         * program inputs.
         */
    const GLSLCprogramInputInfo* pProgramInput = reinterpret_cast<const GLSLCprogramInputInfo *>(pReflectionData + pReflectionHeader->programInputsOffset);
    for(uint32_t j = 0; j < pReflectionHeader->numProgramInputs; ++j)
    {
        std::string name = pStringPool + pProgramInput->nameInfo.nameOffset;

            /*
             * Pass the reflection information (stages attribute is referenced in, binding location, and name)
             * through the function pointer passed in.
             */
        attributeFunction(pProgramInput->stagesReferencedIn, pProgramInput->location, name);
        ++pProgramInput;
    }

        /*
         * The second and last step of shader specialization, the specialized
         * compile function takes the intermediary data stored in the compile
         * object and generates an array of GLSLCoutput objects, each one
         * representing the compiled output of one of the shader programs.
         * The number of output objects corresponds to the number of
         * specialization sets contained in the specialization batch object.
         */
    const GLSLCoutput* const* ppGlslcOutputs = glslcCompileSpecialized(&compileObject, specializationBatch);
    if(ppGlslcOutputs == NULL || compileObject.lastCompiledResults->compilationStatus->success != 1)
    {
        NN_LOG(compileObject.lastCompiledResults->compilationStatus->infoLog);
        NN_ASSERT(0, "Shader specialization compilation failed\n");
    }

        /*
         * Looping through the array of output objects, the data within
         * is placed into ShaderStageHeader structs and passed to the
         * CreateShaderProgram function to be placed into memory pools
         * and NVNprograms created.
         */
    for(uint32_t i = 0; i < specializationBatch->numEntries; ++i)
    {
        NVNProgramData* pProgramData = new NVNProgramData;
        pProgramData->Initialize();
        pProgramData->SetName(shaderProgramName, nameLength);

        const GLSLCoutput* currentGlslcOutput = ppGlslcOutputs[i];

        int shaderSections = currentGlslcOutput->numSections;
        std::vector<ShaderStageHeader> shaderStageHeaders;

        for(int j = 0; j < shaderSections; ++j)
        {
                /*
                 * Each shader stage (vertex, fragment, etc...) will have a glslc section that
                 * contains all the relevant data for it, as denoted by the GLSLC_SECTION_TYPE_GPU_CODE
                 * enum.  Other types of glslc sections can be found in GLSLCsectionTypeEnum in
                 * nvnToolGlslcInterface.h
                 */
            GLSLCsectionTypeEnum type = currentGlslcOutput->headers[j].genericHeader.common.type;
            if (type == GLSLC_SECTION_TYPE_GPU_CODE)
            {
                const GLSLCgpuCodeHeader& pGpuHeader = (currentGlslcOutput->headers[j].gpuCodeHeader);
                const uint8_t* pData = (reinterpret_cast<const uint8_t*>(currentGlslcOutput)) + pGpuHeader.common.dataOffset;

                shaderStageHeaders.push_back(ShaderStageHeader());
                ShaderStageHeader& currentHeader    = shaderStageHeaders.back();
                currentHeader.m_ShaderStage         = pGpuHeader.stage;
                currentHeader.m_ShaderDataSize      = pGpuHeader.dataSize;
                currentHeader.m_ShaderControlSize   = pGpuHeader.controlSize;
                currentHeader.m_ShaderControlOffset = 0;
                currentHeader.m_pShaderData         = const_cast<uint8_t*>(pData + pGpuHeader.dataOffset);
                currentHeader.m_pShaderControl      = const_cast<uint8_t*>(pData + pGpuHeader.controlOffset);

                    /* Save the max scratch memory size needed for a shader stage of this program. */
                pProgramData->m_ShaderScratchMemorySize = std::max(static_cast<size_t>(pGpuHeader.scratchMemBytesRecommended), pProgramData->m_ShaderScratchMemorySize);
            }
        }

            /*
             * Creates the actual NVNprogram using the shader data pulled above.
             * An NVNProgramData object is inserted into this asset loaders's
             * data holder.
             */
        CreateShaderProgram(shaderStageHeaders, pProgramData);
        m_pDataHolder->AddProgramData(pProgramData);
    };

        /*
         * Finalize the compile object, rendering all data it held invalid
         * and preventing reuse of the object until glslcInitialize is called
         * on it again.
         */
    glslcFinalize(&compileObject);
}

/*
 * AssetFileLoadingHelper::LoadShaders
 * -----------------------------------
 * Receives a pointer to the shader block of data and passes
 * a pointer to the start of an individual shader program's
 * block to be parsed.
 */
void AssetFileLoadingHelper::LoadShaders(const char* pShaderBlockHead)
{
    NN_ASSERT(pShaderBlockHead != NULL, "Shader block header pointer NULL");

        /* Grabs the number of shaders and their offsets. */
    const ShaderBlockHeader* pShaderBlockHeader = reinterpret_cast<const ShaderBlockHeader*>(pShaderBlockHead);
    uint32_t numPrograms = pShaderBlockHeader->m_NumShaderPrograms;

    const uint64_t* pShaderProgramHeaderOffsets = reinterpret_cast<const uint64_t*>((pShaderBlockHead + sizeof(uint32_t) * 2));

        /* Creates a program for each shader. */
    for(uint32_t i = 0; i < numPrograms; ++i)
    {
        NVNProgramData* pProgramData = LoadShaderHeaders(m_pFileHead + pShaderProgramHeaderOffsets[i]);
        m_pDataHolder->AddProgramData(pProgramData);
    }
}

/*
 * AssetFileLoadingHelper::LoadShaderHeaders
 * -----------------------------------
 * Parses the pointer passed in for the data necessary to build
 * an NVNprogram.  This data is then passed onto CreateShaderProgram
 * to actually create the NVNprogram and set up the memory pools
 * for the data to be copied into.
 */
NVNProgramData* AssetFileLoadingHelper::LoadShaderHeaders(const char* pShaderProgramHead)
{
    NN_ASSERT(pShaderProgramHead != NULL, "Shader program header pointer NULL");

        /* Grabs the metadata and the list of offsets to individual shader stages. */
    const ShaderProgramHeader* pShaderProgramHeader = reinterpret_cast<const ShaderProgramHeader*>(pShaderProgramHead);

    uint32_t                   numShaderStages      = pShaderProgramHeader->m_NumShaderStages;
    const uint64_t*            pShaderStageOffsets  = reinterpret_cast<const uint64_t*>(pShaderProgramHead + sizeof(uint32_t) * 4);
    const char*                pProgramName         = pShaderProgramHead + sizeof(uint32_t) * 4 + sizeof(uint64_t) * numShaderStages;

    if (pShaderProgramHeader == NULL ||
        pShaderProgramHeader->m_ProgramNameLength == 0 ||
        pProgramName == NULL ||
        numShaderStages == 0 ||
        pShaderStageOffsets == NULL)
    {
        NN_ASSERT(0, "Failed to read ShaderProgramHeader");
    }

    NVNProgramData* pProgramData = new NVNProgramData;
    pProgramData->Initialize();
    pProgramData->SetName(pProgramName, pShaderProgramHeader->m_ProgramNameLength);
    pProgramData->m_ShaderType = ShaderTypes::GetShaderTypeEnum(std::string(pProgramData->m_pProgramName));
    pProgramData->m_ShaderScratchMemorySize = pShaderProgramHeader->m_ShaderScratchMemorySize;

    std::vector<ShaderStageHeader> shaderStageHeaders;
    shaderStageHeaders.resize(numShaderStages, ShaderStageHeader());

        /*
         * Grab the stage metadata and pointers to the control and data section
         * of the compiled shader.
         */
    for(uint32_t i = 0; i < numShaderStages; ++i)
    {
        ShaderStageHeader* pShaderStageHeader = reinterpret_cast<ShaderStageHeader*>(m_pFileHead + pShaderStageOffsets[i]);

        shaderStageHeaders[i].m_ShaderStage         = pShaderStageHeader->m_ShaderStage;
        shaderStageHeaders[i].m_ShaderDataSize      = pShaderStageHeader->m_ShaderDataSize;
        shaderStageHeaders[i].m_ShaderControlSize   = pShaderStageHeader->m_ShaderControlSize;
        shaderStageHeaders[i].m_ShaderControlOffset = pShaderStageHeader->m_ShaderControlOffset;
        shaderStageHeaders[i].m_pShaderData         = m_pFileHead + pShaderStageOffsets[i] + 4 * sizeof(uint32_t);
        shaderStageHeaders[i].m_pShaderControl      = m_pFileHead + shaderStageHeaders[i].m_ShaderControlOffset;
    }

    CreateShaderProgram(shaderStageHeaders, pProgramData);
    return pProgramData;
}

/*
 * AssetFileLoadingHelper::CreateShaderProgram
 * -------------------------------------------
 * Saves the shader metadata in an NVNProgramData object
 * and creates an NVNprogram object from the compiled
 * shader binary.
 */
void AssetFileLoadingHelper::CreateShaderProgram(std::vector<ShaderStageHeader>& shaderStageHeaders, NVNProgramData* pProgramData)
{
    NN_ASSERT(pProgramData != NULL, "Pointer to program data to modify is NULL");

    uint32_t numShaderStages = static_cast<uint32_t>(shaderStageHeaders.size());

    std::vector<size_t> alignmentOffsets;
    alignmentOffsets.resize(numShaderStages, 0);

        /*
         * Grab the stage metadata and pointers to the control and data section
         * of the compiled shader.
         */
    size_t shaderDataBufferMemoryPoolSize = 0;
    int bufferAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &bufferAlignment);
    for(uint32_t i = 0; i < numShaderStages; ++i)
    {
            /* Store offsets for filling the memory pool and initializing buffers. */
        alignmentOffsets[i] = shaderDataBufferMemoryPoolSize;
        shaderDataBufferMemoryPoolSize += shaderStageHeaders[i].m_ShaderDataSize;

            /* Make sure the size is at the correct alignment. */
        shaderDataBufferMemoryPoolSize = Align(shaderDataBufferMemoryPoolSize, bufferAlignment);
    }

        /*
         * Shader code is not allowed to be in the last 1024 bytes of a memory pool,
         * additional padding is added to ensure that does not happen.
         */
    shaderDataBufferMemoryPoolSize += 1024;

        /* Copy the shader data into a contiguous block of memory to wrap a pool around. */
    size_t memoryPoolAlignedSize;

    if (shaderDataBufferMemoryPoolSize < g_MinimumPoolSize)
    {
        memoryPoolAlignedSize = g_MinimumPoolSize;
    }
    else
    {
        memoryPoolAlignedSize = Align(shaderDataBufferMemoryPoolSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

    char* shaderDataArray = reinterpret_cast<char*>(AlignedAllocate(memoryPoolAlignedSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT));

    for(uint32_t i = 0; i < shaderStageHeaders.size(); ++i)
    {
        memcpy(shaderDataArray + alignmentOffsets[i], shaderStageHeaders[i].m_pShaderData, shaderStageHeaders[i].m_ShaderDataSize);
    }

        /*
         * Create new memory pool around the shader data. Any memory pool
         * that wishes to load compiled shader data must have the
         * NVN_MEMORY_POOL_FLAGS_SHADER_CODE_BIT bit set.
         *
         * NOTE: Avoid loading a shader into the last 1kb of memory in the
         * memory pool. GPU shader cores may pre-fetch from instruction
         * memory beyond the last byte of actual shader code, which could
         * fault if the GPU virtual address space beyond the end of the
         * pool is unpopulated.
         */
    pProgramData->m_ShaderMemoryPool.Init(shaderDataArray,
                                          memoryPoolAlignedSize,
                                          NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_SHADER_CODE_BIT,
                                          m_pDevice);

        /*
         * NVNprogram
         * ----------
         * The NVNprogram object is a collection of different shader stages
         * that can be used for rendering or compute operations. The object
         * cannot hold both rendering and compute shaders at the same time
         * and can only have one compute shader. The program is created
         * through a call to nvnProgramInitialize followed by a call to
         * nvnProgramSetShaders. This function takes a list of NVNshaderData
         * structures (described below) that provide the object with shader
         * binary compiled by GLSLC. The shader program is bound using
         * nvnCommandBufferBindProgram which sets the active shader for
         * each stage.
         */

        /*
         * NVNshaderData
         * -------------
         * This structure is used to define an individual shader stage
         * that is to be made a part of an NVNprogram object. The structure
         * holds a pointer to the control section output by the GLSLC
         * compile as well as an NVNbufferAddress that holds the address
         * of an NVNbuffer created from the shader data section of the
         * GLSLC output.
         */

    std::vector<NVNshaderData> nvnShaderData;
    nvnShaderData.resize(numShaderStages, NVNshaderData());
    pProgramData->m_ShaderBuffers.resize(numShaderStages, NULL);

        /*
         * Create NVNbuffer objects out of the memory pool for each shader stage.
         * The NVNbufferAddress for these buffers are placed in a list with the
         * pointers to the corresponding shader control sections.
         */
    for(uint32_t i = 0; i < shaderStageHeaders.size(); ++i)
    {
        pProgramData->m_ShaderStages |= ConvertNVNStageToBitField(static_cast<NVNshaderStage>(shaderStageHeaders[i].m_ShaderStage));
        pProgramData->m_ShaderBuffers[i] = new NVNbuffer;

        nvnBufferBuilderSetDefaults(&m_BufferBuilder);
        nvnBufferBuilderSetStorage(&m_BufferBuilder, pProgramData->m_ShaderMemoryPool.GetMemoryPool(), alignmentOffsets[i], shaderStageHeaders[i].m_ShaderDataSize);

        if (!nvnBufferInitialize(pProgramData->m_ShaderBuffers[i], &m_BufferBuilder))
        {
            delete pProgramData->m_ShaderBuffers[i];
            NN_ASSERT(0, "Failed to initialize shader buffer storage");
        }

        nvnShaderData[i].control = shaderStageHeaders[i].m_pShaderControl;
        nvnShaderData[i].data    = nvnBufferGetAddress(pProgramData->m_ShaderBuffers[i]);
    }

        /* Initialize the shader program and provide it with the compiled shader data. */
    nvnProgramInitialize(&pProgramData->m_Program, m_pDevice);

    if (!nvnProgramSetShaders(&pProgramData->m_Program, numShaderStages, &nvnShaderData[0]))
    {
        NN_ASSERT(0, "Failed to set pre-compiled shaders");
    }
}

/*
 * AssetFileLoadingHelper::LoadTextures
 * ------------------------------------
 * Grabs the texture block header of the asset file and loads
 * the texture data at the specified offsets.
 */
void AssetFileLoadingHelper::LoadTextures(const char* pTextureBlockHead)
{
    NN_ASSERT(pTextureBlockHead != NULL, "Texture block header pointer NULL");

    const TextureBlockHeader* pTextureBlockHeader = reinterpret_cast<const TextureBlockHeader*>(pTextureBlockHead);
    const uint64_t* pTextureOffsets = reinterpret_cast<const uint64_t*>(pTextureBlockHead + sizeof(uint32_t) * 2);

    for (uint32_t i = 0; i < pTextureBlockHeader->m_NumTextures; ++i)
    {
        NVNTextureData* pTextureData = LoadTextureData(m_pFileHead + pTextureOffsets[i]);
        m_pDataHolder->AddTextureData(pTextureData);
    }
}

/*
 * AssetFileLoadingHelper::LoadTextureData
 * ---------------------------------------
 * Loads the packaged texture data and metadata from the texture header.
 */
NVNTextureData* AssetFileLoadingHelper::LoadTextureData(char* pTextureDataHead)
{
    NN_ASSERT(pTextureDataHead != NULL, "Texture data pointer NULL");

    NVNTextureData* pNvnTextureData = new NVNTextureData;
    pNvnTextureData->Initialize();

    uint32_t currentDataEntryOffset = 0;

        /* Grab the texture's metadata. */
    pNvnTextureData->m_TextureDataSize      = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint64_t);
    pNvnTextureData->m_GpuVersion           = *reinterpret_cast<const int*>(pTextureDataHead + currentDataEntryOffset);               currentDataEntryOffset += sizeof(int);
    pNvnTextureData->m_Alignment            = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_Width                = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_Height               = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_Depth                = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_NvnTextureTarget     = *reinterpret_cast<const NVNtextureTarget*>(pTextureDataHead + currentDataEntryOffset);  currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_NvnFormat            = *reinterpret_cast<const NVNformat*>(pTextureDataHead + currentDataEntryOffset);         currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_MipLevels            = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    uint32_t textureNameLength              = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    uint32_t textureDataOffset              = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    const char* pTextureName                = reinterpret_cast<const char*>(pTextureDataHead + currentDataEntryOffset);               currentDataEntryOffset += sizeof(char) * textureNameLength;
    void* pTextureData                      = (m_pFileHead + textureDataOffset);

    if (!textureNameLength ||
        pTextureName == NULL ||
        !pNvnTextureData->m_Width ||
        !pNvnTextureData->m_Height ||
        !pNvnTextureData->m_Depth ||
        pNvnTextureData->m_NvnTextureTarget > NVN_TEXTURE_TARGET_BUFFER ||
        !pNvnTextureData->m_NvnFormat ||
        !pNvnTextureData->m_MipLevels ||
        pTextureData == NULL)
    {
        NN_ASSERT(0, "Failed to read texture data");
    }

    size_t memoryPoolSize;

    if (pNvnTextureData->m_TextureDataSize < g_MinimumPoolSize)
    {
        memoryPoolSize = g_MinimumPoolSize;
    }
    else
    {
        memoryPoolSize = Align(static_cast<size_t>(pNvnTextureData->m_TextureDataSize), NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

    void* memoryPoolData = AlignedAllocate(memoryPoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);

    memcpy(memoryPoolData, pTextureData, static_cast<uint32_t>(pNvnTextureData->m_TextureDataSize));

        /* Wrap a memory pool around the texture data. */
    pNvnTextureData->m_TextureMemoryPool.Init(memoryPoolData,
                                              memoryPoolSize,
                                              NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                                              m_pDevice);

        /* Set up the texture builder based on the loaded metadata. */
    pNvnTextureData->SetName(pTextureName, textureNameLength);
    nvnTextureBuilderSetDefaults(&m_TextureBuilder);
    nvnTextureBuilderSetTarget(&m_TextureBuilder, pNvnTextureData->m_NvnTextureTarget);
    nvnTextureBuilderSetFormat(&m_TextureBuilder, pNvnTextureData->m_NvnFormat);
    nvnTextureBuilderSetSize2D(&m_TextureBuilder, pNvnTextureData->m_Width, pNvnTextureData->m_Height);
    nvnTextureBuilderSetDepth(&m_TextureBuilder, pNvnTextureData->m_Depth);
    nvnTextureBuilderSetLevels(&m_TextureBuilder, pNvnTextureData->m_MipLevels);

        /*
        * On Windows, the packaged texture data cannot be used by the GPU as
        * it will not be compatible with the format used by the device. To
        * remedy this, it can be provided to the texture builder through the
        * SetPackagedTextureData function. The windows driver will decode the
        * texture data and re-encode it to the proper format. On the device, the
        * function is a no-op; the pointer is ignored and its contents unmodified.
        */
    nvnTextureBuilderSetPackagedTextureData(&m_TextureBuilder, pTextureData);

        /* Set the storage for the texture. */
    nvnTextureBuilderSetStorage(&m_TextureBuilder, pNvnTextureData->m_TextureMemoryPool.GetMemoryPool(), 0);

        /* Initialize the texture with the builder's settings. */
    if (!nvnTextureInitialize(&pNvnTextureData->m_Texture, &m_TextureBuilder))
    {
        pNvnTextureData->Finalize();
        delete pNvnTextureData;
        NN_ASSERT(0, "Failed to initialize texture");
        return NULL;
    }
    else
    {
        pNvnTextureData->m_TextureInitialized = true;
    }

        /* Register the texture and hold onto its ID. */
    pNvnTextureData->m_TextureID = m_pTextureIDManager->RegisterTexture(&pNvnTextureData->m_Texture);
    return pNvnTextureData;
}

/*
 * AssetFileLoadingHelper::LoadModels
 * ----------------------------------
 * Grabs the model block header and loads the model data
 * at the specified offsets.
 */
void AssetFileLoadingHelper::LoadModels(const char* pModelBlockHead)
{
    NN_ASSERT(pModelBlockHead != NULL, "Model block header pointer NULL");

    uint32_t numModels = *reinterpret_cast<const uint32_t*>(pModelBlockHead);

    const uint64_t* pModelOffsets = reinterpret_cast<const uint64_t*>(pModelBlockHead + sizeof(uint32_t) * 2);

    for(uint32_t i = 0; i < numModels; ++i)
    {
        NVNModelData* pModelData = LoadModelData(m_pFileHead + pModelOffsets[i]);
        m_pDataHolder->AddModelData(pModelData);
    }
}

/*
 * AssetFileLoadingHelper::LoadModelData
 * -------------------------------------
 * Loads in the model vertex attributes and index buffer as well
 * as some data describing the model.
 */
NVNModelData* AssetFileLoadingHelper::LoadModelData(const char* pModelHead)
{
    NN_ASSERT(pModelHead != NULL, "Model data pointer NULL");

        /* Grab the model's metadata. */
    uint32_t            currentDataEntryOffset  = 0;
    uint32_t            numPrimitives           = *reinterpret_cast<const uint32_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint32_t);
    NVNdrawPrimitive    nvnDrawPrimitiveType    = *reinterpret_cast<const NVNdrawPrimitive*>(pModelHead + currentDataEntryOffset);      currentDataEntryOffset += sizeof(uint32_t);
    uint32_t            numVertexAttributes     = *reinterpret_cast<const uint32_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint32_t);
    uint32_t            modelNameLength         = *reinterpret_cast<const uint32_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint32_t);
    uint64_t            indexBufferOffset       = *reinterpret_cast<const uint64_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint64_t);
    const uint64_t*     pVertexAttributeOffsets = reinterpret_cast<const uint64_t*>(pModelHead + currentDataEntryOffset);               currentDataEntryOffset += sizeof(uint64_t) * numVertexAttributes;
    const char*         pModelName              = pModelHead + currentDataEntryOffset;                                                  currentDataEntryOffset += sizeof(char) * modelNameLength;

    if (modelNameLength == 0 ||
        pModelName == NULL ||
        !numPrimitives ||
        nvnDrawPrimitiveType > NVN_DRAW_PRIMITIVE_PATCHES ||
        !numVertexAttributes ||
        pVertexAttributeOffsets == NULL ||
        !indexBufferOffset)
    {
        NN_ASSERT(0, "Failed to read model data");
    }

    NVNModelData* pModelData = new NVNModelData;
    pModelData->Initialize();
    pModelData->SetName(pModelName, modelNameLength);

    Model* pModel = &pModelData->m_Model;
    pModel->m_NumVertexAttributes = numVertexAttributes;
    pModel->m_NumPrimitives = numPrimitives;
    pModel->m_NvnDrawPrimitiveType = nvnDrawPrimitiveType;

        /* Grab the data for the individual vertex attributes. */
    for(uint32_t i = 0; i < numVertexAttributes; ++i)
    {
        VertexAttributeHeader header;
        char* pAttributeHeader = m_pFileHead + pVertexAttributeOffsets[i];
        uint32_t currVertexEntryOffset  = 0;

        header.m_AttributeStride        = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_NvnFormat              = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_AttributeDataSize      = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_AttributeNameLength    = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_AttributeDataOffset    = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_MagicPadding           = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_pAttributeName         = pAttributeHeader + currVertexEntryOffset;                                         currVertexEntryOffset += sizeof(char) * header.m_AttributeNameLength;
        header.m_AttributeData          = m_pFileHead + header.m_AttributeDataOffset;

        pModelData->AddVertexAttribute(header);
        pModelData->m_VertexBufferSize += header.m_AttributeDataSize;
    }

        /* Grab the index buffer data. */
    char* pIndexBufferHead          = m_pFileHead + indexBufferOffset;
    uint32_t currIndexEntryOffset   = 0;

    uint32_t indexBufferSize        = *reinterpret_cast<const uint32_t*>(pIndexBufferHead + currIndexEntryOffset);        currIndexEntryOffset += sizeof(uint32_t);
    uint32_t numIndices             = *reinterpret_cast<const uint32_t*>(pIndexBufferHead + currIndexEntryOffset);        currIndexEntryOffset += sizeof(uint32_t);
    uint32_t indexType              = *reinterpret_cast<const uint32_t*>(pIndexBufferHead + currIndexEntryOffset);        currIndexEntryOffset += sizeof(uint32_t);
    currIndexEntryOffset            += sizeof(uint32_t); /* Account for padding in struct */
    void* pIndexData                = (pIndexBufferHead + currIndexEntryOffset);

        /* Align the index buffer according to its type. */
    size_t memoryPoolSize = pModelData->m_VertexBufferSize;
    uint32_t indexBufferAlignment = 0;
    switch((NVNindexType)indexType)
    {
        case NVN_INDEX_TYPE_UNSIGNED_BYTE:
            indexBufferAlignment = sizeof(uint8_t);
            break;
        case NVN_INDEX_TYPE_UNSIGNED_SHORT:
            indexBufferAlignment = sizeof(uint16_t);
            break;
        case NVN_INDEX_TYPE_UNSIGNED_INT:
            indexBufferAlignment = sizeof(uint32_t);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    NN_ASSERT(indexBufferSize / numIndices == indexBufferAlignment);
        /* Get the aligned pool size. */
    memoryPoolSize = Align(memoryPoolSize, indexBufferAlignment);
    size_t indexBufferAlignedLocation = memoryPoolSize;
    memoryPoolSize += indexBufferSize;

        /* Save the index buffer data in the model. */
    pModel->m_IndexData.m_IndexType = indexType;
    pModel->m_IndexData.m_DataSize  = indexBufferSize;
    pModel->m_IndexData.m_Stride    = indexBufferAlignment;
    pModel->m_IndexData.m_pData     = new char[indexBufferSize];

    memcpy(pModel->m_IndexData.m_pData, pIndexData, indexBufferSize);

        /* Allocate memory for the pool. */
    if (memoryPoolSize < g_MinimumPoolSize)
    {
        memoryPoolSize = g_MinimumPoolSize;
    }
    else
    {
        memoryPoolSize = Align(memoryPoolSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

    char* modelDataBuffer = reinterpret_cast<char*>(AlignedAllocate(memoryPoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT));
    memset(modelDataBuffer, 0, memoryPoolSize);

    uint32_t currentBufferOffset = 0;

        /* Copy in the vertex data. */
    for(uint32_t i = 0; i < pModel->m_VertexAttributes.size(); ++i)
    {
        memcpy(modelDataBuffer + currentBufferOffset, pModel->m_VertexAttributes[i].m_pData, pModel->m_VertexAttributes[i].m_DataSize);
        pModelData->m_VertexAttributeBufferOffsets.push_back(currentBufferOffset);
        currentBufferOffset += pModel->m_VertexAttributes[i].m_DataSize;
    }

        /* Copy the index data. */
    memcpy(modelDataBuffer + indexBufferAlignedLocation, pIndexData, indexBufferSize);

        /* Create a memory pool around the model data. */
    pModelData->m_BufferMemoryPool.Init(modelDataBuffer,
                                        memoryPoolSize,
                                        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                                        m_pDevice);

    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

        /* Create the vertex buffer from the memory pool at the correct offset. */
    nvnBufferBuilderSetStorage(&m_BufferBuilder, pModelData->m_BufferMemoryPool.GetMemoryPool(), 0, pModelData->m_VertexBufferSize);
    if (!nvnBufferInitialize(&pModelData->m_VertexBuffer, &m_BufferBuilder))
    {
        pModelData->Finalize();
        NN_ASSERT(0, "Failed to initialize vertex buffer storage");
    }

    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

        /* Create the index buffer from the memory pool at the correct offset. */
    nvnBufferBuilderSetStorage(&m_BufferBuilder, pModelData->m_BufferMemoryPool.GetMemoryPool(), indexBufferAlignedLocation, indexBufferSize);
    if (!nvnBufferInitialize(&pModelData->m_IndexBuffer, &m_BufferBuilder))
    {
        pModelData->Finalize();
        NN_ASSERT(0, "Failed to initialize index buffer storage");
    }

    return pModelData;
}//NOLINT(impl/function_size)
