﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/init/init_Malloc.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/socket_SystemConfig.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "ApplicationHeap.h"
#include "GraphicsSystem.h"
#include "FontSystem.h"
#include "Color.h"
#include "Npad.h"
#include "SceneManager.h"
#include "Async.h"

namespace {

const size_t MemoryHeapSize = 512 * 1024 * 1024;

const size_t MallocHeapSize = 256 * 1024 * 1024;

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
void* NvAllocate( size_t size, size_t alignment, void* userPtr ) NN_NOEXCEPT
{
    NN_UNUSED( userPtr );

    return aligned_alloc( alignment, nn::util::align_up(size , alignment));
}

void NvDeallocate( void* addr, void* userPtr ) NN_NOEXCEPT
{
    NN_UNUSED( userPtr );

    free( addr );
}

void* NvReallocate( void* addr, size_t newSize, void* userPtr ) NN_NOEXCEPT
{
    NN_UNUSED( userPtr );
    return realloc( addr, newSize );
}
#endif

nn::socket::SystemConfigDefaultWithMemory<10, 10> g_SocketConfig;

} // namespace


extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// アプリケーションのエントリポイント
extern "C" void nnMain()
{
    nn::socket::Initialize(g_SocketConfig);
    Async::Initialize();

    // デバッグ設定書き込み
    const bool isEnabled = true;
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
    nn::nifm::Initialize();
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);
    // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておく
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    nn::oe::Initialize();

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator( NvAllocate, NvDeallocate, NvReallocate, nullptr );
    nv::SetGraphicsDevtoolsAllocator( NvAllocate, NvDeallocate, NvReallocate, NULL );
    nv::InitializeGraphics( std::malloc( GraphicsMemorySize ), GraphicsMemorySize );
#endif

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize( ApplicationHeapSize );

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(
        &applicationHeap, FrameBufferWidth, FrameBufferHeight);

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&applicationHeap, pGraphicsSystem);

    SceneManager sceneMgr(&applicationHeap);
    sceneMgr.Initialize();

    while( sceneMgr.IsExit() == false )
    {
        sceneMgr.Update();
        sceneMgr.Draw(pGraphicsSystem, pFontSystem);
    }
    NN_LOG("exit, %d\n", __LINE__);
    sceneMgr.Finalize();

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();

    Async::Finalize();
    nn::socket::Finalize();
}
