﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneManager.h"

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_Result.h>

#include "SceneConnected.h"
#include "SceneConnecting.h"
#include "SceneConnectionParam.h"
#include "SceneDirectInput.h"
#include "SceneScanning.h"
#include "SceneScanResult.h"
#include "SceneTitle.h"

namespace {
// スキャンバッファ
const size_t  smBuffSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_smScanBuffer[ smBuffSize ];
}

SceneManager::SceneManager(ApplicationHeap* pApplicationHeap) :
        m_nextScene(eScene_None)
{
    m_scene = (BaseScene*) new SceneTitle(this, this);
    m_prevScene = eScene_Title;
    m_currScene = eScene_Title;
    m_pApplicationHeap = pApplicationHeap;
    m_isExit = false;
}

void SceneManager::Initialize() NN_NOEXCEPT
{
    m_ssid.Set("");
    m_channel = 1;
    memset(&m_security, 0, sizeof(nn::wlan::Security));
    strcpy(reinterpret_cast<char*>(m_security.key), "");
    strcpy(m_password, "");
    memset(g_smScanBuffer, 0, sizeof(g_smScanBuffer));

    // wlanの初期化
    nn::wlan::InitializeInfraManager();

    nn::Result result = nn::wlan::Infra::OpenMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    m_scene->Initialize();
}

void SceneManager::Finalize() NN_NOEXCEPT
{
    if( m_scene != nullptr )
    {
        m_scene->Finalize();
        delete m_scene;
    }

    // wlanの終了処理
    nn::wlan::Infra::CloseMode();
    nn::wlan::FinalizeInfraManager();
}

void SceneManager::Update() NN_NOEXCEPT
{
    if(m_nextScene != eScene_None)  // 次のシーンがセットされているか
    {
        // 現在のシーンの終了処理
        m_scene->Finalize();
        delete m_scene;

        switch( m_nextScene )
        {
        case eScene_Title:
            m_scene = (BaseScene*) new SceneTitle(this, this);
            break;
        case eScene_Scanning:
            m_scene = (BaseScene*) new SceneScanning(this, this);
            break;
        case eScene_ScanResult:
            m_scene = (BaseScene*) new SceneScanResult(this, this);
            break;
        case eScene_ConnectionParam:
            m_scene = (BaseScene*) new SceneConnectionParam(this, this);
            {
                SceneConnectionParam* pScene = reinterpret_cast<SceneConnectionParam*>(m_scene);
                pScene->SetApplicationHeap(m_pApplicationHeap);
            }
            break;
        case eScene_Connecting:
            m_scene = (BaseScene*) new SceneConnecting(this, this);
            break;
        case eScene_Connected:
            m_scene = (BaseScene*) new SceneConnected(this, this);
            break;
        case eScene_DirectInput:
            m_scene = (BaseScene*) new SceneDirectInput(this, this);
            {
                SceneDirectInput* pScene = reinterpret_cast<SceneDirectInput*>(m_scene);
                pScene->SetApplicationHeap(m_pApplicationHeap);
            }
            break;
        case eScene_Preset:
            break;
        case eScene_Exit:
            m_scene = (BaseScene*) new SceneTitle(this, this);
            m_isExit = true;
            break;
        default:
            break;
        }

        //次のシーン情報をクリア
        m_prevScene = m_currScene;
        m_currScene = m_nextScene;
        m_nextScene = eScene_None;
        //シーンを初期化
        m_scene->Initialize();
    }

    //シーンの更新
    m_scene->Update();
}

void SceneManager::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    //シーンの描画
    m_scene->Draw(pGraphicsSystem, pFontSystem);
}

bool SceneManager::IsExit() NN_NOEXCEPT
{
    return m_isExit;
}

// 引数 nextScene にシーンを変更する
void SceneManager::ChangeScene(eScene NextScene) NN_NOEXCEPT
{
    //次のシーンをセットする
    m_nextScene = NextScene;
}

eScene SceneManager::GetPrevScene() NN_NOEXCEPT
{
    return m_prevScene;
}

void SceneManager::SetScanResult(char* pData, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pData);
    memset(g_smScanBuffer, 0, sizeof(g_smScanBuffer));
    memcpy(g_smScanBuffer, pData, size);
}

char* SceneManager::GetScanResult() NN_NOEXCEPT
{
    return g_smScanBuffer;
}

void SceneManager::SetConnectionParam(nn::wlan::Ssid ssid, int16_t channel, nn::wlan::Security security, char* pw) NN_NOEXCEPT
{
    m_ssid.Set(ssid.GetSsidData(), ssid.GetLength());
    m_channel = channel;
    memcpy(&m_security, &security, sizeof(nn::wlan::Security));
    if( pw == NULL )
    {
        strcpy(m_password, "");
    }
    else
    {
        strcpy(m_password, pw);
    }
}

void SceneManager::GetConnectionParam(nn::wlan::Ssid* ssid, int16_t* channel, nn::wlan::Security* security, char* pw) NN_NOEXCEPT
{
    ssid->Set(m_ssid.GetSsidData(), m_ssid.GetLength());
    *channel = m_channel;
    memcpy(security, &m_security, sizeof(nn::wlan::Security));
    strcpy(pw, m_password);
}

void SceneManager::SetOwnIp(nn::wlan::WlanIpv4Address ip) NN_NOEXCEPT
{
    for( int i = 0; i < 4; i++ )
    {
        m_ownIp.addr[i] = ip.addr[i];
    }
}

void SceneManager::GetOwnIp(nn::wlan::WlanIpv4Address* pIp) NN_NOEXCEPT
{
    for( int i = 0; i < 4; i++ )
    {
        pIp->addr[i] = m_ownIp.addr[i];
    }
}
