﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>
#include <iomanip>

#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_SocketApi.h>

#include "Util.h"

using namespace std;

namespace {

char MyToUpper(char c)
{
    return std::toupper(c);
}

char MyToLower(char c)
{
    return std::tolower(c);
}

}

namespace WlanTest {

void TPrintResult(nn::Result result)
{
    // #if NN_VERSION_MAJOR >= 3
    //     NN_DBG_PRINT_RESULT(result);
    // #else
    //     nn::dbg::PrintResult(result);
    // #endif
    NN_LOG("  % Result module      : %d\n", result.GetModule());
    NN_LOG("  % Result description : 0x%x\n", result.GetDescription());
    NN_LOG("  % Result inner value : 0x%08x\n", result.GetInnerValueForDebug());
}

void AlignedDelete( void* p )
{
    std::free( p );
}

bool ReadFile(uint8_t out[], uint32_t outSize, const char* path)
{
    if(path == nullptr)
    {
        return false;
    }

    bool error = false;
    NN_LOG("Read file : %s\n", path);

    nn::Result result = nn::fs::MountHostRoot();
    if(result.IsFailure())
    {
        NN_LOG("  - failed : MountHostRoot()\n");
        return false;
    }

    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    if(result.IsFailure())
    {
        NN_LOG("  - failed : OpenFile()\n");
        NN_LOG("           : %s\n", path);
        nn::fs::UnmountHostRoot();
        return false;
    }

    int64_t fileSize = 0;
    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    if(result.IsFailure())
    {
        NN_LOG("  - failed : GetFileSize()\n");
        error = true;
    }

    if(!error && fileSize <= outSize)
    {
        memset(out, 0x00, outSize);
        result = nn::fs::ReadFile(fileHandle, 0, out, static_cast<size_t>(fileSize));
        if(result.IsFailure())
        {
            NN_LOG(" - failed : ReadFile()\n");
            error = true;
        }

        NN_LOG("\n<< File (%lld byte)>>\n--------\n%s\n--------\n", fileSize, out);
    }
    else
    {
        NN_LOG("  - failed : too large file (> %d)\n", outSize);
        error = true;
    }

    nn::fs::CloseFile(fileHandle);
    nn::fs::UnmountHostRoot();

    return !error;
}

std::string ToString(const nn::wlan::SecurityMode& mode)
{
    switch(mode)
    {
    case nn::wlan::SecurityMode_Open       : return "OPEN";
    case nn::wlan::SecurityMode_Wpa2Aes    : return "WPA2AES";
    case nn::wlan::SecurityMode_StaticAes  : return "StaticAES";
    default : return "Unknown";
    }
}

string ToString(const nn::wlan::DetectHash& hash)
{
    ostringstream oss;
    for(int i=0; i<sizeof(hash.hash); ++i)
    {
        oss << setw(2) << setfill('0') << hex << static_cast<int>(hash.hash[i]);
    }
    return oss.str();
}

std::string ToString(const nn::TimeSpan& timespan, const TimeUnit& fillFrom, const TimeFormat& format)
{
    nn::TimeSpan ts = timespan;
    ostringstream oss;
    bool pre = false;

    // 無限とみなす
    int64_t n = ts.GetDays();
    if( n > 30 )
    {
        return "-";
    }
    else if( format == TimeFormat_Alphabet && (fillFrom == TIME_UNIT_DAYS || n > 0) )
    {
        oss << n << " d ";
        ts -= nn::TimeSpan::FromDays(n);
        pre = true;
    }

    n = ts.GetHours();
    if( n > 0 || pre || fillFrom == TIME_UNIT_HOURS )
    {
        oss << setw(2) << setfill('0') << n;
        switch( format )
        {
        case TimeFormat_Alphabet : oss << " h "; break;
        case TimeFormat_Colon    : oss << ":"; break;
        default : break;
        }
        ts -= nn::TimeSpan::FromHours(n);
        pre = true;
    }

    n = ts.GetMinutes();
    if( n > 0 || pre || fillFrom == TIME_UNIT_MINUTES )
    {
        oss << setw(2) << setfill('0') << n;
        switch( format )
        {
        case TimeFormat_Alphabet : oss << " m "; break;
        case TimeFormat_Colon    : oss << ":"; break;
        default : break;
        }
        ts -= nn::TimeSpan::FromMinutes(n);
        pre = true;
    }

    // 必ず表示
    n = ts.GetSeconds();
    if( true )
    {
        oss << setw(2) << setfill('0') << n;
        switch( format )
        {
        case TimeFormat_Alphabet : oss << " s "; break;
        case TimeFormat_Colon    : break;
        default : break;
        }
        ts -= nn::TimeSpan::FromSeconds(n);
        pre = true;
    }

    return oss.str();
}

std::string ToStringSimpleTimeSpan(const nn::TimeSpan& timespan)
{
    nn::TimeSpan ts = timespan;
    ostringstream oss;
    bool pre = false;

    // 無限とみなす
    int64_t n = ts.GetDays();
    if(n > 30)
    {
        return "Infinity";
    }
    else if(n > 0)
    {
        oss << n << " day ";
        ts -= nn::TimeSpan::FromDays(n);
        pre = true;
    }

    n = ts.GetHours();
    if(n > 0 && !pre)
    {
        oss << n << " hour ";
        ts -= nn::TimeSpan::FromHours(n);
        pre = true;
    }

    n = ts.GetMinutes();
    if(n > 0 && !pre)
    {
        oss << n << " min ";
        ts -= nn::TimeSpan::FromMinutes(n);
        pre = true;
    }

    n = ts.GetSeconds();
    if(n > 0 && !pre)
    {
        oss << n << " sec ";
        ts -= nn::TimeSpan::FromSeconds(n);
        pre = true;
    }

    n = ts.GetMilliSeconds();
    if(n > 0 && !pre)
    {
        oss << n << " ms ";
        ts -= nn::TimeSpan::FromSeconds(n);
        pre = true;
    }

    if( !pre )
    {
        oss << "-";
    }

    return oss.str();
}

string ToString(const nn::wlan::MacAddress& addr)
{
    static char buf[nn::wlan::MacAddress::MacStringSize];
    addr.GetString(buf);
    return buf;
}

string ToString(const nn::btm::BdAddress& bdaddr)
{
    static char buf[18];
    snprintf(buf, sizeof(buf), "%02X:%02X:%02X:%02X:%02X:%02X",
             bdaddr.address[0], bdaddr.address[1], bdaddr.address[2], bdaddr.address[3], bdaddr.address[4], bdaddr.address[5]);
    return buf;
}

string ToString(const nn::btm::BluetoothMode& btMode, const nn::btm::WlanMode& wlMode)
{
    // hid の都合により、Static Local モードにいるときは BlutoothMode が正しく設定されない。
    if( wlMode == nn::btm::WlanMode_Local4 || wlMode == nn::btm::WlanMode_Local8 )
    {
        return "Static";
    }

    switch(btMode)
    {
    case nn::btm::BluetoothMode_Active : return "Active";
    case nn::btm::BluetoothMode_Auto   : return "Auto";
    default                            : return "Unknown";
    }
}

string ToString(const nn::btm::WlanMode& mode)
{
    switch(mode)
    {
    case nn::btm::WlanMode_Local4 : return "Local4";
    case nn::btm::WlanMode_Local8 : return "Local8";
    case nn::btm::WlanMode_None   : return "None";
    default                       : return "Unknown";
    }
}

string ToString(const nn::btm::SniffMode& mode)
{
    switch(mode)
    {
    case nn::btm::SniffMode_5ms    : return "5 ms";
    case nn::btm::SniffMode_10ms   : return "10 ms";
    case nn::btm::SniffMode_15ms   : return "15 ms";
    case nn::btm::SniffMode_Active : return "Active";
    default                        : return "Unknown";
    }
}

string ToString(const nn::btm::SlotMode& mode)
{
    switch(mode)
    {
    case nn::btm::SlotMode_2        : return "2 slots";
    case nn::btm::SlotMode_4        : return "4 slots";
    case nn::btm::SlotMode_6        : return "6 slots";
    case nn::btm::SlotMode_Active   : return "Active";
    default                         : return "Unknown";
    }
}

string ToString(const Button& button)
{
    switch(button)
    {
    case Button::A        : return "A";
    case Button::B        : return "B";
    case Button::X        : return "X";
    case Button::Y        : return "Y";
    case Button::L        : return "L";
    case Button::R        : return "R";
    case Button::ZL       : return "ZL";
    case Button::ZR       : return "ZR";
    case Button::START    : return "START";
    case Button::SELECT   : return "SELECT";
    case Button::LEFT     : return "LEFT";
    case Button::UP       : return "UP";
    case Button::RIGHT    : return "RIGHT";
    case Button::DOWN     : return "DOWN";
    case Button::SL       : return "SL";
    case Button::SL_LEFT  : return "SL_LEFT";
    case Button::SL_UP    : return "SL_UP";
    case Button::SL_RIGHT : return "SL_RIGHT";
    case Button::SL_DOWN  : return "SL_DOWN";
    case Button::SR       : return "SR";
    case Button::SR_LEFT  : return "SR_LEFT";
    case Button::SR_UP    : return "SR_UP";
    case Button::SR_RIGHT : return "SR_RIGHT";
    case Button::SR_DOWN  : return "SR_DOWN";
    case Button::MINUS    : return "-";
    case Button::PLUS     : return "+";
    default               : return "?";
    }
}

std::string ToStringForBinaryPrefix(const double& v, const string& separator, const int8_t precision)
{
    double value = v;
    uint32_t index = 0;
    static char unit[] = {' ', 'K', 'M', 'G'};

    while(value >= 1024.0f)
    {
        value /= 1024.0f;
        ++index;
    }

    ostringstream oss;
    if( precision >= 0 )
    {
        oss << fixed << setprecision(precision) << value << separator;
    }
    else
    {
        oss << value << separator;
    }

    if(index > 0)
    {
        oss << unit[index];
    }

    return oss.str();
}

std::string ToStringForDecimalPrefix(const double& v, const string& separator, const int8_t precision)
{
    double value = v;
    uint32_t index = 0;
    static char unit[] = {' ', 'K', 'M', 'G'};

    while(value >= 1000.0f)
    {
        value /= 1000.0f;
        ++index;
    }

    ostringstream oss;
    if( precision >= 0 )
    {
        oss << fixed << setprecision(precision) << value << separator;
    }
    else
    {
        oss << value << separator;
    }

    if(index > 0)
    {
        oss << unit[index];
    }

    return oss.str();
}

std::string ToStringTrueFalse(const int32_t& v, const CaseOption option)
{
    switch(v)
    {
    case 0 : return ConvertWordCase("False", option);
    case 1 : return ConvertWordCase("True", option);
    default : return ConvertWordCase("UNKNOWN", option);
    }
}

std::string ToStringOnOff(const int32_t& v, const CaseOption option)
{
    switch(v)
    {
    case 0 : return ConvertWordCase("Off", option);
    case 1 : return ConvertWordCase("On", option);
    default : return ConvertWordCase("Unkown", option);
    }
}

std::string ToStringYesNo(const int32_t& v, const CaseOption option)
{
    switch(v)
    {
    case 0 : return ConvertWordCase("No", option);
    case 1 : return ConvertWordCase("Yes", option);
    default : return ConvertWordCase("Unknown", option);
    }
}

std::string ToStringSocketType(const SocketType& type)
{
    switch(type)
    {
    case SOCKET_TYPE_UNKNOWN : return SOCKET_TYPE_UNKNOWN_STR;
    case SOCKET_TYPE_TCP : return SOCKET_TYPE_TCP_STR;
    case SOCKET_TYPE_UDP : return SOCKET_TYPE_UDP_STR;
    default : return "Unknown";
    }
}

std::string ToStringPacketFormat(const PacketFormat& format)
{
    switch(format)
    {
    case PACKET_FORMAT_WIT : return PACKET_FORMAT_WIT_STR;
    case PACKET_FORMAT_RTP : return PACKET_FORMAT_RTP_STR;
    default : return "Unknown";
    }
}

uint64_t ToUint(const nn::wlan::MacAddress& macAddress)
{
    uint64_t address = 0;
    const nn::wlan::MacAddress::MacAddressArray& array = macAddress.GetMacAddressData();
    NN_ASSERT(sizeof(uint64_t) >= sizeof(nn::wlan::MacAddress::MacAddressArray));
    std::memcpy(&address, array, sizeof(nn::wlan::MacAddress::MacAddressArray));

    return address;
}

uint64_t ToUint(const nn::wlan::DetectHash& hash)
{
    uint64_t rtn = 0;
    rtn = ((static_cast<uint64_t>(hash.hash[0]) << 56) & 0xff00000000000000) |
        ((static_cast<uint64_t>(hash.hash[1]) << 48) & 0x00ff000000000000) |
        ((static_cast<uint64_t>(hash.hash[2]) << 40) & 0x0000ff0000000000) |
        ((static_cast<uint64_t>(hash.hash[3]) << 32) & 0x000000ff00000000) |
        ((static_cast<uint64_t>(hash.hash[4]) << 24) & 0x00000000ff000000) |
        ((static_cast<uint64_t>(hash.hash[5]) << 16) & 0x0000000000ff0000) |
        ((static_cast<uint64_t>(hash.hash[6]) << 8)  & 0x000000000000ff00) |
        ((static_cast<uint64_t>(hash.hash[7]) << 0)  & 0x00000000000000ff);

    return rtn;
}

void ToUpper(char* str, int size)
{
    for(int i=0; i<size; ++i)
    {
        str[i] = std::toupper(str[i]);
    }
}

string ToUpper(const string& s)
{
    string str = s;
    transform(str.begin(), str.end(), str.begin(), MyToUpper);
    return str;
}

void ToLower(char* str, int size)
{
    for(int i=0; i<size; ++i)
    {
        str[i] = std::tolower(str[i]);
    }
}

string ToLower(const string& s)
{
    string str = s;
    transform(str.begin(), str.end(), str.begin(), MyToLower);
    return str;
}

string Trim(const string& s, const char* charList)
{
    string str = s;

    string::size_type index = str.find_first_not_of(charList);
    if(index != string::npos)
    {
        str = str.substr(index);
    }

    index = str.find_last_not_of(charList);
    if(index != string::npos)
    {
        str = str.substr(0, index + 1);
    }

    return str;
}

bool GenerateTimeSpan(nn::TimeSpan& timeSpan, const int64_t& value, const TimeUnit& unit)
{
    switch(unit)
    {
    case TIME_UNIT_NANO_SECONDS  : timeSpan = nn::TimeSpan::FromNanoSeconds(value); break;
    case TIME_UNIT_MICRO_SECONDS : timeSpan = nn::TimeSpan::FromMicroSeconds(value); break;
    case TIME_UNIT_MILLI_SECONDS : timeSpan = nn::TimeSpan::FromMilliSeconds(value); break;
    case TIME_UNIT_SECONDS       : timeSpan = nn::TimeSpan::FromSeconds(value); break;
    case TIME_UNIT_MINUTES       : timeSpan = nn::TimeSpan::FromMinutes(value); break;
    case TIME_UNIT_HOURS         : timeSpan = nn::TimeSpan::FromHours(value); break;
    case TIME_UNIT_DAYS          : timeSpan = nn::TimeSpan::FromDays(value); break;
    default : return false;
    }

    return true;
}

bool GenerateTimeSpan(nn::TimeSpan& timeSpan, const int64_t& value, const char& unit)
{
    switch(unit)
    {
    case 'd' :
    case 'D' :
        GenerateTimeSpan(timeSpan, value, TIME_UNIT_DAYS);
        break;

    case 'h' :
    case 'H' :
        GenerateTimeSpan(timeSpan, value, TIME_UNIT_HOURS);
        break;

    case 'm' :
    case 'M' :
        GenerateTimeSpan(timeSpan, value, TIME_UNIT_MINUTES);
        break;

    case 's' :
    case 'S' :
        GenerateTimeSpan(timeSpan, value, TIME_UNIT_SECONDS);
        break;

    default  :
        return false;
    }

    return true;
}

bool GenerateByteScale(uint32_t& scale, const uint32_t& value, const char& unit)
{
    switch(unit)
    {
    case 'b' :
    case 'B' :
        {
            scale = value;
        }
        break;

    case 'k' :
    case 'K' :
        {
            scale = value * 1024;
        }
        break;

    case 'm' :
    case 'M' :
        {
            scale = value * 1024 * 1024;
        }
        break;

    default  :
        {
            return false;
        }
    }

    return true;
}

string ConvertWordCase(const string& from, const CaseOption option)
{
    if( from.size() == 0 )
    {
        return from;
    }

    switch(option)
    {
    case CASE_OPTION_UPPER : return ToUpper(from);
    case CASE_OPTION_LOWER : return ToLower(from);
    case CASE_OPTION_PROPER : return static_cast<char>(toupper(from[0])) + ToLower(from.substr(1, from.size() - 1));
    default : return from;
    }
}

string PackString(const string& from, uint32_t width, uint32_t height)
{
    string to;

    for(int i=0; i<height; ++i)
    {
        to += from.substr(i * width, (i + 1) * width) + '\n';
        if( from.size() <= (i + 1) * width )
        {
            break;
        }
    }

    return to;
}

string DuplicateEscapeSequence(const string& from, const string& escapeCodes)
{
    string to;
    for(int i=0; i<from.size(); ++i)
    {
        to += from[i];
        if( escapeCodes.find(from[i]) != string::npos )
        {
            to += from[i];
        }
    }

    return to;
}

int Count(const string& str, const char c)
{
    int cnt = 0;
    for(int i=0; i<str.size(); ++i)
    {
        if( str[i] == c )
        {
            ++cnt;
        }
    }

    return cnt;
}

double CalculateThroughput(uint64_t sizeByte, int64_t timeMs, ThroughputUnit unit)
{
    double div = 1.0f;
    switch(unit)
    {
    case ThroughputUnit_Bps : div = 1.0f; break;
    case ThroughputUnit_Kbps : div = 1024.0f; break;
    case ThroughputUnit_Mbps : div = 1024.0f * 1024.0f; break;
    default : div = 1.0f; break;
    }
    return 1.0f * sizeByte * 8.0f * 1000.0f / div / timeMs;
}

double CalculateThroughput(uint64_t sizeByte, nn::os::Tick start, nn::os::Tick end, ThroughputUnit unit)
{
    return CalculateThroughput(sizeByte, (end - start).ToTimeSpan().GetMilliSeconds(), unit);
}

string GetMacAddressStr(const string& separator)
{
    ostringstream oss;
    uint8_t size = nn::wlan::MacAddress::MacAddressSize;
    nn::wlan::MacAddress macAddress;
    nn::wlan::Local::GetMacAddress(&macAddress);
    for(int i=0; i<size; ++i)
    {
        if( i != 0 )
        {
            oss << separator;
        }

        oss << hex << setw(2) << setfill('0') << static_cast<uint32_t>(macAddress.GetMacAddressData()[i]);
    }

    return oss.str();
}

string GetTodayStr()
{
    // 現在日時を生成し、ファイル名にする
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );

    // 協定世界時 (UTC) の 西暦1970年1月1日午前0時0分0秒 からの経過秒数で現在時刻を取得します。
    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );

    // PosixTime を CalendarTime へ変換します。
    // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用します。
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
    char buf[30];
    snprintf(buf, 30, "%04d%02d%02d-%02d%02d%02d",
             calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );

    return buf;
}

bool operator!=(const nn::btm::BdAddress& address1, const nn::btm::BdAddress& address2)
{
    return !(address1 == address2);
}

ostream& operator<<(ostream& stream, const nn::bluetooth::BluetoothAddress& address)
{
    stream << WlanTest::ToString(address);
    return stream;
}

bool operator==(const nn::wlan::DetectHash& hash1, const nn::wlan::DetectHash& hash2)
{
    for(int i=0; i<sizeof(hash1.hash); ++i)
    {
        if( hash1.hash[i] != hash2.hash[i] )
        {
            return false;
        }
    }
    return true;
}

bool operator!=(const nn::wlan::DetectHash& hash1, const nn::wlan::DetectHash& hash2)
{
    return !(hash1 == hash2);
}

bool ConvertWepKey(string* to, const string& from, const nn::wlan::SecurityMode mode)
{
    NN_ASSERT(to);

    ssize_t asciiKeySize = 0;
    switch(mode)
    {
    case nn::wlan::SecurityMode_Wep64Open :
    case nn::wlan::SecurityMode_Wep64Shared :
        {
            asciiKeySize = WEP64_ASCII_KEY_SIZE;
        }
        break;

    case nn::wlan::SecurityMode_Wep128Open :
    case nn::wlan::SecurityMode_Wep128Shared :
        {
            asciiKeySize = WEP128_ASCII_KEY_SIZE;
        }
        break;

    default :
        return false;
    }

    ssize_t size = from.size();
    if( size == asciiKeySize )
    {
        // ascii から HEX に変換
        char buf[2 * asciiKeySize + 1];
        for(int i=0; i<asciiKeySize; ++i)
        {
            NN_ASSERT( snprintf(buf + i * 2, 3, "%02x", from[i]) == 2 );
        }

        *to = buf;
    }
    else if( size == 2 * asciiKeySize )
    {
        *to = from;
    }
    else
    {
        return false;
    }

    return true;
}

nn::Result GetTsfTime(int64_t* pTime)
{
    int64_t delta = 0;
    nn::Result result = nn::wlan::Socket::GetDeltaTimeBetweenSystemAndTsf(&delta);
    if( result.IsSuccess() )
    {
        *pTime = nn::os::GetSystemTick().ToTimeSpan().GetMicroSeconds() - delta;
    }
    else
    {
        NN_LOG(" - failed : GetDeltaTimeBetweenSystemAndTsf()\n");
    }
    return result;
}

}
