﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Node.h"
#include "Log.h"

uint8_t buffer[32 * 1024];
uint8_t commandBuffer[2000];

// nn::wlan で未実装部分
namespace nn { namespace wlan { namespace Local {

nn::Result GetBssIndicationEvent(nn::os::SystemEventType* pSystemEvent)
{
    NN_UNUSED(pSystemEvent);

    return ResultSuccess();
}

nn::Result GetBssIndicationInfo(uint8_t (&bssInfo)[BssIndicationBufferSize])
{
    NN_UNUSED(bssInfo);

    return ResultSuccess();
}

} } }

namespace WlanTest {

void GenerateWitHeader(WitHeader* pHeader, const WitCommand command, const uint32_t length)
{
    static nn::os::Mutex mutex(false);
    static uint32_t sequenceNumber = 0;

    mutex.Lock();

    pHeader->dummy   = 0xffff;
    pHeader->version = 0x01;
    pHeader->command = command;
    pHeader->length  = length;
    pHeader->reserved = 0x0;
    pHeader->sequence = sequenceNumber++;

    mutex.Unlock();
}

/*---------------------------------------------------------------------------
　　　　　Ieee802_3_Llc_Snap
---------------------------------------------------------------------------*/

// Ieee802_3_Llc_Snap::Ieee802_3_Llc_Snap()
DixFrame::DixFrame()
{
    // std::memset(m_Destination, 0, 3);
    // std::memset(m_Source, 0, 3);
    // m_Length = 0;
    // m_Dsap = 0xAA;    // SNAP
    // m_Ssap = 0xAA;    // SNAP
    // m_Control = 0x03; // コネクションレス、信頼性オプションなし
    // std::memset(m_Pid, 0, 3);
    // m_Type = 0;
    // std::memset(m_Payload, 0, 1506);
    std::memset(m_Destination, 0x00, sizeof(m_Destination));
    std::memset(m_Source, 0x00, sizeof(m_Source));
    m_Type = 0x0000;
    std::memset(m_Payload, 0, sizeof(m_Payload));
}



/*---------------------------------------------------------------------------
　　　　　Ieee802_3_Llc_Snap
---------------------------------------------------------------------------*/

NintendoIE::NintendoIE(size_t length)
{
    NN_ASSERT(length > 6);
    ElementID = 0xdd;
    Length = length - 2;
    OUI[0] = 0x00;
    OUI[1] = 0x1f;
    OUI[2] = 0x32;
    SubType = 0x15;
    Data = new nn::Bit8[length - 6]; // 6はヘッダ分
    std::memset(Data, 0, length - 6);
}

NintendoIE::~NintendoIE()
{
    delete[] Data;
}

std::string NintendoIE::ToString()
{
    std::string s;
    s.reserve(1000);
    std::ostringstream oss;
    oss << std::setbase(16);
    oss << "ElementID : 0x" << (int)ElementID << std::endl;
    s.append(oss.str());
    oss.str("");

    oss << std::setbase(10);
    oss << "Length    : " << (int)Length << std::endl;
    s.append(oss.str());
    oss.str("");

    oss << std::setbase(16) << std::setfill('0');
    oss << "OUI       : " << std::setw(2) << (int)OUI[0] << ":" << std::setw(2) << (int)OUI[1] << ":" << std::setw(2) << (int)OUI[2] << std::endl;
    s.append(oss.str());
    oss.str("");

    oss << "SubType   : 0x" << (int)SubType<< std::endl;
    oss << "Data      : " << std::endl;
    s.append(oss.str());
    oss.str("");

    for(int i=0; i<Length - 6; ++i)
    {
        if(i % 16 == 0)
            oss << "0x" << std::setfill('0') << std::setw(2) << i << " | ";

        oss << std::setw(2) << (int)Data[i];

        if( i % 16 == 15)
        {
            oss << std::endl;
            s.append(oss.str());
            oss.str("");
        }
        else if( i % 16 == 7)
        {
            oss << "   ";
        }
        else
        {
            oss << " ";
        }
    }
    oss << std::endl;
    s.append(oss.str());
    oss.str("");

    return s;
}

/*---------------------------------------------------------------------------
　　　　　Receiver
---------------------------------------------------------------------------*/
IReceiver::IReceiver()
{
    m_Handler = NULL;
    m_Target = NULL;
}

IReceiver::~IReceiver()
{
}

/*---------------------------------------------------------------------------
　　　　　ProtocolReceiver
---------------------------------------------------------------------------*/

ProtocolReceiver::ProtocolReceiver()
{
    m_ProtocolType = PID_OUI_EXT;
    m_RequestStop = false;
    m_IsStop = true;
}

void ProtocolReceiver::ReceiveThreadFunction(void* receiver)
{
    DBG_VLOG("Thread start - ProtocolReceiver::ReceiveThread");
    reinterpret_cast<ProtocolReceiver*>(receiver)->Receive();
    DBG_VLOG("Thread stop - ProtocolReceiver::ReceiveThread");
}

void ProtocolReceiver::Receive()
{
    if(m_Target == NULL || m_ProtocolType == 0)
    {
        NN_LOG("Target: %d\n ProtocolType : %d\n", m_Target, m_ProtocolType);
        NN_LOG("Can't receive because Target or ProtocolType is not not set.");
        m_RequestStop = true;
        return;
    }

    nn::Result result;
    uint32_t rxId = m_Target->GetRxIdForData();
    ReceiveEventArgs args;
    args.Data = buffer;

    PacketLogger& logger = PacketLogger::GetInstance();
    if( logger.IsEnabled() )
    {
        logger.Write("SourceAddress, Size(byte), Seqno, RxTime(us), TxTime(us), Delta(us), LatencyCounted?\n");
    }

    // IDLE から状態遷移するまで少しまつ
    nn::os::Tick tick = nn::os::GetSystemTick();
    while(!m_Target->IsConnected())
    {
        if((nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds() >= 5000 || m_RequestStop)
        {
            break;
        }

        Sleep(nn::TimeSpan::FromMilliSeconds(100));
    }

    while(!m_RequestStop)
    {
        result = m_Target->WlanGetFrame(rxId, buffer, &args.Size, sizeof(buffer), &args.Rssi);
        if(result.IsFailure())
        {
            m_Statistics.ReceiveErrorCount++;
            Sleep(nn::TimeSpan::FromMicroSeconds(10));    // 切断時の連続受信失敗を避けるため、スリープする
        }
        else
        {
            if(m_Statistics.ReceiveCount == 0)
            {
                m_Statistics.FirstReceiveTime = nn::os::GetSystemTick();
            }
            m_Statistics.LastReceiveTime = nn::os::GetSystemTick();
            m_Statistics.ReceiveCount++;
            m_Statistics.ReceiveSize += args.Size;

            // パケットの中身を表示
            // for(int i=0; i<64; i++)
            // {
            //     NN_LOG("%02hx ", buffer[i]);
            //     if((i + 1)%8==0)
            //     {
            //         NN_LOG("\n");
            //     }
            // }

            args.Receiver = PROTOCOL_RECEIVER;
            // 受信イベントを発生させる
            if(m_Handler != NULL)
            {
                (*m_Handler)(&args);
            }
            else
            {
                //NN_LOG("ReceiveEvent does not exist.\n");
            }
        }
    }

    if( logger.IsEnabled() )
    {
        logger.Write("end\n");
    }

    m_IsStop = true;
}

nn::Result ProtocolReceiver::Start(int32_t priority)
{
    static NN_ALIGNAS(4096) char s_ProtoclReceiveThreadStack[4096 * 5];
    nn::Result result;

    if( !m_IsStop )
    {
        return WlanTest::ResultBusy();
    }
    m_IsStop = false;
    m_RequestStop = false;
    result = nn::os::CreateThread(&m_ReceiveThread, ProtocolReceiver::ReceiveThreadFunction, this, s_ProtoclReceiveThreadStack, sizeof(s_ProtoclReceiveThreadStack), priority);
    if( result.IsFailure() )
    {
        NN_LOG("ReceiveThread can not run.\n");
        return result;
    }

    nn::os::SetThreadNamePointer(&m_ReceiveThread, "WitProtoclRxThread");
    nn::os::StartThread(&m_ReceiveThread);

    return result;
}

nn::Result ProtocolReceiver::Stop()
{
    nn::Result result = nn::ResultSuccess();

    if( !m_IsStop )
    {
        m_RequestStop = true;

        int rxId = m_Target->GetRxIdForData();
        if( rxId > 0 )
        {
            m_Target->CancelGetFrame(rxId);
        }
        nn::os::DestroyThread(&m_ReceiveThread);
    }
    return result;
}

/*---------------------------------------------------------------------------
　　　　　InformationElementReceiver
---------------------------------------------------------------------------*/
InformationElementReceiver::InformationElementReceiver()
{
    m_TypesNum = 4;
    m_Types[0] = 0x15;
    m_Types[1] = 0x16;
    m_Types[2] = 0x17;
    m_Types[3] = 0x18;
    m_RequestStop = false;
    m_IsStop = true;
}

InformationElementReceiver::~InformationElementReceiver()
{
}

void InformationElementReceiver::ReceiveThreadFunction(void* receiver)
{
    DBG_VLOG("Thread start - InformationElementReceiveThread");
    reinterpret_cast<InformationElementReceiver*>(receiver)->Receive();
    DBG_VLOG("Thread stop - InformationElementReceiver");
}

void InformationElementReceiver::Receive()
{
    if(m_Target == NULL || m_TypesNum == 0)
    {
        NN_LOG("Can't receive because Target or SubType is not not set.");
        return;
    }

    nn::Result result;
    ReceiveEventArgs args;
    uint8_t buffer[1300];
    uint8_t buffer2[1024];
    nn::Bit8 oui[3] = {0x00, 0x1f, 0x32};
    args.Data = buffer2;
    nn::os::SystemEventType bssIndicationEvent;

    result = nn::wlan::Local::GetBssIndicationEvent(&bssIndicationEvent);
    if(result.IsFailure())
    {
        NN_LOG("GetBssIndicationEvent failed.\n");
        WlanTest::TPrintResult(result);
        goto End;
    }

    while(!m_RequestStop)
    {
        //bool signaled = e.Wait(nn::TimeSpan::FromMilliSeconds(100));
        //#?? not implemented
        //bool signaled = TimedWaitSystemEvent(&bssIndicationEvent, nn::TimeSpan::FromMilliSeconds(100));
        bool signaled = false;

        if(!signaled)
        {
            //#?? as signaled is always false for not implementing bssIndicationEvent
            Sleep(nn::TimeSpan::FromMilliSeconds(100));
            continue;
        }
        //result = Sap::GetBssIndicationInfo(buffer);
        result = nn::wlan::Local::GetBssIndicationInfo(buffer);
        if(result.IsFailure())
        {
            m_Statistics.ReceiveErrorCount++;
            Sleep(nn::TimeSpan::FromMilliSeconds(1));    // 切断時の連続受信失敗を避けるため、スリープする
            //            NN_LOG("GetBssIndicationInfo failed.\n");
            //            WlanTest::TPrintResult(result);
            continue;
        }

        //BeaconDescriptionReader beaconInfo(reinterpret_cast<BssDescription*>(buffer));
        nn::wlan::BeaconDescriptionReader beaconInfo(reinterpret_cast<nn::wlan::BssDescription*>(buffer));
        //const VendorInfoElementReader* ie[4];
        const nn::wlan::VendorInfoElementReader* ie[m_TypesNum];
        int maxIE;
        for(maxIE = 0; maxIE<m_TypesNum; maxIE++)
        {
            //ie[maxIE] = beaconInfo.GetVendorIe(oui, m_Types[maxIE]);
            ie[maxIE] = beaconInfo.GetVendorIeWithOuiAndType(oui, m_Types[maxIE]);
            if(ie[maxIE] == NULL)
            {
                break;
                // NN_LOG("ie[%d] = %d\n", maxIE, ie[maxIE]->GetLength());
            }
        }

        if(maxIE == 0)
        {
            continue;
        }

        std::memset(buffer2, 0, 1024);
        uint32_t offset = 0;
        for(int i=0; i<maxIE; i++)
        {
            if(ie[i] == NULL)
                break;
            std::memcpy(buffer2 + offset, ie[i]->GetVendorData() + 1, ie[i]->GetLength() - 3 - 1); // 3:ヘッダ分, 1:SubType分
            offset += ie[i]->GetLength() - 3 - 1;
        }

        args.Size = offset;
        args.Rssi = beaconInfo.GetRssi();

        if(m_Statistics.ReceiveCount == 0)
        {
            m_Statistics.FirstReceiveTime = nn::os::GetSystemTick();
        }
        m_Statistics.LastReceiveTime = nn::os::GetSystemTick();
        m_Statistics.ReceiveCount++;
        m_Statistics.ReceiveSize += args.Size;

        //        for(int i=0;i<16;i++)
        //        {
        //            NN_LOG("%02X : ", buffer2[i]);
        //            if(i%8 ==7)
        //                NN_LOG("\n");
        //        }

        args.Receiver = INFORMATION_ELEMENT_RECEIVER;

        // 受信イベントを発生させる
        if(m_Handler != NULL)
        {
            (*m_Handler)(&args);
        }
        else
        {
            //NN_LOG("ReceiveEvent does not exist.\n");
        }
    }
End:
    //e.Finalize();
    //#?? comment out because of no implementation of bssIndicationEvent
    //nn::os::DestroySystemEvent(&bssIndicationEvent);
    m_IsStop = true;
}

nn::Result InformationElementReceiver::Start(int32_t priority)
{
    static NN_ALIGNAS(4096) char s_IEReceiveThreadStack[4096];
    nn::Result result;

    if(!m_IsStop)
    {
        return WlanTest::ResultBusy();
    }
    m_IsStop = false;
    m_RequestStop = false;

    result = nn::os::CreateThread(&m_ReceiveThread, InformationElementReceiver::ReceiveThreadFunction, this, s_IEReceiveThreadStack, sizeof(s_IEReceiveThreadStack), priority);
    if( result.IsFailure() )
    {
        NN_LOG("ReceiveThread can not run.\n");
        return result;
    }

    nn::os::SetThreadNamePointer(&m_ReceiveThread, "WitIeRxThread");
    nn::os::StartThread(&m_ReceiveThread);

    return result;
}

nn::Result InformationElementReceiver::Stop()
{
    nn::Result result = nn::ResultSuccess();

    if(!m_IsStop)
    {
        m_RequestStop = true;
        //#??
        Sleep(nn::TimeSpan::FromMilliSeconds(1000));
        nn::os::DestroyThread(&m_ReceiveThread);
    }

    return result;
}

/*---------------------------------------------------------------------------
　　　　　CommandReceiver
---------------------------------------------------------------------------*/

CommandReceiver::CommandReceiver()
{
    m_RequestStop = false;
    m_IsStop = true;
}

void CommandReceiver::ReceiveThreadFunction(void* receiver)
{
    DBG_VLOG("Thread start - CommandReceiver::ReceiveThread");
    reinterpret_cast<CommandReceiver*>(receiver)->Receive();
    DBG_VLOG("Thread stop - CommandReceiver::ReceiveThread");
}

void CommandReceiver::Receive()
{
    if( m_Target == nullptr )
    {
        NN_LOG("  - failed : No target set\n");
        m_RequestStop = true;
        return;
    }

    // IDLE から状態遷移するまで少しまつ
    nn::os::Tick tick = nn::os::GetSystemTick();
    while( !m_Target->IsConnected() )
    {
        if((nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds() >= 5000 || m_RequestStop)
        {
            break;
        }

        Sleep(nn::TimeSpan::FromMilliSeconds(100));
    }

    size_t size = 0;
    uint32_t rxId = m_Target->GetRxIdForControl();
    nn::Result result;
    WitHeader witHeader;
    while( !m_RequestStop )
    {
        // 継承先の送信関数を使わず nn::wlan::Socket を用いる
        result = nn::wlan::Socket::GetFrameRaw(commandBuffer, sizeof(commandBuffer), &size, rxId);
        if( result.IsFailure() )
        {
            // 切断時の連続受信失敗を避けるため、スリープする
            Sleep(nn::TimeSpan::FromMilliSeconds(1));
        }
        else
        {
            // パケットの中身を表示
            // for(int i=0; i<40; i++)
            // {
            //     NN_LOG("%02hx ", buffer[i]);
            //     if((i + 1)%8==0)
            //     {
            //         NN_LOG("\n");
            //     }
            // }

            std::memcpy(&witHeader, commandBuffer + WlanHeaderSize, WitHeaderSize);
            m_Target->RunCommand(witHeader, commandBuffer + CommandHeaderSize, size - CommandHeaderSize);
        }
    }

    m_IsStop = true;
}

static NN_ALIGNAS(4096) char s_CommandReceiveThreadStack[4096];
nn::Result CommandReceiver::Start(int32_t priority)
{
    nn::Result result;

    if( !m_IsStop )
    {
        return WlanTest::ResultBusy();
    }

    m_IsStop = false;
    m_RequestStop = false;
    result = nn::os::CreateThread(&m_ReceiveThread, CommandReceiver::ReceiveThreadFunction, this, s_CommandReceiveThreadStack, sizeof(s_CommandReceiveThreadStack), priority);
    if( result.IsFailure() )
    {
        NN_LOG("ReceiveThread can not run.\n");
        return result;
    }

    nn::os::SetThreadNamePointer(&m_ReceiveThread, "WitCommandRxThread");
    nn::os::StartThread(&m_ReceiveThread);

    return result;
}

nn::Result CommandReceiver::Stop()
{
    nn::Result result = nn::ResultSuccess();

    if( !m_IsStop )
    {
        m_RequestStop = true;

        int rxId = m_Target->GetRxIdForControl();
        if( rxId > 0 )
        {
            m_Target->CancelGetFrame(rxId);
        }
        nn::os::DestroyThread(&m_ReceiveThread);
    }

    return result;
}


/*---------------------------------------------------------------------------
　　　　　ActionFrameReceiver
---------------------------------------------------------------------------*/

ActionFrameReceiver::ActionFrameReceiver()
{
    m_RequestStop = false;
    m_IsStop = true;

    m_Param.scanType = nn::wlan::ScanType_Passive;
    std::memset(m_Param.channelList, 0, sizeof(m_Param.channelList));
    m_Param.channelCount = 0;
    m_Param.channelScanTime = 110;
    m_Param.homeChannelTime = 0;
    m_Param.ssidList = nullptr;
    m_Param.ssidCount = 0;
    m_Param.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    m_BufferSize = 4096;
    m_pBuffer = new uint8_t[m_BufferSize];

    m_WaitTime = 100;
    m_RandomWaitTime = 0;
}

ActionFrameReceiver::~ActionFrameReceiver()
{
    delete[] m_pBuffer;
}

void ActionFrameReceiver::ReceiveThreadFunction(void* receiver)
{
    DBG_VLOG("Thread start - ActionFrameReceiver::ReceiveThread");
    reinterpret_cast<ActionFrameReceiver*>(receiver)->Receive();
    DBG_VLOG("Thread stop - ActionFrameReceiver::ReceiveThread");
}

void ActionFrameReceiver::Receive()
{
    if( m_Target == NULL )
    {
        NN_LOG("Can't receive because Target or ActionFrameType is not not set.");
        m_RequestStop = true;
        return;
    }

    nn::Result result;
    uint32_t rxId = m_Target->GetRxIdForAf();
    uint16_t channel = 0;
    int16_t rssi = 0;
    AfReceiveEventArgs args;
    args.Data = m_pBuffer;

    // IDLE から状態遷移するまで少しまつ
    nn::os::Tick tick = nn::os::GetSystemTick();
    while( !m_Target->IsAfRxReady() )
    {
        if((nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds() >= 5000 || m_RequestStop)
        {
            break;
        }

        Sleep(nn::TimeSpan::FromMilliSeconds(100));
    }

    nn::wlan::MacAddress macAddress;
    while( !m_RequestStop )
    {
        result = m_Target->WlanGetActionFrame(&macAddress, m_pBuffer, m_BufferSize, &args.Size, rxId, &channel, &rssi);
        if( result.IsFailure() )
        {
            m_Statistics.ReceiveErrorCount++;
            Sleep(nn::TimeSpan::FromMicroSeconds(10));    // 切断時の連続受信失敗を避けるため、スリープする
        }
        else
        {
            if(m_Statistics.ReceiveCount == 0)
            {
                m_Statistics.FirstReceiveTime = nn::os::GetSystemTick();
            }
            m_Statistics.LastReceiveTime = nn::os::GetSystemTick();
            m_Statistics.ReceiveCount++;
            m_Statistics.ReceiveSize += args.Size;

            // パケットの中身を表示
            // for(int i=0; i<64; i++)
            // {
            //     NN_LOG("%02hx ", buffer[i]);
            //     if((i + 1)%8==0)
            //     {
            //         NN_LOG("\n");
            //     }
            // }

            args.Source = macAddress;
            args.Receiver = ACTION_FRAME_RECEIVER;
            args.Channel = channel;
            args.Rssi = rssi;
            // 受信イベントを発生させる
            if(m_Handler != NULL)
            {
                (*m_Handler)(&args);
            }
            else
            {
                //NN_LOG("ReceiveEvent does not exist.\n");
            }
        }
    }

    m_IsStop = true;
}

nn::Result ActionFrameReceiver::Start(int32_t priority)
{
    static NN_ALIGNAS(4096) char s_ProtoclReceiveThreadStack[4096 * 5];
    nn::Result result;

    if( !m_IsStop )
    {
        return WlanTest::ResultBusy();
    }
    m_IsStop = false;
    m_RequestStop = false;
    result = nn::os::CreateThread(&m_ReceiveThread, ActionFrameReceiver::ReceiveThreadFunction, this, s_ProtoclReceiveThreadStack, sizeof(s_ProtoclReceiveThreadStack), priority);
    if( result.IsFailure() )
    {
        NN_LOG("ReceiveThread can not run.\n");
        return result;
    }

    nn::os::SetThreadNamePointer(&m_ReceiveThread, "WitProtoclRxThread");
    nn::os::StartThread(&m_ReceiveThread);

    return result;
}

nn::Result ActionFrameReceiver::Stop()
{
    nn::Result result = nn::ResultSuccess();

    if( !m_IsStop )
    {
        m_RequestStop = true;

        int rxId = m_Target->GetRxIdForAf();
        if( rxId > 0 )
        {
            m_Target->CancelGetActionFrame(rxId);
        }
        nn::os::DestroyThread(&m_ReceiveThread);
    }
    return result;
}


/*---------------------------------------------------------------------------
　　　　　ModeChanger
---------------------------------------------------------------------------*/

Mode ModeChanger::s_Mode = READY;

Mode ModeChanger::GetMode()
{
    return s_Mode;
}

nn::Result ModeChanger::ToReadyMode()
{
    nn::Result result = nn::ResultSuccess();

    switch( GetMode() )
    {
    case READY           : result = nn::ResultSuccess(); break;
    case INFRA           : result = nn::wlan::Infra::CloseMode(); break;
    case LOCAL_MASTER    : result = nn::wlan::Local::CloseMasterMode(); break;
    case LOCAL_CLIENT    : result = nn::wlan::Local::CloseClientMode(); break;
    case LOCAL_SPECTATOR : result = nn::wlan::Local::CloseSpectatorMode(); break;
    case LCS_MASTER      : result = nn::wlan::Local::CloseLcsMasterMode(); break;
    case LCS_CLIENT      : result = nn::wlan::Local::CloseLcsClientMode(); break;
    case DETECTOR        : result = nn::wlan::Detect::CloseMode(); break;
    default: NN_ASSERT(false, "ToReadyMode : %d", s_Mode); break;
    }

    s_Mode = READY;

    return result;
}

nn::Result ModeChanger::ToLocalMasterMode()
{
    if( GetMode() == LOCAL_MASTER )
    {
        return nn::ResultSuccess();
    }

    nn::Result result = ToReadyMode();
    if( result.IsSuccess() )
    {
        result = nn::wlan::Local::OpenMasterMode();
        if( result.IsSuccess() )
        {
            s_Mode = LOCAL_MASTER;
        }
    }

    return result;
}

nn::Result ModeChanger::ToLocalClientMode()
{
    if( GetMode() == LOCAL_CLIENT )
    {
        return nn::ResultSuccess();
    }

    nn::Result result = ToReadyMode();
    if( result.IsSuccess() )
    {
        result = nn::wlan::Local::OpenClientMode();
        if( result.IsSuccess() )
        {
            s_Mode = LOCAL_CLIENT;
        }
    }

    return result;
}

nn::Result ModeChanger::ToLocalSpectatorMode()
{
    if( GetMode() == LOCAL_SPECTATOR )
    {
        return nn::ResultSuccess();
    }

    nn::Result result = ToReadyMode();
    if( result.IsSuccess() )
    {
        result = nn::wlan::Local::OpenSpectatorMode();
        if( result.IsSuccess() )
        {
            s_Mode = LOCAL_SPECTATOR;
        }
    }

    return result;
}

nn::Result ModeChanger::ToLcsMasterMode()
{
    if( GetMode() == LCS_MASTER )
    {
        return nn::ResultSuccess();
    }

    nn::Result result = ToReadyMode();
    if( result.IsSuccess() )
    {
        result = nn::wlan::Local::OpenLcsMasterMode();
        if( result.IsSuccess() )
        {
            s_Mode = LCS_MASTER;
        }
    }

    return result;
}

nn::Result ModeChanger::ToLcsClientMode()
{
    if( GetMode() == LCS_CLIENT )
    {
        return nn::ResultSuccess();
    }

    nn::Result result = ToReadyMode();
    if( result.IsSuccess() )
    {
        result = nn::wlan::Local::OpenLcsClientMode();
        if( result.IsSuccess() )
        {
            s_Mode = LCS_CLIENT;
        }
    }

    return result;
}

nn::Result ModeChanger::ToInfraMode()
{
    if( GetMode() == INFRA )
    {
        return nn::ResultSuccess();
    }

    nn::Result result = ToReadyMode();
    if( result.IsSuccess() )
    {
        result = nn::wlan::Infra::OpenMode();
        if( result.IsSuccess() )
        {
            s_Mode = INFRA;
        }
    }

    return result;
}

nn::Result ModeChanger::ToDetectorMode(uint16_t channel)
{
    if( GetMode() == DETECTOR )
    {
        return nn::ResultSuccess();
    }

    nn::Result result = ToReadyMode();
    if( result.IsSuccess() )
    {
        result = nn::wlan::Detect::OpenMode(channel);
        if( result.IsSuccess() )
        {
            s_Mode = DETECTOR;
        }
    }

    return result;
}


/*---------------------------------------------------------------------------
　　　　　Node
---------------------------------------------------------------------------*/
Node::Node()
{
    m_RequestStop = false;
    m_IsStop = true;
    m_IsInitialized = false;
    m_EnableEcho = false;
    m_Receiver = new ProtocolReceiver();
    m_ReceiverHandler = new ReceiveEventHandler(Node)(*this, &Node::OnReceive);
    m_CommandReceiver = new CommandReceiver();
    m_AfReceiver = new ActionFrameReceiver();
    m_AfReceiverHandler = new ReceiveEventHandler(Node)(*this, &Node::OnAfReceive);
    m_Handler = nullptr;
    m_AfHandler = nullptr;
    m_SocketDataRxId = 0;
    m_SocketControlRxId = 0;
    m_LocalRxId = 0;
    m_AfRxId = 0;
    m_IsAfThreadEnabled = false;
    m_ConnectionTime = 0;
    GetMacAddress();
    m_Statistics.Clear();
}

Node::~Node()
{
    delete m_CommandReceiver;
    delete m_Receiver;
    delete m_ReceiverHandler;
    delete m_AfReceiver;
    delete m_AfReceiverHandler;
}

void Node::MaintainConnection()
{
}

//void Node::MaintainConnectionThreadFunction(Node* node)
void Node::MaintainConnectionThreadFunction(void* node)
{
    DBG_VLOG("Thread start - Node::MaintainConnectionThread");

#ifdef OPENCLOSEMODE_WHEN_OPENCLOSE
    node->Initialize();
#endif

    reinterpret_cast<Node*>(node)->MaintainConnection();

#ifdef OPENCLOSEMODE_WHEN_OPENCLOSE
    node->Finalize();
#endif

    DBG_VLOG("Thread stop - Node::MaintainConnectionThread");
}


nn::Result Node::StartMaintainConnection(int32_t priority)
{
    static NN_ALIGNAS(4096) char s_MaintainConnectionThreadStack[4096 * 5];
    nn::Result result;

    if(!m_IsInitialized)
    {
        return WlanTest::ResultNotInitialized();
    }

    if(!m_IsStop)
    {
        //return nn::Result(nn::Result::LEVEL_PERMANENT, nn::Result::SUMMARY_INVALID_STATE, nn::Result::MODULE_COMMON, nn::Result::DESCRIPTION_BUSY);
        return WlanTest::ResultBusy();
    }

    m_IsStop = false;

    result = nn::os::CreateThread(&m_MaintainConnectionThread, MaintainConnectionThreadFunction, this, s_MaintainConnectionThreadStack, sizeof(s_MaintainConnectionThreadStack), priority);

    nn::os::SetThreadNamePointer(&m_MaintainConnectionThread, "WitMaintainConnectionThread");
    nn::os::StartThread(&m_MaintainConnectionThread);
    if( result.IsFailure() )
    {
        NN_LOG("Can not run MaintainConnectionThread.\n");
        return result;
    }

    //m_MaintainConnectionThread.Detach();
    //m_MaintainConnectionThread.Finalize();

    return result;
}

void Node::StopMaintainConnection()
{
    StopReceiveCommand();
    StopReceiveData();
    //#?? Detach が使えないためここで破棄する
    nn::os::DestroyThread(&m_MaintainConnectionThread);
}

nn::Result Node::StartReceiveData(int32_t priority)
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    m_Receiver->SetTarget(*this);
    m_Receiver->SetHandler(*m_ReceiverHandler);

    nn::Result result;
    result = m_Receiver->Start(priority);
    if( result.IsFailure() )
    {
        NN_LOG("  - failed : cannot start StartReceiveData\n");
        NN_ASSERT(false);
    }

    return nn::ResultSuccess();
}

nn::Result Node::StopReceiveData()
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    nn::Result result;
    result = m_Receiver->Stop();
    if( result.IsFailure() )
    {
        NN_LOG("  - failed : cannot stop StartReceiveData\n");
        NN_ASSERT(false);
    }

    return nn::ResultSuccess();
}

nn::Result Node::StartReceiveCommand(int32_t priority)
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    m_CommandReceiver->SetTarget(*this);

    nn::Result result;
    result = m_CommandReceiver->Start(priority);
    if( result.IsFailure() )
    {
        NN_LOG("  - failed : cannot start StartReceiveCommand\n");
        NN_ASSERT(false);
    }

    return nn::ResultSuccess();
}

nn::Result Node::StopReceiveCommand()
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    nn::Result result;
    result = m_CommandReceiver->Stop();
    if( result.IsFailure() )
    {
        NN_LOG("  - failed : cannot stop StartReceiveCommand\n");
        NN_ASSERT(false);
    }

    return nn::ResultSuccess();
}

nn::Result Node::StartReceiveActionFrame(int32_t priority)
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    if( !m_IsAfThreadEnabled )
    {
        return nn::ResultSuccess();
    }

    m_AfReceiver->SetTarget(*this);
    m_AfReceiver->SetHandler(*m_AfReceiverHandler);

    nn::Result result;
    result = m_AfReceiver->Start(priority);
    if( result.IsFailure() )
    {
        NN_LOG("  - failed : cannot start StartReceiveData\n");
        NN_ASSERT(false);
    }

    return nn::ResultSuccess();
}

nn::Result Node::StopReceiveActionFrame()
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    nn::Result result;
    result = m_AfReceiver->Stop();

    if( result.IsFailure() )
    {
        NN_LOG("  - failed : cannot stop StartReceiveData\n");
        NN_ASSERT(false);
    }

    return nn::ResultSuccess();
}


nn::Result Node::Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize)
{
    nn::Result result;

    if(!m_IsInitialized)
    {
        return WlanTest::ResultNotInitialized();
    }

    //std::memcpy(((IEEE_802_3_LLC_SNAP*)data)->Source, GetMacAddress().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
    std::memcpy(((DIX_FRAME*)data)->Source, GetMacAddress().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

    // パケットの中身を表示する
    // for(int i=0; i<30; i++)
    // {
    //     NN_LOG("%02x ", data[i]);
    //     if( i % 8 == 7)
    //         NN_LOG("\n");
    // }
    // NN_LOG("\n");

    result = WlanPutFrame(data, size);
    if (result.IsFailure())
    {
        m_Statistics.SendErrorCount++;
        Sleep(nn::TimeSpan::FromMilliSeconds(1));  // 切断時の連続送信失敗を避けるため、スリープする
    }
    else
    {
        if(m_Statistics.SendCount == 0)
        {
            //m_Statistics.FirstSendTime = nn::os::Tick::GetSystemCurrent();
            m_Statistics.FirstSendTime = nn::os::GetSystemTick();
        }
        //m_Statistics.LastSendTime = nn::os::Tick::GetSystemCurrent();
        m_Statistics.LastSendTime = nn::os::GetSystemTick();
        m_Statistics.SendCount++;
        m_Statistics.SendSize += size;
        *pSentSize = size;
    }

    return result;
}

void Node::OnReceive(const EventArgs* pArgs)
{
    if( !m_IsInitialized )
    {
        NN_LOG(" - failed : not initialized...\n");
        return;
    }

    const ReceiveEventArgs& args = *reinterpret_cast<const ReceiveEventArgs*>(pArgs);

    m_Statistics.ReceiveCount      = m_Receiver->GetStatistics().ReceiveCount;
    m_Statistics.ReceiveSize       = m_Receiver->GetStatistics().ReceiveSize;
    m_Statistics.ReceiveErrorCount = m_Receiver->GetStatistics().ReceiveErrorCount;
    m_Statistics.FirstReceiveTime  = m_Receiver->GetStatistics().FirstReceiveTime;
    m_Statistics.LastReceiveTime   = m_Receiver->GetStatistics().LastReceiveTime;

    if( m_Handler != nullptr )
    {
        (*m_Handler)(pArgs);
    }

    if( m_EnableEcho )
    {
        // nn::wlan::MacAddress address(((IEEE_802_3_LLC_SNAP*)args.Data)->Source);
        // std::memcpy(((IEEE_802_3_LLC_SNAP*)args.Data)->Destination, address.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        nn::wlan::MacAddress address(((DIX_FRAME*)args.Data)->Source);
        std::memcpy(((DIX_FRAME*)args.Data)->Destination, address.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        size_t sentSize = 0;
        Send(args.Data, args.Size, 0, &sentSize);
    }
}

void Node::OnAfReceive(const EventArgs* pArgs)
{
    if( !m_IsInitialized )
    {
        NN_LOG(" - failed : not initialized...\n");
        return;
    }

    // m_Statistics.ReceiveCount      = m_Receiver->GetStatistics().ReceiveCount;
    // m_Statistics.ReceiveSize       = m_Receiver->GetStatistics().ReceiveSize;
    // m_Statistics.ReceiveErrorCount = m_Receiver->GetStatistics().ReceiveErrorCount;
    // m_Statistics.FirstReceiveTime  = m_Receiver->GetStatistics().FirstReceiveTime;
    // m_Statistics.LastReceiveTime   = m_Receiver->GetStatistics().LastReceiveTime;

    if( m_AfHandler != nullptr )
    {
        (*m_AfHandler)(pArgs);
    }
}

nn::Result Node::CancelGetFrame(uint32_t rxId)
{
    return WlanCancelGetFrame(rxId);
}

nn::Result Node::CancelGetActionFrame(uint32_t rxId)
{
    return WlanCancelGetActionFrame(rxId);
}

nn::Result Node::RunCommand(const WitHeader& witHeader, const uint8_t* data, const size_t& size)
{
    bool isValidCommand = false;
    nn::Result result = nn::ResultSuccess();

    switch( witHeader.command )
    {
    case WitCommand_AssignSrcIpAddress :
        {
            if( 7 <= size && size <= 17 )
            {
                string ipAddress(reinterpret_cast<const char*>(data));
                result = RunAssignSrcIpAddressCommand(ipAddress);
                isValidCommand = true;
            }
        }
        break;

    default :
        {
            isValidCommand = false;
        }
        break;
    }

    if( !isValidCommand )
    {
        NN_LOG("  - failed : Received invalid command [id : %04x, size : %d]\n", witHeader.command, size);
        // NN_LOG("-----\n");
        // for(int i=0; i<32 && i*8<size; ++i)
        // {
        //     NN_LOG("\n  ");
        //     for(int j=0; j<8 && i*8+j<size; ++j)
        //     {
        //         NN_LOG(" %02x", data[i*8 + j]);
        //     }
        // }
        // NN_LOG("-----\n");
    }


    return result;
}

nn::Result Node::SendCommand(const nn::wlan::MacAddress& dst, const WitCommand& command, const uint8_t* data, const uint32_t size)
{
    static uint32_t sequence = 0;
    WitHeader witHeader;
    GenerateWitHeader(&witHeader, command, size);

    DixFrame frame;
    std::memcpy(((DIX_FRAME*)&frame)->Destination, dst.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
    std::memcpy(((DIX_FRAME*)&frame)->Source, GetMacAddress().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
    frame.SetType(PID_LOCAL_EXP1);

    uint32_t offset = frame.GetHeaderSize();

    std::memcpy(((DIX_FRAME*)&frame)->Payload + offset, reinterpret_cast<uint8_t*>(&witHeader), sizeof(WitHeader));
    offset += sizeof(WitHeader);

    std::memcpy(((DIX_FRAME*)&frame)->Payload + offset, data, size);
    offset += size;

    return nn::wlan::Socket::PutFrameRaw(reinterpret_cast<uint8_t*>(&frame), 14 + offset);
}

/*---------------------------------------------------------------------------
　　　　　WlanModuleInfo
---------------------------------------------------------------------------*/
// WlanModuleInfo::WlanModuleInfo()
// {
//     m_Console.Parent = this;
//     ostringstream oss;

//     HardwareInfo info;
//     Ext::GetHardwareInfo(info);
//     oss << "=====Wlan Module Info=====" << endl;
//     oss << hex << showbase;
//     oss << "CHIP ID    : " << info.chipId << endl;
//     oss << "ROM ID     : " << info.romId << endl;
//     oss << "HOST VER   : " << info.hostVersion << endl;
//     oss << "FW Ver     : " << info.firmwareVersion << endl;
//     oss << dec << noshowbase;
//     oss << "DPATCH VER : " << info.datapatchVersion << endl;
//     oss << "EEP FW VER : " << info.eepromFirmVersion << endl;
//     oss << hex << showbase;
//     oss << "REGDOMAIN  : " << info.regulatoryDomain << endl;
//     m_Console.Write(oss.str());
// }

// void WlanModuleInfoShowImpl(WlanTest::Display& display)
// {
//     m_Console.ShowImpl(display);
// }

}
