﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include "UIControl.h"
#include "Selector.h"
#include "Master.h"
#include "Client.h"
#include "DataGenerator.h"
#include "SocketNode.h"
#include "LcsSocket.h"
#include "Detector.h"
#include "Rssi.h"

namespace WlanTest {

static uint8_t MAINTAIN_CONNECTION_THREAD_PRIORITY = 14;
static uint8_t RECORD_RSSI_THREAD_PRIORITY = 15;
static uint8_t RECEIVE_DATA_THREAD_PRIORITY = 0;
static uint8_t RECEIVE_COMMAND_THREAD_PRIORITY = 14;
static uint8_t RECEIVE_ACTION_FRAME_THREAD_PRIORITY = 5;
static uint8_t SEARCH_THREAD_PRIORITY = 5;

template<typename T>
void AddSelectorElement(SelectableValue<T>* sv, const T& value, const T& initialValue)
{
    sv->Add(value, value == initialValue);
}

template<typename T>
void AddSelectorElement(SelectableValue<NamedValue<T>>* sv, const string& text, const T& value, const T& initialValue)
{
    sv->Add(NamedValue<T>(text, value), value == initialValue);
}

void AddSelectorElement(SelectableValue<NamedValue<nn::TimeSpan> >* sv, const nn::TimeSpan& value, const nn::TimeSpan& initialValue);

void GenerateSelectableOnOffValue(SelectableValue<NamedValue<bool> >* sv, const bool& initialValue = false);
void GenerateSelectableChannel(SelectableValue<NamedValue<int16_t> >* sv, const int16_t& initialValue = 1);
void GenerateSelectableSsid(SelectableValue<string>* sv, const string& initialValue = "wireless_test_1");

template<typename T>
class ModelController : public Page//: public IModelController
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
    T* target;
private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    ModelController(){ target = nullptr; }
    virtual ~ModelController(){}

    void SetTarget(T& target)
    {
        this->target = &target;
    }
};

template<typename T>
class SceneController : public ModelController<T>
{
public:
protected:

    map<Button, string> padTexts;
    Label padLabel;

private:
public:

    SceneController()
    {
        SetPadText(Button::START, "Home");
        SetPadText(Button::A, "Set Param");
        ModelController<T>::Add(padLabel);
        padLabel.X = DISPLAY_PAD_START_X;
        padLabel.Y = DISPLAY_PAD_START_Y;
        padLabel.Text = "";
        padLabel.Width = 2000;
    }

    virtual ~SceneController()
    {}

    virtual void ShowImpl(Display& display)
    {
        string text;

        map<Button, string>::iterator it = padTexts.begin();
        while( it != padTexts.end() )
        {
            text += "<" + ToString(it->first) + "> " + it->second + PAD_TEXT_SEPARATOR;
            ++it;
        }
        padLabel.Text = text;

        ModelController<T>::ShowImpl(display);
    }

    void SetPadText(const Button& button, const string& text)
    {
        padTexts[button] = text;
    }

    void ErasePadText(const Button& button)
    {
        padTexts.erase(button);
    }

};

class DataGeneratorController : public SceneController<DataGenerator>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    Label title;
    SelectableValue<NamedValue<nn::wlan::MacAddress> > destination;
    SelectableValue<NamedValue<uint16_t> > type;
    SelectableValue<size_t>                size;
    SelectableValue<uint32_t>              interval;
    SelectableValue<NamedValue<uint32_t> > m_Num;
    SelectableValue<NamedValue<bool> >     m_Clear;

    ValueSelector selector;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Set()
    {
        target->SetDestination( destination );
        target->SetType( type );
        target->SetPayloadSize(size);
        target->SetInterval(nn::TimeSpan::FromMilliSeconds(interval));

        if(m_Clear)
        {
            target->ClearCount();
        }

        //IEデータかデータかの識別子の設定
        if(destination.ToStringSelectingItem() == "ON NintendoIE")
        {
            target->SetIeIndicator(1);
        }
        else
        {
            target->SetIeIndicator(0);
        }
    }

    void Set(const DataGeneratorParam& param)
    {
        size.Add(param.payloadSize, true);
        interval.Add(param.sendInterval, true);

        m_Num.Add(NamedValue<uint32_t>("-user defined-", param.numOfPacket), true);
        type.Add(NamedValue<uint16_t>("-user defined-", param.type), true);
        Set();
    }

    void SetDestination(const nn::wlan::MacAddress& dest)
    {
        target->SetDestination( dest );
    }

    void SetType(const uint16_t& type)
    {
        target->SetType(type);
    }

    void SetPayloadSize(const size_t& size)
    {
        target->SetPayloadSize(size);
    }

    void SetInterval(const nn::TimeSpan& interval)
    {
        target->SetInterval(interval);
    }

    void SetTxCount(const uint32_t& count)
    {
        int index = m_Num.FindInSelectableItems(count);
        if( index == -1 )
        {
            m_Num.Add(NamedValue<uint32_t>("-user defined-", count), true);
        }
        else
        {
            m_Num.Select(index);
        }
    }

    void SetIndicator(const string& str)
    {
        uint8_t indicator = 0;
        if( destination.ToStringSelectingItem() == "ON NintendoIE" )
        {
            indicator = 1;
        }
        else
        {
            indicator = 0;
        }
        target->SetIeIndicator(indicator);
    }

    void ClearCount()
    {
        target->ClearCount();
    }

    void SelectType(uint16_t t)
    {
        size_t size = type.Size();

        for(int i=0; i<size; ++i)
        {
            type.Next(true);
            if( type.GetValue().Value == t )
            {
                return;
            }
        }
    }

    virtual void InputPad(Pad& pad)
    {
    }

    void RenewDestinations()
    {
        destination.Clear();
        destination.Add(NamedValue<nn::wlan::MacAddress>("<Input>", nn::wlan::MacAddress::CreateZeroMacAddress()));
        destination.Add(NamedValue<nn::wlan::MacAddress>("Broadcast", nn::wlan::MacAddress::CreateBroadcastMacAddress()), true);
        destination.Add(NamedValue<nn::wlan::MacAddress>("ON NintendoIE", nn::wlan::MacAddress::CreateZeroMacAddress()));

        nn::wlan::MacAddress list[15];
        uint32_t num = 15;
        target->GetCandidateDestinations(list, num);

#if NN_VERSION_REVISION>=19894
        char8 str[nn::wlan::MacAddress::MAC_STRING_SIZE];
        for( uint32_t i=0; i<num; i++ )
        {
            destination.Add(NamedValue<nn::wlan::MacAddress>(list[i].GetString(str), list[i]));
        }
#endif
    }

    virtual void Start()
    {
        if(target->IsIdle())
        {
            target->Generate(m_Num);
        }
    }

    virtual void Stop()
    {
        if(!target->IsIdle())
        {
            target->Cancel();
        }
    }

    virtual bool IsIdle()
    {
        return target->IsIdle();
    }
protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    DataGeneratorController()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Tx Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = DISPLAY_CONTENT_HEIGHT;

        // selector.Register("Destination", &destination);
        destination.Add(NamedValue<nn::wlan::MacAddress>("Broadcast", nn::wlan::MacAddress::CreateBroadcastMacAddress()));
        destination.Add(NamedValue<nn::wlan::MacAddress>("40:d2:8a:b4:93:7e", nn::wlan::MacAddress(0x40, 0xd2, 0x8a, 0xb4, 0x93, 0x7e)));
        destination.Add(NamedValue<nn::wlan::MacAddress>("40:d2:8a:b4:94:43", nn::wlan::MacAddress(0x40, 0xd2, 0x8a, 0xb4, 0x94, 0x43)));

        // selector.Register("Type", &type);
        type.Add(NamedValue<uint16_t>("0x88B7(OUI Extended Ethertype)", PID_OUI_EXT));
        type.Add(NamedValue<uint16_t>("0xFFFF(No header, for socket)", PID_PAYLOAD_ONLY));

        selector.Register("Payload Size [bytes]", &size);
        size.Add(0);
        size.Add(40);
        for(int i=1; i<=15; ++i)
        {
            uint16_t value = i * 100;
            size.Add(value, value == 500);
        }

        selector.Register("Send Interval [msec]", &interval);
        for(int i=0; i<=10; ++i)
        {
            interval.Add(i);
        }
        interval.Add(16, true);
        interval.Add(33);
        interval.Add(67);
        for(int i=1; i<=10; ++i)
        {
            interval.Add(i * 100);
        }

        selector.Register("Num of Frames", &m_Num);
        m_Num.Add(NamedValue<uint32_t>("1", 1));
        m_Num.Add(NamedValue<uint32_t>("10", 10));
        m_Num.Add(NamedValue<uint32_t>("100", 100));
        m_Num.Add(NamedValue<uint32_t>("1000", 1000));
        m_Num.Add(NamedValue<uint32_t>("5000", 5000));
        m_Num.Add(NamedValue<uint32_t>("10000", 10000));
        m_Num.Add(NamedValue<uint32_t>("30000", 30000));
        m_Num.Add(NamedValue<uint32_t>("50000", 50000));
        m_Num.Add(NamedValue<uint32_t>("Infinity", 0), true);

        selector.Register("Reset SequenceNum", &m_Clear);
        m_Clear.Add(NamedValue<bool>("Yes", true));
        m_Clear.Add(NamedValue<bool>("No", false), true);
    };
private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
};

class FixedSizeDataTransmitterController : public SceneController<FixedSizeDataTransmitter>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    Label title;
    SelectableValue<NamedValue<uint64_t>>    dataSize;
    SelectableValue<NamedValue<uint64_t>>    sendSize;
    SelectableValue<uint32_t>              interval;
    bool cancel;

    ValueSelector selector;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Set()
    {
        cancel = false;

        target->ClearCount();
        target->SetDataSize(dataSize);
        target->SetPayloadSize(sendSize);
        target->SetInterval(nn::TimeSpan::FromMilliSeconds(interval));
    }

    void Set(const DataGeneratorParam& param)
    {
        NN_ASSERT(false);
    }

    virtual void InputPad(Pad& pad)
    {
    }

    virtual void Start()
    {
        if(target->IsIdle())
        {
            target->Generate(0);
        }
    }

    virtual void Stop()
    {
        if(!target->IsIdle())
        {
            target->Cancel();
        }
    }

    virtual bool IsIdle()
    {
        return target->IsIdle();
    }
protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    FixedSizeDataTransmitterController()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Tx Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = DISPLAY_CONTENT_HEIGHT;

        selector.Register("Total Data Size", &dataSize);
        const size_t kb = 1 * 1024;
        const size_t mb = 1 * 1024 * kb;
        const size_t gb = 1 * 1024 * mb;
        for(int i=1; i<10; ++i)
        {
            uint64_t size = i * 100 * mb;
            string sizeName = ToStringForBinaryPrefix(size, " ", 0) + "B";
            dataSize.Add(NamedValue<uint64_t>(sizeName, size), size == 100 * mb);
        }

        for(int i=1; i<=10; ++i)
        {
            uint64_t size = i * gb;
            string sizeName = ToStringForBinaryPrefix(size, " ", 0) + "B";
            dataSize.Add(NamedValue<uint64_t>(sizeName, size));
        }

        // DataGenerator では 20MB まで一度に送信可能
        uint32_t size = 1 * kb;
        selector.Register("Data Size for Send()", &sendSize);
        for(int i=0; i<15; ++i)
        {
            string sizeName = ToStringForBinaryPrefix(size, " ", 0) + "B";
            sendSize.Add(NamedValue<uint64_t>(sizeName, size), size == 32 * kb);
            size *= 2;
        }

        selector.Register("Send Interval [msec]", &interval);
        for(int i=0; i<=20; ++i)
        {
            interval.Add(i, i == 0);
        }
        interval.Add(33);
        interval.Add(67);
        for(int i=1; i<=10; ++i)
        {
            interval.Add(i * 100);
        }

        cancel = false;
    };
private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
};

class MasterController : public SceneController<Master>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    ValueSelector selector;

private:

    Label title;
    SelectableValue<NamedValue<uint32_t> > numOfClients;
    SelectableValue<NamedValue<int16_t> > channel;
    SelectableValue<NamedValue<bool> > hiddenSsid;
    SelectableValue<NamedValue<bool> > defaultAuthorize;
    SelectableValue<int> inactivePeriod;
    SelectableValue<NamedValue<uint32_t> > actionFrame;
    SelectableValue<NamedValue<bool> > actionFrameThread;
    SelectableValue<NamedValue<nn::wlan::RateSetLegacy> > supportedRates;
    SelectableValue<NamedValue<nn::wlan::RateSetLegacy> > basicRates;
    SelectableValue<string> ssid;
    SelectableValue<int> beaconIntervalTu;
    SelectableValue<NamedValue<nn::wlan::SecurityMode> > securityMode;
    SelectableValue<string> key;
    SelectableValue<int> m_CwMin;
    SelectableValue<int> m_CwMax;
    SelectableValue<int> m_Aifsn;
    SelectableValue<int> m_Interval;
    SelectableValue<NamedValue<bool> > m_Echo;
    SelectableValue<NamedValue<bool> > m_RespondWildcard;
    SelectableValue<Oui> m_IeOui;
    SelectableValue<int> m_NumOfDeauth;
    SelectableValue<NamedValue<bool> > m_EnablePs;
    SelectableValue<int> m_Ctw;

    bool m_Active;

    DataGeneratorController* m_DataGeneratorController;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/

public:
    virtual void Set()
    {
        target->SetConnectableCount( numOfClients );

        target->GetMasterBssParameters().channel = channel;
        target->GetMasterBssParameters().hiddenSsid = hiddenSsid;
        target->GetMasterBssParameters().inactivePeriod = (uint32_t)inactivePeriod;
        target->GetMasterBssParameters().supportedRates = supportedRates;
        target->GetMasterBssParameters().basicRates = basicRates;

        target->GetMasterBssParameters().beaconInterval = (uint16_t)beaconIntervalTu;
        target->GetMasterBssParameters().security.privacyMode = securityMode;
        target->GetMasterBssParameters().security.groupPrivacyMode = securityMode;

        std::memset(target->GetMasterBssParameters().security.key, 0x00, 64);
        std::memcpy(target->GetMasterBssParameters().security.key, key.GetValue().c_str(), key.GetValue().size());

        if(ssid.GetValue() == "Auto")
        {
            ostringstream id;
            id << "wireless_test_";
            id << channel;
            id << "ch";
            target->GetMasterBssParameters().ssid = nn::wlan::Ssid(id.str().c_str());
        }
        else
        {
            target->GetMasterBssParameters().ssid = nn::wlan::Ssid(ssid.GetValue().c_str());
        }

        target->GetMasterBssParameters().beaconInterval = static_cast<uint16_t>(beaconIntervalTu);
        target->SetNumOfDeauth(m_NumOfDeauth);
        target->SetActionFrameSize(actionFrame);
        target->SetIsAfThreadEnabled(actionFrameThread);

#if 0
        target->SetResponseParam(m_RespondWildcard, m_IeOui.GetValue().GetRawData());
        target->SetEnableEcho(m_Echo);
        target->SetEnablePs( m_EnablePs );
        target->SetCtw( m_Ctw );
#endif
    }

    void Set(MasterParam param)
    {
        NamedValue<int16_t> ch("- user defined -", param.channel);
        channel.Add(ch, true);

        ssid.Add(param.ssid, true);

        NamedValue<nn::wlan::SecurityMode> mode("- user defined -", param.security.privacyMode);
        securityMode.Add(mode, true);

        key.Add(reinterpret_cast<char*>(param.security.key), true);

        NamedValue<bool> on_off("- user defined -", param.hiddenSsid);
        hiddenSsid.Add(on_off, true);

        inactivePeriod.Add(param.inactivePeriod, true);

        Set();
    }

    virtual void InputPad(Pad& pad)
    {
    }

    virtual void Start()
    {
        if(!m_Active)
        {
            Set();

            nn::Result result;
            result = target->StartMaintainConnection(MAINTAIN_CONNECTION_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                return;
            }

            result = target->StartReceiveActionFrame(RECEIVE_ACTION_FRAME_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                target->StopMaintainConnection();
                return;
            }

            result = target->StartReceiveData(RECEIVE_DATA_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                target->StopReceiveActionFrame();
                target->StopMaintainConnection();
                return;
            }

            result = target->StartReceiveCommand(RECEIVE_COMMAND_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                target->StopReceiveData();
                target->StopReceiveActionFrame();
                target->StopMaintainConnection();
                return;
            }

            m_Active = true;
        }
    }

    void Stop()
    {
        if(m_Active)
        {
            m_Active = false;
            if(!target->IsStop())
            {
                target->StopReceiveCommand();
                target->StopReceiveData();
                target->StopReceiveActionFrame();
                target->StopMaintainConnection();
            }
        }
    }
protected:

private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MasterController()
    {
        Initialize();
    }

    virtual void Initialize()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        m_Active = false;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Master Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = DISPLAY_CONTENT_HEIGHT;

        NamedValue<bool> off ("Off", false);
        NamedValue<bool> on ("On", true);

        selector.Register("Num of Clients", &numOfClients);
        numOfClients.Add(NamedValue<uint32_t>("Any", ConnectableClientCount_Any), true);
        for(int i=1; i<=nn::wlan::ConnectableClientsCountMax; ++i)
        {
            ostringstream oss;
            oss << i;
            numOfClients.Add(NamedValue<uint32_t>(oss.str(), i));
        }

        NamedValue<int16_t> chAuto("Auto", -1);
        selector.Register("Channel", &channel);
        channel.Add(chAuto, 1);
        GenerateSelectableChannel(&channel);

        selector.Register("SSID", &ssid);
        ssid.Add("Auto");
        GenerateSelectableSsid(&ssid, "wireless_test_1");

        NamedValue<nn::wlan::SecurityMode> open("OPEN", nn::wlan::SecurityMode_Open);
        NamedValue<nn::wlan::SecurityMode> staticAes("StaticAES", nn::wlan::SecurityMode_StaticAes);
        selector.Register("Security Mode", &securityMode);
        securityMode.Add(open, true);
        securityMode.Add(staticAes);

        selector.Register("Key", &key);
        key.Add("0123456789abcdef", true);
        key.Add("aaaaaaaaaaaaaaaa");

        selector.Register("Stealth Mode", &hiddenSsid);
        hiddenSsid.Add(off, true);
        hiddenSsid.Add(on);

        //selector.Register("Beacon Interval [TU] (Not Implemented)", &beaconIntervalTu);
        beaconIntervalTu.Add(30);
        beaconIntervalTu.Add(50);
        beaconIntervalTu.Add(100, true);
        beaconIntervalTu.Add(200);
        beaconIntervalTu.Add(500);
        beaconIntervalTu.Add(1000);

        NamedValue<nn::wlan::RateSetLegacy> def("Default", nn::wlan::RateSetLegacy_Default);
        NamedValue<nn::wlan::RateSetLegacy> rate1("1.0Mbps", nn::wlan::RateSetLegacy_1_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate2("2.0Mbps", nn::wlan::RateSetLegacy_2_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate5("5.5Mbps", nn::wlan::RateSetLegacy_5_5m);
        NamedValue<nn::wlan::RateSetLegacy> rate6("6.0Mbps", nn::wlan::RateSetLegacy_6_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate9("9.0Mbps", nn::wlan::RateSetLegacy_9_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate11("11.0Mbps", nn::wlan::RateSetLegacy_11_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate12("12.0Mbps", nn::wlan::RateSetLegacy_12_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate18("18.0Mbps", nn::wlan::RateSetLegacy_18_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate24("24.0Mbps", nn::wlan::RateSetLegacy_24_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate36("36.0Mbps", nn::wlan::RateSetLegacy_36_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate48("48.0Mbps", nn::wlan::RateSetLegacy_48_0m);
        NamedValue<nn::wlan::RateSetLegacy> rate54("54.0Mbps", nn::wlan::RateSetLegacy_54_0m);
        NamedValue<nn::wlan::RateSetLegacy> ratesCCK("CCK Rates", static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_1_0m | nn::wlan::RateSetLegacy_2_0m) );
        NamedValue<nn::wlan::RateSetLegacy> ratesDSSS("DSSS Rates", static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_5_5m | nn::wlan::RateSetLegacy_11_0m) );
        NamedValue<nn::wlan::RateSetLegacy> ratesOFDM("OFDM Rates", static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_6_0m | nn::wlan::RateSetLegacy_9_0m | nn::wlan::RateSetLegacy_12_0m | nn::wlan::RateSetLegacy_18_0m | nn::wlan::RateSetLegacy_24_0m | nn::wlan::RateSetLegacy_36_0m | nn::wlan::RateSetLegacy_48_0m | nn::wlan::RateSetLegacy_54_0m) );
        NamedValue<nn::wlan::RateSetLegacy> rates11b("802.11b Rates", nn::wlan::RateSetLegacy_11bMask);
        NamedValue<nn::wlan::RateSetLegacy> rates11g("802.11g Rates", nn::wlan::RateSetLegacy_11gMask);
        //NamedValue<nn::wlan::RateSetLegacy> ratesUDS("UDS Rates", static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_11_0m | nn::wlan::RateSetLegacy_12_0m | nn::wlan::RateSetLegacy_18_0m | nn::wlan::RateSetLegacy_24_0m | nn::wlan::RateSetLegacy_36_0m | nn::wlan::RateSetLegacy_48_0m | nn::wlan::RateSetLegacy_54_0m) );

        //selector.Register("Supported Rates(Not Implemented)", &supportedRates);
        supportedRates.Add(def, true);
        supportedRates.Add(rate1);
        supportedRates.Add(rate2);
        supportedRates.Add(rate5);
        supportedRates.Add(rate6);
        supportedRates.Add(rate9);
        supportedRates.Add(rate11);
        supportedRates.Add(rate12);
        supportedRates.Add(rate18);
        supportedRates.Add(rate24);
        supportedRates.Add(rate36);
        supportedRates.Add(rate48);
        supportedRates.Add(rate54);
        supportedRates.Add(ratesCCK);
        supportedRates.Add(ratesDSSS);
        supportedRates.Add(ratesOFDM);
        supportedRates.Add(rates11b);
        supportedRates.Add(rates11g);
        //supportedRates.Add(ratesUDS, true);

        //selector.Register("Basic Rates(Not Implemented)", &basicRates);
        basicRates.Add(def, true);
        basicRates.Add(rate1);
        basicRates.Add(rate2);
        basicRates.Add(rate5);
        basicRates.Add(rate6);
        basicRates.Add(rate9);
        basicRates.Add(rate11);
        basicRates.Add(rate12);
        basicRates.Add(rate18);
        basicRates.Add(rate24);
        basicRates.Add(rate36);
        basicRates.Add(rate48);
        basicRates.Add(rate54);
        basicRates.Add(ratesCCK);
        basicRates.Add(ratesDSSS);
        basicRates.Add(ratesOFDM);
        basicRates.Add(rates11b);
        basicRates.Add(rates11g);

        //selector.Register("eCWmin (Not Implemented)", &m_CwMin);
        m_CwMin.Add(1);
        m_CwMin.Add(2);
        m_CwMin.Add(3);
        m_CwMin.Add(4, true);
        m_CwMin.Add(5);
        m_CwMin.Add(6);
        m_CwMin.Add(7);
        m_CwMin.Add(8);
        m_CwMin.Add(9);
        m_CwMin.Add(10);

        //selector.Register("eCWmax (Not Implemented)", &m_CwMax);
        m_CwMax.Add(1);
        m_CwMax.Add(2);
        m_CwMax.Add(3);
        m_CwMax.Add(4);
        m_CwMax.Add(5);
        m_CwMax.Add(6);
        m_CwMax.Add(7);
        m_CwMax.Add(8);
        m_CwMax.Add(9);
        m_CwMax.Add(10, true);

        //selector.Register("AIFSN (Not Implemented)", &m_Aifsn);
        for(int i=1; i<=252; i++)
        {
            if( i==2 )
                m_Aifsn.Add( i, true );
            else
                m_Aifsn.Add(i);
        }

#if 0
        selector.Register("Default Authorize", &defaultAuthorize);
        defaultAuthorize.Add(off);
        defaultAuthorize.Add(on, true);
#endif

        selector.Register("Inactive Period [sec]", &inactivePeriod);
        inactivePeriod.Add(0, true);
        inactivePeriod.Add(1);
        inactivePeriod.Add(10);
        inactivePeriod.Add(30);
        inactivePeriod.Add(60);
        inactivePeriod.Add(90);
        inactivePeriod.Add(120);
        inactivePeriod.Add(150);
        inactivePeriod.Add(180);

        // (SIGLO-71971)
        // SetActionFrameWithBeacon() にセットする最大ペイロードサイズを、
        // WLAN WDM の仕様に規定される 1450byte に合わせる
        selector.Register("AF with Beacon [byte]", &actionFrame);
        actionFrame.Add(NamedValue<uint32_t>("Disable", 0));
        actionFrame.Add(NamedValue<uint32_t>("5", 5));
        actionFrame.Add(NamedValue<uint32_t>("10", 10));
        for(int i=1; i<=29; ++i)
        {
            ostringstream oss;
            auto size = i * 50;
            oss << size;
            actionFrame.Add(NamedValue<uint32_t>(oss.str(), size), size == 1450);
        }

        selector.Register("AF Rx Thread", &actionFrameThread);
        actionFrameThread.Add(NamedValue<bool>("Enable", true), true);
        actionFrameThread.Add(NamedValue<bool>("Disable", false));

        //selector.Register("Number of DeAuth", &m_NumOfDeauth);
        m_NumOfDeauth.Add(0);
        m_NumOfDeauth.Add(1);
        m_NumOfDeauth.Add(2);
        m_NumOfDeauth.Add(3, true);
        m_NumOfDeauth.Add(4);
        m_NumOfDeauth.Add(5);
        m_NumOfDeauth.Add(10);

#if 0
        selector.Register("Respond Wildcard SSID", &m_RespondWildcard);
        m_RespondWildcard.Add(on, true);
        m_RespondWildcard.Add(off);
#endif

        Oui oui[3];
        oui[0].SetRawDataFromU32(0x000000);
        oui[1].SetRawDataFromU32(0x001f32);
        oui[2].SetRawDataFromU32(0xffffff);

        //selector.Register("Respond IE-OUI", &m_IeOui);
        for(int i=0; i<3; i++)
        {
            if(i==1)
                m_IeOui.Add(oui[i], true);
            else
                m_IeOui.Add(oui[i]);
        }

        //selector.Register("Power Save (Not Implemented)", &m_EnablePs);
        m_EnablePs.Add(off, true);
        m_EnablePs.Add(on);

        //selector.Register("CTW [TU] (Not Implemented)", &m_Ctw);
        for(int i=0; i<100; i++)
        {
            if(i==10)
                m_Ctw.Add(i, true);
            else
                m_Ctw.Add(i);
        }

#if 0
        selector.Register("Echo", &m_Echo);
        m_Echo.Add(off, true);
        m_Echo.Add(on);
#endif
    } //NOLINT(impl/function_size)
private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    bool IsIdle(){ return target->IsStop(); }

    void SetDataGeneratorController(DataGeneratorController& dataGeneratorController)
    {
        m_DataGeneratorController = &dataGeneratorController;
    }

};



class ClientController : public SceneController<Client>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
    ValueSelector selector;

private:
    Label title;
    SelectableValue<NamedValue<int16_t> > channel;
    SelectableValue<string> ssid;
    SelectableValue<NamedValue <nn::wlan::MacAddress> > bssid;
    SelectableValue<NamedValue <bool> > bssIndication;
    SelectableValue<int> beaconLostTimeout;
    SelectableValue<NamedValue<nn::wlan::SecurityMode> > securityMode;
    SelectableValue<string> key;
    SelectableValue<NamedValue<bool> > m_DirectBroadcast;
    SelectableValue<int> m_CwMin;
    SelectableValue<int> m_CwMax;
    SelectableValue<int> m_Aifsn;
    SelectableValue<int> m_Interval;
    SelectableValue<NamedValue<bool> > m_EnablePs;
    SelectableValue<NamedValue<bool> > m_EnableClr;
    SelectableValue<int> m_TxWait;
    SelectableValue<NamedValue<bool> > m_Echo;
    SelectableValue<NamedValue<bool> > actionFrameThread;

    bool m_Active;

    DataGeneratorController* m_DataGeneratorController;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Set()
    {
        if(ssid.GetValue() == "Auto")
        {
            ostringstream id;
            id << "basic_test_";
            id << channel;
            id << "ch";
            target->SetSsid(id.str().c_str());
        }
        else
        {
            target->SetSsid(ssid.GetValue().c_str());
        }
        target->SetBssid( bssid );
        target->SetChannel( channel );
        nn::wlan::Security sec = target->GetSecurity();
        {
            sec.privacyMode = securityMode;
            sec.groupPrivacyMode = securityMode;
            std::memset(sec.key, 0x00, 64);
            std::memcpy(sec.key, key.GetValue().c_str(), key.GetValue().size());
        }
        target->SetSecurity(sec);

        target->SetEnableBssIndication( bssIndication );
        target->SetBeaconLostTimeout( beaconLostTimeout );
        target->SetIsAfThreadEnabled(actionFrameThread);

#if 0
        SapTxQueueParam p = target->GetAccessParam( TX_QUEUE_1 );
        {
            p.eCWmin = m_CwMin;
            p.eCWmax = m_CwMax;
            p.aifsn  = m_Aifsn;
            target->SetAccessParam(p);
        }
        target->SetEnableDirectBroadcast( m_DirectBroadcast );

        target->SetEnablePs( m_EnablePs );
        target->SetEnableClr( m_EnableClr );
        target->SetTxWait( m_TxWait );
        target->SetEnableEcho(m_Echo);
#endif
    }

    void Set(ClientParam param)
    {
        NamedValue<int16_t> ch("- user defined -", param.channel);
        channel.Add(ch, true);

        ssid.Add(param.ssid, true);

        NamedValue<nn::wlan::SecurityMode> mode("- user defined -", param.security.privacyMode);
        securityMode.Add(mode, true);

        key.Add(reinterpret_cast<char*>(param.security.key), true);
        beaconLostTimeout.Add(param.beaconLostTimeout, true);
    }

    virtual void InputPad(Pad& pad)
    {
    }

    void Start()
    {
        if(!m_Active)
        {
            Set();

            nn::Result result;
            Sleep(nn::TimeSpan::FromSeconds(1));
            result = target->StartMaintainConnection(MAINTAIN_CONNECTION_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                return;
            }

            result = target->StartReceiveActionFrame(RECEIVE_ACTION_FRAME_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                target->StopMaintainConnection();
                return;
            }

            result = target->StartReceiveData(RECEIVE_DATA_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                target->StopReceiveActionFrame();
                target->StopMaintainConnection();
                return;
            }

            result = target->StartReceiveCommand(RECEIVE_COMMAND_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                NN_ASSERT(false);
                target->StopReceiveData();
                target->StopReceiveActionFrame();
                target->StopMaintainConnection();
                return;
            }

            m_Active = true;
        }
    }

    void Stop()
    {
        if(m_Active)
        {
            m_Active = false;
            if(!target->IsStop())
            {
                target->StopReceiveCommand();
                target->StopReceiveData();
                target->StopReceiveActionFrame();
                target->StopMaintainConnection();
            }
        }
    }
protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    ClientController()
    {
        Initialize();
    }

    virtual void Initialize()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        m_Active = false;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Client Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = DISPLAY_CONTENT_HEIGHT;

        NamedValue<int16_t> chAuto("Auto", -1);
        selector.Register("Channel", &channel);
        channel.Add(chAuto, 1);
        GenerateSelectableChannel(&channel);

        selector.Register("SSID", &ssid);
        ssid.Add("Auto");
        GenerateSelectableSsid(&ssid, "wireless_test_1");

        selector.Register("BSSID", &bssid);
        bssid.Add(NamedValue<nn::wlan::MacAddress>("FF:FF:FF:FF:FF:FF", nn::wlan::MacAddress::CreateBroadcastMacAddress()), true);
        bssid.Add(NamedValue<nn::wlan::MacAddress>("00:00:00:00:00:00", nn::wlan::MacAddress::CreateZeroMacAddress()));
        bssid.Add(NamedValue<nn::wlan::MacAddress>("78:A2:A0:35:D9:0E", nn::wlan::MacAddress(0x78,0xa2,0xa0,0x35,0xd9,0x0e)));
        bssid.Add(NamedValue<nn::wlan::MacAddress>("78:A2:A0:DD:71:EF", nn::wlan::MacAddress(0x78,0xa2,0xa0,0xdd,0x71,0xef)));

        selector.Register("Security Mode", &securityMode);
        NamedValue<nn::wlan::SecurityMode> open("OPEN", nn::wlan::SecurityMode_Open);
        NamedValue<nn::wlan::SecurityMode> staticAes("StaticAES", nn::wlan::SecurityMode_StaticAes);
        securityMode.Add(open, true);
        securityMode.Add(staticAes);

        selector.Register("Key", &key);
        key.Add("0123456789abcdef", true);
        key.Add("aaaaaaaaaaaaaaaa");

        NamedValue<bool> on("On", true);
        NamedValue<bool> off("Off", false);

#if 0
        // DirectBroadcast 機能は不要 (2016/1/14) Broadcast 通信を優先、Unicast は infra で使用されるので Local では優先度低
        selector.Register("Direct Broadcast", &m_DirectBroadcast);
        m_DirectBroadcast.Add(on, true);
        m_DirectBroadcast.Add(off);
#endif

        //selector.Register("eCWmin (Not Implemented)", &m_CwMin);
        m_CwMin.Add(1);
        m_CwMin.Add(2);
        m_CwMin.Add(3);
        m_CwMin.Add(4, true);
        m_CwMin.Add(5);
        m_CwMin.Add(6);
        m_CwMin.Add(7);
        m_CwMin.Add(8);
        m_CwMin.Add(9);
        m_CwMin.Add(10);

        //selector.Register("eCWmax (Not Implemented)", &m_CwMax);
        m_CwMax.Add(1);
        m_CwMax.Add(2);
        m_CwMax.Add(3);
        m_CwMax.Add(4);
        m_CwMax.Add(5);
        m_CwMax.Add(6);
        m_CwMax.Add(7);
        m_CwMax.Add(8);
        m_CwMax.Add(9);
        m_CwMax.Add(10, true);

        //selector.Register("AIFSN (Not Implemented)", &m_Aifsn);
        for(int i=1; i<=252; i++)
        {
            if( i==2 )
            {
                m_Aifsn.Add( i, true );
            }
            else
            {
                m_Aifsn.Add(i);
            }
        }

        //selector.Register("BssIndication", &bssIndication);
        bssIndication.Add(on, true);
        bssIndication.Add(off);

        selector.Register("BeaconLostTimeout", &beaconLostTimeout);
        beaconLostTimeout.Add(1);
        beaconLostTimeout.Add(2);
        beaconLostTimeout.Add(3);
        beaconLostTimeout.Add(4);
        beaconLostTimeout.Add(5);
        beaconLostTimeout.Add(10, true);
        beaconLostTimeout.Add(20);
        beaconLostTimeout.Add(30);

        selector.Register("AF Rx Thread", &actionFrameThread);
        actionFrameThread.Add(NamedValue<bool>("Enable", true), true);
        actionFrameThread.Add(NamedValue<bool>("Disable", false));

        // selector.Register("Power Save (Not Implemented)", &m_EnablePs);
        m_EnablePs.Add(off, true);
        m_EnablePs.Add(on);

#if 0
        selector.Register("CLR", &m_EnableClr);
        m_EnableClr.Add(off, true);
        m_EnableClr.Add(on);

        selector.Register("Tx Wait [TU]", &m_TxWait);
        for(int i=0; i<100; i++)
        {
            if(i==10)
                m_TxWait.Add(i, true);
            else
                m_TxWait.Add(i);
        }

        selector.Register("Echo", &m_Echo);
        m_Echo.Add(off, true);
        m_Echo.Add(on);
#endif
    } //NOLINT(impl/function_size)

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    bool IsIdle(){ return target->IsStop(); }

    void SetDataGeneratorController(DataGeneratorController& dataGeneratorController)
    {
        m_DataGeneratorController = &dataGeneratorController;
    }
};


class SocketNodeController : public SceneController<SocketNode>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    ValueSelector m_Selector;

private:

    Label m_Title;
    SelectableValue<NamedValue<int16_t> > m_Channel;
    SelectableValue<string> m_Ssid;
    SelectableValue<NamedValue<nn::wlan::SecurityMode> > m_SecurityMode;
    SelectableValue<string> m_Key;

    SelectableValue<NamedValue<SocketType> > m_Type;
    SelectableValue<NamedValue<string> > m_IpAddress;
    SelectableValue<NamedValue<string> > m_GwIpAddress;
    SelectableValue<NamedValue<string> > m_RemoteIpAddress;
    SelectableValue<uint16_t> m_Port;
    SelectableValue<NamedValue<PacketFormat> > m_PacketFormat;

    bool m_Active;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/

public:

    virtual void Set()
    {
        target->SetChannel(m_Channel);

        if(m_Ssid.GetValue() == "Auto")
        {
            ostringstream id;
            id << "wireless_test_";
            id << m_Channel;
            id << "ch";
            target->SetSsid(nn::wlan::Ssid(id.str().c_str()));
        }
        else
        {
            target->SetSsid(nn::wlan::Ssid(m_Ssid.GetValue().c_str()));
        }

        nn::wlan::Security security;
        security.privacyMode = m_SecurityMode;
        security.groupPrivacyMode = m_SecurityMode;
        security.keyIdx = 0;

        string key = m_Key.GetValue();
        if( security.groupPrivacyMode == nn::wlan::SecurityMode_Wep64Open ||
            security.groupPrivacyMode == nn::wlan::SecurityMode_Wep64Shared ||
            security.groupPrivacyMode == nn::wlan::SecurityMode_Wep128Open ||
            security.groupPrivacyMode == nn::wlan::SecurityMode_Wep128Shared )
        {
            ssize_t asciiKeySize = 0;
            if( security.groupPrivacyMode == nn::wlan::SecurityMode_Wep64Open ||
                security.groupPrivacyMode == nn::wlan::SecurityMode_Wep64Shared )
            {
                asciiKeySize = WEP64_ASCII_KEY_SIZE;
            }
            else
            {
                asciiKeySize = WEP128_ASCII_KEY_SIZE;
            }

            ssize_t size = key.size();
            if( size != asciiKeySize && size != 2 * asciiKeySize )
            {
                string newKey = string(asciiKeySize, 'a');
                NN_LOG(" WEP key (%s) is invalid, changed to \"%s\"\n", key.c_str(), newKey.c_str());
                key = newKey;
            }

            string newKey;
            NN_ASSERT( ConvertWepKey(&newKey, key, security.groupPrivacyMode) == true );
            key = newKey;
        }

        std::memcpy(security.key, key.c_str(), sizeof(security.key));
        target->SetSecurity(security);

        target->SetSocketType(m_Type);
        target->SetRxPacketFormat(m_PacketFormat);

        if(m_IpAddress.GetValue().Value == "DHCP")
        {
            target->EnableDhcp();
            target->SetIpAddress("0.0.0.0");
        }
        else
        {
            target->DisableDhcp();
            target->SetIpAddress(m_IpAddress.GetValue().Value);
        }
        target->SetGwIpAddress(m_GwIpAddress.GetValue().Value);
        target->SetRemoteIpAddress(m_RemoteIpAddress.GetValue());
        target->SetPort(m_Port);
    }

    void Set(SocketParam param)
    {
        NamedValue<int16_t> ch("- user defined -", param.channel);
        m_Channel.Add(ch, true);

        m_Ssid.Add(param.ssid, true);

        NamedValue<nn::wlan::SecurityMode> mode("- user defined -", param.security.privacyMode);
        m_SecurityMode.Add(mode, true);

        m_Key.Add(reinterpret_cast<char*>(param.security.key), true);

        NamedValue<SocketType> type("- user defined -", param.type);
        m_Type.Add(type, true);

        NamedValue<PacketFormat> format("- user defined -", param.format);
        m_PacketFormat.Add(format, true);

        NamedValue<string> ip("- user defined -", param.ipAddress);
        m_IpAddress.Add(ip, true);

        NamedValue<string> remote("- user defined -", param.remoteIpAddress);
        m_RemoteIpAddress.Add(remote, true);

        NamedValue<string> gw("- user defined -", param.gwIpAddress);
        m_GwIpAddress.Add(gw, true);

        m_Port.Add(param.port, true);

        Set();
    }

    virtual void InputPad(Pad& pad)
    {
    }

    virtual void Start()
    {
        if(!m_Active)
        {
            Set();

            nn::Result result;
            result = target->StartMaintainConnection(MAINTAIN_CONNECTION_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                return;
            }

            result = target->StartReceiveData(RECEIVE_DATA_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                target->StopMaintainConnection();
                return;
            }

            result = target->StartReceiveCommand(RECEIVE_COMMAND_THREAD_PRIORITY);
            if(result.IsFailure())
            {
                target->StopReceiveData();
                target->StopMaintainConnection();
                return;
            }

            m_Active = true;
        }
    }

    void Stop()
    {
        if(m_Active)
        {
            m_Active = false;
            if(!target->IsStop())
            {
                target->StopReceiveCommand();
                target->StopReceiveData();
                target->StopMaintainConnection();
            }
        }
    }
protected:

private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    SocketNodeController()
    {
        Initialize();
    }

    virtual void Initialize()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        m_Active = false;

        Add(m_Title);
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "Socket Parameters";
        m_Title.FitSize();

        Add(m_Selector);
        m_Selector.X = x;
        m_Selector.Y = y;
        m_Selector.Height = DISPLAY_CONTENT_HEIGHT;

        NamedValue<bool> off ("Off", false);
        NamedValue<bool> on ("On", true);

        NamedValue<int16_t> chAuto("Auto", -1);
        m_Selector.Register("Channel", &m_Channel);
        m_Channel.Add(chAuto, 1);
        GenerateSelectableChannel(&m_Channel);

        m_Selector.Register("SSID", &m_Ssid);
        m_Ssid.Add("Auto");
        GenerateSelectableSsid(&m_Ssid, "wireless_test_1");

        NamedValue<nn::wlan::SecurityMode> open("OPEN", nn::wlan::SecurityMode_Open);
        NamedValue<nn::wlan::SecurityMode> wep64Open("WEP64-OPEN", nn::wlan::SecurityMode_Wep64Open);
        NamedValue<nn::wlan::SecurityMode> wep64Shared("WEP64-SHARED", nn::wlan::SecurityMode_Wep64Shared);
        NamedValue<nn::wlan::SecurityMode> wep128Open("WEP128-OPEN", nn::wlan::SecurityMode_Wep128Open);
        NamedValue<nn::wlan::SecurityMode> wep128Shared("WEP128-SHARED", nn::wlan::SecurityMode_Wep128Shared);
        NamedValue<nn::wlan::SecurityMode> wpaAes("WPA-AES", nn::wlan::SecurityMode_WpaAes);
        NamedValue<nn::wlan::SecurityMode> wpa2Aes("WPA2-AES", nn::wlan::SecurityMode_Wpa2Aes);
        NamedValue<nn::wlan::SecurityMode> wpaTkip("WPA-TKIP", nn::wlan::SecurityMode_WpaTkip);
        NamedValue<nn::wlan::SecurityMode> wpa2Tkip("WPA2-TKIP", nn::wlan::SecurityMode_Wpa2Tkip);
        m_Selector.Register("Security Mode", &m_SecurityMode);
        m_SecurityMode.Add(open, true);
        m_SecurityMode.Add(wep64Open);
        m_SecurityMode.Add(wep64Shared);
        m_SecurityMode.Add(wep128Open);
        m_SecurityMode.Add(wep128Shared);
        m_SecurityMode.Add(wpaAes);
        m_SecurityMode.Add(wpa2Aes);
        m_SecurityMode.Add(wpaTkip);
        m_SecurityMode.Add(wpa2Tkip);

        m_Selector.Register("Key", &m_Key);
        m_Key.Add("0123456789abcdef", true);
        m_Key.Add("aaaaaaaaaaaaaaaa");
        m_Key.Add("0123456789abc");
        m_Key.Add("30313233343536373839616263");
        m_Key.Add("01234");
        m_Key.Add("3031323334");

        NamedValue<SocketType> udp("UDP", SOCKET_TYPE_UDP);
        m_Selector.Register("Protocol", &m_Type);
        m_Type.Add(udp, true);

        NamedValue<PacketFormat> witFormat("WIT", PACKET_FORMAT_WIT);
        NamedValue<PacketFormat> rtpFormat("RTP", PACKET_FORMAT_RTP);
        m_Selector.Register("Rx Format", &m_PacketFormat);
        m_PacketFormat.Add(witFormat, true);
        m_PacketFormat.Add(rtpFormat);

        m_Selector.Register("GW IP Address", &m_GwIpAddress);
        m_Selector.Register("IP Address", &m_IpAddress);
        m_Selector.Register("Remote IP Address", &m_RemoteIpAddress);
        for(int i=1; i<=20; ++i)
        {
            ostringstream oss;
            oss << "192.168.11." << i;

            NamedValue<string> ip(oss.str() + "/24", oss.str());
            m_GwIpAddress.Add(ip, i == 1);
            m_IpAddress.Add(ip, i == 2);
            m_RemoteIpAddress.Add(ip, i == 3);
        }
        for(int i=0; i<=20; ++i)
        {
            ostringstream oss;
            oss << "192.168.11." << i + 100;

            NamedValue<string> ip(oss.str() + "/24", oss.str());
            m_IpAddress.Add(ip);
            m_GwIpAddress.Add(ip);
            m_RemoteIpAddress.Add(ip);
        }

        NamedValue<string> ip("DHCP", "DHCP");
        m_IpAddress.Add(ip);

        m_Selector.Register("Port", &m_Port);
        for(int i=0; i<20; ++i)
        {
            m_Port.Add(50000 + i);
        }

    } //NOLINT(impl/function_size)

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    SocketType GetSelectingSocketType(){ return m_Type.GetValue(); }
    bool IsIdle(){ return target->IsStop(); }

};

class LcsSocketController : public SceneController<LcsSocket>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    ValueSelector selector;

private:

    Label title;
    SelectableValue<NamedValue<size_t> > txBuffer;
    SelectableValue<NamedValue<size_t> > rxBuffer;
    SelectableValue<NamedValue<bool> > negle;
    SelectableValue<NamedValue<uint32_t> > txUpdateInterval;
    SelectableValue<NamedValue<uint32_t> > rxUpdateInterval;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/

public:
    virtual void Set()
    {
        target->SetTxBufferSize(txBuffer);
        target->SetRxBufferSize(rxBuffer);
        target->SetNegleAlgorithm(negle);
        target->SetTxStatsUpdateInterval(txUpdateInterval);
        target->SetRxStatsUpdateInterval(rxUpdateInterval);
    }

    virtual void InputPad(Pad& pad)
    {
    }

protected:

private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    LcsSocketController()
    {
        Initialize();
    }

    virtual void Initialize()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Socket Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = DISPLAY_CONTENT_HEIGHT;

        selector.Register("Tx Buffer Size", &txBuffer);
        selector.Register("Rx Buffer Size", &rxBuffer);

        const size_t kb = 1 * 1024;
        size_t size = 1 * kb;
        for(int i=0; i<10; ++i)
        {
            string sizeName = ToStringForBinaryPrefix(size, " ", 0) + "B";
            txBuffer.Add(NamedValue<size_t>(sizeName, size), size == 512 * kb);
            rxBuffer.Add(NamedValue<size_t>(sizeName, size), size == 512 * kb);
            size *= 2;
        }

        selector.Register("Negle Algorithm", &negle);
        negle.Add(NamedValue<bool>("Enabled", true));
        negle.Add(NamedValue<bool>("Disabled", false), true);

        selector.Register("Update Interval (TxStats)", &txUpdateInterval);
        selector.Register("Update Interval (RxStats)", &rxUpdateInterval);
        for(int i=1; i<=10; ++i)
        {
            ostringstream oss;
            oss << i << " sec";
            txUpdateInterval.Add(NamedValue<uint32_t>(oss.str(), i));
            rxUpdateInterval.Add(NamedValue<uint32_t>(oss.str(), i));
        }
    }

private:

};

class ScannerController : public SceneController<Scanner>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
protected:

    enum State
    {
        STATE_IDLE = 0,
        STATE_WAITING,
        STATE_SCANNING,
        STATE_COMPLETED
    };

    Label title;
    Label m_A;
    SelectableValue<NamedValue<nn::wlan::ScanType> >   m_ScanType;
    MultiSelectableValue<NamedValue<int16_t> >         m_Channel;
    SelectableValue<NamedValue<int> >                  m_ScanTime;
    SelectableValue<NamedValue<int> >                  m_HomeChTime;
    SelectableValue<NamedValue<nn::wlan::Ssid> >       m_Ssid;
    SelectableValue<NamedValue<nn::wlan::MacAddress> > m_Bssid;
    SelectableValue<size_t>                            m_BufferSize;

    SelectableValue<NamedValue<bool> >                 m_EnableRepeat;
    SelectableValue<int>                               m_ScanInterval;

    Label                   m_StateLabel;
    ActiveIndicator         m_Indicator;
    MultiValueSelector      m_Selector;
    nn::os::ThreadType      m_Thread;
    bool                    m_IsThreadRunning;
    bool                    m_ExitThread;
    State                   m_State;

    /*---------------------------------------------------------------------------
      　　　　　メンバメソッド
      ---------------------------------------------------------------------------*/
private:

    static void ThreadFunc(void* arg)
    {
        ScannerController* controller = reinterpret_cast<ScannerController *>(arg);
        while( !controller->m_ExitThread )
        {
            controller->Set();
            controller->m_State = STATE_SCANNING;
            controller->target->StartScan();
            controller->target->PrintResult();

            if( controller->m_EnableRepeat == false )
            {
                break;
            }

            controller->m_State = STATE_WAITING;
            Sleep(nn::TimeSpan::FromMilliSeconds(controller->m_ScanInterval));
        }

        controller->m_State = STATE_COMPLETED;
        controller->m_Indicator.Inactivate();
    }

public:

    void Initialize()
    {}

    void Finalize()
    {
        Stop();
        if( m_IsThreadRunning )
        {
            nn::os::DestroyThread(&m_Thread);
            m_IsThreadRunning = false;
        }
    }

    virtual void ShowImpl(Display& display)
    {
        switch( m_State )
        {
        case STATE_IDLE :
            {
                m_StateLabel.Text = "Idle";
                SetPadText(Button::A, "Scan");
            }
            break;

        case STATE_WAITING :
            {
                m_StateLabel.Text = "Sleeping";
                SetPadText(Button::A, "Stop");
            }
            break;

        case STATE_SCANNING :
            {
                m_StateLabel.Text = "Scanning";
                SetPadText(Button::A, "Stop");
            }
            break;

        case STATE_COMPLETED :
            {
                m_StateLabel.Text = "Completed";
                SetPadText(Button::A, "Scan");
            }
            break;

        default :
            {
                m_StateLabel.Text = "Unknown";
            }
            break;
        }

        if( m_Selector.GetSelectingItem() == &m_Channel )
        {
            SetPadText(Button::Y, "Add Item");
            SetPadText(Button::B, "Remove Item");
        }
        else
        {
            ErasePadText(Button::Y);
            ErasePadText(Button::B);
        }

        SceneController<Scanner>::ShowImpl(display);
    }

    virtual void Set()
    {
        target->SetBufferSize( m_BufferSize );
        target->SetEnableActiveScan( m_ScanType == nn::wlan::ScanType_Active ? true : false );
        int16_t chList[nn::wlan::WirelessChannelsCountMax] = {0};
        uint8_t chCount = m_Channel.GetSelectingItemCount();

        if( chCount == 0 )
        {
            // 明示的に選択していない場合は、表示している項目を選択する
            // ただし、All が選択されている場合のカウントは 0
            chList[0] = m_Channel.GetValue();
            if( chList[0] == 0 )
            {
                chCount = 0;
            }
            else
            {
                chCount = 1;
            }
        }
        else if( 0 < chCount && chCount <= nn::wlan::WirelessChannelsCountMax )
        {
            for(int i=0; i<chCount; ++i)
            {
                chList[i] = m_Channel.GetSelectingItems()->at(i).Value;
            }
        }
        else
        {
            // 規定数よりも選択している数が多い場合は全チャネル検索する
            chCount = 0;
        }

        target->SetChannel( chList, chCount );
        target->SetScanTime( m_ScanTime );
        target->SetHomeChannelTime( m_HomeChTime );
        target->SetBssid( m_Bssid );

        nn::wlan::Ssid ssid = m_Ssid.GetValue();
        if( ssid == nn::wlan::Ssid(0, 0) )
        {
            // wildcard
            target->SetSsidList(nullptr, 0);
        }
        else
        {
            target->SetSsid(ssid);
        }
    }

    virtual void InputPad(Pad& pad)
    {
        nn::Result result;

        if( pad.IsTrigger(Button::A) )
        {
            if( IsScanning() )
            {
                Stop();
            }
            else
            {
                Set();
                Start();
            }
        }
    }

    void Start()
    {
        static NN_ALIGNAS(4096) char s_ThreadStack[4096];

        if( !IsScanning() )
        {
            if( m_IsThreadRunning )
            {
                nn::os::DestroyThread(&m_Thread);
                m_IsThreadRunning = false;
            }

            m_State = STATE_SCANNING;
            m_ExitThread = false;
            m_IsThreadRunning = true;

            nn::Result result = nn::os::CreateThread(&m_Thread, ThreadFunc, this, s_ThreadStack, sizeof(s_ThreadStack), nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
            if( result.IsFailure() )
            {
                NN_LOG("ScanThread can not run.\n");
                NN_ASSERT(false);
            }

            nn::os::SetThreadNamePointer(&m_Thread, "WitScanThread");
            nn::os::StartThread(&m_Thread);

            m_Indicator.Activate();
        }
    }

    void Stop()
    {
        if( IsScanning() )
        {
            target->StopScan();
            m_ExitThread = true;
            nn::os::DestroyThread(&m_Thread);
            m_IsThreadRunning = false;
        }
    }

    bool IsScanning()
    {
        return m_State == STATE_SCANNING || m_State == STATE_WAITING;
    }

    /*---------------------------------------------------------------------------
      　　　　　コンストラクタ類
      ---------------------------------------------------------------------------*/
public:

    ScannerController()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t w = Display::GetInstance().GetFixedWidth();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        m_State = STATE_IDLE;
        m_IsThreadRunning = false;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Scan Parameters";
        title.FitSize();

        Add(m_Indicator);
        m_Indicator.X = x;
        m_Indicator.Y = y;
        m_Indicator.Width = 3 * w;
        m_Indicator.Height = h;

        Add(m_StateLabel);
        m_StateLabel.Text = "Idle";
        m_StateLabel.X = x + 30;
        m_StateLabel.Y = y;
        m_StateLabel.Width = 100 * w;

        Add(m_Selector);
        m_Selector.X = x;
        m_Selector.Y = (y += 2 * h);
        m_Selector.Height = DISPLAY_CONTENT_HEIGHT;

        m_Selector.Register("Buffer Size [byte]", &m_BufferSize);
        m_BufferSize.Add(100 * 1024, true);

        m_Selector.Register("Type", &m_ScanType);
        NamedValue<nn::wlan::ScanType> active("Active Scan", nn::wlan::ScanType_Active);
        NamedValue<nn::wlan::ScanType> passive("Passive Scan", nn::wlan::ScanType_Passive);
        m_ScanType.Add(active, true);
        m_ScanType.Add(passive);

        m_Selector.Register("Channel", &m_Channel);
        NamedValue<int16_t> chAll("All", 0);
        m_Channel.SetSelectableItemCountMax(100);
        m_Channel.Add(chAll);
        GenerateSelectableChannel(&m_Channel);

        m_Selector.Register("Scan Time [ms]", &m_ScanTime);
        m_Selector.Register("Home Ch Time [ms]", &m_HomeChTime);
        NamedValue<int> defaultTime("Default", -1);
        m_ScanTime.Add(defaultTime);
        m_HomeChTime.Add(defaultTime);
        for(int i=10; i<=500; i+=10)
        {
            ostringstream oss;
            oss << i;
            NamedValue<int> time(oss.str(), i);
            m_ScanTime.Add(time);
            m_HomeChTime.Add(time);
        }
        for(int i=1000; i<=5000; i+=1000)
        {
            ostringstream oss;
            oss << i;
            NamedValue<int> time(oss.str(), i);
            m_ScanTime.Add(time);
            m_HomeChTime.Add(time);
        }
        NamedValue<int> lTime("0x7fffffff", 0x7fffffff);
        NamedValue<int> mTime("0x80000000", 0x80000000);
        m_ScanTime.Add(lTime);
        m_HomeChTime.Add(lTime);
        m_ScanTime.Add(mTime);
        m_HomeChTime.Add(mTime);

        m_Selector.Register("SSID", &m_Ssid);
        NamedValue<nn::wlan::Ssid> wildcardSsid("Wildcard", nn::wlan::Ssid(0, 0));
        NamedValue<nn::wlan::Ssid> specificSsid1("wireless_test_1", nn::wlan::Ssid("wireless_test_1"));
        NamedValue<nn::wlan::Ssid> specificSsid2("wireless_test_2", nn::wlan::Ssid("wireless_test_2"));
        NamedValue<nn::wlan::Ssid> specificSsid3("wireless_test_3", nn::wlan::Ssid("wireless_test_3"));
        NamedValue<nn::wlan::Ssid> specificSsid4("wireless_test_4", nn::wlan::Ssid("wireless_test_4"));
        NamedValue<nn::wlan::Ssid> specificSsid5("wireless_test_5", nn::wlan::Ssid("wireless_test_5"));
        NamedValue<nn::wlan::Ssid> specificSsid6("wireless_test_6", nn::wlan::Ssid("wireless_test_6"));
        NamedValue<nn::wlan::Ssid> specificSsid7("wireless_test_7", nn::wlan::Ssid("wireless_test_7"));
        NamedValue<nn::wlan::Ssid> specificSsid8("wireless_test_8", nn::wlan::Ssid("wireless_test_8"));
        NamedValue<nn::wlan::Ssid> specificSsid9("wireless_test_9", nn::wlan::Ssid("wireless_test_9"));
        NamedValue<nn::wlan::Ssid> specificSsid10("wireless_test_10", nn::wlan::Ssid("wireless_test_10"));
        m_Ssid.Add(wildcardSsid, true);
        m_Ssid.Add(specificSsid1);
        m_Ssid.Add(specificSsid2);
        m_Ssid.Add(specificSsid3);
        m_Ssid.Add(specificSsid4);
        m_Ssid.Add(specificSsid5);
        m_Ssid.Add(specificSsid6);
        m_Ssid.Add(specificSsid7);
        m_Ssid.Add(specificSsid8);
        m_Ssid.Add(specificSsid9);
        m_Ssid.Add(specificSsid10);

        m_Selector.Register("BSSID", &m_Bssid);
        NamedValue<nn::wlan::MacAddress> zeroMac("Zero", nn::wlan::MacAddress::CreateZeroMacAddress());
        NamedValue<nn::wlan::MacAddress> wildcardMac("Wildcard", nn::wlan::MacAddress::CreateBroadcastMacAddress());
        NamedValue<nn::wlan::MacAddress> specificMac1("40:d2:8a:b4:93:7e", nn::wlan::MacAddress(0x40, 0xd2, 0x8a, 0xb4, 0x93, 0x7e));
        m_Bssid.Add(zeroMac);
        m_Bssid.Add(wildcardMac, true);
        m_Bssid.Add(specificMac1);

        NamedValue<bool> t("Enable", true);
        NamedValue<bool> f("Disable", false);
        m_Selector.Register("AutoRepeat", &m_EnableRepeat);
        m_EnableRepeat.Add(t);
        m_EnableRepeat.Add(f, true);

        m_Selector.Register("Interval [ms]", &m_ScanInterval);
        m_ScanInterval.Add(0);
        m_ScanInterval.Add(10);
        m_ScanInterval.Add(50);
        m_ScanInterval.Add(55);
        m_ScanInterval.Add(60);
        m_ScanInterval.Add(65);
        m_ScanInterval.Add(70);
        m_ScanInterval.Add(75);
        m_ScanInterval.Add(80);
        m_ScanInterval.Add(85);
        m_ScanInterval.Add(90);
        m_ScanInterval.Add(95);
        m_ScanInterval.Add(100, true);
        m_ScanInterval.Add(200);
        m_ScanInterval.Add(400);
        m_ScanInterval.Add(500);
        m_ScanInterval.Add(1000);
        m_ScanInterval.Add(2000);
        m_ScanInterval.Add(2500);
        m_ScanInterval.Add(5000);
    } //NOLINT(impl/function_size)

    virtual ~ScannerController()
    {
        if( m_IsThreadRunning )
        {
            target->StopScan();
            m_ExitThread = true;
            nn::os::DestroyThread(&m_Thread);
            m_IsThreadRunning = false;
        }
    }

private:

    ScannerController(const ScannerController& scannerController);
    ScannerController& operator=(const ScannerController& scannerController);

};

class RssiController : public SceneController<RssiStatistics>
{

public:
protected:

    ValueSelector m_Selector;
    Node *m_pNode;

private:

    bool m_Active;

    SelectableValue<int> m_RecordInterval;
    SelectableValue<int> m_RecordCount;

public:

    RssiController()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t w = Display::GetInstance().GetFixedWidth();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        m_Active = false;

        Add(m_Selector);
        m_Selector.X = x;
        m_Selector.Y = (y += 2 * h);
        m_Selector.Height = DISPLAY_CONTENT_HEIGHT;

        m_Selector.Register("Record Interval", &m_RecordInterval);
        for(int i=1; i<=10; ++i)
        {
            m_RecordInterval.Add(i * 100, i * 100 == 1000);
        }

        m_Selector.Register("Record Count", &m_RecordCount);
        for(int i=1; i<=5; ++i)
        {
            uint32_t time = i * 60;
            m_RecordCount.Add(time, time == 180);
        }
    }

    virtual void Set()
    {
        target->SetNode(m_pNode);
        target->SetRecordInterval(m_RecordInterval);
        target->SetRecordCount(m_RecordCount);
    }

    virtual void InputPad(Pad& pad)
    {
    }

    void Start()
    {
        if( !m_Active )
        {
            NN_ASSERT_NOT_NULL(m_pNode);
            Set();

            nn::Result result;
            result = target->StartRecordRssi(RECORD_RSSI_THREAD_PRIORITY);
            NN_ASSERT( result.IsSuccess() );
            m_Active = true;
        }
    }

    void Stop()
    {
        if( m_Active )
        {
            m_Active = false;
            if( !target->IsStop() )
            {
                target->StopRecordRssi();
                target->Clear();
            }
        }
    }

    void SetNode(Node *pNode)
    {
        m_pNode = pNode;
    }

protected:
private:

};

class AfScannerController : public SceneController<AfTxScanner>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
protected:

    Label title;

    SelectableValue<NamedValue<bool>>                  m_IsAfEnabled;
    MultiSelectableValue<NamedValue<int16_t>>          m_AfChannel;
    SelectableValue<uint32_t>                          m_AfPayloadSize;
    SelectableValue<int32_t>                           m_TxInterval;
    SelectableValue<uint32_t>                          m_TxCount;
    SelectableValue<int32_t>                           m_InitialTxWaitTime;
    SelectableValue<int32_t>                           m_InitialMaxRandomWaitTime;

    SelectableValue<NamedValue<bool>>                  m_IsScanEnabled;
    SelectableValue<NamedValue<nn::wlan::ScanType>>    m_ScanType;
    MultiSelectableValue<NamedValue<int16_t>>          m_Channel;
    SelectableValue<NamedValue<int>>                   m_ScanTime;
    SelectableValue<NamedValue<int>>                   m_HomeChTime;
    SelectableValue<string>                            m_Ssid;
    SelectableValue<NamedValue<nn::wlan::MacAddress>>  m_Bssid;
    SelectableValue<size_t>                            m_BufferSize;
    SelectableValue<int32_t>                           m_InitialScanWaitTime;

    MultiValueSelector      m_Selector;
    nn::os::ThreadType      m_Thread;
    bool                    m_IsThreadRunning;
    bool                    m_ExitThread;
    bool                    m_IsScanning;

    /*---------------------------------------------------------------------------
      　　　　　メンバメソッド
      ---------------------------------------------------------------------------*/
private:

    static void ThreadFunc(void* arg)
    {
        AfScannerController* controller = reinterpret_cast<AfScannerController *>(arg);
        controller->Set();

        bool isEnabled = controller->target->IsAfEnabled() || controller->target->IsScanEnabled();
        while( !controller->m_ExitThread )
        {
            controller->target->StartScan();

            if( !isEnabled )
            {
                Sleep(nn::TimeSpan::FromMilliSeconds(1));
            }
        }
        controller->m_IsScanning = false;
    }

public:

    void Initialize()
    {}

    void Finalize()
    {
        Stop();
        if( m_IsThreadRunning )
        {
            nn::os::DestroyThread(&m_Thread);
            m_IsThreadRunning = false;
        }
    }

    void ShowImpl(Display& display)
    {
        auto pItem = m_Selector.GetSelectingItem();
        if( pItem == &m_Channel || pItem == &m_AfChannel )
        {
            SetPadText(Button::Y, "Add Item");
            SetPadText(Button::B, "Remove Item");
        }
        else
        {
            ErasePadText(Button::Y);
            ErasePadText(Button::B);
        }

        SceneController<AfTxScanner>::ShowImpl(display);
    }

    void SetChannelList(int16_t chList[nn::wlan::WirelessChannelsCountMax], uint8_t *pChCount,
                        const MultiSelectableValue<NamedValue<int16_t>>* pSelectedChannels)
    {
        auto chCount = pSelectedChannels->GetSelectingItemCount();
        if( chCount == 0 )
        {
            // 明示的に選択していない場合は、表示している項目を選択する
            *pChCount = 1;
            chList[0] = pSelectedChannels->GetValue();
        }
        else if( 0 < chCount && chCount <= nn::wlan::WirelessChannelsCountMax )
        {
            *pChCount = chCount;
            for(int i=0; i<chCount; ++i)
            {
                chList[i] = pSelectedChannels->GetSelectingItems()->at(i).Value;
            }
        }
        else
        {
            // 規定数よりも選択している数が多い場合は全チャネル検索する
            pChCount = 0;
        }
    }

    virtual void Set()
    {
        target->SetIsAfEnabled(m_IsAfEnabled);
        ActionFrameParams afParam;
        afParam.payloadSize = m_AfPayloadSize;
        afParam.count = m_TxCount;
        afParam.waitTime = nn::TimeSpan::FromMilliSeconds(m_InitialTxWaitTime);
        afParam.maxRandomWaitTime = m_InitialMaxRandomWaitTime;
        afParam.interval = nn::TimeSpan::FromMilliSeconds(m_TxInterval);
        afParam.payloadSize = m_AfPayloadSize;
        SetChannelList(afParam.channelList, &afParam.channelCount, &m_AfChannel);
        target->SetAfParam(afParam);

        target->SetIsScanEnabled(m_IsScanEnabled);
        target->SetBufferSize( m_BufferSize );
        target->SetEnableActiveScan( m_ScanType == nn::wlan::ScanType_Active ? true : false );
        int16_t chList[nn::wlan::WirelessChannelsCountMax] = {0};
        uint8_t chCount = 0;
        SetChannelList(chList, &chCount, &m_Channel);
        target->SetChannel( chList, chCount );
        target->SetScanTime( m_ScanTime );
        target->SetHomeChannelTime( m_HomeChTime );
        target->SetBssid( m_Bssid );

        string ssidStr = m_Ssid.GetValue().c_str();
        if( ssidStr == "Wildcard" )
        {
            target->SetSsidList(nullptr, 0);
        }
        else
        {
            nn::wlan::Ssid ssid = nn::wlan::Ssid(ssidStr.c_str());
            target->SetSsid(ssid);
        }

        target->ClearStatistics();
    }

    void Start()
    {
        static NN_ALIGNAS(4096) char s_ThreadStack[4096];

        if( !m_IsScanning )
        {
            if( m_IsThreadRunning )
            {
                nn::os::DestroyThread(&m_Thread);
                m_IsThreadRunning = false;
            }

            m_ExitThread = false;
            m_IsScanning = true;
            m_IsThreadRunning = true;

            nn::Result result = nn::os::CreateThread(&m_Thread, ThreadFunc, this, s_ThreadStack, sizeof(s_ThreadStack), SEARCH_THREAD_PRIORITY);
            if( result.IsFailure() )
            {
                NN_LOG("ScanThread can not run.\n");
                NN_ASSERT(false);
            }

            nn::os::SetThreadNamePointer(&m_Thread, "WitScanThread");
            nn::os::StartThread(&m_Thread);
        }
    }

    void Stop()
    {
        if( m_IsScanning )
        {
            m_ExitThread = true;
            target->StopScan();
            nn::os::DestroyThread(&m_Thread);
            m_IsThreadRunning = false;
            m_IsScanning = false;
        }
    }

    /*---------------------------------------------------------------------------
      　　　　　コンストラクタ類
      ---------------------------------------------------------------------------*/
public:

    AfScannerController()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t w = Display::GetInstance().GetFixedWidth();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        m_IsScanning      = false;
        m_IsThreadRunning = false;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Search Parameters";
        title.FitSize();

        Add(m_Selector);
        m_Selector.X = x;
        m_Selector.Y = y;
        m_Selector.Height = DISPLAY_CONTENT_HEIGHT;

        NamedValue<bool> t("Enable", true);
        NamedValue<bool> f("Disable", false);
        NamedValue<int16_t> chAll("All", 0);

        // Action Frame
        m_Selector.Register("AF Transmission", &m_IsAfEnabled);
        m_IsAfEnabled.Add(t);
        m_IsAfEnabled.Add(f, true);

        m_Selector.Register("AF Channel", &m_AfChannel);
        m_AfChannel.SetSelectableItemCountMax(100);
        m_AfChannel.Add(chAll);
        GenerateSelectableChannel(&m_AfChannel);

        // (SIGLO-70910)
        // PutActionFrameOneShot() にセットする最大ペイロードサイズを
        // WLAN FW の仕様に規定される 1640byte に合わせる
        m_Selector.Register("AF Payload Size [byte]", &m_AfPayloadSize);
        for(int i=1; i<=32; ++i)
        {
            auto size = i * 50;
            m_AfPayloadSize.Add(size, size == 1500);
        }
        m_AfPayloadSize.Add(1640);

        m_Selector.Register("AF Tx Count", &m_TxCount);
        for(int i=1; i<=20; ++i)
        {
            m_TxCount.Add(i, i == 8);
        }

        m_Selector.Register("AF Initial Wait [ms]", &m_InitialTxWaitTime);
        for(int i=0; i<20; ++i)
        {
            auto time = i * 50;
            m_InitialTxWaitTime.Add(time, time == 100);
        }

        m_Selector.Register("AF Max Random Wait [ms]", &m_InitialMaxRandomWaitTime);
        for(int i=0; i<20; ++i)
        {
            auto time = i * 50;
            m_InitialMaxRandomWaitTime.Add(time, time == 50);
        }

        m_Selector.Register("AF Tx Interval [ms]", &m_TxInterval);
        for(int i=0; i<=50; ++i)
        {
            auto interval = i * 10;
            m_TxInterval.Add(interval, interval == 100);
        }

        // Search
        m_Selector.Register("Scan", &m_IsScanEnabled);
        m_IsScanEnabled.Add(t);
        m_IsScanEnabled.Add(f, true);

        // m_Selector.Register("Buffer Size [byte]", &m_BufferSize);
        m_BufferSize.Add(100 * 1024, true);

        m_Selector.Register("Type", &m_ScanType);
        NamedValue<nn::wlan::ScanType> active("Active Scan", nn::wlan::ScanType_Active);
        NamedValue<nn::wlan::ScanType> passive("Passive Scan", nn::wlan::ScanType_Passive);
        m_ScanType.Add(active, true);
        m_ScanType.Add(passive);

        m_Selector.Register("Channel", &m_Channel);
        m_Channel.SetSelectableItemCountMax(100);
        m_Channel.Add(chAll);
        GenerateSelectableChannel(&m_Channel);

        m_Selector.Register("Scan Time [ms]", &m_ScanTime);
        m_Selector.Register("Home Ch Time [ms]", &m_HomeChTime);
        NamedValue<int> defaultTime("Default", -1);
        m_ScanTime.Add(defaultTime);
        m_HomeChTime.Add(defaultTime);
        for(int i=0; i<=500; i+=10)
        {
            ostringstream oss;
            oss << i;
            NamedValue<int> time(oss.str(), i);
            m_ScanTime.Add(time, time == 110);
            m_HomeChTime.Add(time, time == 0);
        }
        for(int i=1000; i<=5000; i+=1000)
        {
            ostringstream oss;
            oss << i;
            NamedValue<int> time(oss.str(), i);
            m_ScanTime.Add(time);
        }

        m_Selector.Register("SSID", &m_Ssid);
        m_Ssid.Add("Wildcard", true);
        GenerateSelectableSsid(&m_Ssid, "");

        m_Selector.Register("BSSID", &m_Bssid);
        NamedValue<nn::wlan::MacAddress> zeroMac("Zero", nn::wlan::MacAddress::CreateZeroMacAddress());
        NamedValue<nn::wlan::MacAddress> wildcardMac("Wildcard", nn::wlan::MacAddress::CreateBroadcastMacAddress());
        NamedValue<nn::wlan::MacAddress> specificMac1("40:d2:8a:b4:93:7e", nn::wlan::MacAddress(0x40, 0xd2, 0x8a, 0xb4, 0x93, 0x7e));
        m_Bssid.Add(zeroMac);
        m_Bssid.Add(wildcardMac, true);
        m_Bssid.Add(specificMac1);

        m_Selector.Register("Scan Initial Wait [ms]", &m_InitialScanWaitTime);
        for(int i=0; i<50; ++i)
        {
            m_InitialScanWaitTime.Add(i * 10, i * 10 == 100);
        }
        for(int i=0; i<5; ++i)
        {
            m_InitialScanWaitTime.Add(500 + i * 100);
        }
        for(int i=0; i<=5; ++i)
        {
            m_InitialScanWaitTime.Add(1000 + i * 1000);
        }

    } //NOLINT(impl/function_size)

    virtual ~AfScannerController()
    {
        if( m_IsThreadRunning )
        {
            target->StopScan();
            m_ExitThread = true;
            nn::os::DestroyThread(&m_Thread);
            m_IsThreadRunning = false;
        }
    }

private:

};

class DetectorController : public SceneController<Detector>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    bool m_Active;
    ValueSelector m_Selector;

private:

    Label title;
    SelectableValue<NamedValue<int16_t>> m_Channel;
    SelectableValue<NamedValue<Detector::TxMode>>   m_TxMode;
    SelectableValue<NamedValue<uint32_t>> m_TxInterval;
    SelectableValue<NamedValue<size_t>>   m_PayloadSize;
    //SelectableValue<NamedValue<nn::wlan::DetectHash>>   m_Hash;
    SelectableValue<NamedValue<DetectorHash>>   m_Hash;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/

public:
    virtual void Set()
    {
        NN_ASSERT(target != nullptr);
        target->SetTxMode(m_TxMode);
        target->SetChannel(m_Channel);
        target->SetTxInterval(m_TxInterval);
        target->SetPayloadSize(m_PayloadSize);
        target->SetHash(m_Hash);
        target->SetIsAfThreadEnabled(true);
    }

    virtual void InputPad(Pad& pad)
    {
    }

    virtual void Start()
    {
        if( !m_Active )
        {
            Set();
            nn::Result result;
            result = target->StartReceiveActionFrame(RECEIVE_ACTION_FRAME_THREAD_PRIORITY);
            if( result.IsFailure() )
            {
                NN_ASSERT(false);
                return;
            }

            m_Active = true;
        }
    }

    void Stop()
    {
        if( m_Active )
        {
            target->StopReceiveActionFrame();
            m_Active = false;
        }
    }

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    DetectorController()
    {
        Initialize();
    }

    virtual void Initialize()
    {
        m_Active = false;

        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Aloe Parameters";
        title.FitSize();

        Add(m_Selector);
        m_Selector.X = x;
        m_Selector.Y = y;
        m_Selector.Height = DISPLAY_CONTENT_HEIGHT;

        m_Selector.Register("Channel", &m_Channel);
        GenerateSelectableChannel(&m_Channel);

        m_Selector.Register("Tx Mode", &m_TxMode);
        m_TxMode.Add(NamedValue<Detector::TxMode>("Periodic API (for system test w/o PLR)", Detector::TxMode_Periodic));
        m_TxMode.Add(NamedValue<Detector::TxMode>("OneShot API (for PLR measurement)", Detector::TxMode_OneShot));
        m_TxMode.Add(NamedValue<Detector::TxMode>("No Tx", Detector::TxMode_Disabled));

        m_Selector.Register("Tx Interval", &m_TxInterval);
        for(int i=0; i<10; ++i)
        {
            ostringstream oss;
            auto interval = i * 10;
            oss << interval << " ms";


            m_TxInterval.Add(NamedValue<uint32_t>(oss.str(), interval));
        }
        uint32_t initialValue[] = {90, 100, 120, 150};
        for(int i=0; i<sizeof(initialValue) / sizeof(initialValue[0]); ++i)
        {
            for(int j=0; j<10; ++j)
            {
                ostringstream oss;
                auto interval = initialValue[i] + j;
                oss << interval << " ms";
                m_TxInterval.Add(NamedValue<uint32_t>(oss.str(), interval), interval == 127);
            }
        }
        for(int i=0; i<=16; ++i)
        {
            ostringstream oss;
            auto interval = 200 + i * 50;
            oss << interval << " ms";
            m_TxInterval.Add(NamedValue<uint32_t>(oss.str(), interval));
        }

        m_Selector.Register("Payload Size", &m_PayloadSize);
        for(int i=1; i<=26; ++i)
        {
            ostringstream oss;
            auto size = i * 50;
            oss << size << " byte";
            m_PayloadSize.Add(NamedValue<size_t>(oss.str(), size), size == 1300);
        }

        m_Selector.Register("Hash", &m_Hash);
        m_Hash.Add(NamedValue<DetectorHash>("0x0123456789abcdef", DetectorHash(DetectorHash::HashMode_Fixed, {{0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef}})));
        m_Hash.Add(NamedValue<DetectorHash>("Incremental Hash (OneShot only)", DetectorHash(DetectorHash::HashMode_Incremental, {{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}})));
        m_Hash.Add(NamedValue<DetectorHash>("Random Hash (OneShot only)", DetectorHash(DetectorHash::HashMode_Random, {{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}})));

        m_Hash.Add(NamedValue<DetectorHash>("0x0000000000000000", DetectorHash(DetectorHash::HashMode_Fixed, {{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x1111111111111111", DetectorHash(DetectorHash::HashMode_Fixed, {{0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x2222222222222222", DetectorHash(DetectorHash::HashMode_Fixed, {{0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x3333333333333333", DetectorHash(DetectorHash::HashMode_Fixed, {{0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x4444444444444444", DetectorHash(DetectorHash::HashMode_Fixed, {{0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x5555555555555555", DetectorHash(DetectorHash::HashMode_Fixed, {{0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x6666666666666666", DetectorHash(DetectorHash::HashMode_Fixed, {{0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x7777777777777777", DetectorHash(DetectorHash::HashMode_Fixed, {{0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x8888888888888888", DetectorHash(DetectorHash::HashMode_Fixed, {{0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x9999999999999999", DetectorHash(DetectorHash::HashMode_Fixed, {{0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99}})));
        m_Hash.Add(NamedValue<DetectorHash>("0xaaaaaaaaaaaaaaaa", DetectorHash(DetectorHash::HashMode_Fixed, {{0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa}})));
        m_Hash.Add(NamedValue<DetectorHash>("0xbbbbbbbbbbbbbbbb", DetectorHash(DetectorHash::HashMode_Fixed, {{0xbb, 0xbb, 0xbb, 0xbb, 0xbb, 0xbb, 0xbb, 0xbb}})));
        m_Hash.Add(NamedValue<DetectorHash>("0xcccccccccccccccc", DetectorHash(DetectorHash::HashMode_Fixed, {{0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc}})));
        m_Hash.Add(NamedValue<DetectorHash>("0xdddddddddddddddd", DetectorHash(DetectorHash::HashMode_Fixed, {{0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd}})));
        m_Hash.Add(NamedValue<DetectorHash>("0xeeeeeeeeeeeeeeee", DetectorHash(DetectorHash::HashMode_Fixed, {{0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee}})));
        m_Hash.Add(NamedValue<DetectorHash>("0xffffffffffffffff", DetectorHash(DetectorHash::HashMode_Fixed, {{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x0101010101010101", DetectorHash(DetectorHash::HashMode_Fixed, {{0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x0202020202020202", DetectorHash(DetectorHash::HashMode_Fixed, {{0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x0303030303030303", DetectorHash(DetectorHash::HashMode_Fixed, {{0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x0404040404040404", DetectorHash(DetectorHash::HashMode_Fixed, {{0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04}})));
        m_Hash.Add(NamedValue<DetectorHash>("0x0505050505050505", DetectorHash(DetectorHash::HashMode_Fixed, {{0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05}})));
    }

    void StartPeriodicActionFrame()
    {
        NN_ASSERT(target != nullptr);
        target->StartPeriodicActionFrame();
    }

    void CancelPeriodicActionFrame()
    {
        NN_ASSERT(target != nullptr);
        target->CancelPeriodicActionFrame();
    }

    bool IsPeriodicEnabled()
    {
        return target->IsPeriodicEnabled();
    }

    bool IsOneShotEnabled()
    {
        return target->IsOneShotEnabled();
    }

private:

};


}        // WlanTest
