﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <string>
#include <algorithm>

#include <nn/os/os_Mutex.h>

#include "LocalNode.h"
#include "Util.h"

namespace WlanTest {


/*!--------------------------------------------------------------------------*
  @brief        SAP Masterクラス
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
           MasterParam
---------------------------------------------------------------------------*/
class MasterParam
{
public :

    enum ItemId
    {
        ITEM_ID_UNDEFINED = 0,

        ITEM_ID_NUM_OF_CLIENTS,
        ITEM_ID_CHANNEL,
        ITEM_ID_SSID,
        ITEM_ID_SECURITY_MODE,
        ITEM_ID_SECURITY_KEY,
        ITEM_ID_HIDDEN_SSID,
        ITEM_ID_INACTIVE_PERIOD
    };

    static const char* ITEM_STR_NUM_OF_CLIENTS;
    static const char* ITEM_STR_CHANNEL;
    static const char* ITEM_STR_SSID;
    static const char* ITEM_STR_SECURITY_MODE;
    static const char* ITEM_STR_SECURITY_KEY;
    static const char* ITEM_STR_HIDDEN_SSID;
    static const char* ITEM_STR_INACTIVE_PERIOD;

    uint32_t           numOfClients;
    int                channel;
    char               ssid[nn::wlan::Ssid::SsidLengthMax];
    nn::wlan::Security security;
    bool               hiddenSsid;
    uint32_t           inactivePeriod;

    MasterParam()
    {
        numOfClients                = 1;
        channel                     = 1;
        security.privacyMode        = nn::wlan::SecurityMode_Open;
        security.groupPrivacyMode   = nn::wlan::SecurityMode_Open;
        hiddenSsid                  = false;
        inactivePeriod              = 10;

        nn::util::Strlcpy(ssid, "wireless_test_1", sizeof(ssid));
        std::memset(security.key, 0x00, sizeof(security.key));
    }

    void Print()
    {
        NN_LOG("  Num of clients           : %u\n", numOfClients);
        NN_LOG("  Channel                  : %d\n", channel);
        NN_LOG("  Ssid                     : %s\n", ssid);
        NN_LOG("  Security mode            : %s\n", ToString(security.privacyMode).c_str());
        NN_LOG("  Security key             : %s\n", security.key);
        NN_LOG("  Stealth mode             : %s\n", ToStringYesNo(hiddenSsid).c_str());
        NN_LOG("  Inactive period          : %u\n", inactivePeriod);
    }

    ItemId GetItemId(const string& item)
    {
        if(strcmp(item.c_str(), ITEM_STR_NUM_OF_CLIENTS) == 0)  return ITEM_ID_NUM_OF_CLIENTS;
        if(strcmp(item.c_str(), ITEM_STR_CHANNEL) == 0)         return ITEM_ID_CHANNEL;
        if(strcmp(item.c_str(), ITEM_STR_SSID) == 0)            return ITEM_ID_SSID;
        if(strcmp(item.c_str(), ITEM_STR_SECURITY_MODE) == 0)   return ITEM_ID_SECURITY_MODE;
        if(strcmp(item.c_str(), ITEM_STR_SECURITY_KEY) == 0)    return ITEM_ID_SECURITY_KEY;
        if(strcmp(item.c_str(), ITEM_STR_HIDDEN_SSID) == 0)     return ITEM_ID_HIDDEN_SSID;
        if(strcmp(item.c_str(), ITEM_STR_INACTIVE_PERIOD) == 0) return ITEM_ID_INACTIVE_PERIOD;

        NN_LOG("  - failed : Unrecognized item name (%s)\n", item.c_str());

        return ITEM_ID_UNDEFINED;
    }

    bool SetParam(const string& item, const string& value)
    {
        ItemId id = GetItemId(item);
        if(id == ITEM_ID_UNDEFINED)
        {
            return false;
        }

        switch(id)
        {
        case ITEM_ID_NUM_OF_CLIENTS  : return SetNumOfClients(value);
        case ITEM_ID_CHANNEL         : return SetChannel(value);
        case ITEM_ID_SSID            : return SetSsid(value);
        case ITEM_ID_SECURITY_MODE   : return SetSecurityMode(value);
        case ITEM_ID_SECURITY_KEY    : return SetSecurityKey(value);
        case ITEM_ID_HIDDEN_SSID     : return SetHiddenSsid(value);
        case ITEM_ID_INACTIVE_PERIOD : return SetInactivePeriod(value);
        default : return false;
        }

        return false;
    }

    bool SetNumOfClients(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%u", &numOfClients) == 1)
        {
            if(numOfClients <= 7)
            {
                return true;
            }
            else
            {
                NN_LOG("  - failed : Out of range (%u)\n", numOfClients);
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    bool SetChannel(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%d", &channel) == 1)
        {
            if((1 <= channel && channel <= 13) ||
               (36 <= channel && channel % 4 == 0) ||
               (channel == 60 || channel == 64) ||
               (100 <= channel && channel <= 144 && channel % 4 == 0) ||
               (149 <= channel && channel <= 165 && channel % 4 == 1))
            {
                return true;
            }
            else
            {
                NN_LOG("  - failed : Out of range (%u)\n", channel);
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    bool SetSsid(const string& valueStr)
    {
        if(valueStr.size() > sizeof(ssid))
        {
            NN_LOG("  - failed : Out of range (%s)\n", valueStr.c_str());
            return false;
        }

        nn::util::Strlcpy(ssid, valueStr.c_str(), sizeof(ssid));

        return true;
    }

    bool SetSecurityMode(const string& valueStr)
    {
        nn::wlan::SecurityMode mode;
        string str = ToUpper(valueStr);

        if(str == "OPEN") mode = nn::wlan::SecurityMode_Open;
        else if(str == "WPA2AES") mode = nn::wlan::SecurityMode_Wpa2Aes;
        else if(str == "STATIC_AES") mode = nn::wlan::SecurityMode_StaticAes;
        else return false;

        security.privacyMode = mode;
        security.groupPrivacyMode = mode;

        return true;
    }

    bool SetSecurityKey(const string& valueStr)
    {
        if(sizeof(security.key) >= valueStr.size())
        {
            std::memcpy(security.key, valueStr.c_str(), valueStr.size());
        }
        else
        {
            NN_LOG("  - failed : Out of range (%s)\n", valueStr.c_str());
            return false;
        }

        return true;
    }

    bool SetHiddenSsid(const string& valueStr)
    {
        uint32_t val;
        if(sscanf(valueStr.c_str(), "%u", &val) == 1)
        {
            if(val == 0 || val == 1)
            {
                hiddenSsid = val;
                return true;
            }
            else
            {
                NN_LOG("  - failed : Out of range (%u)\n", hiddenSsid);
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    bool SetInactivePeriod(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%u", &inactivePeriod) == 1)
        {
            if(inactivePeriod <= 180)
            {
                return true;
            }
            else
            {
                NN_LOG("  - failed : Out of range (%u)\n", inactivePeriod);
                return false;
            }
        }
        else
        {
            return false;
        }
    }

};

class Master : public LocalNode
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    nn::wlan::MasterBssParameters      m_Param;
    uint16_t         m_Ctw;
    //SapTxQueueParam  m_TxQueueParam[2];

    nn::wlan::ClientStatus     m_ClientStatus[nn::wlan::ConnectableClientsCountMax];
    nn::wlan::MacAddress       m_ClientList[nn::wlan::ConnectableClientsCountMax];
    nn::Bit32        m_Updated;
    //nn::os::Event    m_BeaconSentEvent;
    uint8_t          m_NumOfDeauth;
    uint8_t          m_NumOfClients;

    bool             m_respondWildcard;
    uint8_t          m_respondOui[3];

    bool m_IsWaiting;
    uint32_t m_ActionFrameSize;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual nn::Result Initialize();
    virtual nn::Result Finalize();
    virtual nn::Result Open();
    virtual nn::Result Close();
    virtual nn::Result Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize);
    virtual void MaintainConnection();
    virtual nn::Result Disconnect( const nn::wlan::DisconnectClient* pClient );
    virtual nn::Result Disconnect();

    virtual nn::Result StartReceiveCommand(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()))
    { return nn::ResultSuccess(); }
    virtual nn::Result StopReceiveCommand()
    { return nn::ResultSuccess(); }

    virtual bool IsConnected();

protected:

    virtual void SetActionFrame();

private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    Master();
    NN_IMPLICIT Master(nn::wlan::MasterBssParameters param);
    virtual ~Master();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    virtual void GetMacAddressListInBss(nn::wlan::MacAddress list[nn::wlan::ConnectableClientsCountMax], uint32_t& num)
    {
        uint32_t count = 0;

        //for(int i=0; i<num; i++)
        m_Cs.Lock();
        for(int i = 0; i<sizeof(m_ClientList) / sizeof(m_ClientList[0]); i++)
        {
            if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Connected)
            {
                list[count] = m_ClientList[i];
                count++;
            }
        }
        m_Cs.Unlock();
        num = count;

        return;
    }

    virtual void SetConnectableCount(uint32_t num)
    {
        m_NumOfClients = num;
    }

    virtual uint32_t GetConnectableCount()
    {
        return m_NumOfClients;
    }

    virtual bool GetRssi(vector<Rssi>* pRssiList)
    {
        NN_ASSERT(pRssiList);
        pRssiList->empty();

        nn::os::Tick tick = nn::os::GetSystemTick();
        for(const auto& status : m_ClientStatus)
        {
            if( status.state == nn::wlan::ConnectionState_Connected )
            {
                Rssi rssi;
                rssi.value = status.rssi;
                rssi.address = status.clientMacAddress;
                rssi.recordTime = tick;
                pRssiList->push_back(rssi);
            }
        }

        return true;
    }

    virtual int16_t GetChannel()
    {
        return m_Param.channel;
    }

    uint32_t GetConnectedCount()
    {
        uint32_t count = 0;

        m_Cs.Lock();
        for(int i = 0; i<sizeof(m_ClientStatus) / sizeof(m_ClientStatus[0]); i++)
        {
            if(m_ClientStatus[i].state == nn::wlan::ConnectionState_Connected)
            {
                count++;
            }
        }
        m_Cs.Unlock();

        return count;
    }

    bool IsConnectableCountConnected()
    {
        if( GetConnectableCount() == ConnectableClientCount_Any )
        {
            return true;
        }

        return GetConnectableCount() == GetConnectedCount();
    }

    void SetSsid(std::string str)
    {
        nn::wlan::Ssid ssid(str.c_str());
        m_Param.ssid = ssid;
    }

    void SetChannel(int ch)
    {
        m_Param.channel = ch;
    }

    nn::wlan::MasterBssParameters& GetMasterBssParameters()
    {
        return m_Param;
    }

    void SetCtw(uint16_t w)
    {
        m_Ctw = w;
    }

    void SetActionFrameSize(const uint32_t& size)
    {
        m_ActionFrameSize = size;
    }

    bool IsActionFrameEnabled()
    {
        return m_ActionFrameSize != 0;
    }

    // SapTxQueueParam GetAccessParam(SapTxQueueType type)
    // {
    //     return m_TxQueueParam[type];
    // }

    // void SetAccessParam(SapTxQueueParam param)
    // {
    //     m_TxQueueParam[param.queueType] = param;
    // }

    virtual nn::wlan::MacAddress GetApAddress()
    {
        return GetMacAddress();
    }

    void SetResponseParam(bool respondWildcard, uint8_t oui[3])
    {
        m_respondWildcard = respondWildcard;
        memcpy(m_respondOui, oui, 3);
    }

    void SetNumOfDeauth(uint8_t num)
    {
        m_NumOfDeauth = num;
    }
};


}
