﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <string>
#include <algorithm>

#include <nn/os/os_Mutex.h>

#include "LocalNode.h"
#include "Util.h"

namespace WlanTest {

class DetectorHash
{
public:

    static const uint8_t HashSize = sizeof(nn::wlan::DetectHash) / sizeof(uint8_t);

    enum HashMode
    {
        HashMode_Fixed = 0x0,
        HashMode_Incremental,
        HashMode_Random
    };

protected:

    HashMode m_HashMode;
    nn::wlan::DetectHash m_Hash;

public:

    DetectorHash();
    DetectorHash(HashMode mode, nn::wlan::DetectHash hash);

    void Update();

    nn::wlan::DetectHash GetHash() const;
    void SetHash(const nn::wlan::DetectHash& hash);
    DetectorHash::HashMode GetMode() const;
    void SetMode(const HashMode& mode);

    bool operator==(const DetectorHash& hash) const;
    bool operator!=(const DetectorHash& hash) const;

protected:
private:

};

/*!--------------------------------------------------------------------------*
  @brief        Detectorクラス
 *---------------------------------------------------------------------------*/

class Detector : public LocalNode
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:

    enum TxMode
    {
        TxMode_Disabled = 0x00,
        TxMode_OneShot,
        TxMode_Periodic
    };

protected:

    TxMode m_TxMode;
    int16_t  m_Channel;
    uint32_t m_TxInterval;
    size_t   m_PayloadSize;
    DetectorHash m_Hash;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual nn::Result Initialize();
    virtual nn::Result Finalize();
    virtual nn::Result Open();
    virtual nn::Result Close();
    virtual nn::Result Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize);
    virtual nn::wlan::MacAddress GetMacAddress();

    void StartPeriodicActionFrame();
    void CancelPeriodicActionFrame();

protected:

    virtual nn::Result WlanGetActionFrame(nn::wlan::MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId, uint16_t* pChannel, int16_t* pRssi);
    virtual nn::Result WlanPutActionFrame(const nn::wlan::MacAddress& dstMac, const uint8_t* pData, size_t size, int16_t channel, uint32_t dwellTime);

    virtual nn::Result WlanCancelGetFrame(uint32_t rxId);
    virtual nn::Result WlanCancelGetActionFrame(uint32_t rxId);

    virtual nn::Result CreateRxEntry();
    virtual nn::Result DeleteRxEntry();

private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    Detector();
    virtual ~Detector();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    virtual bool IsAfRxReady()
    {
        return true;
    }

    void SetTxMode(const TxMode& txMode)
    {
        m_TxMode = txMode;
    }

    bool IsOneShotEnabled()
    {
        return m_TxMode == TxMode_OneShot;
    }

    bool IsPeriodicEnabled()
    {
        return m_TxMode == TxMode_Periodic;
    }

    void SetChannel(const int16_t& channel)
    {
        NN_ASSERT(channel > 0)
        m_Channel = channel;
    }

    virtual int16_t GetChannel()
    {
        return m_Channel;
    }

    void SetTxInterval(const uint32_t& interval)
    {
        m_TxInterval = interval;
    }

    uint32_t GetTxInterval()
    {
        return m_TxInterval;
    }

    void SetPayloadSize(const size_t& size)
    {
        m_PayloadSize = size;
    }

    size_t GetPayloadSize()
    {
        return m_PayloadSize;
    }

    void SetHash(const DetectorHash& hash)
    {
        m_Hash = hash;
    }

    DetectorHash GetHash()
    {
        return m_Hash;
    }

};


}
