﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include "LocalNode.h"
#include "Scanner.h"
#include "Util.h"

namespace WlanTest {

/*!--------------------------------------------------------------------------*
  @brief        SAP Clientクラス
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
           ClientParam
---------------------------------------------------------------------------*/
struct ClientParam
{
public :

    enum ItemId
    {
        ITEM_ID_UNDEFINED = 0,

        ITEM_ID_CHANNEL,
        ITEM_ID_SSID,
        ITEM_ID_SECURITY_MODE,
        ITEM_ID_SECURITY_KEY,
        ITEM_ID_BEACON_LOST_TIMEOUT
    };

    static const char* ITEM_STR_CHANNEL;
    static const char* ITEM_STR_SSID;
    static const char* ITEM_STR_SECURITY_MODE;
    static const char* ITEM_STR_SECURITY_KEY;
    static const char* ITEM_STR_BEACON_LOST_TIMEOUT;

    int                        channel;
    char               ssid[nn::wlan::Ssid::SsidLengthMax];
    nn::wlan::Security           security;
    int                        beaconLostTimeout;

    ClientParam()
    {
        channel                   = 1;
        security.privacyMode      = nn::wlan::SecurityMode_Open;
        security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
        beaconLostTimeout         = 10;

        nn::util::Strlcpy(ssid, "wireless_test_1", sizeof(ssid));
        std::memset(security.key, 0x00, sizeof(security.key));
    }

    void Print()
    {
        NN_LOG("  Channel                  : %d\n", channel);
        NN_LOG("  Ssid                     : %s\n", ssid);
        NN_LOG("  Security mode            : %s\n", ToString(security.privacyMode).c_str());
        NN_LOG("  Security key             : %s\n", security.key);
        NN_LOG("  Beacon lost timeout      : %d s\n", beaconLostTimeout);
    }

    ItemId GetItemId(const string& item)
    {
        if(strcmp(item.c_str(), ITEM_STR_CHANNEL) == 0)             return ITEM_ID_CHANNEL;
        if(strcmp(item.c_str(), ITEM_STR_SSID) == 0)                return ITEM_ID_SSID;
        if(strcmp(item.c_str(), ITEM_STR_SECURITY_MODE) == 0)       return ITEM_ID_SECURITY_MODE;
        if(strcmp(item.c_str(), ITEM_STR_SECURITY_KEY) == 0)        return ITEM_ID_SECURITY_KEY;
        if(strcmp(item.c_str(), ITEM_STR_BEACON_LOST_TIMEOUT) == 0) return ITEM_ID_BEACON_LOST_TIMEOUT;

        NN_LOG("  - failed : Unrecognized item name (%s)\n", item.c_str());

        return ITEM_ID_UNDEFINED;
    }

    bool SetParam(const string& item, const string& value)
    {
        ItemId id = GetItemId(item);
        if(id == ITEM_ID_UNDEFINED)
        {
            return false;
        }

        switch(id)
        {
        case ITEM_ID_CHANNEL             : return SetChannel(value);
        case ITEM_ID_SSID                : return SetSsid(value);
        case ITEM_ID_SECURITY_MODE       : return SetSecurityMode(value);
        case ITEM_ID_SECURITY_KEY        : return SetSecurityKey(value);
        case ITEM_ID_BEACON_LOST_TIMEOUT : return SetBeaconLostTimeout(value);
        default : return false;
        }

        return false;
    }

    bool SetChannel(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%d", &channel) == 1)
        {
            if((1 <= channel && channel <= 13) ||
               (36 <= channel && channel % 4 == 0) ||
               (channel == 60 || channel == 64) ||
               (100 <= channel && channel <= 144 && channel % 4 == 0) ||
               (149 <= channel && channel <= 165 && channel % 4 == 1))
            {
                return true;
            }
            else
            {
                NN_LOG("  - failed : Out of range (%u)\n", channel);
                return false;
            }

        }
        else
        {
            return false;
        }
    }

    bool SetSsid(const string& valueStr)
    {
        if(valueStr.size() > sizeof(ssid))
        {
            NN_LOG("  - failed : Out of range (%s)\n", valueStr.c_str());
            return false;
        }

        nn::util::Strlcpy(ssid, valueStr.c_str(), sizeof(ssid));

        return true;
    }

    bool SetSecurityMode(const string& valueStr)
    {
        nn::wlan::SecurityMode mode;
        string str = ToUpper(valueStr);

        if(str == "OPEN") mode = nn::wlan::SecurityMode_Open;
        else if(str == "WPA2AES") mode = nn::wlan::SecurityMode_Wpa2Aes;
        else if(str == "STATIC_AES") mode = nn::wlan::SecurityMode_StaticAes;
        else return false;

        security.privacyMode = mode;
        security.groupPrivacyMode = mode;

        return true;
    }

    bool SetSecurityKey(const string& valueStr)
    {
        if(sizeof(security.key) >= valueStr.size())
        {
            std::memcpy(security.key, valueStr.c_str(), valueStr.size());
        }
        else
        {
            NN_LOG("  - failed : Out of range (%s)\n", valueStr.c_str());
            return false;
        }

        return true;
    }

    bool SetBeaconLostTimeout(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%d", &beaconLostTimeout) == 1)
        {
            if(1 <= beaconLostTimeout && beaconLostTimeout <= 30)
            {
                return true;
            }
            else
            {
                NN_LOG("  - failed : Out of range (%u)\n", beaconLostTimeout);
                return false;
            }

        }
        else
        {
            return false;
        }
    }


};

class Client : public LocalNode
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    nn::wlan::Ssid m_Ssid;
    nn::wlan::MacAddress m_Bssid;
    int16_t m_Channel;
    nn::wlan::Security m_Security;
    nn::wlan::BeaconIndication m_Indication;
    bool m_EnableDirectBroadcast;
    bool m_EnableClr;
    int  m_BeaconLostTimeout;
    uint16_t  m_TxWait;
    uint32_t  m_TxStartDelay;

    bool m_IsWaiting;

    // 接続維持用
    nn::wlan::ConnectionStatus m_ConnectionStatus;

    IReceiveEventHandler*        m_InformationElementReceiveHandler;
    nn::os::Mutex    m_Cs;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual nn::Result Initialize();
    virtual nn::Result Finalize();
    virtual nn::Result Open();
    virtual nn::Result Close();
    virtual void MaintainConnection();

    virtual nn::Result StartReceiveCommand(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()))
    { return nn::ResultSuccess(); }
    virtual nn::Result StopReceiveCommand()
    { return nn::ResultSuccess(); }

protected:
private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    Client();
    virtual ~Client();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    virtual void GetMacAddressListInBss(nn::wlan::MacAddress list[], uint32_t& num)
    {
        if( num >= 1 )
        {
            // ##? wlan プロセスで BSSID のみ初期化されず、前回の BSSID が残っているため state で判断する
            //if( m_ConnectionStatus.bssid != nn::wlan::MacAddress::CreateZeroMacAddress() )
            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                list[0] = nn::wlan::MacAddress(m_ConnectionStatus.bssid);
                num = 1;
            }
            else
            {
                num = 0;
            }
        }
        return;
    }


    virtual uint32_t GetConnectableCount()
    {
        return 1;
    }

    virtual uint32_t GetConnectedCount()
    {
        return (IsConnected() ? 1 : 0);
    }

    virtual bool GetRssi(vector<Rssi>* pRssiList)
    {
        NN_ASSERT(pRssiList);
        pRssiList->empty();
        if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            Rssi rssi;
            nn::Result result = nn::wlan::Local::GetRssi(&rssi.value);
            nn::os::Tick tick = nn::os::GetSystemTick();
            if( result.IsSuccess() )
            {
                rssi.address = GetBssid();
                rssi.recordTime = tick;
                pRssiList->push_back(rssi);
                return true;
            }
        }

        return false;
    }

    virtual nn::wlan::LinkLevel GetLinkLevel()
    {
        if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            nn::wlan::LinkLevel linkLevel;
            nn::Result result = nn::wlan::Local::GetLinkLevel(&linkLevel);
            if( result.IsSuccess() )
            {
                return linkLevel;
            }
        }
        return nn::wlan::LinkLevel_0;
    }

    nn::wlan::Ssid GetSsid()
    {
        return m_Ssid;
    }

    void SetSsid(std::string str)
    {
        nn::wlan::Ssid ssid(str.c_str());
        m_Ssid = ssid;
    }

    nn::wlan::MacAddress GetBssid()
    {
        return m_Bssid;
    }

    void SetBssid(nn::wlan::MacAddress id)
    {
        m_Bssid = id;
    }

    int16_t GetChannel()
    {
        return m_Channel;
    }

    void SetChannel(int ch)
    {
        m_Channel = ch;
    }

    nn::wlan::Security GetSecurity()
    {
        return m_Security;
    }

    void SetSecurity(nn::wlan::Security sec)
    {
        m_Security = sec;
    }

    bool IsEnableBssIndication()
    {
        if( m_Indication == nn::wlan::BeaconIndication_Enable )
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    void SetEnableBssIndication(bool set)
    {
        if( set )
        {
            m_Indication = nn::wlan::BeaconIndication_Enable;
        }
        else
        {
            m_Indication = nn::wlan::BeaconIndication_Disable;
        }
    }

    void SetBeaconLostTimeout(int timeout)
    {
        m_BeaconLostTimeout = timeout;
    }

    void SetEnableDirectBroadcast(bool b)
    {
        m_EnableDirectBroadcast = b;
    }

    void SetEnableClr( bool c )
    {
        m_EnableClr = c;
    }

    void SetTxWait( uint16_t w )
    {
        m_TxWait = w;
    }

    // SapTxQueueParam GetAccessParam(SapTxQueueType type)
    // {
    //     return m_TxQueueParam[type];
    // }

    // void SetAccessParam(SapTxQueueParam param)
    // {
    //     m_TxQueueParam[param.queueType] = param;
    // }

    virtual void ClearStatistics()
    {
        Node::ClearStatistics();
        //m_InformationElementReceiver->GetStatistics().Clear();
    }

    virtual bool IsConnected();

};


}
