﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Hid.h"
#include <nn/hid/hid_DebugPadMap.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <array>
#include <tuple>


namespace
{

std::array<std::tuple<nn::hid::DebugPadButtonSet, nn::hid::NpadButtonSet>, 14> g_PadButtonMapping =
{ {
    std::make_tuple(nn::hid::DebugPadButton::A::Mask     , nn::hid::NpadButton::A::Mask),
    std::make_tuple(nn::hid::DebugPadButton::B::Mask     , nn::hid::NpadButton::B::Mask),
    std::make_tuple(nn::hid::DebugPadButton::X::Mask     , nn::hid::NpadButton::X::Mask),
    std::make_tuple(nn::hid::DebugPadButton::Y::Mask     , nn::hid::NpadButton::Y::Mask),
    std::make_tuple(nn::hid::DebugPadButton::L::Mask     , nn::hid::NpadButton::L::Mask),
    std::make_tuple(nn::hid::DebugPadButton::R::Mask     , nn::hid::NpadButton::R::Mask),
    std::make_tuple(nn::hid::DebugPadButton::ZL::Mask    , nn::hid::NpadButton::ZL::Mask),
    std::make_tuple(nn::hid::DebugPadButton::ZR::Mask    , nn::hid::NpadButton::ZR::Mask),
    std::make_tuple(nn::hid::DebugPadButton::Start::Mask , nn::hid::NpadButton::Plus::Mask),
    std::make_tuple(nn::hid::DebugPadButton::Select::Mask, nn::hid::NpadButton::Minus::Mask),
    std::make_tuple(nn::hid::DebugPadButton::Left::Mask  , nn::hid::NpadButton::Left::Mask),
    std::make_tuple(nn::hid::DebugPadButton::Up::Mask    , nn::hid::NpadButton::Up::Mask),
    std::make_tuple(nn::hid::DebugPadButton::Right::Mask , nn::hid::NpadButton::Right::Mask),
    std::make_tuple(nn::hid::DebugPadButton::Down::Mask  , nn::hid::NpadButton::Down::Mask),
} };

nn::hid::NpadIdType g_NpadIds[] = {
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::Handheld,
};

}


namespace ApConnectivityTest
{

const unsigned int Hid::g_PadRepeatTime = 20;
const unsigned int Hid::g_PadRepeatCycle = 3;
const unsigned int Hid::g_TouchRepeatTime = 40;


// 初期化
void Hid::Initialize()
{
    // DebugPad の初期化
    nn::hid::InitializeDebugPad();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::settings::DebugPadKeyboardMap padMap;
    padMap.buttonLeft = nn::hid::KeyboardKey::A::Index;
    padMap.buttonUp = nn::hid::KeyboardKey::W::Index;
    padMap.buttonRight = nn::hid::KeyboardKey::D::Index;
    padMap.buttonDown = nn::hid::KeyboardKey::S::Index;
    padMap.buttonA = nn::hid::KeyboardKey::G::Index;
    padMap.buttonB = nn::hid::KeyboardKey::F::Index;
    nn::hid::SetDebugPadKeyboardMap(padMap);
#endif

    // Npad の初期化
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleJoyDual::Mask);
    nn::hid::SetSupportedNpadIdType(g_NpadIds, sizeof(g_NpadIds) / sizeof(*g_NpadIds));

    // 前回 Pad 状態を初期化しておく
    nn::hid::GetDebugPadState(&m_LastPadState);

    for (int i = 0; i < sizeof(g_NpadIds) / sizeof(*g_NpadIds); ++i)
    {
        auto styleSet = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

        if (styleSet.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState fullkeyState = {};
            nn::hid::GetNpadState(&fullkeyState, g_NpadIds[i]);
            for (auto& map : g_PadButtonMapping)
            {
                if ((fullkeyState.buttons & std::get<1>(map)).IsAnyOn())
                {
                    m_LastPadState.buttons |= std::get<0>(map);
                }
            }
        }
        if (styleSet.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState handheldState = {};
            nn::hid::GetNpadState(&handheldState, g_NpadIds[i]);
            for (auto& map : g_PadButtonMapping)
            {
                if ((handheldState.buttons & std::get<1>(map)).IsAnyOn())
                {
                    m_LastPadState.buttons |= std::get<0>(map);
                }
            }
        }
        if (styleSet.Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::NpadJoyDualState handheldState = {};
            nn::hid::GetNpadState(&handheldState, g_NpadIds[i]);
            for (auto& map : g_PadButtonMapping)
            {
                if ((handheldState.buttons & std::get<1>(map)).IsAnyOn())
                {
                    m_LastPadState.buttons |= std::get<0>(map);
                }
            }
        }
    }

    nn::hid::InitializeTouchScreen();

    nn::hid::TouchScreenState<nn::hid::TouchStateCountMax> touchState;
    nn::hid::GetTouchScreenState(&touchState);
    for (int i = 0; i < nn::hid::TouchStateCountMax; ++i)
    {
        m_LastTouchState[i].frameCount = 0;
        if (i < touchState.count)
        {
            m_LastTouchState[i].valid = true;
            m_LastTouchState[i].state = touchState.touches[i];
        }
        else
        {
            m_LastTouchState[i].valid = false;
        }
    }
}


// 終了
void Hid::Finalize()
{
}


// 情報更新
void Hid::Refresh()
{
    nn::hid::DebugPadState padState;
    nn::hid::GetDebugPadState(&padState);

    for (int i = 0; i < sizeof(g_NpadIds) / sizeof(*g_NpadIds); ++i)
    {
        auto styleSet = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

        if (styleSet.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState fullkeyState = {};
            nn::hid::GetNpadState(&fullkeyState, g_NpadIds[i]);
            for (auto& map : g_PadButtonMapping)
            {
                if ((fullkeyState.buttons & std::get<1>(map)).IsAnyOn())
                {
                    padState.buttons |= std::get<0>(map);
                }
            }
        }
        if (styleSet.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState handheldState = {};
            nn::hid::GetNpadState(&handheldState, g_NpadIds[i]);
            for (auto& map : g_PadButtonMapping)
            {
                if ((handheldState.buttons & std::get<1>(map)).IsAnyOn())
                {
                    padState.buttons |= std::get<0>(map);
                }
            }
        }
        if (styleSet.Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::NpadJoyDualState handheldState = {};
            nn::hid::GetNpadState(&handheldState, g_NpadIds[i]);
            for (auto& map : g_PadButtonMapping)
            {
                if ((handheldState.buttons & std::get<1>(map)).IsAnyOn())
                {
                    padState.buttons |= std::get<0>(map);
                }
            }
        }
    }

    // Up, Down, Press の状態を検出
    auto changeButtons = m_LastPadState.buttons ^ padState.buttons;
    m_DownPadButtonSet = ~m_LastPadState.buttons & padState.buttons;
    m_UpPadButtonSet = m_LastPadState.buttons & ~padState.buttons;

    m_PressPadButtonSet.Reset();
    for (int i = 0; i < 14; ++i)
    {
        if (changeButtons.Test(i))
        {
            m_PadStateCounts[i] = 0;
        }
        else
        {
            ++m_PadStateCounts[i];
        }

        m_PressPadButtonSet.Set(i, padState.buttons.Test(i) && (
            !m_PadStateCounts[i] ||
            (m_PadStateCounts[i] >= g_PadRepeatTime && !((m_PadStateCounts[i] - g_PadRepeatTime) % g_PadRepeatCycle))
            ));
    }

    m_LastPadState = padState;

    nn::hid::TouchScreenState<nn::hid::TouchStateCountMax> touchState;
    nn::hid::GetTouchScreenState(&touchState);

    bool continuouslyTouch[nn::hid::TouchStateCountMax] = {};
    for (int lastTouchIndex = 0; lastTouchIndex < nn::hid::TouchStateCountMax; ++lastTouchIndex)
    {
        ++m_LastTouchState[lastTouchIndex].frameCount;
        if (m_LastTouchState[lastTouchIndex].valid)
        {
            bool release = true;
            for (int touchIndex = 0; touchIndex < touchState.count; ++touchIndex)
            {
                if (m_LastTouchState[lastTouchIndex].state.fingerId == touchState.touches[touchIndex].fingerId)
                {
                    m_LastTouchState[lastTouchIndex].state = touchState.touches[touchIndex];
                    release = false;
                    continuouslyTouch[touchIndex] = true;
                }
            }

            if (release)
            {
                m_LastTouchState[lastTouchIndex].valid = 0;
                m_LastTouchState[lastTouchIndex].frameCount = 0;
            }
        }
    }

    for (int touchIndex = 0; touchIndex < touchState.count; ++touchIndex)
    {
        if (!continuouslyTouch[touchIndex])
        {
            for (int lastTouchIndex = 0; lastTouchIndex < nn::hid::TouchStateCountMax; ++lastTouchIndex)
            {
                if (!m_LastTouchState[lastTouchIndex].valid && m_LastTouchState[lastTouchIndex].frameCount)
                {
                    m_LastTouchState[lastTouchIndex].valid = true;
                    m_LastTouchState[lastTouchIndex].frameCount = 0;
                    m_LastTouchState[lastTouchIndex].state = touchState.touches[touchIndex];
                }
            }
        }
    }
} // NOLINT(impl/function_size)


// キー情報取得
const nn::hid::DebugPadButtonSet& Hid::GetKeyDown() const
{
    return m_DownPadButtonSet;
}


const nn::hid::DebugPadButtonSet& Hid::GetKeyUp() const
{
    return m_UpPadButtonSet;
}


const nn::hid::DebugPadButtonSet& Hid::GetKeyPress() const
{
    return m_PressPadButtonSet;
}


const bool Hid::GetTouchDown(nn::hid::TouchState* pOut, size_t index) const
{
    for (int lastTouchIndex = 0; lastTouchIndex < nn::hid::TouchStateCountMax; ++lastTouchIndex)
    {
        if (m_LastTouchState[lastTouchIndex].valid && !m_LastTouchState[lastTouchIndex].frameCount)
        {
            if (index--)
            {
                *pOut = m_LastTouchState[lastTouchIndex].state;
                return true;
            }
        }
    }

    return false;
}


const bool Hid::GetTouchUp(nn::hid::TouchState * pOut, size_t index) const
{
    for (int lastTouchIndex = 0; lastTouchIndex < nn::hid::TouchStateCountMax; ++lastTouchIndex)
    {
        if (!m_LastTouchState[lastTouchIndex].valid && !m_LastTouchState[lastTouchIndex].frameCount)
        {
            if (index--)
            {
                *pOut = m_LastTouchState[lastTouchIndex].state;
                return true;
            }
        }
    }

    return false;
}


const bool Hid::GetTouchPress(nn::hid::TouchState * pOut, size_t index) const
{
    for (int lastTouchIndex = 0; lastTouchIndex < nn::hid::TouchStateCountMax; ++lastTouchIndex)
    {
        if (m_LastTouchState[lastTouchIndex].valid && (
                !m_LastTouchState[lastTouchIndex].frameCount ||
                (m_LastTouchState[lastTouchIndex].frameCount >= g_TouchRepeatTime && !((m_LastTouchState[lastTouchIndex].frameCount - g_TouchRepeatTime) % g_PadRepeatCycle))
                ))
        {
            if (index--)
            {
                *pOut = m_LastTouchState[lastTouchIndex].state;
                return true;
            }
        }
    }

    return false;
}


// コンストラクタ
Hid::Hid() :
        m_PadStateCounts()
{
}


// デストラクタ
Hid::~Hid()
{
}


}
