﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{WebSimple_Main.cpp,PageSampleWebSimple}
 *
 * @brief
 *  nn::web ライブラリを使って Web アプレットを起動し、Web ページを開くシンプルなサンプルプログラム
 */

/**
 * @page PageSampleWebSimple WebSimple
 * @tableofcontents
 *
 * @brief
 *  nn::web ライブラリを使って Web アプレットを起動し、Web ページを開くシンプルなサンプルプログラム
 *
 * @section PageSampleWebSimple_SectionBrief 概要
 *  nn::web ライブラリを使って Web アプレットを起動し、Web ページを開くシンプルなサンプルプログラムです。
 *
 * @section PageSampleWebSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/WebSimple Samples/Sources/Applications/WebSimple @endlink 以下にあります。
 *
 * @section PageSampleWebSimple_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX 実機環境のみサポートしています。
 *
 * @section PageSampleWebSimple_SectionHowToOperate 操作方法
 *  画面上に操作方法が表示されます。
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> A ボタン </td><td> 指定した Web ページを開く </td></tr>
 * <tr><td> B ボタン </td><td> 指定ページに到達したら自動で Web アプレットを終了するように設定して Web ページを開く </td></tr>
 * <tr><td> X ボタン </td><td> 起動時の画面と Web ページの背景を指定して Web ページを開く </td></tr>
 * <tr><td> Y ボタン </td><td> 動画専用起動で Web アプレットを起動して動画コンテンツを再生する </td></tr>
 * <tr><td> + ボタン </td><td> サンプルアプリケーションを終了する </td></tr>
 * </table>
 *
 * @section PageSampleWebSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleWebSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleWebSimple_SectionDetail 解説
 *
 * @subsection PageSampleWebSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  WebSimple_Main.cpp
 *  @includelineno WebSimple_Main.cpp
 *
 * @subsection PageSampleWebSimple_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの主な処理の流れは以下の通りです。
 *  - 描画のための初期化処理と背景色の設定
 *  - メインループ
 *      - キー入力処理の更新
 *      - 文字の表示処理
 *      - 特定のキーの入力があれば、Web アプレットを起動して Web ページを表示
 *
 * @subsection PageSampleWebSimple_SectionAccessibleUrls アクセス可能な URL の指定
 * Web アプレットが開くことができる URL は、accessible-urls.txt という
 * テキストファイルを ROM に付帯情報として含めて指定する必要があります。
 * 本サンプルでは、任天堂の Web サイトと
 * サンプル Web ページが置かれた Web サイトのみを閲覧可能と設定しています。
 */

#include <stdint.h>

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/web.h>

#include "WebSimple_Pad.hpp"
#include "WebSimple_BackgroundDrawer.hpp"

namespace {

const char Title[] = "WebSimple";

SamplePad g_SamplePad;
BackgroundDrawer g_BackgroundDrawer;

}

void PrintUsage() NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter* pWriter = g_BackgroundDrawer.GetTextWriter();
    NN_ASSERT_NOT_NULL(pWriter);
    pWriter->SetCursor(20, 20);
    pWriter->SetFontSize(36);
    pWriter->Print("--------------------------------------------------\n");
    pWriter->Print("%s Sample\n", Title);
    pWriter->Print("--------------------------------------------------\n");
    pWriter->Print("[A]              Show Simple\n");
    pWriter->Print("[B]              Set CallbackUrl\n");
    pWriter->Print("[X]              Set DisplayKind And BackgroundKind\n");
    pWriter->Print("[Y]              Boot As Media Player\n");
    pWriter->Print("[+]              Exit Application\n");
    pWriter->Print("--------------------------------------------------\n");
}

void ShowSimple() NN_NOEXCEPT
{
    static const char* requestUrl = "https://www.nintendo.com/countryselector";
    nn::web::ShowWebPageArg showWebPageArg(requestUrl);
    nn::web::WebPageReturnValue webPageReturnValue;

    const nn::Result result = nn::web::ShowWebPage(&webPageReturnValue, showWebPageArg);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("Exit Reason %d\n", webPageReturnValue.GetWebExitReason());
}

void UseCallbackUrl() NN_NOEXCEPT
{
    // Web アプレット起動時に開く Web ページの URL
    static const char* requestUrl =
        "https://nintendo-sdk-sample.cdn.nintendo.net/page/callback_url_sample/index.html";
    // Callback URL
    static const char* callbackUrl =
        "https://nintendo-sdk-sample.cdn.nintendo.net/page/callback_url_sample/callback_url/index.html";
    // Callback 可能 URL
    static const char* callbackableUrl =
        "https://nintendo-sdk-sample.cdn.nintendo.net/page/callback_url_sample/callbackable_url/index.html";

    nn::web::ShowWebPageArg showWebPageArg(requestUrl);
    // Callback URL を指定した場合は、この URL に Web アプレットが到達するとウェブページの表示を終了します。
    showWebPageArg.SetCallbackUrl(callbackUrl);
    // Callback 可能 URL を指定した場合は、この URL から Callback URL に Web アプレットが到達した時のみウェブページの表示を終了します。
    showWebPageArg.SetCallbackableUrl(callbackableUrl);

    nn::web::WebPageReturnValue webPageReturnValue;
    const nn::Result result = nn::web::ShowWebPage(&webPageReturnValue, showWebPageArg);
    NN_ABORT_UNLESS(result.IsSuccess());

    nn::web::WebExitReason exitReason = webPageReturnValue.GetWebExitReason();
    NN_LOG("Exit Reason %d\n", exitReason);
    if( exitReason == nn::web::WebExitReason_CallbackUrlReached )
    {
        if( webPageReturnValue.GetLastUrlSize() > 0 )
        {
            NN_LOG("LastUrl %s\n", webPageReturnValue.GetLastUrl());
        }
    }
}

void SetBootDisplayAndBackground() NN_NOEXCEPT
{
    static const char* requestUrl = "https://www.nintendo.com/countryselector";
    nn::web::ShowWebPageArg showWebPageArg(requestUrl);
    // 起動時の画面の種類を指定します。
    showWebPageArg.SetBootDisplayKind(nn::web::WebBootDisplayKind_CallerCapture);
    // Web ページの背景を指定します。
    showWebPageArg.SetBackgroundKind(nn::web::WebBackgroundKind_ApplicationCapture);

    nn::web::WebPageReturnValue webPageReturnValue;
    const nn::Result result = nn::web::ShowWebPage(&webPageReturnValue, showWebPageArg);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("Exit Reason %d\n", webPageReturnValue.GetWebExitReason());
}

void BootAsMediaPlayer()
{
    static const char* requestUrl =
        "https://nintendo-sdk-sample.cdn.nintendo.net/movie/sample1.mp4";
    nn::web::ShowWebPageArg showWebPageArg(requestUrl);
    // 動画専用起動を有効にします。Web ページを介さずに直接動画再生を開始します。
    showWebPageArg.SetBootAsMediaPlayer(requestUrl);

    nn::web::WebPageReturnValue webPageReturnValue;
    const nn::Result result = nn::web::ShowWebPage(&webPageReturnValue, showWebPageArg);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("Exit Reason %d\n", webPageReturnValue.GetWebExitReason());
}

bool Update() NN_NOEXCEPT
{
    // シンプルに Web ページを指定して起動
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::A>() )
    {
        ShowSimple();
    }

    // 指定されたページから指定ページに到達したら自動で戻るように指定して起動
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::B>() )
    {
        UseCallbackUrl();
    }

    // 起動時の画面の種類と背景を指定して起動
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::X>() )
    {
        SetBootDisplayAndBackground();
    }

    // 動画専用起動
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::Y>() )
    {
        BootAsMediaPlayer();
    }

    // 終了
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::Start>() )
    {
        return false;
    }
    return true;
}

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain() NN_NOEXCEPT
{
    // 描画のための初期化処理と背景色の設定
    g_BackgroundDrawer.Initialize();
    g_BackgroundDrawer.SetBackgroundColor(125, 125, 125);

    // メインループ
    for(;;)
    {
        // キー入力処理の更新
        g_SamplePad.Update();
        // 文字の表示処理
        PrintUsage();
        g_BackgroundDrawer.Draw();

        // 特定のキーの入力があれば、Web アプレットを使用してページを表示
        if( !Update() )
        {
            break;
        }
    }

    g_BackgroundDrawer.Finalize();
}
