﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>
#include <nn/vi.private.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include "Controller.h"
#include "MenuManager.h"
#include "LumaMenu.h"
#include "Graphics.h"
#include "Color.h"

static void* NvAllocate(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

static void NvFree(void* ptr, void* userPtr)
{
    NN_UNUSED(userPtr);
    std::free(ptr);
}

static void* NvReallocate(void* ptr, size_t size, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::realloc(ptr, size);
}

extern "C" void nnMain()
{
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsServiceName("nvdrv:a");
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::InitializeGraphics( std::malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );

    // This function is a temporary workaround and will be removed at a later date.
    // Eventually, nn::vi::Initialize will handle all use cases.
    // Note nn::vi::Initialize() currently only allows creation of the default display.
    // The other special displays are not available to applications.
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    MenuManager menus;

    {
        std::unique_ptr<IMenu> ptr(new LumaMenu(pDisplay));
        menus.Add(ptr);
    }

    Controller controller;

    menus.Render();
    Graphics g;

    const int LayerWidth = 200;
    const int LayerHeight = 200;

    nn::vi::Layer* pLayer;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay));

    g.RegisterLayer(pLayer, LayerWidth, LayerHeight, Color(.75f, 0.f, 0.f));

    while( true )
    {
        controller.Update();

        menus.Update(controller);

        g.Update();
    }

    nn::vi::Finalize();
}
