﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{VfxViewerSimple/main.cpp,PageSampleVfxViewerSimple}
 *
 * @brief
 *  エフェクト ビューアライブラリを利用したシンプルなサンプルプログラム
 */

/**
 * @page PageSampleVfxViewerSimple vfxビューアライブラリを利用するシンプルなサンプル
 * @tableofcontents
 *
 * @image html Applications\VfxViewerSimple\VfxViewerSimple.png
 *
 * @brief
 *  サンプルプログラム VfxViewerSimple の解説です。
 *
 * @section PageSampleVfxViewerSimple_SectionBrief 概要
 *  nn::vfx::viewer ライブラリを利用してアプリ内でエフェクトビューアを実装するシンプルなサンプルです。
 *  本サンプル起動後に EffectMaker を起動しサンプルに接続することで、
 *  サンプル内で EffectMakerから転送した エフェクトバイナリのプレビューや、
 *  既に、サンプル内に組み込まれているエフェクトバイナリ( Resources\Eset\Fountain.eset )を編集することができます。
 *  また、本サンプルでは、G3dを利用してキャラクタを表示しており、そのキャラクタのボーン情報を EffectMaker に転送し、
 *  EffectMakerから転送したエフェクトをキャラクタに親子付けすることもできます。
 *
 *  nn::vfx::viewer ライブラリの機能の使い方については、
 *  @ref nn::vfx "Vfx の関数リファレンス" やプログラマーズガイドも併せて参照して下さい。
 *
 * @section PageSampleVfxViewerSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/VfxViewerSimple
 *  Samples/Sources/Applications/VfxViewerSimple @endlink 以下にあります。
 *
 * @section PageSampleVfxViewerSimple_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleVfxViewerSimple_SectionHowToOperate 操作方法
 *  このサンプルを実行するとエフェクトが自動的に再生され、一定回数描画した後終了します。
 *
 * @section PageSampleVfxViewerSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleVfxViewerSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleVfxViewerSimple_SectionDetail 解説
 *
 * @subsection PageSampleVfxViewerSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  VfxViewerSimple/main.cpp
 *  @includelineno VfxViewerSimple/main.cpp
 *
 * @subsection PageSampleVfxViewerSimple_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *  - Gfxの初期化
 *  - カメラ・プロジェクション行列の初期化
 *  - VFXランタイム、VFXビューアランタイムの初期化
 *  - VFXバイナリロード
 *  - エフェクト生成
 *  - ループ開始
 *  - VFXランタイムの計算処理
 *  - VFXランタイムの描画処理
 *  - ループ開始に戻る
 *  - VFXランタイム、VFXビューアランタイムの終了処理
 *  - Gfxの終了処理
 *
 */

#include <nn/TargetConfigs/build_Base.h>

#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs.h>
#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nn/vfx.h>
#include <nn/vi.h>
#include <nn/htcs.h>
#include <nn/vfx/viewer/vfx_Viewer.h>
#include <nn/util/util_StringUtil.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include "SimpleModel.h"

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/hws/hws_Message.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

nn::util::Vector3fType          g_CamPosVec = NN_UTIL_VECTOR_3F_INITIALIZER(30, 30, 30); // カメラ位置
nn::util::Vector3fType          g_CamLookAtVec = NN_UTIL_VECTOR_3F_INITIALIZER(0, 0, 0); // カメラ注視点
nn::util::Vector3fType          g_CamUpVec = NN_UTIL_VECTOR_3F_INITIALIZER(0, 1, 0);     // カメラ上方向
nn::util::Matrix4x3fType        g_ViewMtx;                                               // ビュー行列
nn::util::Matrix4x4fType        g_ProjectionMatrix;                                      // 射影変換行列
float                           g_NearZ = 0.1f;                                          // Z Near
float                           g_FarZ = 10000.0f;                                       // Z Far

//------------------------------------------------------------------------------
//  GraphicsFramework
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework::FrameworkInfo g_GraphicsFrameworkInfo;
nns::gfx::GraphicsFramework                g_GraphicsFramework;


//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate(size_t size)
{
    return malloc(size);
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate(void* ptr, size_t size)
{
    NN_UNUSED(size);
    free(ptr);
}

//------------------------------------------------------------------------------
//  nn::fs を利用したファイルロード関数
//------------------------------------------------------------------------------
bool FileLoad(
    void**           pOutBuffer,
    size_t*          pOutBufferSize,
    const char*      filePath,
    nn::vfx::Heap*   pAllocator,
    int              alignment)
{
    char filePathName[nn::fs::EntryNameLengthMax];
    memcpy(reinterpret_cast< void* >(&filePathName), filePath, strlen(filePath) + 1);

    nn::fs::FileHandle fsHandle;
    nn::Result ret = nn::fs::OpenFile(&fsHandle, static_cast< char* >(filePathName), static_cast< nn::fs::OpenMode >(nn::fs::OpenMode_Read));
    if (ret.IsFailure())
    {
        return false;
    }

    int64_t fsSize;
    nn::fs::GetFileSize(&fsSize, fsHandle);
    *pOutBufferSize = static_cast< size_t >(fsSize);

    *pOutBuffer = pAllocator->Alloc(*pOutBufferSize + 4, alignment);
    if (!(*pOutBuffer))
    {
        nn::fs::CloseFile(fsHandle);
        return false;
    }

    memset(*pOutBuffer, 0, *pOutBufferSize + 4);
    nn::fs::ReadFile(fsHandle, 0, *pOutBuffer, *pOutBufferSize);
    nn::fs::CloseFile(fsHandle);

    return true;
}
void*                           g_HeapPtr;              // エフェクトヒープバッファポインタ
void*                           g_ConnectionHeapPtr;    // 通信用ヒープバッファポインタ
nn::vfx::StandardHeap           g_VfxHeap;              // エフェクトヒープ
nn::vfx::StandardHeap           g_ConnectionHeap;       // 通信用ヒープ


//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ確保関数
//!--------------------------------------------------------------------------------------
void* AllocateGpuBuffer(size_t size, size_t alignment, void* pUserData)
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);

    // ユーザーデータポインタでアロケータを渡してもらう.
    nn::vfx::StandardHeap*    pAllocator = static_cast<nn::vfx::StandardHeap*>(pUserData);

    return pAllocator->Alloc(size, alignment);
}

//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ開放関数
//!--------------------------------------------------------------------------------------
void FreeGpuBuffer(void* ptr, void* pUserData)
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);

    // ユーザーデータポインタでアロケータを渡してもらう.
    nn::vfx::StandardHeap*    pAllocator = static_cast<nn::vfx::StandardHeap*>(pUserData);

    pAllocator->Free(ptr);
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 初期化処理
//------------------------------------------------------------------------------
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(AllocateGpuBuffer, &g_VfxHeap);

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity(2);

    // PrimitiveRendererのインスタンス
    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(g_GraphicsFramework.GetDevice(), info);

    g_pPrimitiveRenderer->SetScreenWidth(g_GraphicsFramework.GetDisplayWidth());
    g_pPrimitiveRenderer->SetScreenHeight(g_GraphicsFramework.GetDisplayHeight());
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 破棄処理
//------------------------------------------------------------------------------
void FinalizePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, g_GraphicsFramework.GetDevice(), FreeGpuBuffer, &g_VfxHeap);
}

//------------------------------------------------------------------------------
//  グリッドの描画処理
//------------------------------------------------------------------------------
void GridDraw( nn::gfx::CommandBuffer* commandBuffer, float scaleValue )
{
    g_pPrimitiveRenderer->Update(0);

    // Blend
    commandBuffer->SetBlendState(g_pPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Opacity));

    // Depth Enable
    g_pPrimitiveRenderer->SetDepthStencilState(commandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);

    g_pPrimitiveRenderer->SetProjectionMatrix(&g_ProjectionMatrix);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Vector3fType cameraPos;
    nn::util::VectorMultiply(&cameraPos, g_CamPosVec, scaleValue);

    nn::util::MatrixLookAtRightHanded(&viewMatrix, cameraPos, g_CamLookAtVec, 0.0f);

    g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixIdentity(&modelMatrix);
    const nn::util::Vector3f offset(0, -15, -15);
    nn::util::MatrixSetTranslate(&modelMatrix, offset);
    g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    int cnt = 0;
    for (int i = -5; i < 5; i++)
    {
        for (int j = -5; j < 5; j++)
        {
            nn::util::Vector3fType center = { i * 10.f + 5.f, 0.f, j * 10.f + 5.f };
            nn::util::Vector3fType size = { 10.f, 0.1f, 10.f };

            nn::util::Uint8x4 color = { { 32, 32, 32, 255 } };
            if ((cnt % 2) == 0)
            {
                color.v[0] = 64;
                color.v[1] = 64;
                color.v[2] = 64;
            }
            g_pPrimitiveRenderer->SetColor(color);
            g_pPrimitiveRenderer->DrawCube(commandBuffer,
                nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
            cnt++;
        }
        cnt++;
    }
}


//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForTexture(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(dstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForSampler(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);

    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        int slotIndex = pPool->GetDescriptorSlotIndex(*dstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }
}

//------------------------------------------------------------------------------
//  VfxSimple デモ関連
//------------------------------------------------------------------------------
nn::vfx::System*                g_pVfxSystem        = nullptr;                                      // nn::vfx システム
nn::vfx::viewer::ViewerSystem*  g_pVfxViewerSystem  = nullptr;                                      // nn::vfx::viewer::ViewerSystem システム
void*                           g_StackBase         = nullptr;                                      // packet受信スレッドのスタックポインタ
nn::os::ThreadType              g_Thread;                                                           // packet受信スレッドオブジェクト
SimpleModel                     g_Human;                                                            // Humanモデル
SimpleModel                     g_Torch;                                                            // Torchモデル


//---------------------------------------------------------------------------
//  EffectMakerへのモデル情報転送クラス
//---------------------------------------------------------------------------
class ModelEnumeratorTest : public nn::vfx::viewer::ModelEnumerator
{
    NN_DISALLOW_COPY(ModelEnumeratorTest);
    enum
    {
        ModelMaxCount = 5
    };
public:
    ModelEnumeratorTest() NN_NOEXCEPT : m_ModelCount(0) {}

    virtual ~ModelEnumeratorTest() NN_NOEXCEPT {}

public:
    // モデル名からインデックスを取得する
    int GetIndex(const char* modelName) const
    {
        for ( int i = 0; i < m_ModelCount; i++ )
        {
            if (strcmp(m_Model[i]->GetModelName(), modelName) == 0)
            {
                return i;
            }
        }
        return -1;
    }

    // モデル数を取得する
    int GetModelCount() const
    {
        // g_Human の 1体のみ
        return m_ModelCount;
    }

    // モデル数を取得する
    const char* GetModelName(int index) const
    {
        if ( (index < 0 ) || (index > GetModelCount() ) ) return NULL;
        return m_Model[index]->GetModelName();
    }

    // ボーン名を取得する
    int GetBoneCount(const char* modelName) const
    {
        int index = GetIndex(modelName);
        if ( (index < 0 ) || ( index > GetModelCount() ) ) return 0;
        return m_Model[index]->GetBoneNum();
    }

    // ボーン名を取得する
    virtual const char* GetBoneName(const char* modelName, int boneIndex) const
    {
        int index = GetIndex(modelName);
        if ( (index < 0 ) || ( index > GetModelCount() ) ) return NULL;
        return m_Model[index]->GetBoneName( boneIndex );
    }

    // ボーンマトリクスを取得する
    virtual void GetBoneMatrix(nn::util::Matrix4x3fType* pOutBoneMatrix, const char* modelName, int boneIndex) const
    {
        int index = GetIndex(modelName);
        if ( (index < 0 ) || ( index > GetModelCount() ) ) return;
        *pOutBoneMatrix = *m_Model[index]->GetBoneWorldMatrix(boneIndex);
    }

    // モデルの表示位置を取得する
    void GetModelRootMatrix(nn::util::Matrix4x3fType* pOutRootMatrix, const char* modelName) const
    {
        NN_UNUSED(modelName);
        nn::util::MatrixIdentity( pOutRootMatrix );
    }

    // モデルオブジェクトをセットする
    int SetModel( SimpleModel* model )
    {
        int modelIndex = 0;
        if ( m_ModelCount < ModelMaxCount)
        {
            modelIndex = m_ModelCount;
            m_Model[m_ModelCount++] = model;
        }
        return modelIndex;
    }

    // モデル中のボーン名のIDを探す
    int SearchBoneId( int modelIndex, const char* boneName )
    {
       for ( int i = 0; i < m_Model[modelIndex]->GetBoneNum(); i++ )
        {
            if ( strcmp( m_Model[modelIndex]->GetBoneName( i ), boneName ) == 0 )
            {
                return i;
            }
        }
        return 0;
    }

private:
    SimpleModel*               m_Model[ModelMaxCount];
    int                        m_ModelCount;
};

//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
struct CalculateCallbackUserData
{
    nn::vfx::Handle* pTorchEmitterSetHandle;
    ModelEnumeratorTest* pModelInfo;
    int humanBoneID;
    int torchBoneID;
    nns::gfx::GraphicsFramework::DebugFontTextWriter* pWriter;
    bool* pProcessResult;
};

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    CalculateCallbackUserData* pCallbackUserData = reinterpret_cast<CalculateCallbackUserData*>(pUserData);

    pCallbackUserData->pWriter->object.SetCursor(32, 32);

    //------------------------------------------
    // EffectMaker 接続後にアプリ内モデル情報を転送する
    //------------------------------------------
    static bool bTransfer = false;
    if (g_pVfxViewerSystem->IsConnected() && !bTransfer)
    {
        g_pVfxViewerSystem->UpdateModelInfo(pCallbackUserData->pModelInfo);
        bTransfer = true;
    }

    if (!g_pVfxViewerSystem->IsConnected())
    {
        pCallbackUserData->pWriter->object.Print("Launch EffectMaker.\n");
    }

    //------------------------------------------
    // VFX ビューアランタイムの計算処理
    //------------------------------------------
    *pCallbackUserData->pProcessResult = g_pVfxViewerSystem->ProcessCalculation(1.0f, g_ViewMtx, AllocateDescriptorSlotForTexture, &g_GraphicsFramework);

    float frameCounter = g_pVfxViewerSystem->GetTime();
    pCallbackUserData->pWriter->object.Print("counter : %f\n", frameCounter);

    //------------------------------------------
    // Humanモデルの計算処理
    //------------------------------------------
    nn::util::Vector3fType lightpos = NN_UTIL_VECTOR_3F_INITIALIZER(0, 1, 0);
    nn::util::Vector3fType modelTrans = NN_UTIL_VECTOR_3F_INITIALIZER(20.0f, 0.0f, 0.0f);
    nn::util::Vector3fType modelScale = NN_UTIL_VECTOR_3F_INITIALIZER(0.1f, 0.1f, 0.1f);

    nn::util::Matrix4x3fType drawMatrix;
    nn::util::MatrixIdentity(&drawMatrix);
    nn::util::MatrixSetTranslate(&drawMatrix, modelTrans);
    nn::util::MatrixSetScale(&drawMatrix, modelScale);

    // アニメーションフレームをビューア時間と同期させる

    g_Human.SetAnimationFrame(frameCounter);
    g_Human.SetAnimationIndex(0);
    g_Human.SetDrawMatrix(drawMatrix);
    g_Human.Calc();
    g_Human.CalcBlock(&g_ProjectionMatrix, &g_ViewMtx, &lightpos, 1.0f);

    //------------------------------------------
    // Torchモデルの計算処理
    //------------------------------------------
    nn::util::MatrixIdentity(&drawMatrix);

    // アニメーションフレームをビューア時間と同期させる
    nn::util::VectorSet(&modelScale, 7.0f, 7.0f, 7.0f);
    nn::util::VectorSet(&modelTrans, 0.0f, 0.0f, 0.0f);
    nn::util::Vector3fType modelRotate = NN_UTIL_VECTOR_3F_INITIALIZER(-90.0f, 0.0f, 5.0f);
    const nn::util::Matrix4x3fType *drawMatrixHuman = g_Human.GetBoneWorldMatrix(pCallbackUserData->humanBoneID);

    nn::util::MatrixSetTranslate(&drawMatrix, modelTrans);
    nn::util::MatrixSetScaleRotateXyz(&drawMatrix, modelScale, modelRotate);

    nn::util::MatrixMultiply(&drawMatrix, drawMatrix, *drawMatrixHuman);

    g_Torch.SetAnimationFrame(frameCounter);
    g_Torch.SetAnimationIndex(0);
    g_Torch.SetDrawMatrix(drawMatrix);
    g_Torch.Calc();
    g_Torch.CalcBlock(&g_ProjectionMatrix, &g_ViewMtx, &lightpos, 1.0f);

    //------------------------------------------
    // TorchモデルとTorchFireエミッタセットのコンストレイント処理
    //------------------------------------------
    nn::util::Matrix4x3fType* drawMatrixTorch = (nn::util::Matrix4x3fType*)g_Torch.GetBoneWorldMatrix(pCallbackUserData->torchBoneID);
    nn::util::VectorSet(&modelRotate, 0.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(drawMatrixTorch, modelRotate);

    pCallbackUserData->pTorchEmitterSetHandle->GetEmitterSet()->SetMatrix(*drawMatrixTorch);
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
struct MakeCommandCallbackUserData
{
    int groupID;
    nns::gfx::GraphicsFramework::DebugFontTextWriter* pWriter;
    bool* pProcessResult;
};

void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    MakeCommandCallbackUserData* pCallbackUserData = reinterpret_cast<MakeCommandCallbackUserData*>(pUserData);

    //------------------------------------------
    // VFXランタイムの計算処理
    //------------------------------------------
    g_pVfxSystem->BeginFrame();

    // EffectMakerの再生状況に応じてエフェクトの計算処理を行う
    g_pVfxSystem->Calculate(pCallbackUserData->groupID, 1.0f, true);
    if (!(*pCallbackUserData->pProcessResult))
    {
        g_pVfxSystem->Calculate(nn::vfx::SystemParameters_ViewerGroupId, 1.0f, true);
    }

    nn::gfx::CommandBuffer*    commandBuffer = g_GraphicsFramework.GetRootCommandBuffer(bufferIndex);
    nn::gfx::ColorTargetView*  pColorTargetView = g_GraphicsFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStecilView = g_GraphicsFramework.GetDepthStencilView();

    g_GraphicsFramework.BeginFrame(bufferIndex);
    {
        commandBuffer->ClearColor(pColorTargetView, 0.2f, 0.2f, 0.2f, 0.0f, nullptr);
        commandBuffer->ClearDepthStencil(pDepthStecilView, 1.0f, 0,
            nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        commandBuffer->SetRenderTargets(1, &pColorTargetView, pDepthStecilView);
        commandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());

        // グリッドの描画処理
        GridDraw(commandBuffer, 1.0f);


        pCallbackUserData->pWriter->object.Draw(commandBuffer);

        // Human　モデルの描画処理
        g_Human.Draw(commandBuffer, NULL);

        // Torch　モデルの描画処理
        g_Torch.Draw(commandBuffer, NULL);
        //------------------------------------------
        // VFXランタイムの描画処理
        //------------------------------------------
        int processingIndex = 0;
        nn::vfx::ViewParam view;

        view.Set(g_ViewMtx, g_ProjectionMatrix, g_CamPosVec, g_NearZ, g_FarZ);
        g_pVfxSystem->SwapBuffer();
        g_pVfxSystem->SetViewParam(processingIndex, &view);
        g_pVfxSystem->AddSortBuffer(processingIndex, pCallbackUserData->groupID);
        g_pVfxSystem->AddSortBuffer(processingIndex, nn::vfx::SystemParameters_ViewerGroupId);
        g_pVfxSystem->DrawSortBuffer(processingIndex, commandBuffer, true, nullptr);

    }
    g_GraphicsFramework.EndFrame(bufferIndex);
}


//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // VFXヒープ初期化
    int heapSize = 1024 * 1024 * 108;
    g_HeapPtr = malloc( heapSize );
    NN_ASSERT_NOT_NULL( g_HeapPtr );
    g_VfxHeap.Initialize( g_HeapPtr, heapSize );

    // ファイルシステムのアロケータ設定
    nn::fs::SetAllocator( Allocate, Deallocate );
    nn::fs::MountHostRoot();

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    static const size_t   GraphicsSystemMemorySize = 16 * 1024 * 1024;
    g_GraphicsFramework.InitializeGraphicsSystem(GraphicsSystemMemorySize);
#endif

    //------------------------------------------
    // GraphicsFrameWorkの初期化
    //------------------------------------------
    g_GraphicsFrameworkInfo.SetDefault();
    g_GraphicsFrameworkInfo.SetDisplayWidth( 1280 );
    g_GraphicsFrameworkInfo.SetDisplayHeight( 720 );
    g_GraphicsFrameworkInfo.SetBufferCount(2);
    g_GraphicsFrameworkInfo.SetSwapChainBufferCount(2);

    // RenderTargetMemoryPoolサイズを増やす
    size_t renderTargetMemoryPoolSize = g_GraphicsFrameworkInfo.GetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);
    renderTargetMemoryPoolSize += 1024 * 1024 * 4;
    g_GraphicsFrameworkInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, renderTargetMemoryPoolSize );

    g_GraphicsFrameworkInfo.SetColorBufferFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
    g_GraphicsFramework.Initialize(g_GraphicsFrameworkInfo);

    nns::gfx::GraphicsFramework::DebugFontTextWriter writer;
    g_GraphicsFramework.InitializeDebugFontTextWriter(&writer, 1024, 2);

    // htcs 初期化
#ifdef NN_ENABLE_HTC
    nn::htcs::Initialize( Allocate, Deallocate );
#endif

    // カメラ・プロジェクション行列初期化
    nn::util::MatrixLookAtRightHanded(&g_ViewMtx, g_CamPosVec, g_CamLookAtVec, 0.0f);
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >(g_GraphicsFramework.GetDisplayWidth()) / static_cast< float >(g_GraphicsFramework.GetDisplayHeight());
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&g_ProjectionMatrix, fovy, aspect, g_NearZ, g_FarZ);

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    // モデルシェーダリソースを初期化
    InitializeG3dShaderResource( g_GraphicsFramework.GetDevice(), &g_VfxHeap );

    // MountRom
    char* mountRomCacheBuffer = NULL;
    {
        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ASSERT(result.IsSuccess());

        mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

        result = nn::fs::MountRom("Contents", mountRomCacheBuffer, cacheSize );
        NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    }

    //------------------------------------------
    // VFXランタイムの初期化
    //------------------------------------------
    nn::vfx::Config  config;
    config.SetEffectHeap( &g_VfxHeap );
    config.SetEffectDynamicHeap( &g_VfxHeap );
    config.SetGfxDevice(g_GraphicsFramework.GetDevice());
    config.SetEmitterSetCount( 8 );
    config.SetEmitterCount( 256 );
    config.SetStripeCount( 128 );
    config.SetSuperStripeCount( 128 );
    config.SetGpuBufferSize( 1024 * 1024 * 4 );
    config.SetParticleSortBufferCount( 100 );
    g_pVfxSystem = new nn::vfx::System( config );
    g_pVfxSystem->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxSystem->RegisterSamplerToDescriptorPool(     AllocateDescriptorSlotForSampler, &g_GraphicsFramework);


    //------------------------------------------
    // VFX ビューアランタイムの初期化
    //------------------------------------------
    int connectionHeapSize = 1024 * 1024 * 8;
    g_ConnectionHeapPtr = malloc( connectionHeapSize );
    NN_ASSERT_NOT_NULL( g_ConnectionHeapPtr );
    g_ConnectionHeap.Initialize( g_ConnectionHeapPtr, connectionHeapSize );
    void* viewerBufferPtr = g_VfxHeap.Alloc(sizeof(nn::vfx::viewer::ViewerSystem), 16);

    // nn::vfx::viewer::ViewerSystem()の第4引き数noUseInternalThreadは、nn::vfx::viewer::ViewerSystem内でパケット受信スレッド制御
    // を行う場合はfalse、アプリケーション側で行う場合はtrueを設定します。
    // スレッド制御をアプリケーション側で行いたい場合などはtrueを設定し、スレッド生成/削除を行います。
    // falseの場合はnn::vfx::viewer::ViewerSystem側でスレッド制御を行いますので、アプリケーションで特に意識する必要はありません。
    // このアプリケーションではパケット受信スレッド作成/削除を行いますので、trueにしています。
    g_pVfxViewerSystem = new (viewerBufferPtr) nn::vfx::viewer::ViewerSystem( &g_VfxHeap, &g_ConnectionHeap, g_pVfxSystem, true );

    // パケット受信スレッド作成
    g_StackBase = reinterpret_cast< uint8_t* >( g_VfxHeap.Alloc( nn::vfx::viewer::detail::ToolConnector::taskMemsize,
                                                                 nn::vfx::viewer::detail::ToolConnector::taskAlignSize ) );
    nn::Result result = nn::os::CreateThread( &g_Thread,
                                              g_pVfxViewerSystem->ReadingPacketProcess,
                                              reinterpret_cast< void* >( g_pVfxViewerSystem ),
                                              reinterpret_cast< void* >( g_StackBase ),
                                              nn::vfx::viewer::detail::ToolConnector::taskMemsize,
                                              static_cast< int >( nn::os::LowestThreadPriority ) );
    if ( result.IsSuccess() == true )
    {
        nn::os::SetThreadName( &g_Thread, "NnVfxViewerToolConnector" );
        nn::os::StartThread( &g_Thread );
    }

    // VFXシステムにバイナリを設定し、0番エフェクトを生成
    const char* EffectFilePath = "Contents:/demo.ptcl";
    void* binary = nullptr;
    size_t binarySize = 0;
    bool resultFileLoad = FileLoad(&binary, &binarySize, EffectFilePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment);
    if (resultFileLoad == false) return;

    int resourceId = 0;
    int groupID = 0;
    nn::vfx::Handle emitterSetHandle;
    g_pVfxSystem->EntryResource( &g_VfxHeap, binary, resourceId, true );
    g_pVfxSystem->GetResource( resourceId )->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework);

    // TorchFireエミッタセットの作成
    nn::vfx::Handle torchEmitterSetHandle;
    int emitterSetId = g_pVfxSystem->GetResource( resourceId )->SearchEmitterSetId( "TorchFire" );
    g_pVfxSystem->CreateEmitterSetId( &torchEmitterSetHandle, emitterSetId, resourceId, groupID );


    //------------------------------------------
    // Humanモデル初期化
    //------------------------------------------
    void* modelBinary = nullptr;
    size_t modelBinarySize = 0;
    void* textureBinary = nullptr;
    size_t textureBinarySize = 0;

    const char* modelFilePath   = "Contents:/human.bfres";
    const char* textureFilePath = "Contents:/human.bntx";

    resultFileLoad = FileLoad( &modelBinary, &modelBinarySize, modelFilePath, &g_VfxHeap, 4096 );
    if (resultFileLoad == false) return;
    resultFileLoad = FileLoad( &textureBinary, &textureBinarySize, textureFilePath, &g_VfxHeap, 4096 );
    if (resultFileLoad == false) return;

    g_Human.Initialize( modelBinary, modelBinarySize, textureBinary, textureBinarySize, &g_VfxHeap, g_GraphicsFramework.GetDevice() );
    g_Human.UpdateDescriptorSlot( AllocateDescriptorSlotForTexture, &g_GraphicsFramework, AllocateDescriptorSlotForSampler, &g_GraphicsFramework);

    //------------------------------------------
    // Torchモデル初期化
    //------------------------------------------
    modelBinary = nullptr;
    modelBinarySize = 0;
    textureBinary = nullptr;
    textureBinarySize = 0;

    modelFilePath   = "Contents:/Torch.bfres";
    textureFilePath = "Contents:/Torch.bntx";

    resultFileLoad = FileLoad( &modelBinary, &modelBinarySize, modelFilePath, &g_VfxHeap, 4096 );
    if (resultFileLoad == false) return;
    resultFileLoad = FileLoad( &textureBinary, &textureBinarySize, textureFilePath, &g_VfxHeap, 4096 );
    if (resultFileLoad == false) return;

    g_Torch.Initialize( modelBinary, modelBinarySize, textureBinary, textureBinarySize, &g_VfxHeap, g_GraphicsFramework.GetDevice() );
    g_Torch.UpdateDescriptorSlot( AllocateDescriptorSlotForTexture, &g_GraphicsFramework, AllocateDescriptorSlotForSampler, &g_GraphicsFramework);

    // モデル情報設定クラスはオブジェクトを保持する必要がある
    ModelEnumeratorTest modelInfo;
    int humanModelIndex = modelInfo.SetModel( &g_Human );
    int torchModelIndex = modelInfo.SetModel( &g_Torch );

    // HumanのボーンＩＤを取得
    int humanBoneID = modelInfo.SearchBoneId( humanModelIndex, "finger_r4_3" );

    // HumanのボーンＩＤを取得
    int torchBoneID = modelInfo.SearchBoneId( torchModelIndex, "ParticleL" );

    // フレームワーク設定
    bool processResult;
    CalculateCallbackUserData calculateUserData;
    calculateUserData.pTorchEmitterSetHandle = &torchEmitterSetHandle;
    calculateUserData.pModelInfo = &modelInfo;
    calculateUserData.humanBoneID = humanBoneID;
    calculateUserData.torchBoneID = torchBoneID;
    calculateUserData.pWriter = &writer;
    calculateUserData.pProcessResult = &processResult;

    MakeCommandCallbackUserData makeCommandUserData;
    makeCommandUserData.groupID = groupID;
    makeCommandUserData.pWriter = &writer;
    makeCommandUserData.pProcessResult = &processResult;

    g_GraphicsFramework.SetCalculateCallback(CalculateCallback, &calculateUserData);
    g_GraphicsFramework.SetMakeCommandCallback(MakeCommandCallback, &makeCommandUserData);

    // メインループ
    for( int i = 0; i < 100000; i++ )
    {
        g_GraphicsFramework.ProcessFrame();
    }
    g_GraphicsFramework.QueueFinish();

    //------------------------------------------
    // VFXランタイムの終了処理
    //------------------------------------------
    g_pVfxViewerSystem->StopReadingPacketProcessThread();

    nn::os::DestroyThread( &g_Thread );
    if ( g_StackBase != NULL )
    {
        g_VfxHeap.Free( g_StackBase );
        g_StackBase = NULL;
    }
    g_pVfxViewerSystem->Finalize(FreeDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxViewerSystem->~ViewerSystem();
    g_VfxHeap.Free( viewerBufferPtr );

    g_pVfxSystem->GetResource( resourceId )->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxSystem->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework);
    g_pVfxSystem->UnregisterSamplerFromDescriptorPool( FreeDescriptorSlotForSampler, &g_GraphicsFramework);
    g_pVfxSystem->ClearResource( &g_VfxHeap, resourceId );
    delete g_pVfxSystem;

    nn::fs::Unmount("Contents");
    delete[] mountRomCacheBuffer;
    nn::fs::UnmountHostRoot();

    // プリミティブレンダラの終了処理
    FinalizePrimitiveRenderer();

    // GraphicsFrameWork の破棄
    g_GraphicsFramework.Finalize();

    g_ConnectionHeap.Finalize();
    g_VfxHeap.Finalize();
    free( g_ConnectionHeapPtr );
    free( g_HeapPtr );

#ifdef NN_ENABLE_HTC
    nn::htcs::Finalize();
#endif

    return;
} // NOLINT(readability/fn_size)
