﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d.h>
#include <nn/vfx/vfx_Heap.h>


class SimpleShader;

//------------------------------------------------------------------------------
// ※ グラフィックスに関する処理は、g3dのデモを参考にしてください。
//------------------------------------------------------------------------------
enum MemoryPoolType
{
    MemoryPoolType_CpuInvisible,
    MemoryPoolType_CpuCached,
    MemoryPoolType_CpuUncached
};

//------------------------------------------------------------------------------
//     シェーダを取得します。
//------------------------------------------------------------------------------
nn::gfx::Shader* getSimpleShader();

//------------------------------------------------------------------------------
//     シェーダを取得します。
//------------------------------------------------------------------------------
nn::gfx::Shader* getSkinningShader();

//------------------------------------------------------------------------------
//　　　シェーダリソースを初期化処理を行います。
//------------------------------------------------------------------------------
void InitializeG3dShaderResource( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap );

//------------------------------------------------------------------------------
//　　　シェーダリソースの終了処理を行います。
//------------------------------------------------------------------------------
void FinalizeG3dShaderResource( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap );

//------------------------------------------------------------------------------
//      テクスチャ ディスクリプタスロット確保関数のインタフェースです。
//------------------------------------------------------------------------------
typedef bool(*RegisterTextureViewSlot)( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData );

//------------------------------------------------------------------------------
//      テクスチャ ディスクリプタスロット解放関数のインタフェースです。
//------------------------------------------------------------------------------
typedef void(*UnregisterTextureViewSlot)( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData );

//------------------------------------------------------------------------------
//      サンプラ ディスクリプタスロット確保関数のインタフェースです。
//------------------------------------------------------------------------------
typedef bool(*RegisterSamplerSlot)(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& textureView, void* pUserData);

//------------------------------------------------------------------------------
//      サンプラ ディスクリプタスロット解放関数のインタフェースです。
//------------------------------------------------------------------------------
typedef void(*UnregisterSamplerSlot)(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& textureView, void* pUserData);


// シェーダで使用する各項目の最大数を設定します。
// 必要数を事前に決定しておくことでロケーションを記録するメモリを節約することも可能です。
enum
{
    ShaderSamplerCount = 8,
};

//------------------------------------------------------------------------------
//　　　サンプラー対応情報
//------------------------------------------------------------------------------
struct SamplerTable
{
    int index[ShaderSamplerCount];
};

//------------------------------------------------------------------------------
//　　　ビューの Uniform Block
//------------------------------------------------------------------------------
struct ViewBlock
{
    nn::util::FloatColumnMajor4x4 projMtx;
    nn::util::FloatColumnMajor4x3 cameraMtx;
};

//------------------------------------------------------------------------------
//　　　シンプルモデルクラス
//------------------------------------------------------------------------------
class SimpleModel
{
public:
    enum
    {
        MODEL_ANIM_MAX = 4
    };

    //---------------------------------------------------------------------------
    //! @brief     コンストラクタ
    //---------------------------------------------------------------------------
    SimpleModel();

    //---------------------------------------------------------------------------
    //! @brief     デストラクタ
    //---------------------------------------------------------------------------
    ~SimpleModel();

    //---------------------------------------------------------------------------
    //! @brief     初期化処理を行います。
    //---------------------------------------------------------------------------
    bool Initialize( void* modelResource, size_t modelResourceSize, void* textureResource, size_t textureResourceSize, nn::vfx::Heap* pHeap, nn::gfx::Device* pDevice );

    //---------------------------------------------------------------------------
    //! @brief     終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap );

    //---------------------------------------------------------------------------
    //! @brief     計算処理を行います。
    //---------------------------------------------------------------------------
    void Calc();

    //---------------------------------------------------------------------------
    //! @brief     ブロックの更新処理を行います。
    //---------------------------------------------------------------------------
    void CalcBlock( const nn::util::Matrix4x4fType* projMtx, const nn::util::Matrix4x3fType* viewMtx, const nn::util::Vector3fType* lightPos, const float animaSpeed );

    //---------------------------------------------------------------------------
    //! @brief     描画処理を行います。
    //---------------------------------------------------------------------------
    void Draw( nn::gfx::CommandBuffer* pCommandBuffer, nn::vfx::Heap* pHeap );

    //---------------------------------------------------------------------------
    //! @brief     描画マトリクスを設定します。
    //---------------------------------------------------------------------------
    void SetDrawMatrix( const nn::util::Matrix4x3fType& drawMatrix );

    //---------------------------------------------------------------------------
    //! @brief     アニメーションフレームを設定します。
    //---------------------------------------------------------------------------
    void SetAnimationFrame( float frame );

    //---------------------------------------------------------------------------
    //! @brief     アニメーションインデックスを設定します。
    //---------------------------------------------------------------------------
    void SetAnimationIndex( uint32_t index );

    //---------------------------------------------------------------------------
    //! @brief        モデル名を取得します。
    //---------------------------------------------------------------------------
    const char* GetModelName() const
    {
        return m_ModelObj->GetResource()->GetName();
    }

    //---------------------------------------------------------------------------
    //! @brief        ボーン数を取得します。
    //---------------------------------------------------------------------------
    int32_t GetBoneNum() const
    {
        nn::g3d::SkeletonObj* skeleton = m_ModelObj->GetSkeleton();
        if ( !skeleton ) return 0;
        return skeleton->GetBoneCount();
    }

    //---------------------------------------------------------------------------
    //! @brief        ボーン名を取得します。
    //---------------------------------------------------------------------------
    const char* GetBoneName( int index ) const
    {
        nn::g3d::SkeletonObj* skeleton = m_ModelObj->GetSkeleton();
        return skeleton->GetBoneName( index );
    }

    //---------------------------------------------------------------------------
    //! @brief        ボーンのワールドマトリックスを取得します。
    //---------------------------------------------------------------------------
    const nn::util::Matrix4x3fType* GetBoneWorldMatrix(int index)
    {
        nn::g3d::SkeletonObj* skeleton = m_ModelObj->GetSkeleton();
        int boneCount = skeleton->GetBoneCount();

        if (boneCount <= index)
            return NULL;

        return &skeleton->GetWorldMtxArray()[index];
    }

    //---------------------------------------------------------------------------
    //! @brief        ディスクリプタスロットを更新します。
    //---------------------------------------------------------------------------
    void UpdateDescriptorSlot( RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pTextureUserData,
                               RegisterSamplerSlot     pRegisterSamplerSlotCallback, void* pSamplerUserData );

private:
    //---------------------------------------------------------------------------
    //! @brief        メモリプールから領域を確保します
    //---------------------------------------------------------------------------
    ptrdiff_t AllocateFromMemoryPool(size_t size, size_t alignment);

private:
    nn::g3d::ResFile*           m_ResFile;
    nn::gfx::ResTextureFile*    m_ResTextureFile;
    nn::g3d::ModelObj*          m_ModelObj;
    nn::g3d::ModelAnimObj*      m_ModelAnimObjArray[MODEL_ANIM_MAX];
    nn::gfx::BlendState         m_BlendState;
    nn::gfx::DepthStencilState  m_DepthStencilState;
    nn::gfx::VertexState*       m_pVertexStateArray;
    SamplerTable*               m_pSamplerTableArray;
    nn::g3d::ResSkeletalAnim*   m_pResSkeletalAnim;
    nn::g3d::SkeletalAnimObj    m_SkeletalAnimObj;
    nn::gfx::Buffer             m_ViewBlock;
    void*                       m_pVertexStateBuffer;
    void*                       m_pMemoryPoolPtr;
    nn::gfx::MemoryPool         m_MemoryPool;
    ptrdiff_t                   m_MemoryPoolOffset;
    nn::util::Matrix4x3fType    m_DrawMatrix;
    float                       m_AnimFrame;
    uint32_t                    m_AnimIndex;
    int                         m_CurrentFace;
};

