﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UtilStreamingDecompression.cpp,PageSampleUtilStreamingDecompression}
 *
 * @brief
 *  ストリーミング伸長ライブラリのサンプルプログラム
 */

/**
 * @page PageSampleUtilStreamingDecompression ストリーミング伸長
 * @tableofcontents
 *
 * @brief
 *  ストリーミング伸長ライブラリを用いたサンプルプログラムの解説です。
 *
 * @section PageSampleUtilStreamingDecompression_SectionBrief 概要
 *  ここでは、ストリーミング伸長ライブラリを用いて圧縮データの
 *  ストリーミング伸長を行うサンプルプログラムの説明をします。
 *
 *  ストリーミング伸長ライブラリの使い方については、
 *  @ref nn::util "ユーティリティライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleUtilStreamingDecompression_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/UtilStreamingDecompression
 *  Samples/Sources/Applications/UtilStreamingDecompression @endlink 以下にあります。
 *
 * @section PageSampleUtilStreamingDecompression_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleUtilStreamingDecompression_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleUtilStreamingDecompression_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleUtilStreamingDecompression_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleUtilStreamingDecompression_SectionDetail 解説
 *
 * @subsection PageSampleUtilStreamingDecompression_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  UtilStreamingDecompression.h
 *  @includelineno UtilStreamingDecompression.h
 *
 *  UtilStreamingDecompression.cpp
 *  @includelineno UtilStreamingDecompression.cpp
 *
 * @subsection PageSampleUtilStreamingDecompression_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムではある文字列を zlib 形式で圧縮したデータである SampleData
 *  のストリーミング伸長を行います。@n
 *  ストリーミング伸長ではサイズの小さい入力および出力バッファに、
 *  それぞれ圧縮データと伸長結果を分割して格納しながら逐次的に伸長を行うことができます。@n
 *  ストリーミング伸長ライブラリが対応している他の圧縮形式には gzip 形式と
 *  raw deflate 形式があります。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - 伸長元データである SampleData のサイズを表示
 *      - SampleData の圧縮元データは @ref PageSampleUtilCompression で圧縮した文字列
 *      - @ref PageSampleUtilCompression では gzip 形式で圧縮しているため圧縮データのサイズは異なる
 *  - ストリーミング伸長用コンテキスト pContext のメモリ領域確保
 *      - コンテキストの型は nn::util::StreamingDecompressZlibContext
 *  - ストリーミング伸長ライブラリの nn::util::InitializeStreamingDecompressZlibContext()
 *  を呼び出して pContext を初期化
 *  - ストリーミング伸長のループ処理を実行
 *      - 入力バッファ srcBuf は 32 Byte 用意
 *      - 出力バッファ dstBuf は 64 Byte 用意
 *
 *  ストリーミング伸長のループ処理の流れは以下の通りです。
 *
 *  - srcBuf が空になっているかどうかを判定
 *      - 空の場合は srcBuf に圧縮データである pComressedData の続きを読み込み
 *  - ストリーミング伸長ライブラリの nn::util::StreamingDecompressZlib()
 *  を呼び出してストリーミング伸長
 *      - 渡された入力バッファが空になるか出力バッファが満杯になるまで伸長
 *      - 伸長に使用した入力データサイズ srcConsumedSize と出力データサイズ dstConsumedSize を出力
 *  - srcConsumedSize と dstConsumedSize が両方とも 0 であるか判定
 *      - 両方とも 0 の場合はストリーミング伸長完了であるためループを終了
 *  - dstBuf に格納された伸長データを文字列として表示
 *  - pSrcBufCurrent と remainSrcBufSize を更新
 *      - pSrcBufCurrent は srcBuf 上の次に伸長を開始する位置を示すポインタ
 *      - remainSrcBufSize は srcBuf 内でまだ伸長されていないデータのサイズ
 *
 *  このサンプルプログラムでは簡単のために、ストリームではなく最初からメモリ上に配置された
 *  SampleData を std::memcpy() で srcBuf に分割しながら読み込んでいます。@n
 *  圧縮データのファイルなどを入力バッファに読み込む場合には、該当の記述を変更してください。
 *
 *  このサンプルプログラムの実行結果を以下に示します。@n
 *  ログに表示される伸長データの文字列は @ref PageSampleUtilCompression の文字列 SampleData
 *  と一致します。
 *
 *  @verbinclude  UtilStreamingDecompression_OutputExample.txt
 *
 */

#include <algorithm>
#include <cstring>
#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/util/util_StreamingDecompression.h>

#include "UtilStreamingDecompression.h"

//-----------------------------------------------------------------------------

namespace {

//
// サイズの小さい入力および出力バッファで zlib 形式のストリーミング伸長を行い、
// 伸長結果を文字列としてログに出力するサンプル関数
//
void DisplayWithStreamingDecompressionZlib(const uint8_t* pComressedData,
                                           size_t compressedDataSize)
{
    // 元の圧縮されたデータを差すポインタ
    const uint8_t* pCompressedDataPtr       = pComressedData;
    // 圧縮データの残りサイズ
    size_t         compressedDataRemainSize = compressedDataSize;

    const size_t   SrcBufSize               = 32;      // 入力バッファのサイズ
    char           srcBuf[SrcBufSize];                 // 一時的に圧縮データを格納する入力バッファ
    char*          pSrcBufCurrent           = nullptr; // 入力バッファ上の位置を示すポインタ
    size_t         remainSrcBufSize         = 0;       // 入力バッファの残りサイズ

    const size_t   DstBufSize               = 64;  // 出力バッファのサイズ
    char           dstBuf[DstBufSize];             // 一時的に伸長データを格納する出力バッファ

    size_t         dstConsumedSize; // 出力されたデータ量
    size_t         srcConsumedSize; // 伸長に使用されたデータ量

    // ストリーミング伸長用コンテキストをヒープから取得
    std::unique_ptr<nn::util::StreamingDecompressZlibContext>
        pContext(new nn::util::StreamingDecompressZlibContext);
    NN_ABORT_UNLESS_NOT_NULL(pContext);

    // ストリーミング伸長用コンテキストを初期化
    nn::util::InitializeStreamingDecompressZlibContext(pContext.get());

    // ストリーミング伸長処理のループ
    for(;;)
    {
        // 入力バッファが空になったら圧縮データの続きから読み込む
        if(remainSrcBufSize == 0)
        {
            // 入力バッファサイズもしくは残りの圧縮データサイズの分だけ読み込む
            size_t inputSize = std::min(compressedDataRemainSize, SrcBufSize);

            // 簡単のために本サンプルでは最初からメモリ上に配置された圧縮データを読み込む
            std::memcpy(srcBuf, pCompressedDataPtr, inputSize);

            pCompressedDataPtr       += inputSize;
            compressedDataRemainSize -= inputSize;
            pSrcBufCurrent            = srcBuf;
            remainSrcBufSize          = inputSize;
        }

        // 入力バッファが空になるか出力バッファが満杯になるまでストリーミング伸長
        // 文字列としてログ出力するために出力バッファサイズは終端文字の分減らして渡す
        bool isSuccess = nn::util::StreamingDecompressZlib(&dstConsumedSize, &srcConsumedSize,
            dstBuf, (DstBufSize - sizeof(dstBuf[0])),
            pSrcBufCurrent, remainSrcBufSize, pContext.get());

        // ライブラリからの返り値をチェック
        if(!isSuccess)
        {
            // ライブラリが false を返した場合は入力データが壊れているか圧縮形式が異なる
            NN_LOG("Streaming Decompression (zlib): NG.\n");
            NN_LOG(" (the input data is corrupted or no supported format.)\n");

            return ;
        }

        // 入出力ともバッファが伸長に使用されていなければ処理は終了
        if(dstConsumedSize == 0 && srcConsumedSize == 0)
        {
            return ;
        }

        // 出力バッファに伸長データが入っていればログに出力
        if(dstConsumedSize > 0)
        {
            // 文字列としてログ出力するために得られた伸長結果の後ろに終端文字を挿入
            NN_ABORT_UNLESS(dstConsumedSize < DstBufSize);
            dstBuf[dstConsumedSize] = '\0';

            // 今回のストリーミング伸長で得られた文字列をログ出力
            NN_LOG("< Streaming decompressed data >: %s\n", dstBuf);
        }

        // 続きから入力するために入力バッファへのポインタとサイズを更新
        pSrcBufCurrent   += srcConsumedSize;
        remainSrcBufSize -= srcConsumedSize;
    }
}

} // namespace
//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    // 伸長対象の圧縮データのサイズを表示
    NN_LOG("\nCompressed data size ( %d Bytes )\n\n", SampleDataSize);

    // zlib 形式でストリーミング伸長した結果をログ出力
    DisplayWithStreamingDecompressionZlib(SampleData, SampleDataSize);
}

