﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UtilStreamingCompression.cpp,PageSampleUtilStreamingCompression}
 *
 * @brief
 *  ストリーミング圧縮ライブラリのサンプルプログラム
 */

/**
 * @page PageSampleUtilStreamingCompression ストリーミング圧縮
 * @tableofcontents
 *
 * @brief
 *  ストリーミング圧縮ライブラリを用いたサンプルプログラムの解説です。
 *
 * @section PageSampleUtilStreamingCompression_SectionBrief 概要
 *  ここでは、ストリーミング圧縮ライブラリを用いて圧縮データの
 *  ストリーミング圧縮を行うサンプルプログラムの説明をします。
 *
 *  ストリーミング圧縮ライブラリの使い方については、
 *  @ref nn::util "ユーティリティライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleUtilStreamingCompression_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/UtilStreamingCompression
 *  Samples/Sources/Applications/UtilStreamingCompression @endlink 以下にあります。
 *
 * @section PageSampleUtilStreamingCompression_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleUtilStreamingCompression_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleUtilStreamingCompression_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleUtilStreamingCompression_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleUtilStreamingCompression_SectionDetail 解説
 *
 * @subsection PageSampleUtilStreamingCompression_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  UtilStreamingCompression.h
 *  @includelineno UtilStreamingCompression.h
 *
 *  UtilStreamingCompression.cpp
 *  @includelineno UtilStreamingCompression.cpp
 *
 * @subsection PageSampleUtilStreamingCompression_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムでは文字列 SampleData の zlib 形式でのストリーミング圧縮を行います。@n
 *  ストリーミング圧縮ではサイズの小さい入力および出力バッファに、
 *  それぞれ入力データと圧縮結果を分割して格納しながら逐次的に圧縮を行うことができます。@n
 *  ストリーミング圧縮ライブラリが対応している他の圧縮形式には gzip 形式と
 *  raw deflate 形式があります。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - 圧縮元データである SampleData のサイズを表示
 *  - ストリーミング圧縮用コンテキスト pContext のメモリ領域確保
 *      - コンテキストの型は nn::util::StreamingCompressZlibContext
 *  - ストリーミング圧縮の作業用メモリ領域確保
 *      - 必要なメモリサイズは NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE で計算
 *  - ストリーミング圧縮ライブラリの nn::util::InitializeStreamingCompressZlibContext()
 *  を呼び出して pContext を初期化
 *  - ストリーミング圧縮のループ処理を実行
 *      - 入力バッファ srcBuf は 32 Byte 用意
 *      - 出力バッファ dstBuf は 64 Byte 用意
 *
 *  ストリーミング圧縮のループ処理の流れは以下の通りです。
 *
 *  - srcBuf が空になっているかどうかを判定
 *      - 空の場合は srcBuf に圧縮元データである pSrcData の続きを読み込み
 *      - pSrcData の読み込みが完了している場合は remainSrcBufSize に 0 を指定
 *  - ストリーミング圧縮ライブラリの nn::util::StreamingCompressZlib()
 *  を呼び出してストリーミング圧縮
 *      - 渡された入力バッファが空になるか出力バッファが満杯になるまで圧縮
 *  - srcConsumedSize と dstConsumedSize が両方とも 0 であるか判定
 *      - 両方とも 0 の場合はストリーミング圧縮完了であるためループを終了
 *  - pSrcBufCurrent と remainSrcBufSize を更新
 *      - pSrcBufCurrent は srcBuf 上の次に圧縮を開始する位置を示すポインタ
 *      - remainSrcBufSize は srcBuf 内でまだ圧縮されていないデータのサイズ
 *
 *  このサンプルプログラムでは簡単のために、ストリームではなく最初からメモリ上に配置された
 *  SampleData を std::memcpy() で srcBuf に分割しながら読み込んでいます。@n
 *  ファイルなどを入力バッファに読み込む場合には、該当の記述を変更してください。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  UtilStreamingCompression_OutputExample.txt
 *
 */

#include <algorithm>
#include <cstring>
#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/util/util_Compression.h>
#include <nn/util/util_StreamingCompression.h>

#include "UtilStreamingCompression.h"

//-----------------------------------------------------------------------------

namespace {

//
// サイズの小さい入力および出力バッファで zlib 形式のストリーミング圧縮を行うサンプル関数
//
void StreamingCompressionZlib(size_t* pOutCompressedDataSize,
                                char* pOutCompressedBuffer, size_t compressedBufferSize,
                                const char* pSrcData, size_t srcDataSize)
{
    // 圧縮済みのサイズ
    *pOutCompressedDataSize = 0;

    // 圧縮元データの残りサイズ
    size_t         srcDataRemainSize = srcDataSize;

    const size_t   SrcBufSize               = 32;      // 入力バッファのサイズ
    char           srcBuf[SrcBufSize];                 // 一時的に圧縮元データを格納する入力バッファ
    char*          pSrcBufCurrent           = nullptr; // 入力バッファ上の位置を示すポインタ
    size_t         remainSrcBufSize         = 0;       // 入力バッファの残りサイズ

    const size_t   DstBufSize               = 64;      // 出力バッファのサイズ
    char           dstBuf[DstBufSize];                 // 一時的に圧縮データを格納する出力バッファ

    size_t         dstConsumedSize; // 出力されたデータ量
    size_t         srcConsumedSize; // 圧縮に使用されたデータ量

    nn::util::StreamingCompressZlibContext  context;    // ストリーミング圧縮に利用するコンテキスト

    // ストリーミング圧縮用バッファをヒープから取得
    const size_t bufferSize = nn::util::CompressZlibWorkBufferSizeDefault;
    std::unique_ptr<nn::Bit8> pBuffer(new nn::Bit8[bufferSize]);
    NN_ABORT_UNLESS_NOT_NULL(pBuffer);

    // ストリーミング圧縮用コンテキストを初期化
    nn::util::InitializeStreamingCompressZlibContext(&context, pBuffer.get(), bufferSize);

    // ストリーミング圧縮処理のループ
    for(;;)
    {
        // 入力バッファが空になったら圧縮データの続きから読み込む
        if(remainSrcBufSize == 0)
        {
            // 入力バッファサイズもしくは残りの圧縮元データサイズの分だけ読み込む
            size_t inputSize = std::min(srcDataRemainSize, SrcBufSize);

            // 簡単のために本サンプルでは最初からメモリ上に配置された圧縮元データを読み込む
            std::memcpy(srcBuf, pSrcData, inputSize);

            pSrcData                 += inputSize;
            srcDataRemainSize        -= inputSize;
            pSrcBufCurrent            = srcBuf;
            remainSrcBufSize          = inputSize;
        }

        // 入力バッファが空になるか出力バッファが満杯になるまでストリーミング圧縮
        bool isSuccess = nn::util::StreamingCompressZlib(&dstConsumedSize, &srcConsumedSize,
            dstBuf, DstBufSize,
            pSrcBufCurrent, remainSrcBufSize, &context);

        // ライブラリからの返り値をチェック
        if(!isSuccess)
        {
            // ライブラリが false を返した場合は圧縮に失敗
            NN_LOG("Streaming compression (zlib): NG.\n");

            return;
        }

        // 入出力ともバッファが圧縮に使用されていなければ処理は終了
        if(dstConsumedSize == 0 && srcConsumedSize == 0)
        {
            return;
        }

        // 出力バッファにコピー
        size_t copiableSize = std::min(compressedBufferSize - (*pOutCompressedDataSize), dstConsumedSize);
        std::memcpy(pOutCompressedBuffer + (*pOutCompressedDataSize), dstBuf, copiableSize);
        *pOutCompressedDataSize += dstConsumedSize;

        // 続きから入力するために入力バッファへのポインタとサイズを更新
        pSrcBufCurrent   += srcConsumedSize;
        remainSrcBufSize -= srcConsumedSize;
    }
}

} // namespace
//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    // 圧縮データを格納するバッファ（圧縮元の SampleData が小さいためスタックに確保）
    char pCompressedData[SampleDataSize];
    size_t  compressedDataSize; // 圧縮データのサイズ

    // 圧縮対象のデータのサイズを表示
    NN_LOG("\nOriginal data size ( %d Bytes )\n\n", SampleDataSize);

    // zlib 形式でストリーミング圧縮
    StreamingCompressionZlib(&compressedDataSize, pCompressedData, SampleDataSize, SampleData, SampleDataSize);

    // 圧縮後のデータサイズを表示
    NN_LOG("Compressed data size ( %d Bytes )\n\n", compressedDataSize);
}

