﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UserMigration_ScreenIo.h"

#include <memory>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

void Logger::Printfln(const char* format, ...) NN_NOEXCEPT
{
    const size_t DisplayLengthMax = 128;

    va_list args;
    va_start(args, format);
    const auto L = static_cast<size_t>(nn::util::VSNPrintf(nullptr, 0, format, args));
    const auto BufferSize = L + 2;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);
    nn::util::VSNPrintf(buffer.get(), BufferSize, format, args);
    va_end(args);

    NN_LOG(buffer.get());
    NN_LOG("\n");

    // TODO ログファイル書き出し

    const auto End = std::min(L, DisplayLengthMax);
    std::strncpy(buffer.get() + End, "\n", BufferSize - End);
    nns::console::SimpleConsole::Puts(buffer.get());
}
void Logger::Puts(const char* str) NN_NOEXCEPT
{
    const size_t DisplayLengthMax = 128;

    const auto L = strlen(str);
    const auto BufferSize = L + 2;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);

    std::memcpy(buffer.get(), str, BufferSize);

    NN_LOG(buffer.get());

    // TODO ログファイル書き出し

    const auto End = std::min(L, DisplayLengthMax);
    buffer.get()[End] = '\0';
    nns::console::SimpleConsole::Puts(buffer.get());
}

void Window::SetComponents(const Components* pComponents) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pComponents);
    NN_SDK_ASSERT_NOT_NULL(pComponents->title);

    std::lock_guard<decltype(*this)> lock(*this);
    // 画面の状態をリセット
    m_pComponents = pComponents;
    m_Cursor = 0;
    m_IsSelected = false;
}

void Window::OnButtonReleasedImpl(const nns::console::PadButtonSet& buttons) NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    if (!(m_pComponents && m_pComponents->optionCount > 0))
    {
        m_Cursor = 0;
        return;
    }

    if (buttons.Test<nns::console::PadButton::Up>())
    {
        m_Cursor = std::max(m_Cursor - 1, 0);
    }
    if (buttons.Test<nns::console::PadButton::Down>())
    {
        m_Cursor = std::min(m_Cursor + 1, static_cast<int>(m_pComponents->optionCount) - 1);
    }
    if (buttons.Test<nns::console::PadButton::B>())
    {
        m_Cursor = 0;
    }
    if (buttons.Test<nns::console::PadButton::A>())
    {
        m_IsSelected = true;
    }
}
void Window::Flush() NN_NOEXCEPT
{
    static const size_t BufferSize = 1024 * 32;
    int cursor = 0;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);

    // ヘッダ
    cursor += m_Title[0] != '\0'
        ? nn::util::SNPrintf(buffer.get() + cursor, BufferSize - cursor, "[%s]\n", m_Title)
        : nn::util::SNPrintf(buffer.get() + cursor, BufferSize - cursor, "[%s]\n", m_pComponents->title);

    NN_ABORT_UNLESS_LESS(static_cast<size_t>(cursor), BufferSize);
    if (m_pComponents->description[0] != '\0')
    {
        cursor += nn::util::SNPrintf(buffer.get() + cursor, BufferSize - cursor, "%s\n", m_pComponents->description);
        NN_ABORT_UNLESS_LESS(static_cast<size_t>(cursor), BufferSize);
    }

    // オプション
    if (m_pComponents->optionCount > 0)
    {
        cursor += nn::util::SNPrintf(buffer.get() + cursor, BufferSize - cursor, "--------------------------------\n\n");
        NN_ABORT_UNLESS_LESS(static_cast<size_t>(cursor), BufferSize);

        for (int i = 0; i < static_cast<int>(m_pComponents->optionCount); ++i)
        {
            cursor += nn::util::SNPrintf(buffer.get() + cursor, BufferSize - cursor, "  %c : %s\n", i == m_Cursor ? '*' : ' ', m_pComponents->options[i].label);
            NN_ABORT_UNLESS_LESS(static_cast<size_t>(cursor), BufferSize);
        }
    }
    cursor += nn::util::SNPrintf(buffer.get() + cursor, BufferSize - cursor, "\n");
    NN_ABORT_UNLESS_LESS(static_cast<size_t>(cursor), BufferSize);

    nns::console::SimpleConsole::Clear();
    nns::console::SimpleConsole::Puts(buffer.get());
}
void Window::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    if (!m_pComponents)
    {
        return;
    }

    if (!(m_pComponents->optionCount > 0 && m_Cursor < static_cast<int>(m_pComponents->optionCount)))
    {
        m_Cursor = 0;
    }

    if (m_IsSelected)
    {
        if (m_pComponents->optionCount > 0)
        {
            if (m_pComponents->options[m_Cursor].withIndex)
            {
                m_pComponents->options[m_Cursor].u.optionHandlerWithIndex(m_Cursor, m_pComponents->optionData, m_pComponents->optionDataSize);
            }
            else
            {
                m_pComponents->options[m_Cursor].u.optionHandler(m_pComponents->optionData, m_pComponents->optionDataSize);
            }
        }
        m_IsSelected = false;
    }

    Flush();
}

void Window::Components::Reset(const char* _title, const char* _description) NN_NOEXCEPT
{
    // NNS_MIGRATION_LOG_LN("Components::Reset: \"%s\" - \"%s\"", _title, _description ? _description : "");
    std::strncpy(title, _title, sizeof(title));
    std::strncpy(description, _description ? _description : "", sizeof(description));
    optionData = nullptr;
    optionDataSize = 0;
    optionCount = 0;
}
void Window::Components::Reset(const char* _title, const char* _description, void* data, size_t dataSize) NN_NOEXCEPT
{
    // NNS_MIGRATION_LOG_LN("Components::Reset: \"%s\" - \"%s\"", _title, _description ? _description : "");
    std::strncpy(title, _title ? _title : "", sizeof(title));
    std::strncpy(description, _description ? _description : "", sizeof(description));
    optionData = data;
    optionDataSize = dataSize;
    optionCount = 0;
}
void Window::Components::AddOption(const char* label, OptionHandler handler) NN_NOEXCEPT
{
    NN_SDK_ASSERT(optionCount < std::extent<decltype(options)>::value);
    // NNS_MIGRATION_LOG_LN("Components::AddOption: [%d] \"%s\"", optionCount, label);
    auto& op = options[optionCount];
    std::strncpy(op.label, label, sizeof(op.label));
    op.withIndex = false;
    op.u.optionHandler = handler;
    ++ optionCount;
}
void Window::Components::AddOption(const char* label, OptionHandlerWithIndex handler) NN_NOEXCEPT
{
    NN_SDK_ASSERT(optionCount < std::extent<decltype(options)>::value);
    // NNS_MIGRATION_LOG_LN("Components::AddOption: [%d] \"%s\"", optionCount, label);
    auto& op = options[optionCount];
    std::strncpy(op.label, label, sizeof(op.label));
    op.withIndex = true;
    op.u.optionHandlerWithIndex = handler;
    ++ optionCount;
}
void Window::Components::ClearOption() NN_NOEXCEPT
{
    optionCount = 0;
}
