﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

/**
 * @examplesource{Program.cs,PageSampleTmSimpleNet}
 *
 * @brief
 *  TM ライブラリサンプルプログラム
 */

/**
 * @page PageSampleTmSimpleNet TmSimpleNet
 * @tableofcontents
 *
 * @brief
 *  TM ライブラリの使用方法を例示したサンプルです。
 *
 * @section PageSampleTmSimpleNet_SectionBrief 概要
 *  ここでは、TM ライブラリを使用した、ホスト側ツールの作成方法を説明します。
 *
 * @section PageSampleTmSimpleNet_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/TmSimpleNet Samples/Sources/Applications/TmSimpleNet @endlink 以下にあります。
 *
 * @section PageSampleTmSimpleNet_SectionNecessaryEnvironment 必要な環境
 *  あらかじめ Target Manager をインストールしておく必要があります。
 *
 * @section PageSampleTmSimpleNet_SectionHowToOperate 操作方法
 *  1. 追加可能な機材の一覧が出力されるので、追加したい機材に対応する数値を入力してください
 *  2. サンプルを実行可能な機材の一覧が出力されるので、サンプルを実行したい機材に対する数値を入力してください
 *  数値入力の際は、-1 などの範囲外の数値を入力することで、その操作をスキップすることができます
 *
 * @section PageSampleTmSimpleNet_SectionPrecaution 注意事項
 *
 * @section PageSampleTmSimpleNet_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleTmSimpleNet_SectionDetail 解説
 *
 * @subsection PageSampleTmSimpleNet_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  Program.cs
 *  @includelineno TmSimpleNet/Program.cs
 *
 * @subsection PageSampleTmSimpleNet_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - この PC で動作する Target Manager に登録可能なターゲットのリストアップを行う
 *  - この Target Manager にすでに登録されているターゲットのリストアップを行う
 *  - 登録可能なターゲットを提示し、指定されたターゲットを Target Manager に登録する
 *  - 登録されているターゲットを提示し、指定されたターゲット上でサンプルを実行する
 *
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace TmSimple
{
    class Program
    {
        static void Main(string[] args)
        {
            try
            {
                ShowDiscoveredTargets();
                ShowRegisteredTargets();
                AddTarget();
                RunProgram();
            }
            catch (Exception e)
            {
                Console.WriteLine(e.ToString());
            }
            Console.ReadKey();
        }

        static internal void ShowRegisteredTargetsWithoutDetails()
        {
            var targets = Nintendo.Tm.TargetManager.GetTargets();
            for (int iTarget = 0; iTarget < targets.Length; iTarget++)
            {
                var target = targets[iTarget];
                Console.Write("{0}):", iTarget);
                Console.Write("\tName: {0}", target.GetName());
                Console.Write("\tIPAddress: {0}", target.GetIpAddress());
                Console.Write("\tSerialNumber: {0}\n", target.GetSerialNumber());
            }
        }

        static internal int GetInputNumber()
        {
            while (true)
            {
                try
                {
                    return int.Parse(Console.ReadLine());
                }
                catch
                {
                    Console.WriteLine("Invalid parameter. Try again.");
                }
            }
        }

        static void ShowDiscoveredTargets()
        {
            const int WaitSec = 2;
            Nintendo.Tm.TargetManager.StartDiscovery(WaitSec * 1000);
            for (int i = 0; i < WaitSec; i++)
            {
                System.Threading.Thread.Sleep(1000);
                Console.WriteLine("This is a list of currently found targets");

                var targets = Nintendo.Tm.TargetManager.GetDiscoveryTargets();
                for (int iTarget = 0; iTarget < targets.Length; iTarget++)
                {
                    Console.WriteLine("{0}:", iTarget.ToString("00"));
                    var target = targets[iTarget];

                    Console.WriteLine("\tName: {0}", target.GetName());
                    Console.WriteLine("\tIPAddress: {0}", target.GetIpAddress());
                    Console.WriteLine("\tSerialNumber: {0}", target.GetSerialNumber());
                    Console.WriteLine("\tMAC Address: {0}", target.GetMacAddress());
                    Console.WriteLine("\tConnectionType: {0}", ConnectionTypeName[(int)target.GetConnectionType()]);
                    Console.WriteLine("\tHardwareType: {0}", HardwareTypeName[(int)target.GetHardwareType()]);
                }
            }
        }

        static void ShowRegisteredTargets()
        {
            var targets = Nintendo.Tm.TargetManager.GetTargets();
            Nintendo.Tm.Target defaultTarget;
            Nintendo.Tm.TargetManager.TryGetDefaultTarget(out defaultTarget);

            for (int iTarget = 0; iTarget < targets.Length; iTarget++)
            {
                var target = targets[iTarget];

                Console.WriteLine("{0}:{1}", iTarget.ToString("00"), (defaultTarget == target) ? "(Default Target)" : "");
                Console.WriteLine("\tName: {0}", target.GetName());
                Console.WriteLine("\tIPAddress: {0}", target.GetIpAddress());
                Console.WriteLine("\tSerialNumber: {0}", target.GetSerialNumber());
                Console.WriteLine("\tMAC Address: {0}", target.GetMacAddress());
                Console.WriteLine("\tStatus: {0}", TargetStatusName[(int)target.GetStatus()]);
                Console.WriteLine("\tConnectionType: {0}", ConnectionTypeName[(int)target.GetConnectionType()]);
                Console.WriteLine("\tHardwareType: {0}", HardwareTypeName[(int)target.GetHardwareType()]);
                Console.WriteLine("\tIsSleeping: {0}", (target.GetIsSleeping()) ? "Yes" : "No");
                Console.WriteLine("\tIsConnected: {0}", (target.GetIsConnected()) ? "Yes" : "No");
            }
        }

        static void AddTarget()
        {
            // StartDiscovery should be called in ShowDiscoveredTargets already so just call GetDiscoveryTargetCount and so on
            var targets = Nintendo.Tm.TargetManager.GetDiscoveryTargets();
            Console.WriteLine("Please input the option number to add:");
            for (int iTarget = 0; iTarget < targets.Length; iTarget++)
            {
                var target = targets[iTarget];
                Console.Write("{0}):", iTarget);
                Console.Write("\tName: {0}", target.GetName());
                Console.Write("\tIPAddress: {0}", target.GetIpAddress());
                Console.Write("\tSerialNumber: {0}\n", target.GetSerialNumber());
            }

            int selectedTarget = GetInputNumber();
            if (0 <= selectedTarget && selectedTarget < targets.Length)
            {
                Nintendo.Tm.TargetManager.AddTarget(targets[selectedTarget]);
                Console.WriteLine("Added.");
            }
        }

        static string GetSampleNspPath()
        {
            var sdkRootPath = System.Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");
            return sdkRootPath + "\\Resources\\Firmwares\\NX\\Sample\\Binaries\\GfxPrimitiveRenderer.nsp";
        }

        static void RunProgram()
        {
            var targets = Nintendo.Tm.TargetManager.GetTargets();
            if (targets.Length == 0)
            {
                Console.WriteLine("No target is found. Skip RunProgram");
                return;
            }

            Console.WriteLine("Please input the option number of the target to run:");

            ShowRegisteredTargetsWithoutDetails();
            int selectedTarget = GetInputNumber();
            if (0 <= selectedTarget && selectedTarget < targets.Length)
            {
                var target = targets[selectedTarget];

                // Guarantee that the target is connected
                target.Connect();
                var sampleNspPath = GetSampleNspPath();
                Console.WriteLine("Loading {0}...", sampleNspPath);
                target.LaunchProgram(sampleNspPath);

                System.Threading.Thread.Sleep(3000);

                Console.WriteLine("Running program is {0}", target.GetProgramName());
                Console.WriteLine("Program state is {0}", ProgramStateName[(int)target.GetProgramState() + 1]);

                Console.Write("Terminating the application... ");
                target.KillProgram();
                Console.Write("Success\n");
            }
        }

        static readonly String[] ConnectionTypeName =
        {
            "Unknown",
            "Usb",
            "Ethernet",
            "Future",
        };

        static readonly String[] HardwareTypeName =
        {
            "Unknown",
            "Edev",
            "Sdev",
            "Sim",
            "Future",
        };

        static readonly String[] TargetStatusName =
        {
            "Unknown",
            "Disconnected",
            "Disconnecting",
            "Connected",
            "Connecting",
            "Asleep",
            "PoweringOn",
            "PoweringOff",
        };

        static readonly String[] ProgramStateName =
        {
            "NoProgram",
            "Loading",
            "Loaded",
            "Running",
            "Halted",
        };
    }
}
