﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstdlib>
#include <cstring>
#include <cerrno>

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/init/init_Malloc.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/os/os_MemoryAllocatorForThreadLocal.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>
#include <nn/nn_Log.h>
#include <nn/init.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
namespace {

    const size_t                TlsBufferSize = 128 * 1024 * 1024;
    NN_ALIGNAS(4096) uint8_t    g_TlsBuffer[ TlsBufferSize ];
    nn::util::TypedStorage<nn::mem::StandardAllocator,sizeof(nn::mem::StandardAllocator),NN_ALIGNOF(nn::mem::StandardAllocator)>    g_TlsAllocator;

    void* CustomAllocatorForThreadLocal(size_t size, size_t alignment) NN_NOEXCEPT
    {
        return Get(g_TlsAllocator).Allocate(size, alignment);
    }

    void CustomDeallocatorForThreadLocal(void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        if (p)
        {
            Get(g_TlsAllocator).Free(p);
        }
    }
}
#endif  // NN_BUILD_CONFIG_OS_HORIZON

//---------------------------------------------------------------------------
//  デフォルトの nninitStartup() が適用されないように空関数として定義。
extern "C" void nninitStartup()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    new( &Get(g_TlsAllocator) ) nn::mem::StandardAllocator;
    Get(g_TlsAllocator).Initialize(g_TlsBuffer, TlsBufferSize);

    NN_LOG("malloc customized in %s line %d\n", __FILE__, __LINE__);

    // コンパイラのスレッドローカル実装用のメモリアロケータの登録
    nn::os::SetMemoryAllocatorForThreadLocal(
                            CustomAllocatorForThreadLocal,
                            CustomDeallocatorForThreadLocal);

    NN_LOG("TLS allocators customized in %s line %d\n", __FILE__, __LINE__);
#endif  // NN_BUILD_CONFIG_OS_HORIZON
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
//-----------------------------------------------------------------------------
//  malloc(), free(), calloc(), realloc(), aligned_alloc 関数の定義
//
extern "C" void* malloc(size_t size)
{

    void* p = Get(g_TlsAllocator).Allocate(size);
    if ( p == NULL )
    {
        errno = ENOMEM;
    }

    return p;
}

extern "C" void free(void* p)
{
    if (p)
    {
        Get(g_TlsAllocator).Free(p);
    }
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = std::malloc(sum);

    if ( p != NULL )
    {
        std::memset(p, 0, sum);
    }
    else
    {
        errno = ENOMEM;
    }

    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    // メモリブロックのサイズを変更する
    // Reallocate() の仕様は realloc() の仕様と同じため、そのまま呼ぶ
    void* r = Get(g_TlsAllocator).Reallocate(p, newSize);
    if ( r == NULL )
    {
        errno = ENOMEM;
    }

    return r;
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    void* p = Get(g_TlsAllocator).Allocate(size, alignment);
    if ( p == NULL )
    {
        errno = ENOMEM;
    }

    return p;
}

//-----------------------------------------------------------------------------
#endif  // NN_BUILD_CONFIG_OS_HORIZON
