﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{TimeStandardSteadyClock.cpp,PageSampleTimeStandardSteadyClock}
 *
 * @brief
 *  nn::time::TimeStandardSteadyClock のサンプルプログラム
 */

/**
 * @page PageSampleTimeStandardSteadyClock StandardSteadyClock
 * @tableofcontents
 *
 * @brief
 *  nn::time::StandardSteadyClock を扱うサンプルプログラムの解説です。
 *
 * @section StandardSteadyClock_SectionBrief 概要
 *  ここでは、 nn::time::StandardSteadyClock を扱うサンプルプログラムの解説です。
 *  このクロックの指し示す値は人間の理解できる日時の表現に変換することはできませんが、
 *  ユーザーやシステムによる時刻の操作に関わらず一定間隔でカウントアップされるため、時刻の変更・補正の影響を受けずに時間経過を計算することができます。
 *
 *  @ref nn::time::StandardSteadyClock "StandardSteadyClock の関数リファレンス" も併せて参照して下さい。
 *
 * @section StandardSteadyClock_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/TimeStandardSteadyClock Samples/Sources/Applications/TimeStandardSteadyClock @endlink 以下にあります。
 *
 * @section TimeStandardSteadyClock_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section TimeStandardSteadyClock_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section TimeStandardSteadyClock_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section TimeStandardSteadyClock_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section TimeStandardSteadyClock_SectionDetail 解説
 *
 * @subsection TimeStandardSteadyClock_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  TimeStandardSteadyClock.cpp
 *  @includelineno TimeStandardSteadyClock.cpp
 *
 * @subsection TimeStandardSteadyClock_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 * - TIME ライブラリの初期化
 * - 経過時間の計測を行うサンプル
 * - TIME ライブラリの終了
 *
 * 経過時間の計測を行う機能は、
 * 始点となる nn::time::SteadyClockTimePoint の値をセーブデータ等で永続化することで
 * アプリケーションのシャットダウンや本体の電源断をまたいだ経過時間の計算を行うことができます。
 * @n
 * セーブデータが別デバイスへ移行されたときや、電池切れによるRTCのリセット発生などにより、
 * 起点と終点の nn::time::SteadyClockTimePoint 同士に連続性がなくなった場合には、
 * @ref nn::time::GetSpanBetween() では経過時間を計算することができません。
 * そのような場合にもネットワーク時計の時刻を使って経過時間を計算できるよう、
 * @ref nn::time::ClockSnapshot クラスと @ref nn::time::CalculateSpanBetween() を用意しています。
 * 詳細は API リファレンスを参照してください。
 * @n
 * アプリケーションの終了と起動をまたがない条件下であれば、 nn::os::GetSystemTick 関数から取得できる
 * システムチック値によって、より分解能の高い時間経過の計算が可能です。
 *
 * nn::Result を返す API において、事前条件を満たしていれば必ず成功するものは
 * @ref NN_ABORT_UNLESS_RESULT_SUCCESS を利用してハンドリングしています。
 *
 * このサンプルの実行結果を以下に示します。
 *
 * @verbinclude TimeStandardSteadyClock_ExampleOutput.txt
 *
 */
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardSteadyClock.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_ThreadApi.h>

extern "C" void nnMain()
{
    // TIME ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );
    NN_LOG("nn::time::Initialize\n");

    // 経過時間の計測を行うサンプルの実行
    NN_LOG("---- Sample of getting nn::TimeSpan from two SteadyClockTimePoint's.\n");
    {
        // 起点となる SteadyClockTimePoint を取得します。
        nn::time::SteadyClockTimePoint steadyClockTimePoint1;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint(&steadyClockTimePoint1) );

        // 上記の SteadyClockTimePoint の値をセーブデータ等に保存して永続化し、次回起動時に読み出すことで、
        // アプリケーションのシャットダウンや本体の電源断をまたいで、経過時間を計算することができます。
        NN_LOG("Waiting 3 seconds.\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // 終点となる SteadyClockTimePoint を取得します。
        nn::time::SteadyClockTimePoint steadyClockTimePoint2;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint(&steadyClockTimePoint2) );

        // 2つの SteadyClockTimePoint から経過時間を計算します。
        int64_t elapsedSeconds;
        nn::Result result = nn::time::GetSpanBetween(
            &elapsedSeconds,
            steadyClockTimePoint1,
            steadyClockTimePoint2);

        if(result.IsSuccess())
        {
            // 経過時間が正常に計算できた場合にここにきます。
            NN_LOG("nn::time::GetSpanBetween succeeded. %lld seconds elapsed.\n", elapsedSeconds);

            // 得られた経過時間が一定の時間を超えているかどうかを判定することで、
            // 例えば前回のイベント発生から、○○時間以上が経過していたら再度イベントが発生する、
            // といった用途に利用することができます。

            // StandardSteadyClock はハードウェア的な改竄攻撃に対して強靭ではありません。
            // 経過時間がマイナス値など想定外の値になったとしても、はまりや止まりがないように実装してください。
        }
        else
        {
            // 以下のいずれかに該当すると、経過時間が計算できずここに来ます
            //  - 電池切れによるRTCのリセットが発生し、2つのSteadyClockTimePoint 同士に連続性がなくなった
            //  - セーブデータが別デバイスに移行され、2つのSteadyClockTimePoint 同士に連続性がなくなった
            //  - 計算結果がオーバーフローもしくはアンダーフローした場合
            NN_LOG("nn::time::GetSpanBetween failed.\n");

            // 正常動作として、nn::time::GetSpanBetween() の失敗時のケアは必ず行う必要があります。
            //
            // 例えば、始点となる SteadyClockTimePoint を永続化している場合には、
            // nn::time::GetSpanBetween() が失敗し続けるのを防ぐため、
            // 最新の SteadyClockTimePoint を取得しなおし、
            // その値を次回の起動時に使用できるよう永続化するなどの対応が必要となります。
        }
    }

    // TIME ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
    NN_LOG("nn::time::Finalize\n");
}
