﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/ssl.h>
#include <nn/socket.h>
#include <nn/nifm.h>


/**
 * @examplesource{SslSimple.cpp,PageSampleSslSimple}
 *
 * @brief
 * Simple Https commnucation with SSL library
 */

/**
 * @page PageSampleSslSimple Simple Https commnucation with SSL library
 * @tableofcontents
 *
 * @brief
 * Documentation about the sample program of SSL library
 *
 * @section PageSampleSslSimple_SectionBrief Overview
 * First this sample prepares network connection by using the NIFM library. If network can be used,
 * it establishes the TCP connection on port 443 with the specified server. Then it performs SSL
 * handshake on the TCP connection and sends HTTP GET request. Finally it receives HTTP data which
 * is sent by the server per HTTP GET request.
 *
 * @section PageSampleSslSimple_SectionFileStoructure File Structure
 * This sample program can be found at
 * @link ../../../Samples/Sources/Applications/SslSimple
 * Samples/Sources/Applications/SslSimple @endlink.
 *
 * @section PageSampleSslSimple_SectionNecessaryEnvironment System Requirements
 * It is required to import network setting by SettingManager prior to running this sample.
 * Please refer @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} for further information.
 *
 * By default this sample sends HTTP GET request to example.com on port 443 thus Internet
 * connection is required. Please pass desired host name in the command line argument to run it
 * with other servers. Note that "https://" should not be included when passing the host name in
 * the argument because this sample doesn't parse it.
 *
 * @section PageSampleSslSimple_SectionHowToOperate Operation Procedure
 * None.
 *
 * @section PageSampleSslSimple_SectionPrecaution Precautions
 * None.
 *
 * @section PageSampleSslSimple_SectionHowToExecute Execution Procedure
 * Build the Visual Studio Solution in the desired configuration and run it.
 *
 * @section PageSampleSslSimple_SectionDetail Detail
 * Here is the sequence of this sample program.
 *
 * - Sends request to use network connection by the NIFM library
 * - Initializes the socket library and the SSL library
 * - Resolves host name and establishes TCP connection on port 443
 *   in SimpleHttpsClient::SetupTcpSocket()
 * - Prepares the SSL context in SimpleHttpsClient::SetupSslContext()
 * - Prepares the SSL connection context in SimpleHttpsClient::SetupSslConnection()
 * - Performs SSL handshake with the server on the TCP connection in
 *   SimpleHttpsClient::PerformSslHandshake(), by default this sample verifies server certificate
 *   and its host name
 * - Sends HTTP GET request to the server over the SSL connection in
 *   SimpleHttpsClient::SendHttpGetOverSsl()
 * - Receives HTTP data sent from the server per HTTP GET requst in
 *   SimpleHttpsClient::ReceiveAllHttpData() until the connection is disconnected by the server or
 *   timeout occurs
 * - Print received content in SimpleHttpsClient::PrintReceivedData()
 * - Destroys the SSL connnection context and the SSL context in SimpleHttpsClient::Finalize()
 * - Finalizes the socket library and the SSL library
 *
 * Please enable "#define PRINT_ALL_VERIFY_ERRORS" to demonstrate how get all of the
 * verify certificate errors.
 *
 * Please enable "#define PRINT_SERVER_CERT" to demonstrate how to get the server certificate
 * which is obtained during SSL handshake.
 *
 * Please enable "#define PERFORM_NON_BLOCKING" to demonstrate how to use I/O APIs in
 * non-blocking mode.
 *
 * Please enable "#define IMPORT_SERVER_PKI" to import CA certificate. Please don't forget to
 * supply appropriate CA certificate when enabling the flag otherwise
 * nn::ssl::Context::ImportServerPki will assert since nullptr will be passed. It is imported in
 * SimpleHttpsClient::SetupSslContext().
 *
 * Please enable "#define IMPORT_CLIENT_PKI" to import client PKI information. Please don't forget
 * to supply appropriate pkcs12 data when enabling the flag otherwise
 * nn::ssl::Context::ImportClientPki will assert since nullptr will be passed. It is imported in
 * SimpleHttpsClient::SetupSslContext().
 *
 * Please enable "#define DISABLE_RENEGOTIATION" to disable renegotiation for the ssl handshake.
 *
 * Please enable "#define DISABLE_SESSION_CACHE" to disable ssl session cache.
 * Note: This option is not compatible with SESSION_CACHE_TICKET.
 *
 * Please enable "#define SESSION_CACHE_TICKET" to enable ssl session tickets.
 * Note: This option is not cimpatible with DISABLE_SESSION_CACHE.
 *
 * Please enable "#define FLUSH_SESSION_CACHE" to flush the ssl session cache before
 * destroying the connection."
 *
 * Here is the expected output log of this sample program. Please note that it will not be printed
 * in the Release build.
 *
 * @verbinclude SslSimple_Output.txt
 */

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
// #define PRINT_ALL_VERIFY_ERRORS // Enable this to print all verify errors that occurred
// #define PRINT_SERVER_CERT       // Enable this to print obtained server cert in DER format
// #define PERFORM_NON_BLOCKING    // Enable this to call I/O APIs in non-blocking mode
// #define IMPORT_SERVER_PKI       // Enable this to import CA certificate
// #define IMPORT_CLIENT_PKI       // Enable this to import client PKI information
// #define DISABLE_RENEGOTIATION   // Enable this to disable renegotiation
// #define DISABLE_SESSION_CACHE   // Enable this to disable session cache. This cannot be combined with SESSION_CACHE_TICKET
// #define SESSION_CACHE_TICKET    // Enable this to use session cache tickets. This cannot be combined with DISABLE_SESSION_CACHE
// #define FLUSH_SESSION_CACHE     // Enable this to flush session cache before destroying the connection

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
#define PRINT_HEX_TABLE(pData,len)                                  \
do                                                                  \
{                                                                   \
    unsigned char* pChar = reinterpret_cast<unsigned char*>(pData); \
    for(uint32_t i = 0; i < len; i++) {                             \
        NN_LOG("0x%02x", pChar[i]);                                 \
        if((i + 1) % 16 == 0) {                                     \
            NN_LOG("\n");                                           \
        } else {                                                    \
            if(((i + 1) % 8 == 0) && (i % 16 != 0)) {               \
                NN_LOG(" | ");                                      \
            } else {                                                \
                NN_LOG(" ");                                        \
            }                                                       \
        }                                                           \
    }                                                               \
} while(NN_STATIC_CONDITION(false))

namespace
{
const int                g_TimeoutOnNicConfigurationInSec = 15;
const int                g_TcpPortNumber                  = 443;

#if defined(DISABLE_SESSION_CACHE) && defined(SESSION_CACHE_TICKET)
    #error "Error: DISABLE_SESSION_CACHE and SESSION_CACHE_TICKET are incompatible. Please only define one of them."
#elif defined(DISABLE_SESSION_CACHE)
const nn::ssl::Connection::SessionCacheMode  g_sessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;
#elif defined(SESSION_CACHE_TICKET)
const nn::ssl::Connection::SessionCacheMode  g_sessionCacheMode = nn::ssl::Connection::SessionCacheMode_SessionTicket;
#endif

#if defined (PRINT_ALL_VERIFY_ERRORS)
const bool g_IsPrintAllVerifyErrors = true;
#else
const bool g_IsPrintAllVerifyErrors = false;
#endif
#if defined (PERFORM_NON_BLOCKING)
const bool               g_IsPerformNonBlocking           = true;
#else
const bool               g_IsPerformNonBlocking           = false;
#endif
#if defined(PRINT_SERVER_CERT)
const bool               g_IsPrintServerCert              = true;
#else
const bool               g_IsPrintServerCert              = false;
#endif

// Socket memory
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

// ------------------------------------------------------------------------------------------------
// Setup network interface
// ------------------------------------------------------------------------------------------------
nn::Result SetupNetwork()
{
    nn::Result result = nn::ResultSuccess();
    uint32_t   timeoutSec = 0;
    result = nn::nifm::Initialize();
    if(result.IsFailure())
    {
        return result;
    }

    nn::nifm::SubmitNetworkRequest();
    while(nn::nifm::IsNetworkRequestOnHold())
    {
        NN_LOG("Waiting for network interface availability...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        if(timeoutSec++ > g_TimeoutOnNicConfigurationInSec)
        {
            NN_LOG("Failed to setup network interface.\n");
            result = nn::ssl::ResultErrorLower();
            break;
        }
    }

    return result;
}

void FinalizeNetwork()
{
    nn::nifm::CancelNetworkRequest();
}

// ------------------------------------------------------------------------------------------------
// class SimpleHttpsClient: the class to perform simple HTTP GET request over SSL
// ------------------------------------------------------------------------------------------------
class SimpleHttpsClient
{
private:
    static const uint32_t ServerCertBufferSize     = 1024 * 32;  //!< The size of the buffer for server certificate
    static const uint32_t ReadBufferSize           = 1024 * 512; //!< The size of the buffer for received data
    static const uint32_t MaxAsyncDoHandshakeRetry = 30;         //!< The number of count to retry DoHandshake
    static const uint32_t MsecPollTimeout          = 5000;       //!< Poll timeout
    static const uint32_t VerifyResultArrayLen     = 4;          //!< Max number of nn::Results in the array where verify errors will be stored.
    static const uint32_t HostNameBufLen           = 256;        //!< The size of the buffer for hostname

    nn::ssl::Context    m_SslContext;               //!< SSL context
    nn::ssl::Connection m_SslConnection;            //!< SSL connection context
    char                m_HostName[HostNameBufLen]; //!< The host name of the server
    bool                m_IsNonBlocking;            //!< Blocking mode of I/O APIs
    uint16_t            m_PortNumber;               //!< Post number of the server to establish TCP connection
    int                 m_TcpSocket;                //!< TCP socket descriptor
    bool                m_IsSocketImported;         //!< It becomes true when m_TcpSocket is passed to the SSL library
    char*               m_pReceivedData;            //!< Pointer to the buffer which received data is stored
    uint32_t            m_ReceivedTotalBytes;       //!< The number of bytes received
    bool                m_IsInitialized;            //!< It becomes true when this class is initialized

#if defined(IMPORT_SERVER_PKI)
    nn::ssl::CertStoreId m_serverCertStoreId;
#endif

#if defined(IMPORT_CLIENT_PKI)
    nn::ssl::CertStoreId m_clientCertStoreId;
#endif

    int        SetupTcpSocket();
    nn::Result SetupSslContext(nn::ssl::Context::SslVersion sslVersion);
    bool       SetupSslConnection(nn::ssl::Connection::VerifyOption verifyOption);
    bool       PrintVerifyError(bool isPrintAllEnabled);

public:
    SimpleHttpsClient(bool IsBlocking, const char* pInHostName, uint16_t portNum):
        m_TcpSocket(-1),
        m_IsSocketImported(false),
        m_pReceivedData(nullptr),
        m_ReceivedTotalBytes(0),
        m_IsInitialized(false)
    {
        strcpy(m_HostName, pInHostName);
        m_IsNonBlocking = IsBlocking;
        m_PortNumber    = portNum;
    }

    ~SimpleHttpsClient()
    {
        if(m_IsInitialized == true)
        {
            this->Finalize();
        }
    }

    bool Initialize(
        nn::ssl::Context::SslVersion SslVersion,
        nn::ssl::Connection::VerifyOption verifyOption);
    void Finalize();
    bool PerformSslHandshake(bool g_IsPrintServerCert);
    bool SendHttpGetOverSsl();
    bool ReceiveAllHttpData();
    void PrintReceivedData();
    bool ConfigureHostName();
    bool ConfigureVerifyOption(nn::ssl::Connection::VerifyOption verifyOption);
    bool ConfigureSessionCacheMode();
    bool ConfigureRenegotiationMode();
    bool ConfigureIoMode();
};

// ------------------------------------------------------------------------------------------------
// Create TCP socket and establish connection with the server
//   1. Create TCP socket by nn::socket::Socket()
//   2. Resolve host name by nn::socket::GetHostByName()
//   3. Establish TCP connection with the server by nn::socket::Connect()
// ------------------------------------------------------------------------------------------------
int SimpleHttpsClient::SetupTcpSocket()
{
    int                 tcpSocket;
    nn::socket::InAddr  inetAddr;

    memset(&inetAddr, 0x00, sizeof(inetAddr));

    if(m_HostName[0] == '\0')
    {
        NN_LOG("Host name is not set\n");
        return -1;
    }

    tcpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if(tcpSocket < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d)\n", errno);
        return -1;
    }
    NN_LOG("Created TCP socket (sockfd: %d).\n", tcpSocket);

    NN_LOG("Resolving %s\n", m_HostName);
    nn::socket::HostEnt *pHostEnt = nn::socket::GetHostEntByName(m_HostName);
    if(pHostEnt == nullptr)
    {
        NN_LOG("Failed to resolve host name (errno:%d)\n", errno);
        return -1;
    }

    // Just pick the first one
    memcpy(&inetAddr, pHostEnt->h_addr_list[0], sizeof(inetAddr));

    nn::socket::SockAddrIn serverAddr;
    serverAddr.sin_addr.S_addr = inetAddr.S_addr;
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;
    serverAddr.sin_port        = nn::socket::InetHtons(m_PortNumber);

    int rval = nn::socket::Connect(
        tcpSocket,
        (nn::socket::SockAddr*)&serverAddr,
        sizeof(serverAddr));
    if(rval < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d).\n", errno);
        return -1;
    }
    NN_LOG("Established TCP connection (addr:0x%x on port :%d).\n",
         serverAddr.sin_addr.S_addr, nn::socket::InetNtohs(serverAddr.sin_port));

    m_TcpSocket = tcpSocket;
    return m_TcpSocket;
}

// ------------------------------------------------------------------------------------------------
// Setup SSL context
//   1. Create nn::ssl::Context by nn::ssl::Context::Create()
//   2. Get context ID of created context by nn::ssl::Context::GetContextId() for demonstration
//   3. Import CA certificate when IMPORT_SERVER_PKI is defined
//   4. Import client PKI information when IMPORT_CLIENT_PKI is defined
//
// Note that imported entities like CA certificates and client PKI information are volatile. These
// entities are copied into the SSL library and maintained until the SSL context created in this
// function is destroyed.
// ------------------------------------------------------------------------------------------------
nn::Result SimpleHttpsClient::SetupSslContext(nn::ssl::Context::SslVersion sslVersion)
{
    nn::Result result = nn::ResultSuccess();
    do
    {
        result = m_SslContext.Create(sslVersion);
        if(result.IsFailure())
        {
            NN_LOG("Failed to create context (Description:%d)\n",
                result.GetDescription());
            break;
        }

        nn::ssl::SslContextId contextId;
        result = m_SslContext.GetContextId(&contextId);
        if(result.IsFailure())
        {
            NN_LOG("Failed to get context ID (Description:%d)\n",
                result.GetDescription());
            break;
        }
        else
        {
            NN_LOG("Created SSL context (ID:%d).\n", contextId);
        }

#if defined(IMPORT_SERVER_PKI)
        char*    pCaCert = nullptr; // Please provide CA certificate in PEM format!
        uint32_t caCertSize = 0;    // Please set the size of CA certificate!
        result = m_SslContext.ImportServerPki(
            &m_serverCertStoreId,
            pCaCert,
            caCertSize,
            nn::ssl::CertificateFormat_Pem);
        if(result.IsFailure())
        {
            NN_LOG("Failed to import server PKI (Description:%d)\n",
                result.GetDescription());
            break;
        }
        else
        {
            NN_LOG("Imported server PKI (certStoreID:%d).\n", m_serverCertStoreId);
        }
#endif // IMPORT_SERVER_PKI

#if defined(IMPORT_CLIENT_PKI)
        char*       pPkcs12Data    = nullptr;    // Please set pointer to the data in PKCS12 format!
        uint32_t    pkcs12DataSize = 0;          // Please set the size of PKCS12 data!
        const char* pkcs12Password = "password"; // Please set the appropriate password for the PKCS12 data!
        uint32_t    passwordSize   = strlen(pkcs12Password);

        result = m_SslContext.ImportClientPki(
            &m_clientCertStoreId,
            pPkcs12Data,
            pkcs12Password,
            pkcs12DataSize,
            passwordSize);
        if(result.IsFailure())
        {
            NN_LOG("Failed to import client PKI (Description:%d)\n",
                result.GetDescription());
            break;
        }
        else
        {
            NN_LOG("Imported client PKI (certStoreID:%d).\n", m_clientCertStoreId);
        }
#endif // IMPORT_CLIENT_PKI

    }
    while (NN_STATIC_CONDITION(false));

    return result;
}

// ------------------------------------------------------------------------------------------------
// Setup SSL connection context
//   1. Create nn::ssl::Connection by nn::ssl::Connection::Create()
//   2. Set socket descriptor of TCP socket which alredy establised TCP connection by
//      nn::ssl::Connection::SetSocketDescriptor()
//   3. Set host name to be used for name validation during SSL handshake by
//      nn::ssl::Connection::SetHostName()
//   4. Set verify option by nn::ssl::Connection::SetVerifyOption
//   5. Set session cache mode by nn::ssl::Connection::SetSessionCacheMode
//   6. Set renegotion mode by nn::ssl::Connection::SetRenegotiationMode
//   7. Set blocking mode of I/O APIs (DoHandshake, Read, Write, Peek)
//   8. Get connection ID of created connection by nn::ssl::Context::GetConnectionId() for
//      demonstration
// ------------------------------------------------------------------------------------------------
bool SimpleHttpsClient::SetupSslConnection(nn::ssl::Connection::VerifyOption verifyOption)
{
    nn::Result result = m_SslConnection.Create(&m_SslContext);
    if(result.IsFailure())
    {
        NN_LOG("m_SslConnection.Create failed (Description:%d)\n",
            result.GetDescription());
        return false;
    }

    result = m_SslConnection.SetSocketDescriptor(m_TcpSocket);
    if(result.IsFailure())
    {
        NN_LOG("m_SslConnection.SetSocketDescriptor failed (Description:%d)\n",
            result.GetDescription());
        return false;
    }

    // Mark it as imported not to close it upon cleanup of SimpleHttpsClient
    m_IsSocketImported = true;

    if(ConfigureHostName() == false)
    {
        return false;
    }

    if(ConfigureVerifyOption(verifyOption) == false)
    {
        return false;
    }

    if(ConfigureSessionCacheMode() == false)
    {
        return false;
    }

    if(ConfigureRenegotiationMode() == false)
    {
        return false;
    }

    if(ConfigureIoMode() == false)
    {
        return false;
    }

    nn::ssl::SslConnectionId connectiontId;
    result = m_SslConnection.GetConnectionId(&connectiontId);
    if(result.IsFailure())
    {
        NN_LOG("Failed to get connection ID (Description:%d)\n",
            result.GetDescription());
        return false;
    }
    else
    {
        NN_LOG("Created SSL connection (ID:%d).\n", connectiontId);
    }

    return true;
}

bool SimpleHttpsClient::ConfigureVerifyOption(nn::ssl::Connection::VerifyOption verifyOption)
{
    nn::Result result = m_SslConnection.SetVerifyOption(verifyOption);
    if(result.IsFailure())
    {
        NN_LOG("m_SslConnection.SetVerifyOption failed (Description:%d)\n",
            result.GetDescription());
        return false;
    }

    nn::ssl::Connection::VerifyOption currentVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
    result = m_SslConnection.GetVerifyOption(&currentVerifyOption);
    if(result.IsFailure())
    {
        NN_LOG("m_SslConnection.GetVerifyOption failed (Description:%d)\n",
            result.GetDescription());
        return false;
    }

    NN_LOG("Verify option: %u\n", currentVerifyOption);

    return true;
}

bool SimpleHttpsClient::ConfigureSessionCacheMode()
{
    nn::Result result = nn::ResultSuccess();

#if defined(DISABLE_SESSION_CACHE) || defined(SESSION_CACHE_TICKET)
    result = m_SslConnection.SetSessionCacheMode(g_sessionCacheMode);
    if(result.IsFailure())
    {
        NN_LOG("Failed to set session cache mode.\n");
        return false;
    }
#endif

    nn::ssl::Connection::SessionCacheMode cacheMode = nn::ssl::Connection::SessionCacheMode_None;
    result = m_SslConnection.GetSessionCacheMode(&cacheMode);
    if(result.IsFailure())
    {
        NN_LOG("Failed to get session cache mode. Desc: %d\n", result.GetDescription());
        return false;
    }

    NN_LOG("Session cache mode: %u\n", cacheMode);
    return true;
}

bool SimpleHttpsClient::ConfigureRenegotiationMode()
{
    nn::Result result = nn::ResultSuccess();

#if defined(DISABLE_RENEGOTIATION)
    result = m_SslConnection.SetRenegotiationMode(nn::ssl::Connection::RenegotiationMode_None);
    if(result.IsFailure())
    {
        NN_LOG("Failed to set session renegotiation mode.\n", result.GetDescription());
        return false;
    }
#endif

    nn::ssl::Connection::RenegotiationMode renegotiationMode = nn::ssl::Connection::RenegotiationMode_None;
    result = m_SslConnection.GetRenegotiationMode(&renegotiationMode);
    if(result.IsFailure())
    {
        NN_LOG("Failed to get session renegotiation mode.\n", result.GetDescription());
        return false;
    }

    NN_LOG("Renegotiation Mode: %u\n", renegotiationMode);

    return true;
}

bool SimpleHttpsClient::ConfigureIoMode()
{
    nn::Result result = nn::ResultSuccess();

    if(m_IsNonBlocking)
    {
        result = m_SslConnection.SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
        if(result.IsFailure())
        {
            NN_LOG("Failed to configure I/O blocking mode (Description:%d)\n",
                result.GetDescription());
            return false;
        }
    }

    nn::ssl::Connection::IoMode ioMode = nn::ssl::Connection::IoMode_Blocking;
    result = m_SslConnection.GetIoMode(&ioMode);
    if(result.IsFailure())
    {
        NN_LOG("Failed to get I/O blocking mode (Description:%d)\n",
            result.GetDescription());
        return false;
    }

    NN_LOG("Io mode: %u\n", ioMode);

    return true;
}

bool SimpleHttpsClient::ConfigureHostName()
{
    bool isSuccess = true;

    uint32_t hostNameLen = static_cast<uint32_t>(strlen(m_HostName));
    nn::Result result = m_SslConnection.SetHostName(m_HostName, hostNameLen);
    if(result.IsFailure())
    {
        NN_LOG("m_SslConnection.SetHostName failed (Description:%d)\n",
            result.GetDescription());
        return false;
    }

    char* pOutHostname = new char[hostNameLen + 1];
    if(pOutHostname == nullptr)
    {
        NN_LOG("Failed to allocate hostname buffer\n");
        return false;
    }
    else
    {
        do
        {
            uint32_t outHostNameLen = 0;
            result = m_SslConnection.GetHostName(pOutHostname, &outHostNameLen, hostNameLen + 1);
            if(result.IsFailure())
            {
                NN_LOG("m_SslConnection.GetHostName failed (Description:%d)\n",
                    result.GetDescription());
                isSuccess = false;
                break;
            }

            if(outHostNameLen > 0)
            {
                NN_LOG("HostNameLen: %u, HostName: %s\n", outHostNameLen, pOutHostname);
            }
            else
            {
                NN_LOG("HostNameLen: %u\n", outHostNameLen);
            }
        } while(NN_STATIC_CONDITION(false));

        delete [] pOutHostname;
    }

    return isSuccess;
}

bool SimpleHttpsClient::PrintVerifyError(bool isPrintAllEnabled)
{
    bool isSuccess = true;
    nn::Result result = nn::ResultSuccess();

    if(isPrintAllEnabled)
    {
        nn::Result* pErrorArray = new nn::Result[VerifyResultArrayLen];
        if(pErrorArray != nullptr)
        {
            uint32_t errorsWritten = 0;
            uint32_t totalErrors = 0;

            result = m_SslConnection.GetVerifyCertErrors(pErrorArray, &errorsWritten, &totalErrors, VerifyResultArrayLen);
            if(result.IsFailure())
            {
                if(nn::ssl::ResultBufferTooShort::Includes(result))
                {
                    NN_LOG("nn::ssl::GetVerifyErrors() returned ResultBufferTooShort! ErrorsWritten: %u TotalErrors: %u\n\n", errorsWritten, totalErrors);
                }
                else
                {
                    NN_LOG("nn::ssl::GetVerifyErrors() failed! Desc: %d\n\n", result.GetDescription());
                }

                isSuccess = false;
            }
            else
            {
                NN_LOG("ErrorsWritten: %u TotalErrors: %u\n", errorsWritten, totalErrors);
                for(uint32_t iError = 0; iError < errorsWritten; ++iError)
                {
                    NN_LOG("Verify error %u: desc: %d\n", iError, pErrorArray[iError].GetDescription());
                }
            }

            delete [] pErrorArray;
            pErrorArray = nullptr;
        }
        else
        {
            NN_LOG("CurlSslConnectionAfterFunction: Failed to allocate Verify result array\n\n");
            isSuccess = false;
        }
    }
    else
    {
        nn::Result verifyCertError;
        result = m_SslConnection.GetVerifyCertError(&verifyCertError);
        if(result.IsFailure())
        {
            NN_LOG("Failed to get verify cert error (Description:%d)\n",
                result.GetDescription());
        }
        else
        {
            NN_LOG("Certificate validation failed (Description:%d)\n",
                verifyCertError.GetDescription());
        }
    }

    return isSuccess;
}

// ------------------------------------------------------------------------------------------------
// Initialize SimpleHttpsClient
//   1. Allocate buffer to store received data
//   2. Setup TCP socket
//   3. Setup SSL context
//   4. Setup SSL connection context
// ------------------------------------------------------------------------------------------------
bool SimpleHttpsClient::Initialize(
    nn::ssl::Context::SslVersion SslVersion,
    nn::ssl::Connection::VerifyOption verifyOption)
{
    bool       IsSuccess = true;
    nn::Result result;

    if(m_IsInitialized == true)
    {
        return false; // already initialized
    }

    do
    {
        m_pReceivedData = new char[ReadBufferSize];
        if(m_pReceivedData == nullptr)
        {
            NN_LOG("Failed to allocate memory for read buffer.\n");
            IsSuccess = false;
            break;
        }
        NN_LOG("Allocated %d bytes for receive buffer.\n", ReadBufferSize);
        memset(m_pReceivedData, 0x00, ReadBufferSize);

        if(this->SetupTcpSocket() < 0)
        {
            NN_LOG("Failed to setup TCP socket.\n");
            IsSuccess = false;
            break;
        }

        if(SetupSslContext(SslVersion).IsFailure())
        {
            NN_LOG("Failed to setup SSL context.\n");
            IsSuccess = false;
            break;
        }

        if(SetupSslConnection(verifyOption) == false)
        {
            NN_LOG("Failed to setup SSL connection context.\n");
            IsSuccess = false;
            break;
        }
    }
    while (NN_STATIC_CONDITION(false));

    if((m_pReceivedData!= nullptr) && (IsSuccess == false))
    {
        delete[] m_pReceivedData;
        m_pReceivedData = nullptr;
    }

    NN_LOG("Initialized SimpleHttpsClient.\n");
    m_IsInitialized = true;
    return IsSuccess;
}

// ------------------------------------------------------------------------------------------------
// Finalize SimpleHttpsClient
//   1. Close TCP socket (if it's created and already imported into the SSL library)
//   2. Destroy nn::ssl::Connection
//   3. Destroy nn::ssl::Context
//   4. Free allocated memory
// ------------------------------------------------------------------------------------------------
void SimpleHttpsClient::Finalize()
{
    // Cloes TCP socket
    if(m_TcpSocket > 0 && m_IsSocketImported != true)
    {
        nn::socket::Close(m_TcpSocket);
    }

    // Destroy nn::ssl::Connection first
    nn::ssl::SslConnectionId connectiontId;
    if(m_SslConnection.GetConnectionId(&connectiontId).IsSuccess())
    {
        if(connectiontId != 0)
        {
#if defined(FLUSH_SESSION_CACHE)
            nn::Result flushResult = m_SslConnection.FlushSessionCache();
            if(flushResult.IsFailure())
            {
                NN_LOG("Warning: Failed to flush session cache during SimpleHttpsClient::Finalize! Desc: %d\n", flushResult.GetDescription());
            }
#endif
            m_SslConnection.Destroy();
        }
    }

    // Destroy nn::ssl::Context after nn::ssl::Connection is destroyed
    nn::ssl::SslContextId contextId;
    if(m_SslContext.GetContextId(&contextId).IsSuccess())
    {
        if(contextId != 0)
        {
#if defined(IMPORT_SERVER_PKI)
            nn::Result removeServerResult = m_SslContext.RemovePki(m_serverCertStoreId);
            if(removeServerResult.IsFailure())
            {
                NN_LOG("Warning: Failed to remove server pki during SimpleHttpsClient::Finalize! Desc: %d\n", removeServerResult.GetDescription());
            }
#endif

#if defined(IMPORT_CLIENT_PKI)
            nn::Result removeClientResult = m_SslContext.RemovePki(m_clientCertStoreId);
            if(removeClientResult.IsFailure())
            {
                NN_LOG("Warning: Failed to remove client pki during SimpleHttpsClient::Finalize! Desc: %d\n", removeClientResult.GetDescription());
            }
#endif
            m_SslContext.Destroy();
        }
    }

    // Free memory
    if(m_pReceivedData != nullptr)
    {
        delete[] m_pReceivedData;
        m_pReceivedData = nullptr;
    }

    m_IsInitialized = false;
    NN_LOG("Finalized SimpleHttpsClient.\n");
}

// ------------------------------------------------------------------------------------------------
// Perform SSL handshake
//   1. Allocate buffer to store server certificate and set it to the SSL library by
//      nn::ssl::Connection::SetServerCertBuffer() if required
//   2. Start SSL handshake by nn::ssl::Connection::DoHandshake()
//   3. If the I/O mode of nn::ssl::Connection is non-blocking and nn::ssl::ResultIoWouldBlock is
//      returned from nn::ssl::Connection::DoHandshake(), try DoHandshake again
//   4. Print server certificate when it is requried
// ------------------------------------------------------------------------------------------------
bool SimpleHttpsClient::PerformSslHandshake(bool isPrintServerCert)
{
    nn::Result result              = nn::ResultSuccess();
    bool       isSuccess           = true;
    char*      serverCertBuff      = nullptr;
    uint32_t   asyncHandshakeCount = 0;
    uint32_t   serverCertLength    = 0;

    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    if(isPrintServerCert == true)
    {
        // Allocate buffer for server certificate and pass it to the SSL library
        serverCertBuff = new char[this->ServerCertBufferSize];
        if(serverCertBuff == nullptr)
        {
            return false; // Failed to allocate buffer
        }
        memset(serverCertBuff, 0x00, this->ServerCertBufferSize);

        result = m_SslConnection.SetServerCertBuffer(serverCertBuff, this->ServerCertBufferSize);
        if(result.IsFailure())
        {
            NN_LOG("SetServerCertBuffer failed (Description:%d)\n", result.GetDescription());
            delete[] serverCertBuff;
            return false;
        }
    }

    do
    {
        result = (serverCertBuff)?
            m_SslConnection.DoHandshake(&serverCertLength, nullptr):m_SslConnection.DoHandshake();
        if(result.IsSuccess())
        {
            NN_LOG("SSL Handshake completed.\n");
            break;
        }

        if(nn::ssl::ResultBufferTooShort::Includes(result))
        {
            uint32_t neededBufSize = 0;
            nn::Result tempResult = m_SslConnection.GetNeededServerCertBufferSize(&neededBufSize);
            if(tempResult.IsFailure())
            {
                NN_LOG("Error: GetNeededServerCertBufferSize() failed with desc: %d\n", tempResult.GetDescription());
                isSuccess = false;
                break;
            }

            // Failed to store server certificate, but it is ok to continue
            NN_LOG("Provided buffer was not long enough to store server cert. Required buffer lenth: %u\n", neededBufSize);
        }
        else if(nn::ssl::ResultIoWouldBlock::Includes(result))
        {
            if(asyncHandshakeCount++ < this->MaxAsyncDoHandshakeRetry)
            {
                NN_LOG("Handshake would block, try again after 100 msec (retry count:%d)\n",
                    asyncHandshakeCount);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
            else
            {
                NN_LOG("Handshake failed after %d times of execution\n", asyncHandshakeCount);
                break;
            }
        }
        else
        {
            NN_LOG("SSL Handshake failed (Description:%d)\n", result.GetDescription());
            if(nn::ssl::ResultVerifyCertFailed::Includes(result))
            {
                if( PrintVerifyError(g_IsPrintAllVerifyErrors) == false )
                {
                    isSuccess = false;
                }
            }
            break;
        }
    }
    while (NN_STATIC_CONDITION(true));

    if(result.IsSuccess() && (serverCertLength > 0))
    {
        NN_LOG("Obtained server certiticate (%d bytes) ---------------------\n", serverCertLength);
        PRINT_HEX_TABLE(serverCertBuff, serverCertLength);
        NN_LOG("\n\n");
    }

    if(serverCertBuff != nullptr)
    {
        delete[] serverCertBuff;
    }

    return isSuccess;
}

// ------------------------------------------------------------------------------------------------
// Send HTTP GET request over SSL connection
//   1. Setup HTTP GET request
//   2. Send HTTP GET request over SSL by nn::ssl::Connection::Write()
// ------------------------------------------------------------------------------------------------
bool SimpleHttpsClient::SendHttpGetOverSsl()
{
    char httpReqBuff[64] = {0};

    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    sprintf(httpReqBuff, "GET / HTTP/1.0\r\nHost: %s\r\n\r\n", m_HostName);

    int sentBytes     = 0;
    nn::Result result = m_SslConnection.Write(
        httpReqBuff,
        &sentBytes,
        static_cast<uint32_t>(strlen(httpReqBuff)));
    if(result.IsFailure() || sentBytes < 0)
    {
        NN_LOG("Failed to write data. (Description:%d)\n", result.GetDescription());
    }
    else
    {
        NN_LOG("Sent HTTP request via SSL (%d bytes).\n", sentBytes);
    }

    return (result.IsSuccess())?(true):(false);
}

// ------------------------------------------------------------------------------------------------
// Perform SSL read until:
//   1. If the I/O mode of nn::ssl::Connection is non-blocking Call nn::ssl::Connection::Poll() to
//      wait for the SSL connection becomes ready to read
//   2. Read received data by nn::ssl::Connection::Read() until:
//     - There's a space to store the data in the buffer (m_pReceivedData)
//     - connection gets closed by the peer
//     - timeout happens on nn::ssl::Connection::Poll
// ------------------------------------------------------------------------------------------------
bool SimpleHttpsClient::ReceiveAllHttpData()
{
    nn::Result                     result = nn::ResultSuccess();
    nn::ssl::Connection::PollEvent pollEvent;
    nn::ssl::Connection::PollEvent pollOutEvent;

    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    memset(m_pReceivedData, 0x00, ReadBufferSize);
    do
    {
        int receivedBytes  = 0;
        int ReadSize       = ReadBufferSize - m_ReceivedTotalBytes;

        if(ReadSize <= 0)
        {
            NN_LOG("There's no more space to store received data (%d bytes received).\n",
                m_ReceivedTotalBytes);
            break;
        }

        if(m_IsNonBlocking)
        {
            pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;  // reset
            pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Read; // set read event

            result = m_SslConnection.Poll(&pollOutEvent, &pollEvent, MsecPollTimeout);
            if(result.IsFailure())
            {
                NN_LOG("Failed due to Poll error (Description:%d)\n", result.GetDescription());
                break;
            }

            if(nn::ssl::ResultIoTimeout::Includes(result))
            {
                NN_LOG("Finished receiving data for the timeout.\n");
                break;
            }

            if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read)
               == nn::ssl::Connection::PollEvent::PollEvent_Read)
            {
                result = m_SslConnection.Read(
                    (m_pReceivedData + m_ReceivedTotalBytes),
                    &receivedBytes,
                    ReadSize);
                if(nn::ssl::ResultIoWouldBlock::Includes(result))
                {
                    continue;
                }
            }
        }
        else
        {
            result = m_SslConnection.Read(
                (m_pReceivedData + m_ReceivedTotalBytes),
                &receivedBytes,
                ReadSize);
        }

        if(result.IsFailure() || receivedBytes < 0)
        {
            NN_LOG("Failed to read data (Description:%d)\n", result.GetDescription());
            break;
        }

        if(receivedBytes == 0)
        {
            NN_LOG("Connection closed by the server.\n");
            break;
        }

        m_ReceivedTotalBytes += receivedBytes;
        NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, m_ReceivedTotalBytes);
    } while(NN_STATIC_CONDITION(true));

    return (result.IsSuccess())?(true):(false);
}

// ------------------------------------------------------------------------------------------------
// Print received data on the console
// ------------------------------------------------------------------------------------------------
void SimpleHttpsClient::PrintReceivedData()
{
    if(m_IsInitialized == false)
    {
        return; // not initialized
    }

    if(m_ReceivedTotalBytes == 0)
    {
        NN_LOG("Nothing is received yet.\n");
        return;
    }

    NN_LOG("\n------------ Received body (total: %d bytes received)---------------\n",
            m_ReceivedTotalBytes);
    NN_LOG("%.*s", m_ReceivedTotalBytes, m_pReceivedData);
    NN_LOG("\n--------------------------------------------------------------------\n");
}

} // Un-named namespace

// ------------------------------------------------------------------------------------------------
// Startup function
// ------------------------------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    // Setup the size of total memory heap
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    NN_ASSERT( result.IsSuccess() );

    // Allcate memory block from the heap for malloc
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess() );

    // Set memory chunk for malloc
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// nnMain
// ------------------------------------------------------------------------------------------------
extern "C" void nnMain()
{
    int        argc                   = nn::os::GetHostArgc();
    char**     argv                   = nn::os::GetHostArgv();
    char       pHostName[64]          = {0};
    uint32_t   hostNameLen            = 0;
    bool       isSocketLibInitialized = false;
    bool       isSslLibInitialized    = false;

    if(argc == 2)
    {
        NN_LOG("Will use %s as a host name\n", argv[1]);
        strncpy(pHostName, argv[1], sizeof(pHostName));
    }
    else
    {
        strncpy(pHostName, "example.com", sizeof(pHostName));
    }
    hostNameLen = static_cast<uint32_t>(strnlen(pHostName, sizeof(pHostName)));

    // Setup network interface
    if(SetupNetwork().IsFailure())
    {
        NN_LOG("Failed to enable network interface\n");
        return;
    }

    do
    {
        nn::Result result = nn::ResultSuccess();
        // Initialize the socket library
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        if(result.IsFailure())
        {
            NN_LOG("Failed to initialize socket library.\n");
            break;
        }
        isSocketLibInitialized = true;

        // Initialize the SSL library
        result = nn::ssl::Initialize();
        if(result.IsFailure())
        {
            NN_LOG("Failed to initialize SSL library.\n");
            break;
        }
        isSslLibInitialized = true;

        SimpleHttpsClient httpsClient(g_IsPerformNonBlocking, pHostName, g_TcpPortNumber);
        do{
            // By default, this sample verifies the server certificate and its host name
            nn::ssl::Connection::VerifyOption verifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
            verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa;
            verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_HostName;

            // Initialize SSL context and connection context
            if(httpsClient.Initialize(nn::ssl::Context::SslVersion::SslVersion_Auto, verifyOption) == false)
            {
                break;
            }

            // Perform SSL handshake
            if(httpsClient.PerformSslHandshake(g_IsPrintServerCert) == false)
            {
                break;
            }

            // Send HTTP GET request over SSL connection
            if(httpsClient.SendHttpGetOverSsl() == false)
            {
                break;
            }

            // Receive HTTP data via SSL connection
            if(httpsClient.ReceiveAllHttpData() == false)
            {
                break;
            }

            // Print out received contents on the console
            httpsClient.PrintReceivedData();
        } while (NN_STATIC_CONDITION(false));

        // Finalize ssl connection context and context
        httpsClient.Finalize();
    } while (NN_STATIC_CONDITION(false));

    if(isSocketLibInitialized)
    {
        nn::socket::Finalize();
    }
    if(isSslLibInitialized)
    {
        nn::ssl::Finalize();
    }

    FinalizeNetwork();
}
