﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{SpyAudio.cpp,PageSampleSpyAudio}
 *
 * @brief
 *  nn::spy, nn::spy::audio ライブラリのサンプルプログラム
 */

/**
 * @page PageSampleSpyAudio SpyAudio
 * @tableofcontents
 *
 * @brief
 *  nn::spy, nn::spy::audio ライブラリ サンプルプログラムの解説です。
 *
 * @section PageSampleSpyAudio_SectionBrief 概要
 *  nn::spy, nn::spy::audio ライブラリを使って、 nn::audio の状態をモニタするサンプルです。
 *
 * @section PageSampleSpyAudio_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/SpyAudio Samples/Sources/Applications/SpyAudio @endlink 以下にあります。
 *
 * @section PageSampleSpyAudio_SectionNecessaryEnvironment 必要な環境
 *  オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleSpyAudio_SectionHowToOperate 操作方法
 *  キーボードや DebugPad による入力を用いて SE を再生し、 nn::audio の処理負荷や出力波形を Spy ツールで確認することができます。
 *  サンプルを起動するとコンソールに操作方法が表示されます。
 *  Spy ツールのリボンから接続先を設定して接続トグルボタンを ON にすると、通信が開始されます。
 *
 * @section PageSampleSpyAudio_SectionPrecaution 注意事項
 *  Release ビルドでは、Spy.exe と通信できません。
 *  開発用機能である Spy は、誤って製品に含まれないようにするために、Release ビルドでは無効化されています。
 *  Release ビルドで Spy を利用したい場合は、spy_Config.h を編集して NN_ENABLE_SPY マクロを定義する必要があります。
 *
 * @section PageSampleSpyAudio_SectionHowToExecute 実行手順
 *  1. サンプルプログラムを Debug or Develop ビルドし、実行します。
 *  2. NintendoTargetManager が起動していない場合は、起動します。
 *  3. PC 上で Spy ツールを起動して、サンプルプログラムと接続します。
 *
 * @section PageSampleSpyAudio_SectionDetail 解説
 *
 * @subsection PageSampleSpyAudio_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  SpyAudio.cpp
 *  @includelineno SpyAudio.cpp
 *
 * @subsection PageSampleSpyAudio_SectionSampleDetail サンプルプログラムの解説
 *  このサンプルプログラムの流れは、以下の通りです。
 *
 * - htcs, audio などの初期化
 * - InitializeSpy()
 *   - nn::spy::SpyController の初期化
 * - InitializeSpyAudio()
 *   - nn::spy::audio::AudioSpyModule の初期化
 *   - nn::spy::audio::AudioSpyModule を nn::spy::SpyController に登録
 * - StartSpy()
 *   - Spy ツールとの通信を開始
 * - メインループ
 *   - (必要に応じて) nn::spy::SpyController::SetCurrentApplicationFrame() でアプリケーションフレームを送信
 *   - nn::spy::audio::AudioSpyModule::PushPerformanceMetrics() で audio パフォーマンス情報を送信
 *   - nn::spy::audio::AudioSpyModule::PushWaveform() で出力波形を送信
 *   - nn::spy::DebugModule::PushDataBufferUsage() で Spy のデバッグ情報（データバッファ使用量）を送信
 * - StopSpy()
 *   - Spy ツールとの通信を終了
 * - FinalizeSpyAudio()
 *   - nn::spy::audio::AudioSpyModule を nn::spy::SpyController から登録解除
 *   - nn::spy::audio::AudioSpyModule の終了
 * - FinalizeSpy()
 *   - nn::spy::SpyController の終了
 * - htcs, audio などの終了
 *
 */

#include <algorithm>
#include <climits>
#include <cmath>

#include <nn/nn_Assert.h>
#include <nns/nns_Log.h>

#include <nn/fs.h>
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/htcs.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/settings.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/util/util_FormatString.h>

#include <nn/spy.h>
#include <nn/spy/audio/spy_AudioSpyModule.h>

#include "Audio.h"

#if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic ignored "-Wunused-const-variable"
#pragma clang diagnostic ignored "-Wunused-function"
#endif

//-----------------------------------------------------------------------------

namespace {

    const size_t HeapSize = 128 * 1024 * 1024;
    const size_t SpyControllerDataBufferLength = 1024 * 1024;

    char g_Heap[HeapSize];
    nn::mem::StandardAllocator g_Allocator;

    const nn::hid::NpadIdType g_NpadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::Handheld,
    };

    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

    nn::hid::DebugPadButtonSet g_PadButtonSet;
    nn::hid::DebugPadButtonSet g_PadLastButtonSet;

    nn::spy::SpyController g_SpyController;
    nn::spy::audio::AudioSpyModule g_AudioSpyModule;

    const nn::spy::audio::AudioSpyModule::ChannelType g_ChannelTypes[] =
    {
        nn::spy::audio::AudioSpyModule::ChannelType_FrontLeft,
        nn::spy::audio::AudioSpyModule::ChannelType_FrontRight,
    };

    void* g_WorkBufferForSpyController = nullptr;
    void* g_WorkBufferForAudioSpyModule = nullptr;

    int g_AppFrameCount = 0;

    void* g_MountRomCacheBuffer = nullptr;

    void* Alloc(size_t size)
    {
        return g_Allocator.Allocate(size);
    }

    void Free(void* addr, size_t size)
    {
        NN_UNUSED(size);
        g_Allocator.Free(addr);
    }

    void InitializeHid()
    {
        nn::hid::InitializeDebugPad();

        //キーボードのキーを DebugPad のボタンに割り当てます。
        nn::settings::DebugPadKeyboardMap map;
        nn::settings::GetDebugPadKeyboardMap(&map);
        map.buttonA = nn::hid::KeyboardKey::A::Index;
        map.buttonB = nn::hid::KeyboardKey::B::Index;
        map.buttonX = nn::hid::KeyboardKey::X::Index;
        map.buttonY = nn::hid::KeyboardKey::Y::Index;
        map.buttonL = nn::hid::KeyboardKey::L::Index;
        map.buttonR = nn::hid::KeyboardKey::R::Index;
        map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
        map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
        map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
        map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
        map.buttonStart = nn::hid::KeyboardKey::Space::Index;
        nn::settings::SetDebugPadKeyboardMap(map);

        nn::hid::InitializeNpad();

        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
        nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);
    }

    void InitializeHtcs()
    {
        nn::htcs::Initialize(Alloc, Free);
    }

    void FinalizeHtcs()
    {
        nn::htcs::Finalize();
    }

    void InitializeFileSystem()
    {
        nn::fs::SetAllocator(Alloc, Free);

        size_t cacheSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
        g_MountRomCacheBuffer = g_Allocator.Allocate(cacheSize);
        NN_ABORT_UNLESS(cacheSize == 0 || g_MountRomCacheBuffer != nullptr);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
        );
    }

    void FinalizeFileSystem()
    {
        nn::fs::Unmount("asset");

        g_Allocator.Free(g_MountRomCacheBuffer);
        g_MountRomCacheBuffer = nullptr;
    }

    void InitializeSpy()
    {
        // nn::spy::SpyController を初期化します。
        nn::spy::SpyController::InitializeArg initializeArg;
        initializeArg.dataBufferLength = SpyControllerDataBufferLength;

        size_t bufferSize = nn::spy::SpyController::GetRequiredMemorySize(initializeArg);
        g_WorkBufferForSpyController = g_Allocator.Allocate(bufferSize);
        NN_ABORT_UNLESS(bufferSize == 0 || g_WorkBufferForSpyController != nullptr);

        g_SpyController.Initialize(initializeArg, g_WorkBufferForSpyController, bufferSize);

        g_AppFrameCount = 0;
    }

    void FinalizeSpy()
    {
        g_SpyController.Finalize();

        if (g_WorkBufferForSpyController)
        {
            g_Allocator.Free(g_WorkBufferForSpyController);
            g_WorkBufferForSpyController = nullptr;
        }
    }

    void InitializeSpyAudio()
    {
        // バッファを確保して、AudioSpyModule を初期化
        const nn::audio::AudioRendererParameter audioRendererParameter = GetAudioRendererParameter();

        nn::spy::audio::AudioSpyModule::InitializeArg initializeArg;
        initializeArg.pAudioRendererParameter = &audioRendererParameter;
        initializeArg.waveformChannelCount = sizeof(g_ChannelTypes) / sizeof(g_ChannelTypes[0]);
        initializeArg.pWaveformChannelTypes = g_ChannelTypes;
        initializeArg.waveformSampleFormat = nn::audio::SampleFormat_PcmInt16;
        size_t bufferSize = nn::spy::audio::AudioSpyModule::GetRequiredMemorySize(initializeArg);
        g_WorkBufferForAudioSpyModule = g_Allocator.Allocate(bufferSize);
        NN_ABORT_UNLESS(bufferSize == 0 || g_WorkBufferForAudioSpyModule != nullptr);
        g_AudioSpyModule.Initialize(initializeArg, g_WorkBufferForAudioSpyModule, bufferSize);

        // AudioSpyModule の登録
        g_SpyController.RegisterModule(g_AudioSpyModule);
    }

    void FinalizeSpyAudio()
    {
        // AudioSpyModule の登録解除
        g_SpyController.UnregisterModule(g_AudioSpyModule);

        // AudioSpyModule の終了処理
        g_AudioSpyModule.Finalize();

        // AudioSpyModule のバッファ解放
        if (g_WorkBufferForAudioSpyModule)
        {
            g_Allocator.Free(g_WorkBufferForAudioSpyModule);
            g_WorkBufferForAudioSpyModule = nullptr;
        }
    }

    // Spy ツールと通信を開始します。
    void StartSpy()
    {
        nn::spy::SpyController::OpenArg openArg;
        bool openResult = g_SpyController.Open(openArg);
        NN_ABORT_UNLESS(openResult);
    }

    // Spy ツールとの通信を終了します。
    void StopSpy()
    {
        g_SpyController.Close();
    }

    // AudioSpyModule を使って、nn::audio から取得したパフォーマンスデータを Spy ツールに送信します。
    void PushAudioPerformanceMetrics()
    {
        void* audioPerformanceBuffer = SwitchAudioPerformanceFrameBuffer();
        g_AudioSpyModule.PushPerformanceMetrics(audioPerformanceBuffer, GetRequiredBufferSizeForAudioPerformanceFrames());
    }

    // AudioSpyModule を使って、nn::audio から取得した出力波形を Spy ツールに送信します。
    void PushWaveform(nn::os::Tick tick)
    {
        const void* buffer;
        size_t readSize = ReadCircularBufferSink(&buffer);
        if (readSize > 0)
        {
            g_AudioSpyModule.PushWaveform(buffer, readSize, tick);
        }
    }

    // DebugModule を使って、Spy のデバッグ情報を Spy ツールに送信します。
    void PushSpyDebugInfo()
    {
        // データバッファ使用量を送信します。
        g_SpyController.GetDebugModule().PushDataBufferUsage();
    }

    // Npad のボタン状態を DebugPad のボタン状態に変換します。
    const nn::hid::DebugPadButtonSet MapNpadButtonSet(const nn::hid::NpadButtonSet& buttons)
    {
        static const struct {
            nn::hid::NpadButtonSet from;
            nn::hid::DebugPadButtonSet to;
        } buttonMapList[] = {
            {
                nn::hid::NpadButton::A::Mask,
                nn::hid::DebugPadButton::A::Mask,
            },
            {
                nn::hid::NpadButton::B::Mask,
                nn::hid::DebugPadButton::B::Mask,
            },
            {
                nn::hid::NpadButton::X::Mask,
                nn::hid::DebugPadButton::X::Mask,
            },
            {
                nn::hid::NpadButton::Y::Mask,
                nn::hid::DebugPadButton::Y::Mask,
            },
            {
                nn::hid::NpadButton::L::Mask,
                nn::hid::DebugPadButton::L::Mask,
            },
            {
                nn::hid::NpadButton::R::Mask,
                nn::hid::DebugPadButton::R::Mask,
            },
            {
                nn::hid::NpadButton::Left::Mask,
                nn::hid::DebugPadButton::Left::Mask,
            },
            {
                nn::hid::NpadButton::Right::Mask,
                nn::hid::DebugPadButton::Right::Mask,
            },
            {
                nn::hid::NpadButton::Up::Mask,
                nn::hid::DebugPadButton::Up::Mask,
            },
            {
                nn::hid::NpadButton::Down::Mask,
                nn::hid::DebugPadButton::Down::Mask,
            },
            {
                nn::hid::NpadButton::Plus::Mask,
                nn::hid::DebugPadButton::Start::Mask,
            },
        };

        nn::hid::DebugPadButtonSet result = {};

        for (const auto& buttonMap : buttonMapList)
        {
            if ((buttons & buttonMap.from) == buttonMap.from)
            {
                result |= buttonMap.to;
            }
        }

        return result;
    }

    // デバッグパッドを更新します。
    nn::hid::DebugPadButtonSet UpdateDebugPad()
    {
        g_PadLastButtonSet = g_PadButtonSet;

        {
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            g_PadButtonSet = state.buttons;
        }

        for (auto id : g_NpadIds)
        {
            const auto styles = nn::hid::GetNpadStyleSet(id);

            if (styles.Test<nn::hid::NpadStyleFullKey>())
            {
                nn::hid::NpadFullKeyState state;
                nn::hid::GetNpadState(&state, id);
                g_PadButtonSet |= MapNpadButtonSet(state.buttons);
            }

            if (styles.Test<nn::hid::NpadStyleHandheld>())
            {
                nn::hid::NpadHandheldState state;
                nn::hid::GetNpadState(&state, id);
                g_PadButtonSet |= MapNpadButtonSet(state.buttons);
            }
        }

        return ~g_PadLastButtonSet & g_PadButtonSet;
    }

    void PrintUsage()
    {
        NNS_LOG("-----------------------------\n");
        NNS_LOG(" SpyAudio:\n");
        NNS_LOG("\n");
#if !defined(NN_BUILD_CONFIG_SPY_ENABLED)
        NNS_LOG(" Warning!\n");
        NNS_LOG("   nn::spy is disabled.\n");
        NNS_LOG("   nn::spy is only available on Debug or Develop build.\n");
        NNS_LOG("\n");
#endif
        NNS_LOG(" [A]              play SE 0.\n");
        NNS_LOG(" [B]              play SE 1.\n");
        NNS_LOG(" [X]              play SE 2.\n");
        NNS_LOG(" [Y]              play SE 3.\n");
        NNS_LOG(" [L]              enable / disable Reverb.\n");
        NNS_LOG(" [+/Start][Space] exit.\n");
        NNS_LOG("-----------------------------\n");
    }

    void PrintMessageLine(const char* message)
    {
        NNS_LOG(message);
        NNS_LOG("\n");
        nn::spy::LogModule::Write(g_SpyController, message);
    }

    void PrintMessage(const char* message)
    {
        NNS_LOG(message);
        nn::spy::LogModule::Write(g_SpyController, message);
    }

    void PrintResult(bool result)
    {
        const char* strResult = result ? " (success)" : " (failure)";
        NNS_LOG(strResult);
        NNS_LOG("\n");
        nn::spy::LogModule::Write(g_SpyController, strResult);
    }

    void Mainloop()
    {
        PrintUsage();

        while (NN_STATIC_CONDITION(true))
        {
            WaitAudioRendererEvent();

            // アプリケーションフレームの開始時間を記録します。
            g_SpyController.SetCurrentApplicationFrame(g_AppFrameCount++);

            nn::hid::DebugPadButtonSet triggered = UpdateDebugPad();

            // SE を再生します。
            if (triggered.Test<nn::hid::DebugPadButton::A>())
            {
                PrintMessage("PlaySe(0) ...");
                bool result = PlaySe(0);
                PrintResult(result);
            }
            if (triggered.Test<nn::hid::DebugPadButton::B>())
            {
                PrintMessage("PlaySe(1) ...");
                bool result = PlaySe(1);
                PrintResult(result);
            }
            if (triggered.Test<nn::hid::DebugPadButton::X>())
            {
                PrintMessage("PlaySe(2) ...");
                bool result = PlaySe(2);
                PrintResult(result);
            }
            if (triggered.Test<nn::hid::DebugPadButton::Y>())
            {
                PrintMessage("PlaySe(3) ...");
                bool result = PlaySe(3);
                PrintResult(result);
            }

            // リバーブの ON / OFF を切り替えます。
            if (triggered.Test<nn::hid::DebugPadButton::L>())
            {
                PrintMessageLine(IsEnableReverb() ? "EnableReverb(false)" : "EnableReverb(true)");
                EnableReverb(!IsEnableReverb());
            }

            if (triggered.Test<nn::hid::DebugPadButton::Start>())
            {
                PrintMessageLine("Exit");
                return;
            }

            RequestUpdateAudioRenderer();
            nn::os::Tick tick = nn::os::GetSystemTick();

            PushAudioPerformanceMetrics();
            PushWaveform(tick);
            PushSpyDebugInfo();
        }
    }

} // namespace {anonymous}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    g_Allocator.Initialize(g_Heap, sizeof(g_Heap));

    InitializeHid();
    InitializeHtcs();
    InitializeFileSystem();

    // Audio を初期化します。
    InitializeAudio(g_Allocator);
    InitializeAudioPerformanceMetrics(g_Allocator);
    InitializeCircularBufferSink(g_Allocator);

    // Spy の初期化して開始します。
    InitializeSpy();
    InitializeSpyAudio();
    StartSpy();

    // メインループです。
    Mainloop();

    // Spy を停止して、終了します。
    StopSpy();
    FinalizeSpyAudio();
    FinalizeSpy();

    // Audio を終了します。
    FinalizeCircularBufferSink(g_Allocator);
    FinalizeAudioPerformanceMetrics(g_Allocator);
    FinalizeAudio(g_Allocator);

    FinalizeFileSystem();
    FinalizeHtcs();
}

