﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{PrepoSystemSimple.cpp,PageSamplePrepoSystemSimple}
 *
 * @brief
 *  システムプレイレポートのサンプルプログラム
 */

/**
 * @page PageSamplePrepoSystemSimple システムプレイレポートの利用
 * @tableofcontents
 *
 * @brief
 *  システム用のプレイレポートを生成して保存するサンプルプログラムの解説です。
 *
 * @section PageSamplePrepoSystemSimple_SectionBrief 概要
 *  ここでは、プレイレポートライブラリの利用方法について説明します。@n
 *  プレイレポートの概要は、PlayReportKit のドキュメント、および、プレイレポートガイドをご一読ください。
 *
 *  一般向けと異なるのは、ターゲットとなるアプリケーション ID を自ら指定する点です。@n
 *  複数のシステムアプリケーションにまたがって共通のレポートデータを記録する場合、代表となるアプリケーション ID を指定してください。
 *
 * @section PageSamplePrepoSystemSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/PrepoSystemSimple Samples/Sources/Applications/PrepoSystemSimple @endlink 以下にあります。
 *
 * @section PageSamplePrepoSystemSimple_SectionNecessaryEnvironment 必要な環境
 *  ユーザーを指定してプレイレポートを保存する場合、ニンテンドーアカウントが紐付いたユーザーアカウントを作成する必要があります。@n
 *  デバッグ機能を利用することで、ニンテンドーアカウントを紐付けなくてもプレイレポートが保存可能です。@n
 *  詳しくは、PlayReportKit のドキュメントを参照してください。
 *
 * @section PageSamplePrepoSystemSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSamplePrepoSystemSimple_SectionPrecaution 注意事項
 *  ネットワーク接続が可能な環境が用意できない場合、レポートデータはサーバーに送信されません。
 *
 * @section PageSamplePrepoSystemSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSamplePrepoSystemSimple_SectionDetail 解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - nn::prepo::SystemPlayReport オブジェクトにイベント ID とワークバッファを指定する。
 *  - レポートするキーとバリューを書き込む。
 *  - 保存する。
 */

#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

// 乱数を取得します。
int64_t GetRandom() NN_NOEXCEPT
{
    nn::Bit64 value;
    nn::os::GenerateRandomBytes(&value, sizeof (value));

    return static_cast<int64_t>(value & 0x7FFFFFFFFFFFFFFull);
}

extern "C" void nnMain()
{
    nn::prepo::SystemPlayReport report("system_sample_event");

    // 一般向けプレイレポートサンプル用のアプリケーション ID
    nn::ApplicationId sampleAppId = {0x0100c0d000494000};

    // 一般向けプレイレポートサンプルと同一のアプリケーション ID としてレポートデータを保存します。
    // 本サンプル自体は、meta を指定していないためデフォルトのアプリケーション ID が割り当てられています。
    report.SetApplicationId(sampleAppId);

    // レポートデータの一時的に保存するバッファです。
    // 必要なバッファサイズはキーバリューの数によって変動します。（サンプルでは、3 つのデータを記録するので 3 を指定しています）
    static nn::Bit8 buffer[nn::prepo::PlayReport::BufferSizeMin + 3 * nn::prepo::KeyValueSizeMax];

    report.SetBuffer(buffer, sizeof (buffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("x", GetRandom() % 100));
    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("y", GetRandom() % 100));
    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("z", GetRandom() % 100));

    // レポートデータを保存する際、エラーが発生してもエラービューア等でエラーをユーザーに伝えてはいけません。
    // プレイレポートはアプリケーションの進行に必須の機能ではなく、エラーを伝えるとユーザー体験の妨げになるからです。
    report.Save();

    NN_LOG("[sample] End.\n");
}
