﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{PrepoSimple.cpp,PageSamplePrepoSimple}
 *
 * @brief
 *  プレイレポートのサンプルプログラム
 */

/**
 * @page PageSamplePrepoSimple プレイレポートの利用
 * @tableofcontents
 *
 * @brief
 *  プレイレポートを生成して保存するサンプルプログラムの解説です。
 *
 * @section PageSamplePrepoSimple_SectionBrief 概要
 *  ここでは、プレイレポートライブラリの利用方法について説明します。@n
 *  プレイレポートの概要は、PlayReportKit のドキュメント、および、プレイレポートガイドをご一読ください。
 *
 * @section PageSamplePrepoSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/PrepoSimple Samples/Sources/Applications/PrepoSimple @endlink 以下にあります。
 *
 * @section PageSamplePrepoSimple_SectionNecessaryEnvironment 必要な環境
 *  ユーザーを指定してプレイレポートを保存する場合、ニンテンドーアカウントが紐付いたユーザーアカウントを作成する必要があります。@n
 *  デバッグ機能を利用することで、ニンテンドーアカウントを紐付けなくてもプレイレポートが保存可能です。@n
 *  詳しくは、PlayReportKit のドキュメントを参照してください。
 *
 * @section PageSamplePrepoSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSamplePrepoSimple_SectionPrecaution 注意事項
 *  ネットワーク接続が可能な環境が用意できない場合、レポートデータはサーバーに送信されません。
 *
 * @section PageSamplePrepoSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSamplePrepoSimple_SectionDetail 解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - nn::prepo::PlayReport オブジェクトにイベント ID とワークバッファを指定する。
 *  - レポートするキーとバリューを書き込む。
 *  - 保存する。
 */

#include <nn/prepo.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/account.h>

// 乱数を取得します。
int64_t GetRandom() NN_NOEXCEPT
{
    nn::Bit64 value;
    nn::os::GenerateRandomBytes(&value, sizeof (value));

    return static_cast<int64_t>(value & 0x7FFFFFFFFFFFFFFull);
}

// レポートデータを保存する。
void Save(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    nn::prepo::PlayReport report("sample_event");

    // レポートデータの一時的に保存するバッファです。
    // 必要なバッファサイズはキーバリューの数によって変動します。（サンプルでは、3 つのデータを記録するので 3 を指定しています）
    static nn::Bit8 buffer[nn::prepo::PlayReport::BufferSizeMin + 3 * nn::prepo::KeyValueSizeMax];

    report.SetBuffer(buffer, sizeof (buffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("x", GetRandom() % 100));
    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("y", GetRandom() % 100));
    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("z", GetRandom() % 100));

    nn::account::Uid user;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&user, handle));

    // レポートデータを保存する際、エラーが発生してもエラービューア等でエラーをユーザーに伝えてはいけません。
    // プレイレポートはアプリケーションの進行に必須の機能ではなく、エラーを伝えるとユーザー体験の妨げになるからです。
    report.Save(user);
}

// ユーザーアカウントを指定せずにレポートデータを保存する。
void SaveWithoutUser() NN_NOEXCEPT
{
    nn::prepo::PlayReport report("sample_event_without_user");

    static nn::Bit8 buffer[nn::prepo::PlayReport::BufferSizeMin + 3 * nn::prepo::KeyValueSizeMax];

    report.SetBuffer(buffer, sizeof (buffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("x", GetRandom() % 100));
    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("y", GetRandom() % 100));
    NN_ABORT_UNLESS_RESULT_SUCCESS(report.Add("z", GetRandom() % 100));

    report.Save();
}

extern "C" void nnMain()
{
    nn::account::Initialize();

    nn::account::Uid users[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    if (count >= 1)
    {
        // サンプルでは、実装簡略化のため先頭のユーザーのプレイレポートとしてレポートデータを保存します。
        // 実際は、プレイヤー選択で選ばれたユーザーを指定してください。
        nn::account::UserHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, users[0]));

        Save(handle);

        nn::account::CloseUser(handle);
    }
    else
    {
        NN_LOG("[sample] Warning: Please create user account.\n");
    }

    SaveWithoutUser();

    NN_LOG("[sample] End.\n");
}
