﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @example OsUserExceptionHandlerSample.cpp
 *  ユーザ例外ハンドラ機能のサンプルプログラムにおけるハンドラの実装です。
 *  ハンドラ実装の詳細な説明は省略します。@n
 *  サンプルの説明は @ref PageSampleOsUserExceptionHandler を参照して下さい。
 *
 */

#include <array>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/diag.h>
#include <nn/os.h>

using namespace nn;

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    #if defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_32)
        #define NNS_OS_BUILD_NX32
    #elif defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_64)
        #define NNS_OS_BUILD_NX64
    #endif
#endif

#if defined(NNS_OS_BUILD_NX32) || defined(NNS_OS_BUILD_NX64)

namespace {

NN_ALIGNAS(32) char UserExceptionHandlerStack[0x4000];

const size_t StackGuardHintSize = os::MemoryPageSize * 4;

const std::pair<nn::os::UserExceptionType, const char*> g_exceptionTypeString[] = {
    std::make_pair( nn::os::UserExceptionType_InvalidInstructionAccess,    "Invalid Instruction Access"),
    std::make_pair( nn::os::UserExceptionType_InvalidDataAccess,           "Invalid Data Access"),
    std::make_pair( nn::os::UserExceptionType_UnalignedInstructionAccess,  "Unaligned Instruction Access"),
    std::make_pair( nn::os::UserExceptionType_UnalignedDataAccess,         "Unaligned Data Access"),
    std::make_pair( nn::os::UserExceptionType_UndefinedInstruction,        "Undefined Instruction"),
    std::make_pair( nn::os::UserExceptionType_ExceptionalInstruction,      "Exceptional Instruction"),
    std::make_pair( nn::os::UserExceptionType_MemorySystemError,           "Memory System Error"),
    std::make_pair( nn::os::UserExceptionType_FloatingPointException,      "Floating Point Exception"),
    std::make_pair( nn::os::UserExceptionType_InvalidSystemCall,           "Invalid System Call"),
};

#if defined(NNS_OS_BUILD_NX32)
const char* registerName[16]={
    "    ", "    ", "    ", "    ", "    ", "    ", "    ", "    ",
    "    ", "(sb)", "(sl)", "(fp)", "(ip)", "(sp)", "(lr)", "(pc)"
};
#endif

// pos:   取り出す最下位ビットの位置
// width: 取り出すビット幅（32 は指定不可）
inline Bit32 GetBits( Bit32 bits, int pos, int width )
{
    Bit32 mask = (1u << width) - 1;
    return (bits >> pos) & mask;
}

//-----------------------------------------------------------------------------
//  例外種別の表示
//
void PrintExceptionType(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    const char* exceptionTypeString = "Unknown ExceptionType";
    for (auto&& e : g_exceptionTypeString)
    {
        if (info->exceptionType == e.first)
        {
            exceptionTypeString = e.second;
            break;
        }
    }

    NN_LOG("\n");
    NN_LOG("UserException handler is called.\n");
    NN_LOG("==============================================================================\n");
    if (info->exceptionType == nn::os::UserExceptionType_InvalidSystemCall)
    {
        Bit32 esrISS = GetBits(info->detail.esr, 0, 25);
        NN_UNUSED(esrISS);
        NN_LOG("  Invalid System Call(No=%u) (ExceptionType=0x%04X)\n", esrISS, info->exceptionType);
        NN_UNUSED(esrISS);
    }
    else
    {
        NN_LOG("  %s  (ExceptionType=0x%04X)\n", exceptionTypeString, info->exceptionType);
        if (info->exceptionType == nn::os::UserExceptionType_InvalidDataAccess)
        {
            Bit32 esrIssDfsc = GetBits(info->detail.esr, 0, 6);
            NN_UNUSED(esrIssDfsc);

            const char* p;
            NN_UNUSED(p);
            switch (esrIssDfsc)
            {
            case 0x00 ... 0x07:
                    p = "Accessed to unmapped memory space.";
                    break;

            case 0x0d ... 0x0f:
                    p = "Memory permission violation";
                    break;

            case 0x21:
                    p = "Misaligned memory access";
                    break;

            default:
                    p = "Unexpected internal error";
                    break;
            }

            NN_LOG("  %s (DFSC=0x%02X)\n", p, esrIssDfsc);
            NN_LOG("  Accessed to: 0x%P\n", info->detail.far);
        }
    }
    NN_LOG("==============================================================================\n");
    NN_LOG("\n");

    NN_UNUSED(exceptionTypeString);
}

//-----------------------------------------------------------------------------
//  スレッド/ファイバー及びスタック情報の表示
//
void PrintThreadInformation(nn::os::UserExceptionInfo* info, uintptr_t* outStackTop, uintptr_t* outStackBottom) NN_NOEXCEPT
{
    uintptr_t stackTop;
    uintptr_t stackBottom;
    nn::os::FiberType* fiber = nn::os::GetCurrentFiber();
    if (fiber == NULL)
    {
        // スレッド実行中だった場合
        nn::os::ThreadType* thread = nn::os::GetCurrentThread();
        const char* threadName     = nn::os::GetThreadNamePointer(thread);
        stackTop    = reinterpret_cast<uintptr_t>(thread->_stack);
        stackBottom = reinterpret_cast<uintptr_t>(thread->_stack) + thread->_stackSize;
        NN_LOG("Thread         : 0x%P (%s)\n", thread, threadName ? threadName : "");
        NN_UNUSED(threadName);
    }
    else
    {
        // ファイバー実行中だった場合
        stackTop    = reinterpret_cast<uintptr_t>(fiber->_stack);
        stackBottom = reinterpret_cast<uintptr_t>(fiber->_stack) + fiber->_stackSize;
        NN_LOG("Fiber          : 0x%P\n", fiber);
    }

    // スタックオーバーフローの可能性が高い場合の処理
    if (info->exceptionType == nn::os::UserExceptionType_InvalidDataAccess)
    {
        auto accessedAddress = info->detail.far;
        if ((accessedAddress >= stackTop - StackGuardHintSize) &&
            (accessedAddress < stackTop))
        {
            NN_LOG("  Accessed to  : 0x%P (Detect stack overflow on guard pages)\n", info->detail.far);
        }
        else if ((accessedAddress >= stackTop - 1 * 1024 * 1024 /* 1MB */) &&
                 (accessedAddress < stackTop))
        {
            NN_LOG("  Accessed to  : 0x%P (Perhaps stack overflow ?)\n", info->detail.far);
        }
    }

    NN_LOG("  Stack Top    : 0x%P\n", stackTop);
    NN_LOG("  Stack Bottom : 0x%P (size=0x%zx)\n", stackBottom, stackBottom - stackTop);
    NN_LOG("\n");

    *outStackTop    = stackTop;
    *outStackBottom = stackBottom;
}

//-----------------------------------------------------------------------------
//  汎用レジスタ等の表示
//
void PrintGeneralPurposeRegisters(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
#if defined(NNS_OS_BUILD_NX32)
    for (int i=0; i<16; i++)
    {
        NN_LOG("r%-2d%s = 0x%08X (%11d)\n", i, registerName[i], info->detail.r[i], info->detail.r[i]);
        NN_UNUSED(registerName);
    }
#elif defined(NNS_OS_BUILD_NX64)
    for (int i=0; i<30; ++i)
    {
        NN_LOG("r%-2d     = 0x%016llX (%21lld)\n", i, info->detail.r[i], info->detail.r[i]);
    }
    NN_LOG("r30(lr) = 0x%016llX (%21lld)\n", info->detail.lr, info->detail.lr);
    NN_LOG("sp      = 0x%016llX (%21lld)\n", info->detail.sp, info->detail.sp);
    NN_LOG("pc      = 0x%016llX (%21lld)\n", info->detail.pc, info->detail.pc);
#endif
    NN_LOG("\n");
}

//-----------------------------------------------------------------------------
//  特別なレジスタの表示
//
void PrintSpecialRegisters(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    Bit32 psr = info->detail.pstate;
    const char* psrName = "pstate";
    Bit32 psrN  = GetBits(psr, 31, 1);
    Bit32 psrZ  = GetBits(psr, 30, 1);
    Bit32 psrC  = GetBits(psr, 29, 1);
    Bit32 psrV  = GetBits(psr, 28, 1);
    Bit32 psrQ  = GetBits(psr, 27, 1);
    Bit32 psrJ  = GetBits(psr, 24, 1);
    Bit32 psrE  = GetBits(psr,  9, 1);
    Bit32 psrT  = GetBits(psr,  5, 1);
    Bit32 psrIT = GetBits(psr, 25, 2) | (GetBits(psr, 10, 6) << 2);
    Bit32 psrGE = GetBits(psr, 16, 4);
    NN_LOG("%-7s = 0x%08X (N=%d Z=%d C=%d V=%d Q=%d J=%d E=%d T=%d IT=0x%02X GE=0x%X)\n",
            psrName, psr, psrN, psrZ, psrC, psrV, psrQ, psrJ, psrE, psrT, psrIT, psrGE);

    NN_UNUSED(psrN);
    NN_UNUSED(psrZ);
    NN_UNUSED(psrC);
    NN_UNUSED(psrV);
    NN_UNUSED(psrQ);
    NN_UNUSED(psrJ);
    NN_UNUSED(psrE);
    NN_UNUSED(psrT);
    NN_UNUSED(psrIT);
    NN_UNUSED(psrGE);
    NN_UNUSED(psrName);

    Bit32 esrEC  = GetBits(info->detail.esr, 26, 6);
    Bit32 esrIL  = GetBits(info->detail.esr, 25, 1);
    Bit32 esrISS = GetBits(info->detail.esr, 0, 25);
    NN_LOG("afsr0   = 0x%08X\n", info->detail.afsr0);
    NN_LOG("afsr1   = 0x%08X\n", info->detail.afsr1);
    NN_LOG("esr     = 0x%08X (EC=0x%02X IL=%d ISS=0x%06X)\n", info->detail.esr, esrEC, esrIL, esrISS);
    NN_LOG("far     = 0x%P\n",   info->detail.far);

    NN_UNUSED(esrEC);
    NN_UNUSED(esrIL);
    NN_UNUSED(esrISS);
    NN_LOG("\n");
}

//-----------------------------------------------------------------------------
//  浮動小数点レジスタの表示
//
void PrintFloatingPointRegisters(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
#if defined(NNS_OS_BUILD_NX32)
    for (int i=0; i<NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS; ++i)
    {
        NN_LOG("D%-2d = 0x%016llX (%E)\n", i, info->detail.D[i], info->detail.D[i]);
    }
#elif defined(NNS_OS_BUILD_NX64)
    for (int i=0; i<32; ++i)
    {
        NN_LOG("V%-2d = 0x%016llX%016llX (%LE)\n", i, static_cast<nn::Bit64>(info->detail.V[i] >> 64), static_cast<nn::Bit64>(info->detail.V[i]), info->detail.V[i]);
    }
#endif
    NN_LOG("\n");
}

//-----------------------------------------------------------------------------
//  モジュール一覧の表示
//
void PrintModuleList() NN_NOEXCEPT
{
    const auto bufferSize = nn::diag::GetRequiredBufferSizeForGetAllModuleInfo();
    auto buffer = reinterpret_cast<nn::Bit8*>(std::malloc(bufferSize));
    NN_UTIL_SCOPE_EXIT { std::free(buffer); };

    nn::diag::ModuleInfo* modules;
    const auto moduleCount = nn::diag::GetAllModuleInfo(&modules, buffer, bufferSize);

    NN_LOG("Modules:\n");
    NN_LOG("  %-*s   %-*s   path\n", sizeof(uintptr_t) * 2, "base", sizeof(uintptr_t) * 2, "size");
    for (auto i = 0; i < moduleCount; i++)
    {
        const auto& module = modules[i];
        NN_LOG("  0x%P 0x%P %s\n", module.baseAddress, module.size, module.path);
    }
    NN_LOG("\n");
}

//-----------------------------------------------------------------------------
//  スタックトレースの表示
//
template <typename T>
T GetAddressValue(uintptr_t address)
{
    if (!address)
    {
        return 0;
    }
    if (address & (sizeof(T) - 1))
    {
        return 0;
    }
    return *reinterpret_cast<T*>(address);
}

void PrintAddressAndSymbol(uintptr_t address)
{
    char symbolName[64];
    const auto ret = address ? nn::diag::GetSymbolName(symbolName, sizeof(symbolName), address) : 0;
    if (ret == 0)
    {
        NN_LOG("  0x%P (unknown)\n", address);
    }
    else
    {
        const auto symbolSize = nn::diag::GetSymbolSize(ret);

        // noreturn 関数の場合は、関数末尾の bl 命令の直後に、次の関数の先頭が来るため、
        // 次の関数の先頭アドレスまでは関数内とみなす。
        const auto isOutOfFunction = address > ret + symbolSize;

        NN_LOG("  0x%P %s + 0x%P%s\n", address, symbolName, address - ret, isOutOfFunction ? " (too far)" : "");
    }
}

void PrintStackTrace(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    const int addressCountMax = 32;
    auto addresses = reinterpret_cast<uintptr_t*>(std::malloc(sizeof(uintptr_t) * addressCountMax));
    NN_UTIL_SCOPE_EXIT { std::free(addresses); };

    const int addressCount = nn::diag::GetBacktrace(addresses, addressCountMax, info->detail.fp, info->detail.sp, info->detail.pc);

    NN_LOG("Stack trace:\n");
    for (int i = 0; i < addressCount; i++)
    {
        PrintAddressAndSymbol(addresses[i]);
    }
    if (addressCount >= addressCountMax)
    {
        NN_LOG("  : Maybe more than addresses are containing...\n");
    }
    NN_LOG("\n");
}

//-----------------------------------------------------------------------------
//  スタックダンプの表示
//
void PrintStackDump(nn::os::UserExceptionInfo* info, uintptr_t stackTop, uintptr_t stackBottom) NN_NOEXCEPT
{
    uintptr_t dumpTop    = std::max(static_cast<uintptr_t>(nn::util::align_down(info->detail.sp, 16)), stackTop);
    uintptr_t dumpBottom = std::min(dumpTop + 16 * 16, stackBottom);

    NN_LOG("Stack dump:\n");

    for (uintptr_t ptr=dumpTop; ptr < dumpBottom; ++ptr)
    {
        if (ptr % 16 == 0)
        {
            NN_LOG("0x%P: ", ptr);
        }

        if (ptr < info->detail.sp)
        {
            NN_LOG("   ");
        }
        else
        {
            NN_LOG("%02X ", *reinterpret_cast<uint8_t*>(ptr));
        }

        if (ptr % 4 == 3)
        {
            NN_LOG(" ");
        }

        if (ptr % 16 == 15)
        {
            for (int i=-15; i<=0; i++)
            {
                uint8_t c = *reinterpret_cast<uint8_t*>(ptr + i);
                NN_LOG("%c", (c >= 0x20 && c <= 0x7e)? c : '.');
                NN_UNUSED(c);
            }
            NN_LOG("\n");
        }
    }
    NN_LOG("\n");
}

//-----------------------------------------------------------------------------
//  ユーザー例外ハンドラの実装サンプル
//-----------------------------------------------------------------------------

void UserExceptionHandlerSample(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    uintptr_t stackTop;
    uintptr_t stackBottom;

    PrintExceptionType(info);
    PrintThreadInformation(info, &stackTop, &stackBottom);
    PrintGeneralPurposeRegisters(info);
    PrintSpecialRegisters(info);
    PrintFloatingPointRegisters(info);
    PrintStackDump(info, stackTop, stackBottom);
    PrintStackTrace(info);
    PrintModuleList();
}

}   // namespace anonymous

#endif // defined(NNS_OS_BUILD_NX32) || definde(NNS_OS_BUILD_NX64)

//-----------------------------------------------------------------------------
//  ユーザー例外ハンドラを登録する
//-----------------------------------------------------------------------------

void SetOriginalUserExceptionHandler() NN_NOEXCEPT
{
#if defined(NNS_OS_BUILD_NX32) || defined(NNS_OS_BUILD_NX64)
    nn::os::SetUserExceptionHandler( UserExceptionHandlerSample, UserExceptionHandlerStack, sizeof(UserExceptionHandlerStack), nn::os::UserExceptionInfoUsesHandlerStack );
#else
    NN_LOG("User exception handler is not supported on windows.\n");
#endif
}

