﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsTransferMemory.cpp,PageSampleOsTransferMemory}
 *
 * @brief
 *  Transfer memory 機能のサンプルプログラム
 */

/**
 * @page PageSampleOsTransferMemory Transfer memory
 * @tableofcontents
 *
 * @brief
 *  Transfer memory 機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsTransferMemory_SectionBrief 概要
 *  ここでは、Transfer memory 機能を使ったサンプルプログラムの説明をします。
 *
 *  Transfer memory 機能の使い方については、Transfer memory 機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsTransferMemory_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsTransferMemory Samples/Sources/Applications/OsTransferMemory @endlink 以下にあります。
 *
 * @section PageSampleOsTransferMemory_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsTransferMemory_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsTransferMemory_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsTransferMemory_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsTransferMemory_SectionDetail 解説
 *
 * @subsection PageSampleOsTransferMemory_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsTransferMemory.cpp
 *  @includelineno OsTransferMemory.cpp
 *
 * @subsection PageSampleOsTransferMemory_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成
 *  - 2 つのスレッドは、 ServerThreadFunc() と ClientThreadFunc()
 *  - ClientThreadFunc() で Transfer memory を作成し ServerThreadFunc() に渡します。
 *  - ServerThreadFunc() では、Transfer memory をマップし、使用します。
 *  - 2 つのスレッドが終了するまで nnMain() は待機
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsTransferMemory_OutputExample.txt
 *
 *  本サンプルでは、MessageQueue を使って NativeHandle をクライアントスレッドからサーバスレッドへ渡していますが、
 *  実際にはシステムで定められたプロセス間通信の仕組みを使ってクライアントプロセスからサーバプロセスへ
 *  NativeHandle を転送して利用することを想定しています。
 *
 */

#include <stdint.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemory.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // Transfer memory 操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のTransfer memory 操作スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のTransfer memory 操作スレッドのスタック

nn::os::MessageQueueType g_MessageQueue;  // サーバー・クライアント通信用のメッセージキュー
const size_t MessageQueueBufferCount = 1; // メッセージキューのバッファ数
uintptr_t g_MessageQueueBuffer[ MessageQueueBufferCount ];     // メッセージキュー用のバッファ

// Transfer memory にするメモリ領域
// アドレス・サイズ共にページサイズにアライメントされている必要がある
const size_t BufferSize = nn::os::MemoryPageSize * 500;
NN_ALIGNAS(4096) char  g_Buffer[ BufferSize ];


}   // namespace

//-----------------------------------------------------------------------------

//
//  サーバー想定のスレッド（Transfer memory を受け取り、利用する）
//
void ServerThreadFunc(void *arg)
{
    NN_UNUSED(arg);

    // サーバーが使用する TransferMemoryType オブジェクト
    nn::os::TransferMemoryType serverTransferMemory;

    uintptr_t data;
    void* address;

    // クライアントから Tansfer memory ハンドルが送られるくるのを待つ
    nn::os::ReceiveMessageQueue(&data, &g_MessageQueue);
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    auto handle = reinterpret_cast<nn::os::NativeHandle>(data);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto handle = static_cast<nn::os::NativeHandle>(data & 0xffffffff);
#endif
    NN_LOG("Server: Receieved handle.\n");

    // 受け取ったハンドルを TransferMemoryType オブジェクトにアタッチ
    // Transfer memory のサイズは、サーバー・クライアント間で予め共有されているはず
    nn::os::AttachTransferMemory(&serverTransferMemory, BufferSize, handle, true);
    NN_LOG("Server: Attached transfer memory handle.\n");

    // Transfer memory を仮想アドレス空間にマップ
    auto result = nn::os::MapTransferMemory(&address, &serverTransferMemory, nn::os::MemoryPermission_None);
    NN_ASSERT(result.IsSuccess(), "Cannot map trasfer memory.");
    NN_LOG("Server: Mapped transfer memory. address=0x%p\n", address);

    // マップした領域はサーバーから使用可能になる
    char* ptr = reinterpret_cast<char*>(address);
    for (size_t i=0; i<BufferSize; ++i)
    {
        ptr[i] = i % 64;
    }

    // 必要がなくなったら Transfer memory をアンマップし、破棄する
    nn::os::UnmapTransferMemory(&serverTransferMemory);
    NN_LOG("Server: Unmapped transfer memory.\n");
    nn::os::DestroyTransferMemory(&serverTransferMemory);
    NN_LOG("Server: Destroyed transfer memory.\n");
}


//
//  クライアント想定のスレッド（Transfer memory を作成し、サーバーに渡す）
//
void ClientThreadFunc(void *arg)
{
    NN_UNUSED(arg);

    // クライアントが使用する TransferMemoryType オブジェクト
    nn::os::TransferMemoryType clientTransferMemory;

    // Transfer memory を作成する
    // この時点でクライアントから g_Buffer へのアクセス権が変更される
    auto result = nn::os::CreateTransferMemory(&clientTransferMemory, g_Buffer, BufferSize, nn::os::MemoryPermission_None);
    NN_ASSERT(result.IsSuccess(), "Cannot create transfer memory.");
    NN_LOG("Client: Created transfer memory.\n");

    // TransferMemoryType オブジェクトからハンドルをデタッチ
    auto handle = nn::os::DetachTransferMemory(&clientTransferMemory);
    NN_LOG("Client: Detached transfer memory.\n");

    // ハンドルをメッセージキューに送る
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    nn::os::SendMessageQueue(&g_MessageQueue, reinterpret_cast<uintptr_t>(handle));
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::os::SendMessageQueue(&g_MessageQueue, handle);
#endif
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::Result      result;

    // メッセージキューを初期化する
    nn::os::InitializeMessageQueue( &g_MessageQueue, g_MessageQueueBuffer, MessageQueueBufferCount);

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ServerThreadFunc, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create server thread." );

    result = nn::os::CreateThread( &thread2, ClientThreadFunc, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create client thread." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // メッセージキューをファイナライズする
    nn::os::FinalizeMessageQueue(&g_MessageQueue);
}

