﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeBatteryDrainingPerformance.cpp,PageSampleOeBatteryDrainingPerformance}
 *
 * @brief
 *  ノーマルモードに電力消費量の大きい性能を設定するサンプルプログラム
 */

/**
 * @page PageSampleOeBatteryDrainingPerformance 電力消費量の大きい性能の選択
 * @tableofcontents
 *
 * @brief
 *  アプリケーション動作環境操作ライブラリを用いた性能選択操作の解説です。
 *
 * @section PageSampleOeBatteryDrainingPerformance_SectionBrief 概要
 *  アプリケーション動作環境操作ライブラリを用いて性能を選択する方法を解説します。
 *
 *  アプリケーション動作環境操作ライブラリの使い方については、
 *  @ref nn::oe "アプリケーション動作環境操作ライブラリの関数リファレンス" も併せて参照して下さい。
 *
 *  本サンプルではノーマルモードに電力消費量の大きい性能を設定します。@n
 *  本サンプルで扱う性能は基本的に開発用途でのみ使用可能であり、製品ではノーマルモード、ブーストモードのどちらでも使用できません。
 *
 * @section PageSampleOeBatteryDrainingPerformance_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OeBatteryDrainingPerformance
 *  Samples/Sources/Applications/OeBatteryDrainingPerformance @endlink 以下にあります。
 *
 * @section PageSampleOeBatteryDrainingPerformance_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeBatteryDrainingPerformance_SectionHowToOperate 操作方法
 *  特に無し。
 *
 * @section PageSampleOeBatteryDrainingPerformance_SectionPrecaution 注意事項
 *  本サンプルプログラムは画面上に何も表示しません。実行結果はログに出力されます。
 *
 * @section PageSampleOeBatteryDrainingPerformance_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOeBatteryDrainingPerformance_SectionDetail 解説
 *
 * @subsection PageSampleOeBatteryDrainingPerformance_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeBatteryDrainingPerformance.cpp
 *  @includelineno OeBatteryDrainingPerformance.cpp
 *
 * @subsection PageSampleOeBatteryDrainingPerformance_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムはノーマルモードで使用する性能を更新します。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - ノーマルモード時の GPU のクロックレートを、デフォルトの性能よりも高い値にします。
 *      - 対象の性能モードを @ref nn::oe::PerformanceMode 列挙型で引数に指定してください。
 *      - 使用したい性能を @ref nn::oe::PerformanceConfiguration 型で引数に指定してください。
 *      - @ref nn::oe::PerformanceConfiguration 型の定義は今後追加される可能性があります。
 *      - @ref nn::oe::SetPerformanceConfiguration 関数は通常成功します。
 *  - プログラム終了時に性能設定は全て破棄されデフォルトの性能設定に戻ります。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OeBatteryDrainingPerformance_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/oe.h>
#include <nn/oe/oe_BatteryDrainingPerformance.h>

//-----------------------------------------------------------------------------

namespace {

struct ConfigurationInfo
{
    nn::oe::PerformanceConfiguration configuration;
    char name[128];
};

const ConfigurationInfo ConfigurationInfoList[] =
{
    { nn::oe::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz, "<Cpu1020Mhz Gpu307Mhz Emc1331Mhz>" },
    { nn::oe::PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1331Mhz, "<Cpu1020Mhz Gpu384Mhz Emc1331Mhz>" },
    { nn::oe::PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1331Mhz, "<Cpu1020Mhz Gpu460Mhz Emc1331Mhz>" },
    { nn::oe::PerformanceConfiguration_Cpu1020MhzGpu768MhzEmc1600Mhz, "<Cpu1020Mhz Gpu768Mhz Emc1600Mhz>" },
};

const char* GetConfigurationName(
    nn::oe::PerformanceConfiguration performanceConfiguration)
{
    for ( auto& configurationInfo : ConfigurationInfoList )
    {
        if ( configurationInfo.configuration == performanceConfiguration )
        {
            return configurationInfo.name;
        }
    }

    return nullptr;
}

void SetPerformanceConfigurationLocal(
    nn::oe::PerformanceMode performanceMode,
    nn::oe::PerformanceConfiguration performanceConfiguration)
{
    const char* pConfigurationName = GetConfigurationName(performanceConfiguration);

    nn::oe::SetPerformanceConfiguration(performanceMode, performanceConfiguration);

    NN_LOG("Set configuration %s to %s.\n",
        pConfigurationName != nullptr ? pConfigurationName : "<invalid performance>",
        performanceMode == nn::oe::PerformanceMode_Normal ? "<Normal Mode>" : "<Boost Mode>");
}

} // namespace

//
//  メイン関数
//
extern "C" void nnMain()
{
    // ノーマルモード時の GPU のクロックレートを、デフォルトの性能より高い値にします。
    SetPerformanceConfigurationLocal(nn::oe::PerformanceMode_Normal, nn::oe::PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1331Mhz);

    // 設定した性能はアプリケーション終了時に自動的に破棄されデフォルト値に戻ります。
}
