﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{Texpkg_Helper.cpp,PageSampleNvnTutorial02}
 *
 * @brief
 *  This file defines a function that loads the needed
 *  texpkg libraries and sets up some global helper objects
 *  for texture file I/O and format conversion.
 */


#include "Texpkg_Helper.h"
#include <iostream>

nvnTool::texpkg::RawImageLib g_NvnImageLib;
nvnTool::texpkg::RawImageFormatManager* g_ImageLoader;
nvnTool::texpkg::RawImageHardwareTextureExporter* g_HwTextureExporter;

bool TexpkgHelperLoadLibraries(const wchar_t* imageDLLFile, const wchar_t* texpkgDLLFile)
{
        /*
         * Loads the NVN image library. This library is responsible
         * for operations on pitch-linear images like generating
         * mipmaps, stitching arrays, performing format
         * conversions, etc.
         */
    if(!LoadNVNImageLibrary(&g_NvnImageLib, imageDLLFile))
    {
        std::cout << "NVN Image Library failed to load\n";
        return false;
    }

        /*
         * Loads the texture packager core library. This library is
         * responsible for converting textures to the hardware texture
         * format and also provides helper classes for texture file I/O.
         */
    nvnTool::texpkg::TexpkgLib tpkgLib;
    if(!LoadTexpkgLibrary(&tpkgLib, texpkgDLLFile))
    {
        std::cout << "NVN Texpkg Library failed to load\n";
        return false;
    }

        /*
         * Creates an image loader helper class to load in the texture
         * files. Supports both TGA and DDS files. More specifics on
         * exact formats supported can be found in the NVN texture
         * packager documenation.
         */
    g_ImageLoader = tpkgLib.createRawImageFormatManager(&g_NvnImageLib,
                                                        TEXPKG_NUM_DEFAULT_FORMAT_LOADERS,
                                                        nvnTool::texpkg::RawImageFormatDefaultHandlerLibraries);

    if(!g_ImageLoader)
    {
        std::cout << "Failed to create image loader\n";
        return false;
    }

        /*
         * Creates the hardware texture exporter object that is used to
         * convert the given texture data to an optimized format to be
         * used directly by the hardware.
         */
    g_HwTextureExporter = tpkgLib.createRawImageHardwareTextureExporter(&g_NvnImageLib);

    if(!g_HwTextureExporter)
    {
        std::cout << "Failed to create texture exporter\n";
        return false;
    }

    return true;
}
